////////////////////////////////////////////////////////////
// -- Dll provides sndRecordSound() function.             //
// -- Written by Paul R. Bonneau 03/04/95.                //
////////////////////////////////////////////////////////////
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include "record.h"

#define cbWavBuf 0x10000 // Wave buffer size.

typedef struct
    {
    DWORD lwTag; // Tag.
    DWORD cb;    // Size of data.
    } CNK;       // ChuNK.

typedef struct
    {
    CNK           cnkRiff;     // 'RIFF' chunk.
    DWORD         lwWaveId;    // Must be 'WAVE'.
    CNK           cnkFormat;   // Wave specific fmt. chunk.
    PCMWAVEFORMAT wft;         // PCM stuff.
    CNK           cnkData;     // PCM Data chunk.
    BYTE          rgb[];       // The sound data.
    } AFH, FAR *LPAFH;         // Audio File Header.

////////////////////////////////////////////////////////////
// Globals.                                               //
////////////////////////////////////////////////////////////
AFH           ahf;        // File header.
HFILE         hfil;       // File handle.
PCMWAVEFORMAT wft;        // Audio format.
HWAVEIN       hwvi;       // Input device handle.
LPWAVEHDR     rglpwvh[2]; // Audio buffers.
HWND          hwnd;       // Notification window.
char          szClass[] = "RecordToFile";
HINSTANCE     hinsThis;   // DLL's instance handle.

////////////////////////////////////////////////////////////
// Prototypes.                                            //
////////////////////////////////////////////////////////////
// Private.
void Cleanup(void);
void CloseFil(HFILE FAR *lphfil, LPAFH lpahf);
UINT ErrCreateFile(HFILE FAR *lphfil, LPSTR lpsz,
     LPAFH lpahf, LPPCMWAVEFORMAT lpwft);
UINT ErrFlushWvh(LPWAVEHDR lpwvh, HFILE hfil);
UINT ErrInitWft(LPPCMWAVEFORMAT lpwft);
UINT ErrPrepareWvhHwvi(LPWAVEHDR FAR *lplpwvh,
     HWAVEIN hwvi);
UINT ErrStartRecording(LPSTR lpszFile,
     LPPCMWAVEFORMAT lpwft);
UINT ErrStopRecording(void);
void ReleaseWvh(LPWAVEHDR FAR *lplpwvh, HWAVEIN hwvi);

// Exports.
LRESULT CALLBACK __export LwWndProc(HWND hwnd, UINT wm,
  WPARAM wParam, LPARAM lParam);

#ifdef __BORLANDC__
    #pragma argsused
#endif
int CALLBACK
LibMain(HINSTANCE hins, WORD ds, WORD cbHeap, LPSTR lsz)
    {
    WNDCLASS wcs;

    hinsThis = hins;
    hfil = HFILE_ERROR;

    wcs.style = 0;               // Register the class for
    wcs.lpfnWndProc = LwWndProc; // the window used to
    wcs.cbClsExtra = 0;          // receive messages from
    wcs.cbWndExtra = 0;          // mmsystem.
    wcs.hInstance = hins;
    wcs.hIcon = NULL;
    wcs.hCursor = NULL;
    wcs.hbrBackground = NULL;
    wcs.lpszMenuName = NULL;
    wcs.lpszClassName = szClass;
    return RegisterClass(&wcs) ? 1 : 0;
    }

#ifdef __BORLANDC__
    #pragma argsused
#endif
int CALLBACK WEP(int wCode)
    {
    Cleanup();
    return 1;
    }

UINT WINAPI __export
sndRecordSound(LPSTR lpszFile, LPPCMWAVEFORMAT lpwft)
////////////////////////////////////////////////////////////
// -- Record a sound.                                     //
// -- pszFile : Sound to record file to.  If this is NULL //
//              stops recording.                          //
// -- lpwft   : Format of requested sound.  If null the   //
//              default is used (lowest bandwidth format  //
//              supported by first input device found).   //
////////////////////////////////////////////////////////////
    {
    return NULL == lpszFile ? ErrStopRecording() :
      ErrStartRecording(lpszFile, lpwft);
    }

UINT
ErrStartRecording(LPSTR lpszFile, LPPCMWAVEFORMAT lpwft)
////////////////////////////////////////////////////////////
// -- Start recording a sound.                            //
// -- lpszFile : Output file (see ahfRecordSound).        //
// -- lpwft    : Sound format (see ahfRecordSound).       //
////////////////////////////////////////////////////////////
    {
    PCMWAVEFORMAT wft; // Audio format.
    UINT          err; // Error code.

    if (NULL != hwvi)
        return errBusy; // Already recoding something.

    if (NULL == lpwft)
        {
        lpwft = &wft;   // Use the default format.
        if (errNil != (err = ErrInitWft(lpwft)))
            return err;
        }

    // Window will receive data-ready messages.
    if (NULL == (hwnd = CreateWindow(szClass, NULL,
      WS_OVERLAPPED, 0, 0, 0, 0, NULL, NULL, hinsThis,
      NULL)))
        return errOutOfMemory;

    // Open the input device.
    if (errNil != (err = waveInOpen(&hwvi,
      (UINT)WAVE_MAPPER, &lpwft->wf, (DWORD)(UINT)hwnd, 0,
      CALLBACK_WINDOW)))
        goto LCleanup;

    // Allocate and "prepare" the audio buffers.
    if (errNil != (err =
      ErrPrepareWvhHwvi(&rglpwvh[0], hwvi)) ||
        errNil != (err =
      ErrPrepareWvhHwvi(&rglpwvh[1], hwvi)))
        goto LCleanup;

    // Create the output file, and reserve space for the
    // header.
    if (errNil != (err = ErrCreateFile(&hfil, lpszFile,
      &ahf, lpwft)))
        goto LCleanup;

    // Queue up the audio buffers.
    if (errNil != (err = waveInAddBuffer(hwvi, rglpwvh[0],
          sizeof *rglpwvh[0])) ||
      errNil != (err = waveInAddBuffer(hwvi, rglpwvh[1],
          sizeof *rglpwvh[1])))
        goto LCleanup;

    ahf.cnkData.cb = 0;
    err = waveInStart(hwvi); // Here we go!

LCleanup:
    if (errNil != err)
        Cleanup();
    return err;
    }

UINT ErrStopRecording(void)
////////////////////////////////////////////////////////////
// -- Finish recording a sound.                           //
////////////////////////////////////////////////////////////
    {
    UINT err; // Error code.

    if (NULL == hwvi)
        return errNotRecording;

    // Release the audio buffers.
    if (errNil != (err = waveInReset(hwvi)))
        goto LCleanup;

    // Write the remainder of the samples to file.
    if (errNil != (err = ErrFlushWvh(rglpwvh[0], hfil)) ||
      errNil != (err = ErrFlushWvh(rglpwvh[1], hfil)))
        goto LCleanup;

    ReleaseWvh(&rglpwvh[0], hwvi); // Free the buffers.
    ReleaseWvh(&rglpwvh[1], hwvi);
    CloseFil(&hfil, &ahf);       // Update header and close.
    waveInClose(hwvi);           // Close input device.
    hwvi = NULL;
    DestroyWindow(hwnd);         // Nuke notification wnd.
    hwnd = NULL;

LCleanup:
    if (errNil != err)
        Cleanup();
    return err;
    }

UINT ErrCreateFile(HFILE FAR *lphfil, LPSTR lpsz,
  LPAFH lpahf, LPPCMWAVEFORMAT lpwft)
////////////////////////////////////////////////////////////
// -- Create (or overwrite) the file to receive the audio //
//    recording.                                          //
// -- Reserve space for the file header.                  //
// -- lphfil : Return file handle here.                   //
// -- lpsz   : Name of audio output file.                 //
// -- lpahf  : File header (partially filled in by this   //
//             function.                                  //
// -- lpwft  : Format of audio data.                      //
////////////////////////////////////////////////////////////
    {
    if (HFILE_ERROR == (*lphfil = _lcreat(lpsz, 0)))
       return errCreateFile;

    lpahf->cnkRiff.lwTag = *(long *)"RIFF";   // Initialize
    lpahf->cnkRiff.cb =                       // file header
      sizeof *lpahf - sizeof lpahf->cnkRiff;  // with
    lpahf->lwWaveId = *(long *)"WAVE";        // constant
    lpahf->cnkFormat.lwTag = *(long *)"fmt "; // stuff.
    lpahf->cnkFormat.cb = sizeof lpahf->wft;
    lpahf->wft = *lpwft;
    lpahf->cnkData.lwTag = *(long *)"data";

    return sizeof *lpahf == _hwrite(hfil, lpahf,
      sizeof *lpahf) ? errNil : errWriteFile;
    }

UINT ErrInitWft(LPPCMWAVEFORMAT lpwft)
////////////////////////////////////////////////////////////
// -- Enumerate the audio input devices, query their      //
//    capabilities, and return the capabilities of the    //
//    first one found.                                    //
// -- lpwft : Audio format returned here.                 //
////////////////////////////////////////////////////////////
    {
    WAVEINCAPS wic;      // Wave input capabilities struct.
    int        sh;       // Bit shifter.
    UINT       cbSample; // Size of one sample of audio.
    UINT       err;      // Error code.

    if (0 == waveInGetNumDevs())
        return errNoDevice;

    if (0 != (err = waveInGetDevCaps(0, &wic, sizeof wic)))
        return err;

    for (sh = 0; 32 > sh; sh++) // Find the first set bit.
        if (wic.dwFormats & (1 << sh))
            break;

    // Extract the sound format from the position of the
    // bit.  See the WAVE_FORMAT list in mmsystem.h.
    lpwft->wf.wFormatTag = WAVE_FORMAT_PCM;
    lpwft->wf.nChannels = (sh & 1) ? 2 : 1;
    lpwft->wf.nSamplesPerSec = (sh & 0xc) == 0 ? 11025 :
      ((sh & 0xc) == 4 ? 22050 : 44100);
    cbSample = (sh & 2) ? 2 : 1;
    lpwft->wf.nAvgBytesPerSec =
      lpwft->wf.nSamplesPerSec * cbSample;
    lpwft->wf.nBlockAlign = lpwft->wf.nChannels * cbSample;
    lpwft->wBitsPerSample = cbSample << 3;

    return errNil;
    }

UINT ErrPrepareWvhHwvi(LPWAVEHDR FAR *lplpwvh, HWAVEIN hwvi)
////////////////////////////////////////////////////////////
// -- Allocate and "prepare" a sound buffer.              //
// -- Allocate space for the header and buffer, and call  //
//    the driver to "prepare" the combination.            //
// -- lplpwvh : Return "prepared" sound buffer here.      //
// -- hwvi    : Audio input device handle.                //
////////////////////////////////////////////////////////////
    {
    // Get space for header.
    if (NULL == (*lplpwvh = GlobalAllocPtr(GMEM_MOVEABLE |
      GMEM_ZEROINIT | GMEM_SHARE, sizeof **lplpwvh)))
        return errOutOfMemory;

    // Get space for audio buffer.
    if (NULL == ((*lplpwvh)->lpData = GlobalAllocPtr(
      GMEM_MOVEABLE | GMEM_SHARE, cbWavBuf)))
        return errOutOfMemory;

    (*lplpwvh)->dwBufferLength = cbWavBuf;     // "Prepare"
    return waveInPrepareHeader(hwvi, *lplpwvh, // header.
      sizeof **lplpwvh);
    }

void ReleaseWvh(LPWAVEHDR FAR *lplpwvh, HWAVEIN hwvi)
////////////////////////////////////////////////////////////
// -- Free the memory associated with the wave header.    //
// -- Calls the driver to "unprepare" the header.         //
// -- lplpwvh : Pointer (to pointer) to wave header.      //
////////////////////////////////////////////////////////////
    {
    if (NULL == *lplpwvh)
        return; // Nothing to do.

    if (NULL != (*lplpwvh)->lpData)
        { // "Unprepare" the header.
        if ((*lplpwvh)->dwFlags & WHDR_PREPARED)
            waveInUnprepareHeader(hwvi, *lplpwvh,
              sizeof **lplpwvh);

        GlobalFreePtr((*lplpwvh)->lpData); // Free buffer.
        }

    GlobalFreePtr(*lplpwvh);               // Free header.
    *lplpwvh = NULL;
    }

UINT ErrFlushWvh(LPWAVEHDR lpwvh, HFILE hfil)
////////////////////////////////////////////////////////////
// -- Flush the data in the audio buffer to the file.     //
// -- Return the amount recorded (bytes).                 //
// -- lpwvh : Audio buffer to flush.                      //
// -- hfil  : Output file.                                //
////////////////////////////////////////////////////////////
    {
    long cbWritten, cbRecorded; // Byte counts.
    UINT err        = errNil;   // Error code.

    if (0 == (cbRecorded = (long)lpwvh->dwBytesRecorded))
        return errNil;

    if (cbRecorded != (cbWritten =
      _hwrite(hfil, lpwvh->lpData, cbRecorded)))
        {
        if (0 < cbWritten)
            cbWritten = 0;
        err = errWriteFile;
        }

    ahf.cnkData.cb += cbWritten;
    lpwvh->dwBytesRecorded = 0;
    lpwvh->dwFlags &= ~WHDR_DONE;
    return err;
    }

void CloseFil(HFILE FAR *lphfil, LPAFH lpahf)
////////////////////////////////////////////////////////////
// -- Write the amount of data recorded back to the       //
//    header and close the file.                          //
// -- lphfil : Audio file handle.                         //
// -- lpahf  : Audio file header.                         //
////////////////////////////////////////////////////////////
    {
    lpahf->cnkRiff.cb += lpahf->cnkData.cb;
    _llseek(*lphfil, 0, 0);
	_hwrite(*lphfil, lpahf, sizeof *lpahf);
    _lclose(*lphfil);
    *lphfil = HFILE_ERROR;
    }

void Cleanup(void)
////////////////////////////////////////////////////////////
// -- Release all the resources for recording a sound.    //
////////////////////////////////////////////////////////////
    {
    if (NULL != hwvi)
        {
        waveInReset(hwvi);
        ReleaseWvh(&rglpwvh[0], hwvi);
        ReleaseWvh(&rglpwvh[1], hwvi);
        waveInClose(hwvi);
        hwvi = NULL;
        }

    if (IsWindow(hwnd))
        {
        DestroyWindow(hwnd);
        hwnd = NULL;
        }

    if (HFILE_ERROR != hfil)
        {
        _lclose(hfil);
        hfil = HFILE_ERROR;
        }
    }

LRESULT CALLBACK __export LwWndProc(HWND hwnd, UINT wm,
  WPARAM wParam, LPARAM lParam)
////////////////////////////////////////////////////////////
// -- Window procedure to receive data-ready              //
//    notifications.                                      //
////////////////////////////////////////////////////////////
    {
    if (MM_WIM_DATA == wm && NULL != hwvi)
        {
        // Write the buffer to disk, and place it back in
        // the queue.
        if (errNil == ErrFlushWvh((LPWAVEHDR)lParam, hfil))
            waveInAddBuffer(hwvi, (LPWAVEHDR)lParam,
              sizeof(WAVEHDR));
        else
            ErrStopRecording();
        }

    return DefWindowProc(hwnd, wm, wParam, lParam);
    }

BOOL WINAPI __export
FGetErrSz(UINT err, LPSTR lpsz, UINT cb)
////////////////////////////////////////////////////////////
// -- Return the text associated with an error value.     //
// -- err  : Error code.                                  //
// -- lpsz : Return error text here.                      //
// -- cb   : Size of buffer.                              //
////////////////////////////////////////////////////////////
    {
    return errBase > err ?
      (0 == waveInGetErrorText(err, lpsz, cb)) :
      (0 != LoadString(hinsThis, err, lpsz, cb));
    }
