/*
COPA 1.0 - gui control panel kit for shell, C, perl etc.  
*              **      **                              *
Copyright 1991-1996 by Stephen C. Grubb.

Permission is hereby granted to USE this software for free.
Permission is hereby granted to MODIFY and/or REDISTRIBUTE 
this software for free, provided that no monetary gain is 
realized, and all attached authorship and copyright notices 
are preserved.

Inclusion of any portion (or derivation) of this software, 
including ancillary files, in any commercial product is not 
allowed without prior written permission of the author.  

See also the file 'Copyright'. 
*/



/* API - all copa functions are here.  All elib calls are from here. */


#include "elib.h"
#include <ctype.h> /* for isspace() */

#define MAXSTACK 5	/* LIMIT */
#define NONE ".!none" 

static int window_exists = 0;
static char *nowin_msg = "No window: copa_start never run.";
static char tagbuf[40]; 
static char tinybuf[8];
static int  evkey = -1;  /* most recent key event.. */
static double evx = 0.0, evy = 0.0;  /* mouse location at most recent key/mouse event */
static int  evp = -1;	/* proc id of most recent mouse click which produced a selection */
static char errstring[80];
static int  curde = -1; /* the "current" data entry area.. */
static int  nlimit = 0;
static int  limlist[12];
static int  window_visible = 1;
static char panstack[MAXSTACK][21];    /* LIMIT (tag length)*/
static int npanstack = 0;
static int popmsg_exists = 0; /* 0 when a popmsg is not on the screen, 1 when is */
static char echotag[21] = "";  /* current mechanism being echoed to.. */
static char *bigbufp; 
static int holdrefresh = 0; /* if 1 dont do any 'genall's */

/* ========== COPA_WINDOW ============== */
/* Create display window.
 * If window already exists, do a resize (ignore new title). 
 */
/* copa_window( int win_x, int win_y, double win_w, double win_h, char *title, double bkclr ) */
copa_window( win_x, win_y, win_w, win_h, title, bkclr )
int win_x; 
int win_y; 
double win_w; 
double win_h; 
char *title; 
double bkclr;
{

if( win_w > 30.0 || win_h > 30.0 ) 
  return( cerr( 2003, "Window too large. Specify width/height in inches." ) );


/* if window already exists, update window size, pos, color, etc.. */
/* -1 can be sent for any of these parameters. -1 causes the current value
	to remain in effect. */
if( window_exists ) {
	copa_resize( win_x, win_y, win_w, win_h, bkclr );
	return( 0 );
	}

/* this goes here because -1 is allowed for win size when window already exists.. */
if( win_w < 1.0 || win_h < 1.0 ) 
  return( cerr( 2002, "Window too small. Specify width/height in inches.", "" ) );

Einitall( title, 'x', win_w, win_h, win_x, win_y ); 

if( bkclr < 0.0 || bkclr > 1.0 ) bkclr = 1.0;
Eseteraseshade( bkclr );
Ebkcolor0 = bkclr; /* remember original color.. */

/* we need to do the align and gen here because copa_load may, in theory, not ever be called */
/* this may change though.. */
/* align all preset mechanisms.. */
Ealignall();
/* display anything that's visible (probably just window background) */
/* Eclr(); */
Egenall();

window_exists = 1;
return( 0 );
}
/* =========== COPA_RESIZE ===================== */
/* Resize or move window.. */
/* copa_resize( int win_x, int win_y, double win_w, double win_h, double bkclr ) */
copa_resize( win_x, win_y, win_w, win_h, bkclr )
int win_x;
int win_y;
double win_w; 
double win_h; 
double bkclr;
{
int stat;
Eresize( win_w, win_h, win_x, win_y );
Eflush();
Eusleep( E_WAITFOR_WM ); /* unknown problem - need to wait for window manager.. */
if( bkclr >= 0 )Eseteraseshade( bkclr );
Ealignall();
if( !holdrefresh) Egenall();
return( 0 );
}

/* =========== COPA_DISPLAY ======================== */
/* Execute various actions on the display window. */
/* copa_display( char *action ) */
copa_display( action )
char *action;
{
if( strcmp( action, "disappear" )==0 ) 
	{ Ewindisappear(); window_visible = 0; return( 0 ); }

else if( strcmp( action, "appear" )==0 ) {
	Ewinappear(); 
	window_visible = 1; 
	Eflush(); 
	Eusleep( E_WAITFOR_WM );  /* unknown problem - need to wait for window manager.. */
	/* Eclr(); */ 
	if( !holdrefresh ) Egenall(); 
	return( 0 ); 
	}

else if( strcmp( action, "refresh" )==0 ) 
	{ /* Eclr(); */ Egenall(); holdrefresh = 0; return( 0 ); }

else if( strncmp( action, "regen", 5 )==0 ) 
	{ Ealignall(); Eclr(); holdrefresh = 0; Egenall(); return( 0 ); }

else if( strcmp( action, "holdrefresh" )==0 ) {
	holdrefresh = 1;
	return( 0 );
	}
else if( strcmp( action, "dump" )==0 ) {
	EIwriteall( "copadump.pan" , 3 ); /* save everything except presets */
	return( 0 );
	}
}

/* =========== COPA_POP =============== */
/* copa_pop( char *msg, char *choices, char *response ) */
copa_pop( msg, choices, response )
char *msg;
char *choices;
char *response;
{
int status;
if( strlen( choices ) < 1 ) return( cerr( 2026, "No buttons specified","" ) );
status = copa_fpopup( "b", msg, choices, response, "" );
return( status );
}
/* =========== COPA_POPLIST =============== */
/* copa_poplist( char *msg, char *choices, char *buttons, char *response ) */
copa_poplist( msg, choices, buttons, response )
char *msg;
char *choices; 
char *buttons;
char *response;
{
int status;
if( strlen( choices ) < 1 ) return( cerr( 2027, "No selection list specified","" ) );
status = copa_fpopup( "l", msg, choices, response, buttons );
return( status );
}
/* =========== COPA_POPENTRY =============== */
/* copa_popentry( char *msg, char *def, int len, char *response ) */
copa_popentry( msg, def, len, response )
char *msg;
char *def;
int len;
char *response;
{
int status;
char parm[10];
sprintf( parm, "%d", len );

status = copa_fpopup( "d", msg, def, response, parm );
return( status );
}
/* =========== COPA_POPMSG =============== */
/* copa_popmsg( char *msg, char *buttons ) */
copa_popmsg( msg, buttons )
char *msg;
char *buttons;
{
int status;
status = copa_fpopup( "m", msg, buttons );
popmsg_exists = 1;
return( status );
}
/* ============ COPA_POPCLEAR ============ */
copa_popclear()
{
int status;
status = copa_fpopup( "c" );
popmsg_exists = 0;
return( status );
}

/* ============= COPA_FPOPUP (internal) ========== */
/* full popup (popup with various options).. */
/* (central point where popup prep/cleanup can be done in one place) */
static int
/* copa_fpopup( char *mectype, char *msg, char *choices, char *response, char *parm ) */
copa_fpopup( mectype, msg, choices, response, parm )
char *mectype;
char *msg;
char *choices;
char *response;
char *parm;
{
int status;
int pop;
if( !window_exists ) return( cerr( 2003, nowin_msg, "" ) );
pop = 0;
if( !window_visible ) {
	pop = 1;
	Ewinappear();
	}
if( mectype[0] == 'b' ) status = copa_btnpop( msg, choices, response );
else if( mectype[0] == 'd' ) status = copa_depop( msg, choices, parm, response );
else if( mectype[0] == 'l' ) status = copa_lbpop( msg, choices, parm, response );
else if( mectype[0] == 'm' ) { status = copa_mopop( msg, choices ); return( 0 ); }
else if( mectype[0] == 'c' ) status = copa_clpop();

if( status == 2100 ) return( cerr( 2100, "presets.pan file not loaded.", "" ) );
if( pop ) { Ewindisappear(); return( 0 ); }
if( mectype[0] != 'c' ) {
	if( all_whitespace( response ) ) strcpy( response, NONE );
	}
/* Eclr(); */
if( !holdrefresh ) Egenall(); /* restore current contents */
Eflush();
return( 0 );
}
/* =========== COPA_PANLOAD ============ */
/* copa_panload( char *fnm, char *tag ) */
copa_panload( fnm, tag )
char *fnm;
char *tag;
{
int status;
status = copa_panel( "load", fnm, tag );
return( status );
}

/* =========== COPA_PANUSE ============ */
copa_panuse( tag )
char *tag;
{
int status;
status = copa_panel( "use", "", tag );
return( status );
}

/* =========== COPA_PANADD ============= */
copa_panadd( fnm )
char *fnm;
{
int stat;
stat = copa_load( fnm ); /* load */
Ealignall(); 
if( !holdrefresh )Egenall();
return( stat );
}


/* =========== COPA_PANQUIT ============ */
copa_panquit( )
{
int status;
status = copa_panel( "return", "", "" );
return( status );
}

/* =========== COPA_PANEL (internal) =============== */
/* Control panel management (called by one of the above copa_pan* routines) 
   act can be: 
	load = load a new copy from the pan file
	use  = use existing panel (in its current state) if found, otherwise load a new one
	return = save current panel, reload previous panel, if any.
 */
static int
/* copa_panel( char *act, char *fnm, char *tag ) */
copa_panel( act, fnm, tag )
char *act;
char *fnm;
char *tag;
{
char buf[120];
int stat;
if( act[0] == 'r' ) { /* return */
	if( npanstack < 1 ) return( cerr( 2015, "warning, mismatched panquit", "" ) ); 

	/* save current panel in its current state.. */
	sprintf( buf, "%s/copa%05d%s", E_TMPDIR, getpid(), panstack[ npanstack-1 ] );
	EIwriteall( buf, 2 ); /* save everything, including presets */

	if( npanstack == 1 ) {  /* restore to nothing (except presets) */
		npanstack--;
		EIclear(); 
		Eload_presets(); /* since there was nothing else there.. */
		}
	else if( npanstack > 1 ) {  /* restore to previous panel (and presets) */
		npanstack--;
		EIclear(); 
		sprintf( buf, "%s/copa%05d%s", E_TMPDIR, getpid(), panstack[ npanstack-1 ] );
		/* read tmp file.. */
		stat = EIreaduserlist( buf );
		if( stat != 1 ) return( cerr( 2018, "Cannot restore previous panel", buf ) );
		}
	Eclr(); 
	if( !holdrefresh )Egenall();
	return( 0 );
	}

/* load or use... */
if( strlen( tag ) < 1 ) return( cerr( 2014, "copa panel: invalid tag", fnm ));

/* if there's an existing panel, write out current control panel in /tmp.. */
if( npanstack > 0 ) {
	sprintf( buf, "%s/copa%05d%s", E_TMPDIR, getpid(), panstack[ npanstack-1 ] );
	EIwriteall( buf, 2 ); /* save everything, including presets */
	EIclear(); 
	}

/* make sure there is room in the stack..*/
if( npanstack >= MAXSTACK-1 ) return( cerr( 2019, "Max panel nesting depth exceeded.", tag ) );

/* add tag to stack */
/* see if we're currently using it.. then just return.. */
if( strcmp( tag, panstack[ npanstack-1 ] )==0 ) return( 0 ); /* its already the
								current panel */
strncpy( panstack[ npanstack ], tag, 20 ); panstack[ npanstack ][20] = '\0';
npanstack++;

if( act[0] == 'l' ) {
	if( npanstack > 1 )Eload_presets(); /* get presets.. (scg 1-23-96) */
	stat = copa_load( fnm ); /* load */
	Ealignall(); 
	/* Eclr(); */  
	if( !holdrefresh )Egenall();
	return( stat );
	}
	
else if( act[0] == 'u' ) {    /* use */
	/* don't need to get presets; they were written out with tmp file.. */
	sprintf( buf, "%s/copa%05d%s", E_TMPDIR, getpid(), tag );
	/* read tmp file.. */
	stat = EIreaduserlist( buf );
	/* if tmp file doesn't exist, return error */
	if( stat != 1 ) return( cerr( 2016, "copa panuse: panel hasn't been loaded yet", tag ) );

	/* Eclr(); */  
	if( !holdrefresh )Egenall();
	return( 0 );
	}
}

/* =========== COPA_LOAD (internal, used by copa_panel) ============== */
static int
copa_load( fnm )
char *fnm;
{
int stat;
int ip;
int nde, firstde;
char fullfnm[100], *cf, *getenv();
FILE *fp;

/* locate file.. */
/* first check local directory.. */
strcpy( fullfnm, fnm );
fp = fopen( fnm, "r" );
if( fp == NULL ) {
	/* now check COPA_FILES directory.. */
	cf = getenv( "COPA_FILES" );
	if( cf != NULL && strlen( cf ) > 0 ) { 
		sprintf( fullfnm, "%s/%s", cf, fnm ); 
		fp = fopen( fullfnm, "r" ); 
		}
	if( fp == NULL ) {
		/* now check COPA_HOME directory.. */
		cf = getenv( "COPA_HOME" );
		if( cf != NULL  && strlen( cf ) > 0 ) {
			sprintf( fullfnm, "%s/%s", cf, fnm );
			fp = fopen( fullfnm, "r" );
			}
		if( fp == NULL ) return( cerr( 2011, "Control panel def file not found.", fnm ) );
		}
	}
fclose( fp );

/* read the file.. */
stat = EIreaduserlist( fullfnm );
if( stat != 1 ) cerr( 2004, "Error in control file.", "" );


count_de( &nde, &firstde ); /* get info on visible de areas.. added scg 12-7-95 */
if( nde == 1 ) EImodattr( firstde, "", "active", "x" ); /* make cursor dark */

return( 0 );
}


/* ============ COPA_TEXT - THIS VERSION USED BY C API ONLY ======== */
/* NOTE Shell "copa text" functionality is in cli.c and srv.c        */
/* capture text + results of any embedded command into bigbuf;
   then update the text attribute of the given mechanism. */

/* copa_text( char *tag, char *txt ) */
copa_text( tag, txt )
char *tag;
char *txt;
{
int i;
char *c;
char *copa_txt();
int stat;
char *s, *copa_fillbuf();

if( !window_exists ) return( cerr( 2003, nowin_msg, "" ) );

/* if this is the server, all command exansion is done by the client..
   so don't do it here.. */
if( !Eservermode ) s = copa_fillbuf( txt, 0 );
else s = txt;
stat = update_zood( tag, "text", s );
if( stat != 0 ) return( cerr( 2020, "Internal update error.", tag ) );
if( tag[0] != '_' ) redraw( tag ); /* don't redraw for popups.. */
return( 0 );
}

/* =========== COPA_ECHO (used by C API only) =========================== */

copa_echo( txt )
char *txt;
{
int stat;
char *copa_fillbuf();

if( !window_exists ) return( cerr( 2003, nowin_msg, "" ) );

if( strncmp( txt, ".!begin", 7 )==0 ) {   /* starting a new run.. */
	echotag[0] = '\0';
	sscanf( txt, "%*s %s", echotag );
	if( strlen( echotag ) < 1 )
		return( cerr( 2022, "copa_echo: No tag found with .!begin", "" ));
	if( EItaglookup( echotag, E_NIL) < 0 )
		return( cerr( 2024, "copa_echo: invalid mechanism tag", echotag ) );
	copa_fillbuf( "", 1 ); /* initialize buffer.. */
	return( 0 );
	}

else if( strcmp( txt, ".!end" )==0 ) {
	stat = update_zood( echotag, "text", bigbufp );
	if( echotag[0] != '_' ) redraw( echotag ); /* don't redraw for popups.. */
	return( 0 );
	}
	
else	{
	if( strlen( echotag ) < 1 )
		return( cerr( 2023, "copa_echo: a mechanism tag needs to be specified using .!begin","" ));
	bigbufp = copa_fillbuf( txt, 2 );
	return( 0 );
	}
}

/* =========== COPA_GETENTRY ======================= */
/* Activate the named data entry area and get a user response */
copa_getentry( tag, response )
char *tag;
char *response;
{
int p;
p = Egetid( tag );
if( p < 0 || !visible( p ) ) return( cerr( 2029, "getentry: bad mechanism tag", "" ) );
else if( strcmp( EIgetobjname( p ) , "dataentry" )!=0 )
	return( cerr( 2030, "getentry: may be used for dataentry mechanisms only", "" ) );

Eexec( p, 0.0, 0.0, E_MOUSE_LEFT, "" ); /* activate it.. */

evp = p;

/* get the terminating event (could be used by copa rinfo).. */
Eretrievekey( &evx, &evy, &evkey );

if( all_whitespace( Eselection ) ) strcpy( response, NONE );
else strcpy( response, Eselection );
return( 0 );
}

/* =========== COPA_GET  =========================== */
/* Get user response - normal mode */
copa_get( response )
char *response;
{
int stat;
stat = copa_mget( "standard", response );
return( stat );
}


/* =========== COPA_MGET ============================ */
/* Get user response - various input modes.
   Args: 
     rtyp = response type, one of the following:
	s(tandard) = Standard mode, suitable for most situations.  Receive results 
		produced by mouse click or data entry.
		If there is only one visible data entry area, all key presses
		will be captured by it.  If there is more than one data entry
		area on the screen, the "current" one (see below) will receive 
		keystrokes by default.  Other dataentry areas must be selected 
		before typing (see below).  In any case, keystrokes which aren't 
		captured in a data entry area are ignored.
	k(eys) = Intercept and return all keystrokes.  Key presses will not
		be received by data entry area(s).  Ignore mouse selections. 
	a(ny)  = Receive results produced by mouse click or data entry. 
		For keystrokes to be captured by a dataentry area, the
		dataentry area must be selected before typing (see below). 
		Keystrokes not captured by a dataentry area are returned.
		Mouse clicks which do not produce results are ignored.
	m(ousepos) = produce only a string containing the tag of the mechanism
		the mouse was in (or ".!none" ) when mouse click or keystroke
		occurs.  get_uinfo can be subsequently called to get x and y 
		location info.
	n(onblocking) = same as standard mode, except that mget will not wait
		for mouse click or data entry; if neither of these has occurred,
		mget when used in this mode will return immediately, with
		response set to ".!none".

    NOTE: The "current" data entry area is generally the one most recently 
	used.  If none have yet been used, it is the first-defined one which
	is visible.

    NOTE: A data entry area may be "selected" by the user clicking on it 
	with mouse.

    NOTE: Data entry responses of zero length or which contain only
	white-space characters are returned as ".!none", to avoid problems 
	in shell scripts.

    NOTE: Keystroke representation: see keyrep() below.

*/
/* copa_mget( char *rtyp, char *response ) */
copa_mget( rtyp, response )
char *rtyp;
char *response;
{
int p, len, bp;
char *keyrep();
int nde, firstde;  /* # de visible areas, proc id of first-defined de area */
int status;
char *next;
char buf[256];

/* evkey = key which was pressed 'A'=A, or mouse button 1001=left 1002=mid 1003=right */

if( !window_exists ) return( cerr( 2003, nowin_msg, "" ) );

bp = EItaglookup( "_POPB1", E_NIL); /* for popmsg */

strcpy( Eselection, NONE );
evp = -1;
while( 1 ) {

	if( rtyp[0] == 'n' ) {
		Easync(); /* check for events; if there is one EEvent will be set to non-zero */
		if( EEvent == 0 ) { /* no mouse click or data entry has occurred */
			strcpy( response, NONE );
			return( 0 );
			}
		else	{
			evx = EEventx;
			evy = EEventy;
			evkey = EEvent;
			EEvent = 0; /* reset */
			}
		}

	else Egetkey( &evx, &evy, &evkey );

	/* keys mode-- intercept keys */
	if( rtyp[0] == 'k' ) {
		if( evkey < 1000 ) {
			sprintf( response, "%s", keyrep( evkey ) ); /* key representation.. */
			return( 0 );
			}
		else continue;
		}

	/* mousepos mode-- */
	else if( rtyp[0] == 'm' ) {
		p = Ewhichobj( evx, evy );
		if( p < 0 ) strcpy( response, NONE );
		else strcpy( response, EIgetc( p, "", "*/tag", &len ) );
		if( all_whitespace( response ) ) strcpy( response, NONE ); /* for safety */
		return( 0 );
		}

	/* normal mode-- */
	else if( rtyp[0] == 's' || rtyp[0] == 'n' || rtyp[0] == 'a' ) {

		/* mouse click -- get results.. */
		if( evkey > 1000 ) {  
			p = Ewhichobj( evx, evy );
			if( p < 0 ) continue;
			if( rtyp[0] == 'n' && popmsg_exists && p != bp ) {
				strcpy( response, NONE );
				return( 0 );
				}
			if( !popmsg_exists && offlimits( p ) ) continue;
        		Eexec( p, evx, evy, evkey, "" );
        		if( strcmp( Eselection, NONE ) == 0 ) continue;
			}

		/* key press */
		else	{
			if( popmsg_exists ) {
				strcpy( response, NONE );
				return( 0 );
				}
			if( rtyp[0] == 'a' ) {  /* any-- just return key */
				sprintf( response, "%s", keyrep( evkey ) );
				return( 0 );
				}
			/* count_de( &nde, &firstde ); */ /* get info on visible de areas.. */
			if( nde >= 1 ) {
				if( nde == 1 ) p = curde = firstde; 
				else if( curde < 0 || !visible( curde ) ) p = curde = firstde;
				else p = curde;
				/* The following Eexec call will get its own keystrokes
				   and return the ascii of the key which terminated it  */
				status = Eexec( p, evx, evy, evkey, "" );

				/* update curde, based on "next" de attribute, etc. //
				/*
				 * if( nde > 1 && status != 27 ) { // look for next //
 			         *     next = EIgetc( p, "", "next", &len );
				 *     if( len > 0 ) {
				 *	  curde = Egetid( next );
				 *	  }
				 *     // else curde remains the same.. //
				 *     }
				 */
				}
			else if( nde < 1 ) continue;
			}
		/* set up for secondary response.. */
		if( strcmp( EIgetobjname( p ), "dataentry" )==0 ) {
			/* get the terminating event of the dataentry area for rinfo.. */
			Eretrievekey( &evx, &evy, &evkey );
			} 
		evp = p;
		if( sscanf( Eselection, "%s", buf ) > 0 ) strcpy( response, Eselection );
		else strcpy( response, NONE );
		return( 0 );
		} 
	}
}

/* ============= COPA_RLIMIT =============================== */
/* limit input to certain mechanism(s) */
copa_rlimit( meclist )
char *meclist;
{
int i, p;
char *tok;
if( strcmp( meclist, ".!all" )==0 || strlen( meclist ) < 1 ) nlimit = 0;
else	{
	for( i = 0; i < strlen( meclist ); i++ ) if( meclist[i] == ',' ) meclist[i] = ' ';
	for( i = 0; i < strlen( meclist ); ) {
		tok = Egetok( meclist, &i );
		p = Egetid( tok );
		if( p < 0 ) cerr( 2008, "copa_rlimit: tag not defined - ignored.", tok );
		limlist[ nlimit++ ] = p;
		}
	}
return( 0 );
}


/* ============= COPA_RINFO ============================ */
/* return requested info related to previous selection/mouse click . */
/* copa_rinfo( char *inftyp, char *results ) */
copa_rinfo( inftyp, results )
char *inftyp,  *results;
{
int len;
char *keyrep();
int ip;

if( !window_exists ) return( cerr( 2003, nowin_msg, "" ) );
if( strcmp( inftyp, "tag" )==0 ) {
	if( evp < 0 ) strcpy( results, NONE );
	else strcpy( results, EIgetc( evp, "", "*/tag", &len ) );
	if( all_whitespace( results ) ) strcpy( results, NONE ); /* safety */
	return( 0 );
	}
else if( strcmp( inftyp, "mectype" )==0 ) {
	if( evp < 0 ) strcpy( results, NONE );
	else strcpy( results, EIgetobjname( evp ) );
	return( 0 );
	}

else if( strcmp( inftyp, "secondary_tag" ) == 0 ) {
	results[0] = '\0';
	if( evp < 0 ); 
	else if( strcmp( EIgetobjname( evp ), "dataentry" )==0 && evkey >= 1000 ) {
		ip = Ewhichobj( evx, evy );
		if( evp >= 0 ) strcpy( results, EIgetc( ip, "", "*/tag", &len ) );
		}
	if( all_whitespace( results ) ) strcpy( results, NONE );
	return( 0 );
	}

else if( strcmp( inftyp, "secondary_result" ) == 0 ) {
	results[0] = '\0';
	if( evp < 0 );
	else if( strcmp( EIgetobjname( evp ), "dataentry" ) ==0 && evkey >= 1000 ) {
		ip = Ewhichobj( evx, evy );
		if( ip >= 0 && strcmp( EIgetobjname( ip ), "dataentry" )!= 0 ) {
			Eexec( ip, evx, evy, evkey, "" );
			strcpy( results, Eselection );
			}
		}
	if( all_whitespace( results ) ) strcpy( results, NONE );
	return( 0 );
	}

else if( strcmp( inftyp, "key" )==0 ) {
	if( evkey < 0 ) strcpy( results, NONE );
	else sprintf( results, "%s", keyrep( evkey ) ); 
	return( 0 );
	}
else if( strcmp( inftyp, "keytype" )==0 ) {
	if( evkey < 0 ) strcpy( results, NONE );
	else if( evkey >= 1000 ) strcpy( results, "mouse" );
	else strcpy( results, "key" );
	return( 0 );
	}
else if( strcmp( inftyp, "x" )==0 ) {
	sprintf( results, "%g", evx );
	return( 0 );
	}
else if( strcmp( inftyp, "y" )==0 ) {
	sprintf( results, "%g", evy );
	return( 0 );
	}

return( cerr( 2007, "Unrecognized parameter in copa_rinfo.", inftyp ) );
}
/* ============= COPA_SETVALUE =========================== */
copa_setvalue( tag, value )
char *tag, *value;
{
char buf[ E_SELECTION_LEN ];
char intag[40];
int stat, nt;
FILE *fp;
if( strcmp( tag, ".!file" )==0 ) {
	fp = fopen( value, "r" );
	if( fp == NULL ) return( cerr( 2033, "Cannot open tag/values file", value ) );
	while( fgets( buf, E_SELECTION_LEN-1, fp ) != NULL ) {
		buf[ strlen( buf ) - 1 ] = '\0'; /* rm newline */
		nt = sscanf( buf, "%s", intag ); 
		if( nt < 1 ) continue;
		if( intag[0] == '#' ) continue;
		stat = copa_setsel( intag, &buf[ strlen( intag ) + 1 ] );
		if( stat != 0 ) 
			fprintf( stderr, "Warning, setvalue failed for a data line (%s)\n", buf );
		}
	fclose( fp );
	if( !holdrefresh ) Egenall();
	return( 0 );
	}
stat = copa_setsel( tag, value );
redraw( tag );
return( stat );
}

/* ============= COPA_GETVALUE =========================== */
copa_getvalue( tag, results )
char *tag;
char *results;
{
int ip, len;
char *typ, *outtag, *val;

if( strcmp( tag, ".!all" )==0 ) {   /* do a tag/value dump.. */
	EIgetnextobj( E_AT_BEGINNING, 0 );
	results[0] = '\0';
	while( 1 ) {
        	ip = EIgetnextobj( E_NEXT, 0 );
        	if( ip < 0 ) break;
		if( strlen( results ) >= E_SELECTION_LEN - 10 )
			return( cerr( 2031, "rinfo allmecs - results too long", "" ) );
		typ = EIgetobjname( ip );
		if( strcmp( typ, "textb" )==0 ||
		    strcmp( typ, "button" )==0 ||
		    strcmp( typ, "slider" )==0 ||
		    strcmp( typ, "dataentry" )==0 ) {
			outtag = EIgetc( ip, "", "*/tag", &len );
			if( outtag[0] == '_' ) continue; /* don't do popups */
			if( strlen( outtag ) < 1 ) continue; /* null tag.. */
			if( strlen( results ) + len >= E_SELECTION_LEN - 3 )
			  return( cerr( 2031, "rinfo allmecs - results too long", "" ) );
			Eaddb( results, outtag );
			Eaddb( results, " " );
			val = EIgetc( ip, "", "pick/value", &len );
			if( strlen( results ) + len >= E_SELECTION_LEN - 3 )
			  return( cerr( 2031, "rinfo allmecs - results too long", "" ) );
		
			Eaddb( results, val );
			Eaddb( results, "\n" );
			}
		}
	EIgetnextobj( E_DONE, 0 );
	if( all_whitespace( results ) ) strcpy( results, NONE );
	return( 0 );
	}
else	{
	results[0] = '\0';
	ip = Egetid( tag );
	if( ip < 0 ) return( cerr( 2032, "Invalid tag", tag ) );
	strcpy( results, EIgetc( ip, "", "pick/value", &len ) );
	if( all_whitespace( results ) ) strcpy( results, NONE );
	return( 0 );
	}
}

/* ============= COPA_SETMEC ============================= */
/* Set a characteristic of a mechanism. */
/* copa_setmec( char *tag, char *attr, char *value ) */
copa_setmec( tag, attr, value )
char *tag, *attr, *value;
{
int status;
char buf[40];
if( strcmp( attr, "value" )==0) strcpy( buf, "pick/value" );
else strcpy( buf, attr );
status = update_zood( tag, buf, value );
if( status < 0 ) return( cerr( 2013, "setmec failed.", buf ) );
redraw( tag );
return( 0 );
}

/* ============= COPA_GETMEC ============================= */
/* Get a characteristic of a mechanism. */
/* copa_getmec( char *tag, char *attr, char *value ) */
copa_getmec( tag, attr, value )
char *tag, *attr, *value;
{
int len;
char buf[40];
int stat;
strcpy( buf, attr );
stat = EIget( value, Egetid( tag ), "", buf );
if( stat != 1 ) return( cerr( 2021, "No mechanism by this name has been defined.", tag ) );
if( all_whitespace( value ) ) strcpy( value, NONE );
return( 0 );
}

/* ============= COPA_PS ============================== */
/* THIS DOESNT WORK WELL FOR CONTROL PANELS  */
/* Do a vector EPS representation of current copa screen. 
   Place it in file fnm ("-" = stdout ).
   Use paper orientation: "L" = landscape  "P" =portrait
 */ 
/* copa_ps( char *name, char *fnm, char *paperway ) */
copa_ps( name, fnm, paperway )
char *name, *fnm, *paperway;
{
Eprintall( name, fnm, paperway[0] );
return( 0 );
}

/* ============= COPA_QUIT ============================ */
copa_quit()
{
char buf[120];
/* remove any tmp files */
sprintf( buf, "rm %s/copa%05d* 2>/dev/null", E_TMPDIR, getpid() );
system( buf );
return( 0 );
}


/* =============== INTERNAL ROUTINES -================= */

/* ============= KEYREP (internal) ============= */
/* Does translation on non-printable and white-space characters.

Printable non-whitespace keystrokes are represented as a string 
containing the character.  All others are represented as a string 
containing ".!char", followed by the character's decimal ascii value, 
e.g.  space = ".!char32".  
Mouse button representations are: ".!mouseleft", ".!mousemiddle" 
	and ".!mouseright".
*/

		
static char *
keyrep( key )
int key;
{

/*       enter      space       backsp      del           esc          tab  */
if( key == 1001 ) sprintf( tinybuf, ".!mouseleft" );
else if( key == 1002 ) sprintf( tinybuf, ".!mousemiddle" );
else if( key == 1003 ) sprintf( tinybuf, ".!mouseright" );
else if( key == 13 || key == 32 || key == 8 || key == 127 || key == 27 || key == '\t'  
	|| ! isprint( key ) ) {
	sprintf( tinybuf, ".!char%d", key );
	return( tinybuf );
	}
else sprintf( tinybuf, "%c", key );

return( tinybuf );
}
/* ============= UPDATE_ZOOD (internal) ============= */
/* Update mechanisms database */
static int
/* update_zood( char *tag, char *attrib, char *value ) */
update_zood( tag, attrib, value )
char *tag, *attrib, *value;
{
int p, status;
p = Egetid( tag );
if( p < 0 ) return( cerr( 2025, "Bad mechanism tag", tag ) );
Enulloutbb( p );
status = EImodattr( p, "", attrib, value );
if( status < 1 ) return( -1 );
updatehooks( p, attrib ); /* check dependencies */
return( 0 );
}


/* ============= REDRAW (internal) ==================== */
/* Redraw a mechanism. */
static int
redraw( tag )
char *tag;
{
int p;
p = Egetid( tag );
Enulloutbb( p );
Egen( p );
return( 0 );
}
/* ============= COUNT_DE ========================= */
/* count how many data entry mechanisms exist and are visible */
static int
/* count_de( int *nde, int *firstde ) */
count_de( nde, firstde )
int *nde, *firstde;
{
int ip, len;
char *c;

*nde = 0; *firstde = -1;
EIgetnextobj( E_AT_BEGINNING, 0 );
while( 1 ) {
        ip = EIgetnextobj( E_NEXT, 0 );
        if( ip < 0 ) break;

	/* if limits are in effect and ip cannot respond, don't include it.. */
	if( offlimits( ip ) ) continue;
	if( strcmp( EIgetobjname( ip ) , "dataentry" )==0 ) {
		c = EIgetc( ip, "", "*/display", &len );
		if( strlen( c ) > 0 ) {
			/* we've found one.. */
			if( *nde == 0 ) *firstde = ip;
			(*nde) ++;
			}
		}
        }
EIgetnextobj( E_DONE, 0 );
return( 0 );
}

/* ============= OFFLIMITS ========================= */
/* see if a given proc id is off limits (not responsive) to user.. */
/* return 1 if the proc is not responsive; 0 if it is responsive */
static int
offlimits( p )
int p;
{
int i;
if( nlimit < 1 ) return( 0 );
for( i = 0; i < nlimit; i++ ) {
	if( p == limlist[i] ) return( 0 );
	}
return( 1 );
}


/* ============= VISIBLE ========================= */
static int
visible( p )
int p;
{
int len;
char *c;

c = EIgetc( p, "", "*/display", &len );
if( strlen( c ) > 0 ) return( 1 );
else return( 0 );
}

/* ============= UPDATEHOOKS ========================= */
static int
/* updatehooks( int p, char *attr ) */
updatehooks( p, attr )
int p;
char *attr;
{
char *class;
class = EIgetobjname( p );
 
/* If text in a text box changes, the textbox auto-scaling may need to be rerun.. */
/* Or if text box becomes scrollable/not scrollable, scrolling control may need to be
   added/removed. */
if( strcmp( class, "textb" )== 0  && strcmp( attr, "text" )==0 ) {
        Eexec( p, 0.0, 0.0, E_MESSAGE, "REINIT" );
#ifdef PLOT
        EPrefreshplotdata( p ); /* notifiy ipl that the plot data may have changed. */
#endif
        }
}

/* ============= CERR ========================= */
static int
/* cerr( int errno, char *msg, char *parm ) */
cerr( errno, msg, parm )
int errno;
char *msg;
char *parm;
{
return( Eerrmsg( errno, msg, parm ) );
}

/* ============= ALL_WHITESPACE ================ */
/* check string, return 1 if it's all white space or 0 length.  Return 0 otherwise */
static int
all_whitespace( s )
char *s;
{
int i;
for( i = 0; i < strlen( s ); i++ ) {
	if( s[i] == ' ' || s[i] == '\t' || s[i] == '\n' );
	else return( 0 );
	}
return( 1 );
}
