/* PCODE - all text/graphics requests (except init) are processed
		via this function.  Many E function calls such
		as Emov and Elin are actually macros defined in
		elib.d, which call pcode with a single character
		op code and perhaps some parameters.

   ==========================================================================
   Device codes (Edev):
	p		postscript
	x		X11

   Op codes:
	L		lineto
	M		moveto
	P		path to
	T, C, J		text, centered text, right justified text
	S		shade
	O		paper (ps only)
	I		text size
	F		text font 
	D		text direction (ps only)
	Y		line properties
	Z		print/eject (ps only)
	B, b		batching on/off for efficiency (sv only)
	R		line/text color
	W		wait for a key or mouse click (win only)
	w		cycle notifier from within a loop (win only)
	.		dot
	z		clear screen 
	d		make window disappear (x11 only)
	a		make window re-appear (x11 only)
	e		set text scale factor (x11 only, when user resizes window)
	Q		end of file (tells ps driver to append trailer)


   ==========================================================================

   NOTE: When Ewidth is returned by the x11 text functions, it is ignored.
*/

#include "elib.x"

static int Ejournal = 0;			/* journal on/off */
static int Ejournalsusp = 0;			/* journal suspend counter */
static int Enew = 0, Edrawing = 0;		/* used by postscript section */
static int Evertchar = 0; 			/* true if character direction is vertical..*/
						/* ..used by x11 to do vertical characters */
static double Ecurx;				/* real starting place of centered &rt justified text */
static double Ewidth;				/* width of most recently drawn text line. */
static char Eprevop;				/* previous op */
static double Etx = 0, Ety = 0;			/* current translation vector */
static int Edoingobject = 0;			/* are we inside an object begin/end? 1=yes */
static int Esquelched = 0;			/* is output being squelched? 1=yes, 0=no */
static int Epspage = 1;			/* current page # */
static int Epsvirginpage = 1;		/* 1 when nothing has yet been drawn on current page */
static int bbexempt = 0;
					
static double Epsx1 = 999;		/* coords of bounding box for entire run */
static double Epsy1 = 999;
static double Epsx2 = -999;
static double Epsy2 = -999;


Epcode( op, x, y, s )
char op; /* op code */
double x, y;  /* coordinates */
char s[];     /* optional character string */
{
char buf[E_GPBUF];


/* Do object package related translation.. if any..  */
if( Emember( op, "MLP." )) { 
	x += Etx; y += Ety; /* translate.. */
	}

/* Postscript: intercept all 'clear' ops - they're not necessary and
   mess up bounding box calculation.. */
if( Edev == 'p' && op == 'z' ) return( 0 );


/* Postscript: inform driver we're beginning a new page.. */
if( Edev == 'p' && Epsvirginpage && !Emember( op, "Q" ) ) {
	EPSnewpage( Epspage );
	Epsvirginpage = 0;
	}

/* if output is squelched, do not do any draw operations, except textsize, which
   returns text width, which is necessary for bb calculations.. */
if( Esquelched ) {
	if( op == 'I' ) {
#ifndef NOX11
		if( Edev == 'x' ) { EXWpointsize( (int)(x), &Ecurtextwidth ); }
#endif
		}
	}


#ifndef NOX11
else if( Edev == 'x' ) {
	switch (op) {
		case 'L' : EXWlineto( x, y ); break;
		case 'M' : EXWmoveto( x, y ); break;
		case 'P' : EXWpath( x, y ); break;
		case 'T' : if( !Evertchar ) EXWtext( s, &Ewidth ); 
			   break;
		case 'U' : EXWflush(); break;
		case 'I' : EXWpointsize( (int)(x), &Ecurtextwidth ); break;
		case 'S' : EXWshade( x ); break;
		case 'C' : if( !Evertchar ) { 
				EXWmoveto( Ex1-6.0, Ey1 ); 
				EXWcentext( s, 12.0, &Ecurx, &Ewidth ); 
				}
			   break;
		case 'J' : if( !Evertchar ) {
				EXWmoveto( Ex1-12.0, Ey1 ); 
				EXWrightjust( s, 12.0, &Ecurx, &Ewidth );
				}
			   break;
		case 'W' : EXWwait(); break;
		case 'Y' : EXWlinetype( s, x, y ); break;
		case 'D' : if( x == 90 || x == 270 ) Evertchar = 1;
			   else Evertchar = 0;
			   break;
		case 'R' : EXWcolor( (int)(x)); break;
		case 'w' : EXWasync(); break;
		case '.' : EXWdot( x, y ); break;
		case 'd' : EXWdisappear(); break;
		case 'a' : EXWappear(); break;
		case 'e' : EXWscaletext( x ); break;
		}
	}
else
#endif

/* interface to postscript driver */
if( Edev == 'p'  ) {

	if( op != 'L' ) { 
		if( Edrawing ) EPSstroke(); 
		Edrawing = 0;
		}

	switch( op ) {
		case 'L' : if( Enew ) EPSmoveto( Ex1, Ey1 ); 
			    EPSlineto( x, y );
			    Enew = 0;
			    Edrawing = 1;
			    break;
		case 'M' : Enew = 1; break;
		case 'P' : if( Enew ) EPSmoveto( Ex1, Ey1 ); 
			   EPSpath( x, y ); 
			   Enew = 0;
			   break;
		case 'T' : EPStext( op, Ex1, Ey1, s, 0.0 ); break;
	
		case 'C' : if( !Evertchar ) EPStext( op, Ex1-6.0, Ey1, s, 12.0 );
			   else if( Evertchar )EPStext( op, Ex1, Ey1-6.0, s, 12.0 );
			   break;
		case 'J' : if( !Evertchar ) EPStext( op, Ex1-12.0, Ey1, s, 12.0 );
			   else if( Evertchar ) EPStext( op, Ex1, Ey1-12.0, s, 12.0 );
			   break;
	
		case 'S' : EPSshade( x ); break;
		case 'O' : EPSpaper( (int)x ); break;
		case 'I' : EPSpointsize( (int)x ); break;
		case 'F' : EPSfont( s ); break;
		case 'D' : EPSchardir( (int)x );
			   if( x == 90 || x == 270 ) Evertchar = 1;
			   else Evertchar = 0;
			   break;
		case 'Y' : EPSlinetype( s, x, y ); break;
		case 'Z' : EPSshow(); Epspage++; Epsvirginpage = 1; break;
		case 'Q' : if( !Epsvirginpage ) { EPSshow(); Epspage++; }
			   /* the 0.2 margin is to be generous in cases of fat lines, etc. */
			   EPStrailer( Epspage - 1, Epsx1-0.2, Epsy1-0.2, Epsx2+0.2, Epsy2+0.2 );
			   break;
		/* case '.' : EPSdot( x, y ); ??? */
		/* case 'R' : ??? */
		}
	}



else { 
	if( Edev == '\0' ) fprintf( stderr, "Elib has not been initialized (device is null).\n" );
	else fprintf( stderr, "Unrecognised graphic environment (%c).\n", Edev ); 
	exit();
	}



/* non device-dependent actions.. */

if( op == 'M' ) { Ex1 = x; Ey1 = y; } /* remember most recent 'move' */

else if( op == 'L' ) { Ex2 = x; Ey2 = y; } /* remember most recent 'line' */


/* add to journal.. */
if( Ejournal && (!Ejournalsusp) && Edoingobject && Edev != 'n' ) {
	if( ! Emember( op, "zWwBb#" )) {
		sprintf( buf, "%c %g %g %s", op, x, y, s );
		EIaddcontline( buf );
		}
	}

/* figure text dimensions */
if( Emember( op, "TCJ" )) {
	if( Emember( Edev, "xs" )) Ewidth = strlen( s ) * Ecurtextwidth;
	/* else if( Edev == 'p' ) Ewidth = EPStextwidth( s )??? */
	}



/* keep bounding box info (minima and maxima) */
if( Emember( op, "LP" ) ) {
	if( Eprevop == 'M' ) Ebb( Ex1, Ey1 );
	Ebb( x, y );
	}
/* normal (horizontal) text operations.  (vertical text below) */
else if( op == 'T' && !Evertchar ) {
	if( Eprevop == 'M' ) Ebb( Ex1, Ey1 );
	Ebb( Ex1+Ewidth+0.05, Ey1+Ecurtextheight );
	}
else if( op == 'C' && !Evertchar ) { 
	Ebb( Ex1-((Ewidth/2.0)+0.05), Ey1 );
	Ebb( Ex1+((Ewidth/2.0)+0.05), Ey1+Ecurtextheight );
	}
else if( op == 'J' && !Evertchar ) { 
	Ebb( Ex1-(Ewidth+0.05), Ey1 );
	Ebb( Ex1, Ey1+Ecurtextheight );
	}
Eprevop = op;


/* clear screen (recursive) */
if( op == 'z' && Edev != 'n' ) { 
	Eobjoutline( E_NIL ); /* reset */
	Etentativeoutline( E_NIL, 0.0, 0.0, "" ); /* reset */
	Eblock( 0.0, 0.0, EWinx, EWiny, x, 0 ); 
	/* if( status != E_DOING_LOAD )Edumpgraphenv(); */
	}

/* vertical text simulation (recursive) */
else if( Evertchar && Emember( Edev, "sxn" ) && Emember( op, "TCJ" )) Everttextsim( op, s );
}

/* ============================================= */
/* EBB - keep an overall bounding box for the entire image.
	 Also call Echeckbb() to maintain nested object bounding boxes.. */
/* Ebb( double x, double y ) */
Ebb( x, y )
double x, y;
{
if( bbexempt ) return( 0 );

if( x < Epsx1 ) Epsx1 = x;
if( x > Epsx2 ) Epsx2 = x;
if( y < Epsy1 ) Epsy1 = y;
if( y > Epsy2 ) Epsy2 = y;

Echeckbb( x, y );
return( 0 );
}


/* ============================================== */
/* EGETTEXTSIZE - get width and height of last text item.. */

Egettextsize( w, h )
  double *w, *h;
{
*w = Ewidth;
*h = Ecurtextheight;
}


/* ============================================== */
/* turn journalling on/off */
Ejournalon()
{
Ejournal = 1;
Ejournalsusp = 0;
}

/* ============================================== */
Ejournaloff()
{
Ejournal = 0;
Ejournalsusp = 0;
}
/* ============================================== */
/* suspend / resume journalling.. */
Ejournalsuspend()
{
Ejournalsusp++;
}
/* ============================================== */
Ejournalresume()
{
if( Ejournalsusp > 0 ) Ejournalsusp--;
}
/* ================================================ */
/* vertical text simulation for window displays */
Everttextsim( op, s )
char op, s[];
{
double dist, x1, y1, x2, y2;
Ejournalsuspend();
dist = strlen(s) * (Ecurtextheight/2.0);
x1 = Ex1-Ecurtextheight; 
x2 = Ex1;
if( op == 'T' ) { y1 = Ey1; y2 = Ey1 + dist; }
else if( op == 'C' ) { y1 = Ey1-(dist/2); y2 = Ey1 + (dist/2); }
else if( op == 'J' ) { y1 = Ey1-dist; y2 = Ey1; }
Eblock( x1, y1, x2, y2, 0.9, 1 );
Ejournalresume();
}

/* ================================================== */
Etrladd( x, y )
double x, y;
{
Etx += x;  Ety += y;
}
/* ================================================== */
Etrlset( x, y )
double x, y;
{
Etx = x; Ety = y;
}
/* ================================================== */
Etrlget( x, y )
double *x, *y;
{
*x = Etx; *y = Ety;
}
/* =================================================== */
Esetobjectflag( mode )
int mode; /* 1 = doing an object-- ok to write journal.  0 - not */
{
int ip;
if( mode == 1 && mode == Edoingobject && Ejournal ) { 
	fprintf( stderr, "Warning, nested objects are not allowed while in journal mode --ignored.\n" );
	return( 1 );
	}
Edoingobject = mode;
}
/* =================================================== */
/* This routine should be used to begin an elib journal object. */
/* Eend_journal_object() should be used to end the object. */
/* It returns the proc_id of the journal object. */
Ebegin_journal_object()
{
int ip;
Ejournalon();
ip = EIcreate( "elib_journal" );
EImodattr( ip, "", "text", "# 0 0" ); /* will be appended to by subsequent journal writes. */
Eobjbegin( ip );
return( ip );
}
/* =================================================== */
/* This routine should be used to end an elib journal object. */
/* typ is the type of bounding zone: "B" = box (usually used)  "L" = line */
Eend_journal_object( typ )
char typ[];
{
Eobjend( typ );
Ejournaloff( );
}
/* ==================================================== */
/* 1 = squelch all display activity, 0 restore to normal */
/* handles nested calls. */
Esquelch_display( mode )
int mode;
{
static int snest = 0;
if( mode == 1 ) {
	snest++;
	Esquelched = 1;
	}
else if( mode == 0 ) { 
	if( snest > 0 ) snest--;
	if( snest == 0 )Esquelched = 0;
	}
}

/* ================================== */
/* 1 = make what follows exempt from bb checking, 0 restore to normal */
Ebbexempt( mode )
{
bbexempt = mode;
}
