/*
COPA 1.0 - gui control panel kit for shell, C, perl etc.  
*              **      **                              *
Copyright 1991-1996 by Stephen C. Grubb.

Permission is hereby granted to USE this software for free.
Permission is hereby granted to MODIFY and/or REDISTRIBUTE 
this software for free, provided that no monetary gain is 
realized, and all attached authorship and copyright notices 
are preserved.

Inclusion of any portion (or derivation) of this software, 
including ancillary files, in any commercial product is not 
allowed without prior written permission of the author.  

See also the file 'Copyright'. 
*/





/* ==================================================================== */
/* ==================================================================== */
/* This is the OO database.						*/
/* ==================================================================== */
/* ==================================================================== */
#include <stdio.h>
#define TMP_DIR "/usr/tmp/"



#define NIL -1			/* nil indicator */

/* Egetnextobj sequences.. */ 
#define E_AT_BEGINNING 0
#define E_AT_END 2
#define E_NEXT 1
#define E_PREVIOUS -1
#define E_DONE 3

#define MAXO 200       		/* max number of user objects */ /* LIMIT */
#define MAXD 200      		/* max number of default objects */ /* LIMIT */
#define MAXA 2500     		/* max number of attributes (both user and default) */ /* LIMIT */
#define MAXNP 3000		/* max size of name parts list */
#define MAXPACK 10		/* max number of object packages */ /* LIMIT */

#define STRINGPOOLLEN 300000  	/* capacity of the string pool */  /* LIMIT */
#define DLISTLEN     10000     	/* capacity of double list */	   /* LIMIT */
#define ILISTLEN     500     	/* capacity of int list */	   /* LIMIT */
#define ANAMELEN     80



/* Packages */
static int EIPname[MAXPACK];          	/* package name (pointer into string pool) */
static int EIPn;			/* number of packages */

/* Display order */
static int  EIorder[MAXO];		/* contains procids, in display order (-1 == skip) */

/* User Objects */
/* cell 0 corresponds to procid 0, 1 to 1, etc. */
static int  EIOname[MAXO]; 		/* proc name (pointer into string pool) */
static int  EIOinheritid[MAXO]; 	/* procid of proc from which inheritance may happen, or NIL */
static int  EIOattrkey[MAXO]; 		/* points to first attribute, or nil */
static int  EIOdefkey[MAXO];  		/* points to entry in default list */
static int  EIOn;    			/* how many */

/* Attributes (both user and default) */
static int  EIAnparts[MAXA];		/* number of parts in name, or NIL for hard default */
static int  EIAnamep[MAXA];		/* points to first entry in namepart pool, or to s.p. for hard def. */
static int  EIAnext[MAXA];            	/* points to next attribute or NIL */
static char EIAtype[MAXA];            	/* I, D, or C */
static int  EIApos[MAXA];             	/* position in storage list */
static int  EIAlen[MAXA];             	/* amount of storage list occupied */
static int  EIAalloc[MAXA];		/* amount of storage list reserved */
static int  EIAconkey[MAXA];            /* (defaults only) points to constraint in the string pool, or NIL */
static int  EIAdeskey[MAXA];		/* (defaults only) points to description in the string pool, or NIL */
static int  EIAxkey[MAXA];		/* for 'X'types this points to called obj. in the EIO list */
static int  EIAdef[MAXA];		/* attributes: points to default entry.  defaults: contains flags */
static int  EIAavail;                 	/* next available slot-- just a counter for now.. */  

/* Name parts */
static char *EINP[MAXNP];		/* point to a name part in the string pool. */
static int  EINPn;			/* how many */

/* Default Object index */
static int  EIDname[MAXD]; 		/* default object name (pointer into string pool) */
static int  EIDattrkey[MAXD];      	/* points to first attribute */
static int  EIDpack[MAXD];		/* index into packages list */
static int  EIDoverrideflag[MAXD];	/* indicates whether any overrides were used */
static int  EIDn;			/* how many */

/* Storage lists */
/* String */
static char  EISs[STRINGPOOLLEN];  	/* string pool */
static int   EISsn;                 	/* next available slot in string pool */

/* Double */
static double EISd[DLISTLEN];	     	/* doubles */
static int  EISdn;

/* Int */
static int EISi[ILISTLEN]; 		/* ints */
static int EISin;


static int EIinitstate[8];		/* State of structure before user stuff.. */	

/* input file line number */
static int EILineno;
static int EIerrorflag = 0;

static int EIcontplace = NIL;		/* placeholder for continuation lines */
static int EIoccur = 1;			/* occurrence selector-- usually the first occurrence */
static int EIaddalways = 0;		/* flag, if 1, EImodattr adds as a new occurrence unconditionally. */
static int EImustalloc = NIL;		/* if non-NIL, indicates amount of storage allocation by EImodattr */
static int EIsilentflag = 0;		/* flag for silencing error messages */
static int EIwhere = NIL;		/* set by EIlocate, 1 = user-specified, 2 = inherited, 3 = default */

static int EIflag = 0; 			/* for debugging */
static int EIulookups, EIinhlookups, EIdeflookups, EIdeflookupfaults;


extern char *EIgetobjname(), *EIgetattrname();

/* ========================================================== */
/* ===== move an object to the beginning of the display list  */
/* ===== so that it is displayed at the bottom of the stacking order. */
/* ========================================================== */
EIlower( p )
int p;
{
int i, riffling;
/* start at end of order list and look for p.  When found, riffle.. */
riffling = 0;
for( i = EIOn-1; i > 0; i-- ) {
	if( EIorder[i] == p ) riffling = 1;
	if( riffling ) EIorder[i] = EIorder[i-1];
	}
EIorder[0] = p;
}

/* ========================================================== */
/* ===== move an object to the end of the display list ====== */
/* ===== so that it is displayed at the top of the stacking order. */
/* ========================================================== */
EIraise( p )
int p;
{
int i, riffling;
/* start at beginning of order list and look for p.  When found, riffle.. */
riffling = 0;
for( i = 0; i < EIOn-1; i++ ) {
	if( EIorder[i] == p ) riffling = 1;
	if( riffling ) EIorder[i] = EIorder[i+1];
	}
EIorder[EIOn-1] = p;
}




/* ========================================================== */
/* ==== initialize structures =============================== */
/* ========================================================== */
EIinit()
{
EIOn = 0;  EIAavail = 0;  EIDn = 0;  

/* put friendly null values in first slot of each type.. */
EISs[0] ='\0'; EISsn = 1; 
EISd[0] = 0.0; EISdn = 1;  
EISi[0] = 0;   EISin = 1; 

EIPn = 0;
EINPn = 0;
EIdeflookups = 0;
EIdeflookupfaults = 0;
EIulookups = 0;
EIinhlookups = 0;
}

/* ========================================================== */
/* ===== save state before any objects are defined.. ============ */
/* ========================================================== */
EIsaveinitstate()
{
EIinitstate[0] = EIAavail; EIinitstate[1] = EISsn; EIinitstate[2] = EISdn; EIinitstate[3] = EISin;
EIinitstate[4] = EIOn; EIinitstate[5] = EINPn;
}

/* ========================================================== */
/* ===== clear all objects (defaults list remains) ============================ */ 
/* ========================================================== */
EIclear()
{
int i;
EIAavail = EIinitstate[0]; EISsn = EIinitstate[1]; EISdn = EIinitstate[2]; EISin = EIinitstate[3];
EIOn = EIinitstate[4]; EINPn = EIinitstate[5];

/* We need to rebuild order list for presets, since the order list
   can be rearranged during zood use.. Presets will be raised when used
   so it doesn't matter. */
for( i = 0; i < EIOn; i++ ) EIorder[i] = i;

}

/* ============================================================ */
/* ============================================================ */
/* = look up the defaults info (type, given length) for an attribute= */
/* = Low level routine.. should never be called by an application */
/* 								*/
/* = returns a pointer into the attributes list which can be == */
/* = used to get alloc, constraint, and description info. ===== */
/* = Returns NIL on failure. ================================== */
/* = Understands external procs. ============================== */ 
/* = Fills the array nameslots with pointers to attribute names,*/
/* = for every part.  ========================================= */
/* ============================================================ */
static
EIgetinfo( defkey, attr, typ, nameslots, nslots )
int defkey; /* points to a slot in the defaults list */
char attr[]; /* attribute name */
char *typ;  /* type - returned */
int nameslots[]; /* returned- up to 5 pointers into attr list, so complete 
			attribute names need not be stored */
int *nslots;   /* returned- number of name slots filled */
{
int i, j, k, ia, id, ix;
int nparts;
char part[5][20];

ia = EIDattrkey[defkey];

/* see if attribute is multi-level -- break up into parts (delimited by / ) */
ix = 0;
for( i = 0; i < 5 ; i++ ) {
	if( ix >= strlen( attr )) break;
	Egetchunk( part[i], attr, &ix, "/" ); 
	}
nparts = i;
if( nparts >= 5 ) { fprintf( stderr, "(%s) too many parts.\n", attr ); EIerrorflag = 1; return( NIL ); }

for( i = 0; i < nparts; i++ ) {
	/* look up the part.. */
	while( ia != NIL ) {
		if( EIAnparts[ ia ] != NIL ) { 
			fprintf( stderr, "Internal error: nparts != NIL\n" ); 
			EIerrorflag = 1;
			return( NIL ); 
			}
		if( strcmp( part[i], &EISs[ EIAnamep[ ia ]] ) == 0 ) break;
		else ia = EIAnext[ia];
		}
	nameslots[i] = ia;
	if( ia == NIL ) { 
		fprintf( stderr, "%s: not found in the default template.\n", part[i] ); 
		EIerrorflag = 1;
		return(NIL);
		}
	if( EIAtype[ia] == 'X' && i < nparts-1 ) {   /* an exec */
		id = EIAxkey[ ia ];
		ia = EIDattrkey[ id ];
		}
	else break;
	}
*typ = EIAtype[ia];
*nslots = i+1;
return( ia );
}


/* ============================================================== */
/* Get the constraint for the specified attr list entry */
/* ============================================================== */
char *
EIgetconstraint( ia )
int ia;
{
return( &EISs[ EIAconkey[ ia ] ] );
}
/* ============================================================== */
/* Get the description for the specified attr list entry */
/* ============================================================== */
char *
EIgetdesc( ia )
int ia;
{
return( &EISs[ EIAdeskey[ ia ] ] );
}

/* ============================================================ */
/* = Get successive attributes of the given procid, in the ==== */
/* ==order given in the template.  Returns a null string ("") when */
/* ==there are no more..                                        */
/* ============================================================ */
char *
EIgetnextattr( procid, ia )
int procid;
int *ia;  /* pointer into attribute list to access the default - returned */
{
static int oldprocid = NIL;
int idef;

if( procid == NIL ) { oldprocid = NIL; return( 0 ); } /* reset */
*ia = NIL;
if( procid != oldprocid ) {
	idef = EIDattrkey[ EIOdefkey[ procid ] ];
	oldprocid = procid;
	}
else 	{
	if( idef == NIL ) return( "" );
	idef = EIAnext[ idef ];
	}
if( idef == NIL ) return( "" );

*ia = idef;
return( &EISs[ EIAnamep[ idef ] ] );
}
/* ============================================================== */
/* Get the value of an attribute.
/* ============================================================== */
EIget( rtn, procid, path, attr )
char rtn[];
int procid;
char path[], attr[];
{
char typ;
int pos, len, stat;
double *dval;
int *ival;
int i;

stat = EIlocate( procid, path, attr, &typ, &pos, &len );
if( stat < 1 ) return( 0 );

if( typ == 'C' || typ == 'X' ) {
	strcpy( rtn, &EISs[ pos ] );
	}
else if( typ == 'I' ) {
	rtn[0] = '\0';
	ival = &EISi[ pos ];
	for( i = 0; i < len; i++ ) { sprintf( rtn, "%s%d ", rtn, *ival ); ival++; }
	if( strlen( rtn ) > 0 ) rtn[ strlen( rtn ) - 1 ] = '\0'; /* remove last blank */
	}
else if( typ == 'D' ) {
	rtn[0] = '\0';
	dval = &EISd[ pos ];
	for( i = 0; i < len; i++ ) { sprintf( rtn, "%s%g ", rtn, *dval ); dval++; }
	if( strlen( rtn ) > 0 ) rtn[ strlen( rtn ) - 1 ] = '\0'; /* remove last blank */
	}
return( 1 );
}

/* ============================================================== */
/* Get the value associated with the specified attr list entry == */
/* Value is in string format. =================================== */
/* ============================================================== */
char *
EIgetval( ia )
int ia;
{
static char rtn[240];
double *dval;
int *ival;
int i;
int pos, len;
char typ;

pos = EIApos[ia];
typ = EIAtype[ia];
len = EIAlen[ia];

if( typ == 'C' || typ == 'X' ) return( &EISs[ pos ] );
else if( typ == 'I' ) {
	rtn[0] = '\0';
	ival = &EISi[ pos ];
	for( i = 0; i < len; i++ ) { sprintf( rtn, "%s%d ", rtn, *ival ); ival++; }
	return( rtn );
	}
else if( typ == 'D' ) {
	rtn[0] = '\0';
	dval = &EISd[ pos ];
	for( i = 0; i < len; i++ ) { sprintf( rtn, "%s%g ", rtn, *dval ); dval++; }
	return( rtn );
	}
}

/* ============================================================ */
/* See if the named attribute exists. Returns 0 if there is no
   such attribute.  Returns 1 if found in user-spec, 2 if inherited,
   3 if coming from default.  */
/* ============================================================ */
EIexists( procid, path, attr )
int procid;
char path[];
char attr[];
{
int len, pos, stat;
char typ;
if( procid < 0 ) return( 0 );

EIsilence( 1 );
stat = EIlocate( procid, path, attr, &typ, &pos, &len );
EIsilence( 0 );
if( stat >= 1 ) return( stat );
else return( 0 );
}


/* ============================================================ */
/* ============================================================ */
/* Given a proc id and attribute name, return a pointer into C list.*/
/* Len (number of characters) is also returned. =============== */
/* Len will be returned as 0 if the attribute is unspecified. */
/* ============================================================ */
char *
EIgetc( procid, path, attr, len )
int procid;
char path[];
char attr[];
int *len;
{
char typ;
int pos;

if( EIlocate( procid, path, attr, &typ, &pos, len ) < 1 ) { EIerrorflag = 1; return( &EISs[0] ); }
if( !Emember( typ, "CX" )) { 
	fprintf( stderr, "Attempt to use EIgetc() for %s %s (type=%c).\n", 
		EIgetobjname( procid ), attr, typ ); 
	EIerrorflag = 1;
	return( &EISs[0] );
	}
(*len)--; /* compensate for null terminator */
return( &EISs[ pos ] );
}

/* ============================================================ */
/* ============================================================ */
/* Given a proc id and attribute name, return an I value. = */
/* Len (number of members) is also returned. ================== */
/* Len will be returned as 0 if the attribute is unspecified. */
/* ============================================================ */
int *
EIgeti( procid, path, attr, len )
int procid;
char path[];
char attr[];
int *len;
{
char typ;
int pos;

if( EIlocate( procid, path, attr, &typ, &pos, len ) < 1 ) { EIerrorflag = 1; return( &EISi[0] ); }
if( typ != 'I' ) { 
	fprintf( stderr, "Attempt to use EIgeti() for %s %s (type=%c).\n", 
		EIgetobjname( procid ), attr, typ ); 
	EIerrorflag = 1;
	return( &EISi[0] );
	}
return( &EISi[ pos ] );
}

/* ============================================================ */
/* ============================================================ */
/* Given a proc id and attribute name, return a pointer into D list. = */
/* Len (number of members) is also returned. ================== */
/* Len will be returned as 0 if the attribute is unspecified. */
/* ============================================================ */
double *
EIgetd( procid, path, attr, len )
int procid;
char path[];
char attr[];
int *len;
{
char typ;
int pos;

if( EIlocate( procid, path, attr, &typ, &pos, len ) < 1 ) { EIerrorflag = 1; return( &EISd[0] ); }
if( typ != 'D' ) { 
	fprintf( stderr, "EIgetd(): wrong type %s %s (type=%c).\n", EIgetobjname( procid ), attr, typ ); 
	EIerrorflag = 1;
	return( &EISd[0] );
	}

return( &EISd[ pos ] );
}


/* ============================================================ */
/* ============================================================ */
/* Algorithm for matching against a user attribute name ======= */
/* Returns 1 if a match, 0 if not ============================= */
/* Coded for speed. =========================================== */
/* ============================================================ */
static
EInamematch( attr, ia )
char attr[];
int ia;   /* points to attribute list entry */
{
int j;
char cmpattr[40];

j = EIAnamep[ ia ];
strcpy( cmpattr, EIgetattrname( ia ));
if( attr[0] != EINP[j][0] ) return( 0 ); /* quick escape */
else if( strcmp( attr, cmpattr ) ==0 ) return( 1 );
return( 0 );
}

/* ============================================================ */
/* ============================================================ */
/* Get an attribute name. */
/* ============================================================ */
char *
EIgetattrname( ia )
int ia;
{
int j;
int nparts;
static char buf[80];
nparts = EIAnparts[ ia ]; 
j = EIAnamep[ ia ];
buf[0] = '\0';
if( nparts == 1 ) return( EINP[j] );
else if( nparts == 2 ) sprintf( buf, "%s/%s", EINP[ j ], EINP[ j+1 ] );
else if( nparts == 3 ) sprintf( buf, "%s/%s/%s", EINP[ j ], EINP[ j+1 ], EINP[ j+2 ] );
else if( nparts == 4 ) sprintf( buf, "%s/%s/%s/%s", EINP[ j ], EINP[ j+1 ], EINP[ j+2 ], EINP[ j+3 ] );
else if( nparts == 5 ) sprintf( buf, "%s/%s/%s/%s/%s", EINP[j], EINP[j+1], EINP[j+2], EINP[j+3], EINP[j+4] );
else if( nparts == NIL ) return( &EISs[ j ] ); /* default attributes */
else { fprintf( stderr, "Internal error, nparts == %d\n", nparts ); EIerrorflag = 1; return( "" ); }
return( buf );
}
/* ============================================================ */
/* ============================================================ */
/* Silence certain error messages. */
/* ============================================================ */
static
EIsilence( mode )
int mode; /* 1 = silence, 0 = normal; */
{
EIsilentflag = mode;
}

/* ============================================================ */
/* ============================================================ */
/* This routine makes it possible to specify an attribute more */
/* than once.  This routine sets the "occurrance" value N.  The */
/* next EImodattr or EIlocate call will then put/get the Nth   */
/* occurance of the specified attribute. ===================== */
/* The application must keep track of setting the occurrence, */
/* and resetting to 1.

EIsetoccur( n )
int n;
{
EIoccur = n;
}

/* ============================================================ */
/* ============================================================ */
/* This routine allows an application to control the amount of list space */
/* allocated during the subsequent EImodattr call.  It will be automatically */
/* reset in EImodattr(). */
EIsetalloc( n )
int n;
{
EImustalloc = n;
}


/* ============================================================ */
/* ============================================================ */
/* Given a proc id and attribute name, find DATA location info.  */
/* Low level routine, should never be called by application. */
/* Attributes must be requested in the order given in the template. */
/* Returns 1 if found in user list, 2 if found in inheritance, 3 */
/* if found in defaults. ======================================= */
/* Returns 0 on fail. ========================================== */
/* ============================================================ */
/* ============================================================ */
static 
EIlocate( procid, path, aname, typ, pos, len )
int procid;
char path[];
char aname[];
char *typ;
int *pos;
int *len;
{
int i, j, k, ipp;
int nmatches;
char attr[ANAMELEN];
int iflag;

*len = 0;

if( procid < 0 ) { fprintf( stderr, "EIlocate: object does not exist.\n" ); return(0); }

/* build attr name */
if( strlen( path ) > 0 ) {
	if( path[0] == '/' )sprintf( attr, "%s/%s", &path[1], aname );
	else sprintf( attr, "%s/%s", path, aname );
	}
else strcpy( attr, aname );
if( attr[0] == '/' ) 
	{ fprintf( stderr, "%s: unrecognized attribute.\n", attr ); EIerrorflag = 1; return(0); }


/* see if the requested attribute was specified by the user.. */
EIwhere = 1;
i = EIOattrkey[procid];
nmatches = 0;
while( i != NIL ) {
	if( EInamematch( attr, i ) == 1 ) nmatches++;
	if( nmatches >= EIoccur ) { EIulookups++; break; }
	i = EIAnext[i];
	}


/* check the defaults.. */
if( i == NIL ) {
	EIwhere = 3;
	i = EIgetdefaultvalue( procid, attr );
	}

/* its not there.. */
if( i == NIL ) { EIwhere = NIL; return( 0 ); }

/* if attribute is inheritable and there is an inheritable proc, check it.. */
/* if( EIwhere == 1 ) iflag = EIAdef[ EIAdef[i] ]; */
/* else if( EIwhere == 3 ) iflag = EIAdef[i]; */

if( EIwhere == 3 ) {
	iflag = EIAdef[i];
	
	if( iflag % 2 != 1 && EIOinheritid[procid] != NIL ) { 
		ipp = procid;
		j = NIL;
		for( k = 0; k < 4; k++ ) {
			if( j == NIL && EIOinheritid[ipp] != NIL ) {
				EIwhere = 2;
				ipp = EIOinheritid[ ipp ];
				j = EIOattrkey[ ipp ];
				nmatches = 0;
				while( j != NIL ) {
					if( EInamematch( attr, j ) == 1 ) nmatches++;
					if( nmatches >= EIoccur ) { EIinhlookups++; break; }
					else j = EIAnext[j];
					}
				}
			else break;
			}
		if( k >= 4 ) {
			fprintf( stderr, "Too many inheritances (%s %s).\n", EIgetobjname( procid ), attr );
			EIerrorflag = 1;
			}
	
		if( j != NIL ) i = j;  /* put location back into i.. */
		}
	}



*typ = EIAtype[i];
*pos = EIApos[i];
*len = EIAlen[i];
return( EIwhere );
}
/* ============================================================ */
/* ============================================================ */
/* query to get the value of Ewhere (where the last EIlocate found its value) */
EIwhere_was_it()
{
return( EIwhere );
}

/* ============================================================ */
/* == Get an attribute's value from the defaults. ============= */
/* == Low level routine-- should never be called by an application. */
/* == Handles multi-part attribute names. ===================== */
/* == Tries to increase efficiency by keeping a pointer for === */
/* == each part level. ======================================== */
/* ============================================================ */
static
EIgetdefaultvalue( procid, attr )  
int procid;	     
char attr[];
{
int i, k, ix, ip;
int nparts;
char part[5][20];
int div[5];
int nsearch;
int override_exists = 0;
static int prev_idef[5];
static int prevkey[5] = { NIL, NIL, NIL, NIL, NIL };
static int second_try[5];
static int oldprocid = NIL;
int idef;
int dkey;

EIdeflookups++;

/* change of proc, re-initialize.. */
if( procid != oldprocid ) {
	for( i = 0; i < 5; i++ ) prevkey[i] = NIL;
	oldprocid = procid;
	}

/* break up attribute name on dash (/) */
ix = 0;
for( i = 0; i < 5; i++ ) {
	if( ix >= strlen( attr )) break;
	div[i] = ix;
	Egetchunk( part[i], attr, &ix, "/" ); 
	}
nparts = i;

for( i = 0; i < 5; i++ )second_try[i] = 0;
ip = procid;
for( i = 0; i < nparts; i++ ) {

	SEARCH:
	if( i == 0 ) { /* level 0 */
		dkey = EIOdefkey[ip];
		override_exists = EIDoverrideflag[ dkey ]; 
		if( EIDattrkey[ dkey ] == NIL ) idef = NIL;
		else if( prevkey[i] != EIDattrkey[ dkey ] || override_exists ) { 
			/* we've started a new proc */
			idef = EIDattrkey[ dkey ]; 
			prevkey[i] = idef;
			}
		else idef = prev_idef[i];
		}
	else 	{     /* called procs */
		override_exists = EIDoverrideflag[ ip ]; 
		if( EIDattrkey[ ip ] == NIL ) idef = NIL;
		else if( prevkey[i] != EIDattrkey[ ip ] || override_exists ) {  
			/* we've started a new sub-proc in the tree walk */
			idef = EIDattrkey[ ip ]; 
			prevkey[i] = idef;
			}
		else idef = prev_idef[i];
		}

	nsearch = 0;
	while( idef != NIL ) {
		nsearch++;
		if( EISs[EIAnamep[idef]] == part[i][0] ) {     /* opt */
			if( override_exists ) {
				if( strcmp( &EISs[EIAnamep[ idef ] ], &attr[ div[i] ] )==0 ) break;
				}
			if( strcmp( &EISs[ EIAnamep[ idef ] ], part[i] )== 0 ) break;
			}
		idef = EIAnext[ idef ];
		}
	if( idef == NIL ) {
		if( ! second_try[i] ) {
			second_try[i] = 1;
			prevkey[i] = NIL;
			/* fprintf( stderr, "[r:%s]", attr ); */
			EIdeflookupfaults++;
			goto SEARCH;   /* indicates an 'out-of-sequence request' */
			}
		if( EIoccur == 1 && !EIsilentflag ) {
			fprintf( stderr, "Proc %s: default attribute (%s) not found.\n", 
				EIgetobjname( procid ), attr );
			EIerrorflag = 1;
			}
		second_try[i] = 0;
		return( NIL );
		}

	/* if we reach here, we found it.. */
	prev_idef[i] = idef; /* save pointer */
	second_try[i] = 0;

	/* see if we need to check another default.. */
	if( EIAtype[ idef ] == 'X' ) {
		ip = EIAxkey[ idef ];
		continue;
		}
	else break;
	}

return( idef );
}

/* ============================================================ */
/* Return number of user procs to run. ======================== */
/* ============================================================ */
/* ============================================================ */
EIgetn()
{
return( EIOn );
}

/* ============================================================ */
/* ============================================================ */
/* Walk through the order list (upon successive calls) ======== */
/* Must be initialized before use and "closed" after use.. */
/*   mode     effect
   -------  ------------
E_AT_BEGINNING      0    initialize at beginning of list 
E_NEXT      	    1    return next entry
E_AT_END            2    initialize at end of list
E_PREVIOUS         -1    return previous entry
E_DONE              3    finished (necessary to prevent collisions)
 */
/* ============================================================ */
EIgetnextobj( mode, c )
int mode;
int c; /* channel */
{
static int pointer[5] = { 0, 0, 0, 0, 0 };
static int flag[5] = { 0, 0, 0, 0, 0 };  /* 0 = off, 1 = on */

/* check for collisions */
if( ( mode == E_AT_BEGINNING || mode == E_AT_END ) && flag[c] == 1 ) /* collision! */
	fprintf( stderr, "Collision on EIgetnextobj(), channel %d.\n", c );

if( mode == E_AT_BEGINNING ) { pointer[c] = -1; flag[c] = 1; return( NIL ); }
else if( mode == E_AT_END ) { pointer[c] = EIOn; flag[c] = 1; return( NIL ); }
else if( mode == E_DONE ) { flag[c] = 0; return( NIL ); }
while( 1 ) {
	if( mode == 1 ) { 
		pointer[c]++; 
		if( pointer[c] >= EIOn ) { return( NIL ); } 
		}
	else if( mode == -1 ) { 
		pointer[c]--; 
		if( pointer[c] < 0 ) { return( NIL ); } 
		}

	if( EIorder[ pointer[c] ] >= 0 ) break;
	}
return( EIorder[ pointer[c] ] );
}

/* ============================================================ */
/* Mark a proc as deleted by negating it in the order list ==== */
/* ============================================================ */
/* ============================================================ */
EIdelete( procid )
int procid;
{
int i;
for( i = 0; i < EIOn; i++ ) {
	if( EIorder[i] == procid ) {
		EIorder[i] = NIL;
		return( 1 );
		}
	}
fprintf( stderr, "can't find %d for delete.\n", procid );
EIerrorflag = 1;
}


/* ============================================================ */
/* given a proc id, return proc name. */
/* Should really be called EIgetobjecttype() */
/* ============================================================ */
/* ============================================================ */
char *
EIgetobjname( procid )
int procid;
{
int i;
if( procid < 0 || procid >= EIOn ) return( "" );
else return( &EISs[ EIOname[procid] ] );
}

/* ============================================================ */
/* given a proc id, return package name. */
/* ============================================================ */
/* ============================================================ */
char *
EIgetpackagename( procid )
int procid;
{
int i;
if( procid < 0 || procid >= EIOn ) return( "" );
else return( &EISs[ EIPname[ EIDpack[ EIOdefkey[ procid ] ] ] ] );
}

/* ============================================================ */
/* ============================================================ */
/* Write out all procs in order 				*/
/* ============================================================ */
EIwriteall( fnm, mode )
char fnm[];
int mode; /* 0 for user dump, 2 for everything.. 3 everything but popups */
{
FILE *fp;
int ip, len;
char *tag;

fp = fopen( fnm, "w" );
if( fp == NULL ) { fprintf( stderr, "cannot open %s for writing.\n", fnm ); exit(); }
EIgetnextobj( E_AT_BEGINNING, 0 );
while( 1 ) {
	ip = EIgetnextobj( E_NEXT, 0 );
	if( ip < 0 ) break;

	tag = EIgetc( ip, "", "*/tag", &len );
	if( mode == 3 && len > 0 && tag[0] == '_' ) continue; /* skip popups */

	/* write the object */
	EIwriteobj( ip, fp, mode );
	}
EIgetnextobj( E_DONE, 0 );
fclose( fp );
chmod( fnm, 00600 ); /* change file protection */
}


/* ============================================================ */
/* ============================================================ */
/* Write a proc to the given file descriptor. ================= */
/* Returns 1 on success, 0 on fail. =========================== */
/* ============================================================ */
EIwriteobj( procid, fp, mode )
int procid;
FILE *fp;
int mode; /* 0 for user dump, 1 for gc, 2 for all.. */
{
int ip, i, j;
int *ival;
double *dval;
char *name;

if( fp == NULL ) { 
	fprintf( stderr, "Null file descriptor.\n" ); 
	EIerrorflag = 1;
	return( 0 ); 
	}
	
ip = procid;
i = EIOattrkey[ ip ];

fprintf( fp, "\nProc %s\n", EIgetobjname( procid ) );
while( i != NIL ) {
	name = EIgetattrname( i );

	/* supress silent attributes if writing a user dump.. */
	if( mode >= 1 || EIAdef[ EIAdef[ i ] ] < 2 ) {
		fprintf( fp, "%s: ", name );
		if( Emember ( EIAtype[i], "CX" )) {
			/* enclose in quotes */
			fprintf( fp, "\"" );
			for( j = EIApos[i]; j < EIApos[i]+EIAlen[i]; j++ ) {
				if( EISs[j] == '\0' ) break;
				if( EISs[j] == '\n' ) fprintf( fp, "\"\n\"" );
				else fprintf( fp, "%c", EISs[j] );
				}
			fprintf( fp, "\"" );
			}
		else if( EIAtype[i] == 'I' ) {
			ival = &EISi[ EIApos[i] ];
			for( j = 0; j < EIAlen[i]; j++ ) { fprintf( fp, "%d ", *ival ); ival++; }
			}
		else if( EIAtype[i] == 'D' ) {
			dval = &EISd[ EIApos[i] ];
			for( j = 0; j < EIAlen[i]; j++ ) { fprintf( fp, "%g ", *dval ); dval++; }
			}
		fprintf( fp, "\n" );
		}
	i = EIAnext[i];
	}
return( 1 );
}


/* ==== read a defaults file ================================== */
/* ============================================================ */
/* ==== Return 1 on success, 0 on failure. ==================== */
/* ============================================================ */
/* Defaults file format:
 *  @@begin <objname>
 *  <attr-name> <attr-type> [attr-length] : <default-val> : <attr-constraint> : <description>
 *  ...
 *  @@end
 */
EIreaddeflist( fnm )
char fnm[];
{
FILE *fp;
char buf[512];
int state;
int ix;
char onm[50];
char attr[50];
char constraint[40];
char typ[4];
char lenstr[40];
char val[200];  
char str[300];
int len;
int na;
int lasta;
int itmp;
int i;
int n;
int pack;
char flags[12];
int iflag;

fp = fopen( fnm, "r" );
if( fp == NULL ) { 
	fprintf( stderr, "Cannot open defaults list %s.\n", fnm ); 
	EIerrorflag = 1;
	return( 0 ); 
	}

state = 0;
pack = NIL;
EILineno = 0;
EIerrorflag = 0;
while( fgets( buf, 512, fp ) != NULL ) {
	EILineno++;
	if( state == 0 ) {      /* looking for @@package or @@begin */
		if( strncmp( buf, "@@package", 9 )==0 ) {
			sscanf( buf, "%*s %s", onm );
			for( i = 0; i < EIPn; i++ ) if( strcmp( onm, &EISs[EIPname[i]] )==0 )break;
			if( i == EIPn ) {
				EIPname[i] = EIaddss( onm );
				EIPn++;
				}
			pack = i;
			}
		else if( strncmp( buf, "@@begin", 7 )==0 ) {  /* get a def obj name */
			state = 1; 
			sscanf( buf, "%*s %s", onm );
			for( i = 0; i < EIDn; i++ ) if( strcmp( onm, &EISs[EIDname[i]] )==0 ) break;
			if( i != EIDn ) continue;
			EIDname[ EIDn ] = EIaddss( onm );
			EIDattrkey[ EIDn ] = NIL;
			EIDpack[ EIDn ] = pack;
			EIDoverrideflag[ EIDn ] = 0;
			EIDn++;
			na = 0;
			}
		continue;
		}
	
	else if( state == 1 ) {   /* getting attributes, and looking for @@end */
		if( strncmp( buf, "@@end", 5 )==0 ) { state = 0; continue; }
		else if( buf[0] == '#' ) continue; /* comment */

		ix = 0;
		Egetchunk( str, buf, &ix, ":" );

		if( ix >= strlen( buf ) ) continue; /* blank line.. */

		/* bad continuation line or some other problem.. */
		if( strlen( str ) > 60 ) { 
			fprintf( stderr, "%3d: illegal text\n", EILineno ); 
			EIerrorflag = 1;
			continue; 
			}

		n = sscanf( str, "%s %s %s", attr, typ, lenstr );

		EIAxkey[ EIAavail ] = NIL;
		if( typ[0] == 'X' ) { /* external proc include.. */
			len = 1;
			/* lookup the referenced proc, which is in lenstr */
			for( i = 0; i < EIDn; i++ ) if( strcmp( &EISs[ EIDname[i] ], lenstr )==0 ) break;
			if( i == EIDn ) { 
				fprintf( stderr, "%3d: Referenced Proc (%s) not defined.\n", EILineno, lenstr ); 
				EIerrorflag = 1;
				continue;
				}
			EIAxkey[ EIAavail ] = i; /* save handle on default for future quick access */
			}

		else if( n == 2 ) len = 1;
		else if( n == 3 )len = atoi( lenstr );
		else { 
			fprintf( stderr, "%3d: attribute name and type required.\n", EILineno ); 
			EIerrorflag = 1;
			continue; 
			}

		/* add room for null terminator.. */
		if( Emember( typ[0], "CX" )) len++;
	
		Egetchunk( val, buf, &ix, ":" );
		Estrip_ws( val );

		if( Emember( '/', attr )) { /* an override, no constraint or description.. */
			constraint[0] = '\0';
			str[0] = '\0';
			EIDoverrideflag[ EIDn-1 ] = 1;
			/* note-- override names will be stored in the string pool */
			}

		else	{		    /* everything else.. */

			/* flag */
			Egetchunk( flags, buf, &ix, ":" );
			iflag = 0;
			if( Emember( 'U', flags )) iflag += 1;
			if( Emember( 'H', flags )) iflag += 2;

			/* constraint */
			Egetchunk( constraint, buf, &ix, ":" );
			Estrip_ws( constraint );
			if( strlen( constraint ) < 1 ) { 
				fprintf( stderr, "%3d: missing constraint field\n", EILineno ); 
				EIerrorflag = 1;
				continue; 
				}

			/* description */
			Egetchunk( str, buf, &ix, ":" );
			Estrip_ws( str );
			if( strlen( str ) < 1 ) {
				fprintf( stderr, "%3d: missing description field\n", EILineno ); 
				EIerrorflag = 1;
				continue; 
				}
			}

		/* continuation desc lines.. */
		if( str[ strlen( str )-1 ] == '\\' ) {
			str[ strlen( str )-1 ] = '\0'; /* get rid of backslash */
			while( 1 ) {
				fgets( buf, 512, fp );
				EILineno++;
				Estrip_ws( buf );
				sprintf( str, "%s %s", str, buf );
				if( buf[ strlen( buf ) - 1 ] != '\\' ) break;
				else str[ strlen( str )-1 ] = '\0'; /* get rid of backslash */
				}
			}
		
		itmp = EIAavail;
		if( na == 0 ) {
			EIDattrkey[ EIDn-1 ] = EIAavail; 
			EIadda( NIL, attr, typ[0], len, constraint, val, str, NIL, NULL, NIL, iflag ); 
			}
		else EIadda( NIL, attr, typ[0], len, constraint, val, str, lasta, NULL, NIL, iflag );
		na++;
		lasta = itmp;
		}
	}
fclose( fp );
if( pack < 0 ) { fprintf( stderr, "No @@package declaration was found.\n" ); EIerrorflag = 1; }
else if( EIDn < 1 ) { fprintf( stderr, "Expecting @@begin.  No objects found in file.\n" ); EIerrorflag = 1; }
else if ( state == 1 ) { fprintf( stderr, "Missing @@end.\n" ); EIerrorflag = 1; }
if( EIerrorflag ) return( 0 );
else return( 1 );
}

/* ==== read a user file ============================= */
/* =================================================== */
/* ==== Returns 1 on success, 0 on failure. ========== */
/* =================================================== */
/* Syntax is Proc <procname> 
   	     <attr-name> : <attr-value>
	     <attr-name> : <attr-value>
	     ..
   We also need to handle old-style files which have just procname.
 */
EIreaduserlist( s )
char s[];
{
char buf[512];
int state;
int firstproc;
int firstattr;
char str[200];
char procname[200];
char val[512];
int nprms, id;
double tx, ty, scx, scy;
int len;
int itmp, lasta;
int ip; /* always points to the current proc.. */
int i;
char typ;
int found;
int ix;
int nameslots[5], nslots;
int idef;
int sourceflag; /* 0=file   1=in the string s */
int readix; 
FILE *fp;
int stat;

/* inheritflag = 0; */

firstproc = 1;
firstattr = 1;
EILineno = 0;
EIerrorflag = 0;

/* try to open s as a filename.. */
fp = NULL;
if( strlen( s ) < 80 ) fp = fopen( s, "r" );
if( fp != NULL ) sourceflag = 0;
else sourceflag = 1;

readix = 0;

	

while( 1 ) {
	EILineno++;
	str[0] = '\0'; procname[0] = '\0'; /* null out tokens */

	if( sourceflag == 0 ) {
		if( fgets( buf, 512, fp ) == NULL ) strcpy( str, "Proc" ); /* indicate that we're done.. */
		buf[ strlen( buf ) - 1 ] = '\0'; /* strip off newline.. */
		}
	else if( sourceflag == 1 ) {
		if( readix >= strlen( s ) ) strcpy( str, "Proc" );  /* indicate that we're done.. */
		Egetchunk( buf, s, &readix, "\n" );
		}

	if( str[0] == '\0' ) { /* everything but the end.. */
		nprms = sscanf( buf, "%s %s", str, procname ); /* check first two tokens.. */
		if( nprms < 1 ) continue;
		}

	if( str[0] == '#' ) continue;  /* comments */

	if( strcmp( str, "Proc" )==0 ) {  
		if( procname[0] == '\0' ) {
			if( EIOn < 1 ) { 
				fprintf( stderr, "No objects were found.\n" ); 
				EIerrorflag = 1;
				}
			if( sourceflag == 0 ) fclose( fp );
			if( EIerrorflag != 0 ) return( 0 );  /* normal exit */
			else return( 1 );  /* normal exit */
			}

		/* add a new proc */
		/* ============== */
		/* strip off colon if there is one.. */
		if( procname[ strlen( procname )-1 ] == ':' ) procname[ strlen( procname )-1 ] = '\0';
		ip = EIcreate( procname );
		firstproc = 0;
		firstattr = 1;
		}

	else if( !firstproc ) { 	

		/* add an attribute */
		/* ================ */
		ix = 0;
		Egetchunk( str, buf, &ix, ":" );
		Estrip_ws( str );
		len = strlen( str );

		/* its a "tag: value" line.. */
		if( str[0] != '"' && str[len-1] != '\\' && ix != strlen( buf ) ) { 

			strcpy( val, &buf[ix+1] );
	
			Estrip_ws( val ); /* get rid of leading & trailing white space */

			/* get rid of leading and trailing quote.. */
			if( val[ strlen( val ) -1 ] == '"' ) val[ strlen( val ) -1 ] = '\0';
			if( val[0] == '"' ) {
				strcpy( val, &val[1] );
				}

			/* take special action if it is a "built-in" attribute.. */
			if( strcmp( str, "*/inherit" )==0 ) EIaddinherit( val, ip );
			
			EIaddalways = 1; /* for the sake of multiple occurrences */
			stat = EImodattr( ip, "", str, val );
			if( stat != 1 ) fprintf( stderr, "Error in line %d of control file.\n",
						EILineno );	
			EIaddalways = 0;
			
			}	
		else 	{
			Estrip_ws( buf );
			EIaddcontline( buf );
			}
		firstattr = 0;
		}
	}
}

/* =========================================================== */
/* =========================================================== */
/* Create a user instance of an object.  ===================== */
/* Returns procid. =========================================== */
/* =========================================================== */

EIcreate( procname )
char procname[];
{
int i, ip;
char tag[20];

ip = EIOn;
/* verify proc name and get index of first default attribute.. */
for( i = 0; i < EIDn; i++ ) if( strcmp( procname, &EISs[EIDname[i]] )==0 ) break;
if( i == EIDn ) { 
	fprintf( stderr, "%3d: Unrecognized Proc name (%s).\n", EILineno, procname );
	EIerrorflag = 1;
	return( 0 ); 
	}
else EIOdefkey[ip] = i;

EIOname[ip] = EIaddss( procname );   	/* save the name */

EIOinheritid[ip] = NIL; /* initialize */
EIOattrkey[ip] = NIL; 
EIorder[ip] = ip;
EIOn++;

return( ip );
}

/* =========================================================== */
/* =========================================================== */
/* Copy a user object, creating a new instance. ============== */
/* Returns the new procid. =================================== */
/* =========================================================== */

EIcopy( procid )
int procid;
{
int i, newprocid;
char attr[40];

newprocid = EIcreate( &EISs[ EIOname[ procid ]] );
i = EIOattrkey[ procid ];
while( i != NIL ) {
	strcpy( attr, EIgetattrname( i ));
	EImodattr( newprocid, "", attr, EIgetval( i ) );
	if( Esmember( attr, "Clone */inherit", " " )) {
		EIaddinherit( EIgetval( i ), newprocid );
		}
	i = EIAnext[i];
	}

return( newprocid );
}

/* ============================================================ */
/* ============================================================ */
/* Add an inheritance relationship */
/* ============================================================ */
static
EIaddinherit( val, ip )
char val[];
int ip;
{
int itag;

itag = EItaglookup( val, ip );
if( itag == NIL ) { 
	fprintf( stderr, "%3d: inherited Proc (%s) not recognized.\n", 
		EILineno, val );
	EIerrorflag = 1;
	}
else EIOinheritid[ip] = itag;
}

/* ============================================================ */
/* Get the proc_id associated with the tag.  Starts with current */
/* procid (ip), and searches backward through the order list until a */
/* match is found.  If ip is NIL, scanning just goes through list linearly. */
/* Returns the proc_id, or NIL if none found. */
EItaglookup( tag, ip )
char tag[];
int ip;       /* current procid */
{
int  itag, len;
char *s;

/* simple linear scan.. */
if( ip == NIL ) {
	EIgetnextobj( E_AT_BEGINNING, 3 );
	while( 1 ) {
		itag = EIgetnextobj( E_NEXT, 3 ); 
		if( itag == NIL ) { EIgetnextobj( E_DONE, 3 ); return( NIL ); }
		if( EIexists( itag, "", "*/tag" ) == 1 ) { /* must be in user-spec, not inherited. */
			s = EIgetc( itag, "", "*/tag", &len );
			if( len > 0 ) {
				if( strcmp( s, tag )==0 ) break;
				}
			}
		}
	EIgetnextobj( E_DONE, 3 );
	return( itag );
	}

/* scan from current proc backward.. */
EIgetnextobj( E_AT_END, 3 );
while( 1 ) {
	itag = EIgetnextobj( E_PREVIOUS, 3 );
	if( itag == NIL ) { EIgetnextobj( E_DONE, 3 ); return( NIL ); }
	else if( itag == ip ) break;
	}
while( 1 ) {
	itag = EIgetnextobj( E_PREVIOUS, 3 ); /* now, search back until the desired tag is found.. */
	if( itag == NIL ) { EIgetnextobj( E_DONE, 3 ); return( NIL ); }
	if( EIexists( itag, "", "*/tag" ) == 1 ) { /* must be in user-spec, not inherited. */
		s = EIgetc( itag, "", "*/tag", &len );
		if( len > 0 ) {
			if( strcmp( s, tag )==0 ) break;
			}
		}
	}
EIgetnextobj( E_DONE, 3 );
return( itag );
}


/* ============================================================ */
/* ============================================================ */
/* Set an attribute to a value. =============================== */
/* Returns 1 on success, 0 on fail. =========================== */
/* ============================================================ */
EImodattr( procid, path, aname, val )
int procid;    /* the proc id */
char path[];   /* attribute path */
char aname[];   /* attribute name */
char val[];    /* the new value */
{
int ip, i, lasta, status, idef, alloc, len, itmp, stat, dogc;
char typ;
int nameslots[5], nslots;
char attr[ ANAMELEN ];
int nmatches;
int gc_once = 0;
char gcfnm[80];

RETRY:

if( procid < 0 ) { fprintf( stderr, "Unrecognized tag." ); return( 0 ); }

/* build attr name */
if( strlen( path ) > 0 ) {
	if( path[0] == '/' )sprintf( attr, "%s/%s", &path[1], aname );
	else sprintf( attr, "%s/%s", path, aname );
	}
else strcpy( attr, aname );

ip = procid;
i = EIOattrkey[ip];
lasta = NIL;

/* look up attribute in default list to get type and allocation */
idef = EIgetinfo( EIOdefkey[ip], attr, &typ, nameslots, &nslots );
if( idef == NIL ) {
  	/* fprintf( stderr, "Unrecognized attribute (%s) in %s.\n", attr, EIgetobjname( procid ) ); */
	EIerrorflag = 1;
  	return( 0 );
  	}
alloc = EIAalloc[ idef ];

/* find length */
if( Emember( typ, "CX" ) ) len = strlen( val ) +1;
else { len = Ecounttokens( val ); }
		
/* check for full list */
dogc = 0;
if( Emember( typ, "CX" ) && (EISsn + len) > STRINGPOOLLEN ) dogc = 1;
else if( typ == 'D' && (EISdn + len) > DLISTLEN ) dogc = 1;
else if( typ == 'I' && (EISin + len) > ILISTLEN ) dogc = 1;
if( dogc ) { 
	if( gc_once ) {
		fprintf( stderr, "Internal capacity exceeded even after gc.\n\
	Type: %c; Length: %d; String: %d; Float: %d; Int: %d.\n", typ, len, EISsn, EISdn, EISin );
		return( 0 );
		}
	
	fprintf( stderr, "Garbage collection.." );
	/* do garbage collection and retry.. */
	sprintf( gcfnm, "%szoodgc%05d", TMP_DIR, getpid() );
	/* EIwriteall( "/usr/tmp/zood.gc", 2 ); */
	EIwriteall( gcfnm, 2 );
	EIinit();
	EIreaduserlist( gcfnm );
	fprintf( stderr, "completed.\n" );
	/* unlink( gcfnm ); */
	gc_once = 1;
	goto RETRY;
	}


/* look up attribute in user list */
nmatches = 0;
while( i != NIL ) {
	if( EInamematch( attr, i )) nmatches++;
	if( nmatches >= EIoccur && !EIaddalways ) break;
	else {
		lasta = i;
		i = EIAnext[ i ];
		}
	}

if( EImustalloc != NIL ) { len = EImustalloc; EImustalloc = NIL; }

if( i == NIL ) { /* if not found we will add it to the end of the user attribute chain.. */
	len = alloc; 
	if( lasta == NIL ) {
		EIOattrkey[ ip ] = EIAavail; 
		status = EIadda( NIL, attr, typ, len, "", val, "", NIL, nameslots, nslots, idef );
		}
	else status = EIadda( NIL, attr, typ, len, "", val, "", lasta, nameslots, nslots, idef );
	return( status );
	}
else if( len > EIAalloc[i] ) {  /* or, if new length is too big we will have to add it and link it in.. */
	/* fprintf( stderr, "." ); fflush( stderr ); */
	if( lasta == NIL ) {
		itmp = EIOattrkey[ip];
		EIOattrkey[ip] = EIAavail;
		status = EIadda( NIL, attr, typ, len, "", val, "", NIL, nameslots, nslots, idef );
		EIAnext[ EIAavail-1 ] = itmp; /* preserve the 'next' pointer */
		}
	else	{
		status = EIadda( NIL, attr, typ, len, "", val, "", lasta, nameslots, nslots, idef );
		EIAnext[ EIAavail-1 ] = EIAnext[i];
		/* return( status );  */ /* moved down two lines.. scg 11-17-95 */
		}
	return( status ); 
	}

/* or, just modify the data.. */
status = EIadda( i, attr, typ, 0, "", val, "", NIL, nameslots, nslots, idef );
return( status );
}

/* =================================================== */
/* =================================================== */
/* Low-level routine to add info to the database. ==== */
/* Used to add both defaults and user attributes. ==== */
/* Should never be called by an application. ========= */
/* =================================================== */
static
EIadda( slot, attr, typ, alloc, constraint, val, desc, lasta, nameslots, nslots, idef )
int slot;          /* NIL = insert,   else update into this slot */
char attr[];       /* attribute name */
char typ;	   /* attribute storage type */
int alloc;         /* (insert mode only).  If > 0, this many slots will be reserved in the storage list.
				For user values this will generally be the alloc of the default entry.
				If not supplied (passed as 0), storage reservation is based on the value. */
char constraint[]; /* required for defaults only.. */
char val[];	   /* always requred */
char desc[];       /* required for defaults only.. */
int lasta; 	   /* previous attribute in chain.. */
int nameslots[];   
int nslots;
int idef;	   /* attributes only-- points to the default entry */
{
int i, ia, n, k, ix;
char str[100];
double atof();
int isnum;

if( slot == NIL ) { /* inserting.. */
	/* add attribute to attr-list */
	ia = EIAavail;
	if( lasta != NIL ) EIAnext[lasta] = ia;
	/* process the attribute name */
	if( nslots == NIL ) EIAnamep[ia] = EIaddss( attr ); /* default */
	else if( nslots == 0 ) { fprintf( stderr, "Internal error. nslots == %d\n", nslots ); return( 0 ); }
	else	{   /* add to name part list */
		EIAnamep[ia] = EINPn;
		for( i = 0; i < nslots; i++ ) EINP[ EINPn++ ] = &EISs[ EIAnamep[ nameslots[i] ] ]; 
		if( EINPn > MAXNP ) fprintf( stderr, "Name part list overflow.\n" ); 
		}
	EIAnparts[ia] = nslots;
	EIAtype[ia] = typ;
	EIAdef[ia] = idef;
	EIAnext[ia] = NIL;
	EIAavail++;
	}
else	{
	ia = slot;
	}

EIcontplace = NIL;

/* put constraint into string pool.. */
if( slot == NIL && strlen( constraint ) > 0 ) EIAconkey[ia] = EIaddss( constraint );
else EIAconkey[ia] = NIL;


/* put descriptor into string pool.. */
if( slot == NIL && strlen( desc ) > 0 ) EIAdeskey[ia] = EIaddss( desc );
else EIAdeskey[ia] = NIL;

/* put val into one of the lists.. */

/* string */
if( typ == 'C' || typ == 'X' ) { 
	/* a null string ("") will go into the string list and occupy one spot. */
	/* we can't use the utility routine because other stuff is going on.. */
	if( slot == NIL ) { i = EISsn; EIApos[ia] = i; }
	else i = EIApos[ia];

	strcpy( &EISs[i], val );
	n = strlen( val ) + 1;
	EIAlen[ia] = n;
	if( slot == NIL ) { /* we're inserting.. */
		if( n > alloc ) EIAalloc[ia] = n;
		else EIAalloc[ia] = alloc;
		EIcontplace = EISsn + n; /* remember exact end of string in case there's a continuation.. */
		EISsn += EIAalloc[ia];
		}
	if( EISsn > STRINGPOOLLEN ) { 
		fprintf( stderr, "%3d: String list overflow.\n", EILineno ); 
		EIerrorflag = 1;
		return( 0 ); 
		}
	}	

/* int */
else if( typ == 'I' ) {
	if( slot == NIL ) { i = EISin; EIApos[ia] = i; }
	else i = EIApos[ia];
	ix = 0;
	for( n = 0; ; n++ ) {
		if( ix >= strlen( val )) break;
		Egetchunk( str, val, &ix, " 	," );
		if( str[0] == '\0' ) break;
		EISi[i+n] = atoi( str );
		}
	EIAlen[ia] = n;
	if( slot == NIL ) { /* we're inserting.. */
		if( n > alloc ) EIAalloc[ia] = n;
		else if( n < 1 && alloc < 1 ) EIAalloc[ia] = 1;
		else EIAalloc[ia] = alloc;
		EISin += EIAalloc[ia];
		}
	if( EISin > ILISTLEN ) { 
		fprintf( stderr, "%3d: Int list overflow.\n", EILineno ); 
		EIerrorflag = 1;
		return( 0 ); 
		}
	}

/* double */
else if( typ == 'D' ) {
	if( slot == NIL ) { i = EISdn; EIApos[ia] = i; }
	else i = EIApos[ia];
	ix = 0;
	for( n = 0; ; n++ ) {
		if( ix >= strlen( val )) break;
		Egetchunk( str, val, &ix, " 	," );
		if( str[0] == '\0' ) break;
		EISd[i+n] = atof( str );
		}
	EIAlen[ia] = n;
	if( slot == NIL ) { /* we're inserting.. */
		if( n > alloc ) EIAalloc[ia] = n;
		else if( n < 1 && alloc < 1 ) EIAalloc[ia] = 1;
		else EIAalloc[ia] = alloc;
		EISdn += EIAalloc[ia];
		}
	if( EISdn > DLISTLEN ) { 
		fprintf( stderr, "%3d: Double list overflow.\n", EILineno ); 
		EIerrorflag = 1;
		return( 0 ); 
		}
	}


return( 1 );
}

/* ========================================================== */
/* ==== add a continuation line to the previous attribute..== */
/* ==== Only when adding new attributes (not modifying existing ones) */
/* ==== C type only..  Must be called IMMEDIATELY after first row is added. */
/* ============================================================ */
EIaddcontline( line )
char line[];	/* continuation line to be appended to string.. */
{
int ia;
int len;

ia = EIAavail - 1;
if( ! Emember( EIAtype[ia], "CX" )) return(0); /* continuations for char only */

if( EIcontplace == NIL ) { 
	fprintf( stderr, "%3d. Illegal continuation.", EILineno );
	EIerrorflag = 1;
	return( 0 );
	}

/* strip off trailing quote which may be used to maintain white space in text */
if( line[ strlen( line ) - 1 ] ==  '"' ) line[ strlen( line )-1 ] = '\0';

/* handle the case of empty first line */
if( (EIcontplace-1) == EIApos[ia] ) EIApos[ia]++;
else EISs[ EIcontplace -1 ] = '\n';

/* strip off leading quote */
if( line[0] == '"' ) strcpy( line, &line[1] );

/* add the string */
if( strlen( line ) == 0 ) strcpy( line, " " );
len = strlen( line ) + 1;
strcpy( &EISs[ EIcontplace ], line );

EIcontplace += len;
EISsn += len; 
EIAlen[ia] += len; 
if( EIAlen[ia] > EIAalloc[ia] ) EIAalloc[ia] = EIAlen[ia];  /* update storage list reservation..*/
}

/* ========================================================= */
/* add a static string to the string pool. ================= */
/* function returns an index into string pool ============== */
/* ========================================================= */
/* ========================================================= */
static
EIaddss( s )
char s[];
{
int i;
i = EISsn;
strcpy( &EISs[ i ], s );
EISsn += strlen( s ) + 1;
if( EISsn > STRINGPOOLLEN ) { 
	fprintf( stderr, "%3d: String list overflow.\n", EILineno ); 
	EIerrorflag = 1;
	return( -1 ); 
	}
return( i );
}

/* ========================================================== */
EIcheckerror()
{
return( EIerrorflag );
}
/* ========================================================== */
EIseterror()
{
EIerrorflag = 0;
}
/* =================================================== */
/* =================================================== */
/* =================================================== */
/* =================================================== */
EIdump( fnm )
char *fnm;
{
int i;
FILE *fp;
int detailed;

detailed = 1;

fp = fopen( fnm, "w" );
if( fp == NULL ) { fprintf( stderr, "can't write to dump file.\n" ); return( 0 ); }

fprintf( fp, "Default lookups: %d.   Retries: %d   User lookups: %d   Inh lookups: %d.\n", 
	EIdeflookups, EIdeflookupfaults, EIulookups, EIinhlookups );
fprintf( fp, "----------------------------\n**** Defaults:\n" );
for( i = 0; i < EIDn; i++ ) fprintf( fp, "%2d. %20s attr=%d override=%d\n", 
	i, &EISs[EIDname[i]], EIDattrkey[i], EIDoverrideflag[i] );

fprintf( fp, "**** Procs:\n" );
for( i = 0; i < EIOn; i++ ) fprintf( fp, "%2d. %20s attr@%d def@%d clone=%d\n", i, &EISs[ EIOname[i] ],
				EIOattrkey[i], EIOdefkey[i], EIOinheritid[i] );

fprintf( fp, "**** Order list:\n" );
for( i = 0; i < EIOn; i++ ) {
	if( i % 30 == 0 ) fprintf( fp, "\n" );
	fprintf( fp, "%d ", EIorder[i] );
	}
fprintf( fp, "\n" );

fprintf( fp, "**** Attributes: (def names go in string list, others via name part list)\n" );
for( i = 0; i < EIAavail; i++ ) {
	fprintf( fp, "%02d. %-12s [%s] %d/%d", i, EIgetattrname( i ), EIgetval( i ), EIAlen[i], EIAalloc[i] );

	if( detailed ) fprintf( fp, " def/fl=%d np=%d ptr=%d typ=%c pos=%d next=%d con=%d des=%d xkey=%d",
				EIAdef[i], EIAnparts[i], EIAnamep[i], EIAtype[i], EIApos[i], 
				EIAnext[i], EIAconkey[i], EIAdeskey[i], EIAxkey[i] );
	fprintf( fp, "\n" );
	}
fprintf( fp, "\n**** Name part list:\n" );
for( i = 0; i < EINPn; i++ ) {
	if( i % 10 == 0 ) fprintf( fp, "\n" );
	fprintf( fp, "(%d)%s ", i, EINP[i] );
	}


fprintf( fp, "\n**** String list\n" );
EISs[ EISsn ] = '\0';
for( i = 0; i < EISsn; i++ ) {
	if( i % 100 == 0 ) fprintf( fp, "\n %5d.", i );
	if( EISs[i] == '\0' && EISs[i+1] != '\0' ) fprintf( fp, "|" );
	else if( EISs[i] == '\0' ) fprintf( fp, "_" );
	else if( EISs[i] == '\n' ) fprintf( fp, "\\n" );
	else fprintf( fp, "%c", EISs[i] );
	}

fprintf( fp, "\n\n**** Double list" );
for( i = 0; i < EISdn; i++ ) {
	if( i % 10 == 0 ) fprintf( fp, "\n" );
	fprintf( fp, "(%d)%g ", i, EISd[i] );
	}
 
fprintf( fp, "\n\n**** Int list" );
for( i = 0; i < EISin; i++ ) {
	if( i % 10 == 0 ) fprintf( fp, "\n" );
	fprintf( fp, "%(%d)%d ", i, EISi[i] );
	}

fprintf( fp, "\n\n" );
fclose( fp );
fprintf( stderr, "Dump completed.\n" ); fflush( stderr );
}
