/*
   example.c: GdkMagick example program
  
   This program loads an image specified on the command line and displays
   it in a GDK window.
 
   Compile this program with the following command:
 
   gcc -o example example.c `gdkmagick-config --cflags --libs`

   Be sure to enclose the gdkmagick-config statement in backticks (`)
   instead of normal quotes (').
 
   Feel free to copy and use this example program in your application.
 */

#include <gdk_magick.h>

int main (int argc, char *argv [])
{
    gchar window_title [] = "GdkMagick Example";
    ImageInfo image_info;
    Image *image;
    GdkVisual *visual;
    GdkColormap *colormap;
    GdkWindowAttr gdk_window_attr;
    GdkWindow *gdk_window;
    GdkPixmap *gdk_pixmap;
    int width, height;

    /* Provide usage information. */

    if (argc != 2)
    {
        fprintf (stderr, "Usage: %s image_data_file\n", argv [0]);
        exit (-1);
    }
    
    /* Initialize GDK and the GdkMagick library. */
   
    gdk_init (&argc, &argv); 
    gdk_magick_init (&argc, &argv);

    /*
     * Allow GdkMagick to dither images into GDK structures when not
     * enough colors are available.
     */
    
    gdk_magick_set_dither (TRUE);
    
    /* Use GdkMagick's visual and colormap. */

    visual = gdk_magick_get_visual ();
    colormap = gdk_magick_get_colormap ();
    
    /* Load the image into an ImageMagick Image structure. */

    GetImageInfo (&image_info);
    strcpy (image_info.filename, argv [1]);
    image = ReadImage (&image_info);
    if (image == NULL)
    {
        fprintf (stderr, "Error: Unable to load image file %s!\n", argv [1]);
        exit (-1);
    }

    /* Get the width and height of the image. */

    width = image -> columns;
    height = image -> rows;
    
    /* Convert the ImageMagick image into a GDK pixmap. */

    gdk_pixmap = gdk_magick_to_gdkpixmap (image);
    if (gdk_pixmap == NULL)
    {
        fprintf (stderr, "Error: Unable to create pixmap!\n");
        if (image != NULL) DestroyImage (image);
        exit (-1);
    }

    /* Create a GDK window to display the pixmap. */

    gdk_window_attr.title = (gchar *) window_title;
    gdk_window_attr.event_mask = GDK_STRUCTURE_MASK;
    gdk_window_attr.width = width;
    gdk_window_attr.height = height;
    gdk_window_attr.wclass = GDK_INPUT_OUTPUT;
    gdk_window_attr.visual = visual;
    gdk_window_attr.colormap = colormap;
    gdk_window_attr.window_type = GDK_WINDOW_TOPLEVEL;
        
    gdk_window = gdk_window_new
    (
        NULL,
        &gdk_window_attr,
        GDK_WA_TITLE | GDK_WA_VISUAL | GDK_WA_COLORMAP
    );
    if (gdk_window == NULL)
    {
        fprintf (stderr, "Error: Unable to create window!\n");
        if (gdk_pixmap != NULL) gdk_pixmap_unref (gdk_pixmap);
        if (image != NULL) DestroyImage (image);
        exit (-1);
    }
    
    /* Place the pixmap into the window's background. */
    
    gdk_window_set_back_pixmap (gdk_window, gdk_pixmap, 0);

    /* Display the window. */

    gdk_window_show (gdk_window);
    gdk_flush ();
    
    /* Handle window events. */

    while (TRUE)
    {
        GdkEvent *event;

        gdk_events_pending ();
        event = gdk_event_get ();
        
        if ((event != NULL)
            && (event -> type == GDK_DELETE)
            && (((GdkEventAny *) event) -> window == gdk_window))
        {
            /* Clean up. */
            
            if (gdk_pixmap != NULL) gdk_pixmap_unref (gdk_pixmap);
            if (image != NULL) DestroyImage (image);
            
            /* Done! */
            
            exit (0);
        }
    }
}
