/*
 *  This file is part of the GdkMagick library.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; see the file COPYING.LIB.  If not,
 *  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#ifndef __GDK_MAGICK_PRIVATE_H__
#define __GDK_MAGICK_PRIVATE_H__

/* __BEGIN_DECLS should be used at the beginning of your declarations,
   so that C++ compilers don't mangle their names.  Use __END_DECLS at
   the end of C declarations. */
#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

/* __P is a macro used to wrap function prototypes, so that compilers
   that don't understand ANSI C prototypes still work, and ANSI C
   compilers can issue warnings about type mismatches. */
#undef __P
#if defined (__STDC__) || defined (_AIX) \
    || (defined (__mips) && defined (_SYSTYPE_SVR4)) \
    || defined(WIN32) || defined(__cplusplus)
# define __P(protos) protos
#else
# define __P(protos) ()
#endif

__BEGIN_DECLS

/*********************************************************************
 *
 * MACROS AND CONSTANTS
 */

/*
 * The _qlass macro expands to class in C and c_class in C++. This
 * ensures that both C and C++ are able to access the image class
 * field in the ImageMagick image structure.
 */

#ifdef _qlass
#undef _qlass
#endif /* _qlass */

#ifdef __cplusplus
#define _qlass c_class
#else
#define _qlass class
#endif /* __cplusplus */

/*
 * If we are using a version of ImageMagick less than 5.0, then
 * define PixelPacket as RunlengthPacket.
 */

#ifdef MAGICK5
#else
#define PixelPacket RunlengthPacket
#endif /* MAGICK5 */

/* Set a parameter to a value if ImageMagick 5.0 is used. */

#ifdef MAGICK5
#define GDK_MAGICK_SET_IF_MAGICK5(PARAM, VAL) ((PARAM) = (VAL))
#else
#define GDK_MAGICK_SET_IF_MAGICK5(PARAM, VAL)
#endif /* MAGICK5 */

/* Set a parameter to a value if ImageMagick 4.2.x is used. */

#ifdef MAGICK5
#define GDK_MAGICK_SET_IF_NOT_MAGICK5(PARAM, VAL)
#else
#define GDK_MAGICK_SET_IF_NOT_MAGICK5(PARAM, VAL) ((PARAM) = (VAL))
#endif /* MAGICK5 */

/* Write color information to an ImageMagick pixel. */

#ifdef MAGICK5
#define GDK_MAGICK_SET_PIXEL(PIX, R, G, B, O, L, CMAP) \
    PIX.red = (R); \
    PIX.green = (G); \
    PIX.blue = (B); \
    PIX.opacity = (O);
#else
#define GDK_MAGICK_SET_PIXEL(PIX, R, G, B, O, L, CMAP) \
    PIX.red = (R); \
    PIX.green = (G); \
    PIX.blue = (B); \
    PIX.index = (CMAP); \
    PIX.length = (L);
#endif /* MAGICK5 */

/*
 * Macros/functions for managing the ImageMagick image pixel cache
 * when ImageMagick 5.0 is used.
 */

#ifdef MAGICK5
#define GDK_MAGICK_GET_PIXEL_CACHE(IMG, X, Y, W, H) \
    GetPixelCache ((IMG), (X), (Y), (W), (H))
#define GDK_MAGICK_SET_PIXEL_CACHE(IMG, X, Y, W, H) \
    SetPixelCache ((IMG), (X), (Y), (W), (H))
#else
#define GDK_MAGICK_GET_PIXEL_CACHE(IMG, X, Y, W, H) (TRUE)
#define GDK_MAGICK_SET_PIXEL_CACHE(IMG, X, Y, W, H) (TRUE)
#endif /* MAGICK5 */

/*
 * Macros/functions for managing a runlength-encoded ImageMagick image
 * when ImageMagick 4.2.x is used.
 */

#ifdef MAGICK5
#define GDK_MAGICK_CONDENSE(IMG)
#define GDK_MAGICK_UNCONDENSE(IMG)
#else
#define GDK_MAGICK_CONDENSE(IMG) (CondenseImage (IMG))
#define GDK_MAGICK_UNCONDENSE(IMG) (UncondenseImage (IMG))
#endif /* MAGICK5 */

/* Execute a block of code while reading a new runlength packet. */

#ifdef MAGICK5
#define GDK_MAGICK_RLE_WRAPPER(RLE, LEN, CODE) CODE
#else
#define GDK_MAGICK_RLE_WRAPPER(RLE, LEN, CODE) \
    if ((RLE) > 0) { (RLE)--; } \
    else { (RLE) = (LEN); CODE }
#endif /* MAGICK5 */

/* Determine the minimum and maximum of two values. */
/* TODO: Use the functions included in glib instead. */

#define GDK_MAGICK_MIN(A, B) (((A) < (B)) ? (A) : (B))
#define GDK_MAGICK_MAX(A, B) (((A) > (B)) ? (A) : (B))

/* Convert an 8-bit color component to 16 bits. */

#define GDK_MAGICK_HIGH_QUANTUM(C) \
    ((((guint16) (C)) << 8) + ((guint16) (C)))

/* Convert a 16-bit color component to 8 bits. */

#define GDK_MAGICK_LOW_QUANTUM(C) \
    ((guint8) ((C) >> 8))

/* Convert RGB color components to a truecolor GDK pixel value. */

#define GDK_MAGICK_TO_GDK_PIXEL(R, G, B, RP, GP, BP, RS, GS, BS, D) \
    ((guint32) \
    (((R) >> ((D) - (RP))) << (RS)) | \
    (((G) >> ((D) - (GP))) << (GS)) | \
    (((B) >> ((D) - (BP))) << (BS)))

/* Convert a truecolor GDK pixel value to an RGB color component. */

#define GDK_MAGICK_FROM_GDK_PIXEL(PIX, M, P, S, D) \
    ((guint16) ((((PIX) & (M)) >> (S)) << ((D) - (P))))

/* Convert RGB color components to a grayscale GDK pixel value. */

#define GDK_MAGICK_GRAY_TO_GDK_PIXEL(R, G, B, VD, D) \
    ((guint32) ((((R) + (G) + (B)) / 3) >> ((D) - (VD))))

/* Convert a grayscale GDK pixel value to an RGB color component. */

#define GDK_MAGICK_GRAY_FROM_GDK_PIXEL(PIX, VD, D) \
    ((guint16) ((PIX) << ((D) - (VD))))

/*********************************************************************
 *
 * DATA TYPES
 */

typedef struct _GdkMagickData
{
    GdkVisual *visual;
    GdkColormap *colormap;
    GdkMagickColorList *alloc_color_list;
    GdkMagickColorList *alloc_color_tail;
    gboolean dither;
} GdkMagickData;

/*********************************************************************
 *
 * GLOBAL VARIABLES
 */

extern GdkMagickData __gdk_magick_data;

/*********************************************************************
 *
 * FUNCTIONS
 */

/* Private functions */

void gdk_magick_stub __P ((gchar *func));
Image *gdk_magick_new_image __P ((gint width, gint height));

__END_DECLS

#endif /* __GDK_MAGICK_PRIVATE_H__ */

