/*
 *  This file is part of the GdkMagick library.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; see the file COPYING.LIB.  If not,
 *  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

/*
 *  gdkmagick_convert.c: Routines to convert an image between an
 *  ImageMagick Image and a GdkPixmap, GdkBitmap, GdkDrawable,
 *  or GdkImage.
 */

#include "config.h"
#include <gdk_magick.h>
#include <gdk_magick_strings.h>
#include <gdk_magick_private.h>

/*********************************************************************/
/**
   Convert a GDK drawable (pixmap, bitmap, or window) to an
   ImageMagick image. The drawable must be created with the same
   visual and colormap as those of GdkMagick. The converted must later
   be freed by calling DestroyImage.

   @param gdk_drawable GDK drawable to convert
   @return The converted ImageMagick image
 **/

Image *gdk_magick_from_gdkdrawable (GdkDrawable *gdk_drawable)
{
    GdkImage *gdk_image;
    Image *magick_image;
    gint x, y, width, height, depth;
    
    /* Retrieve the GDK image from the GDK drawable. */

    gdk_window_get_geometry
    (
        (GdkWindow *) gdk_drawable,
        &x,
        &y,
        &width,
        &height,
        &depth
    );
    gdk_image = gdk_image_get
    (
        (GdkWindow *) gdk_drawable,
        0,
        0,
        width,
        height
    );

    /* Create a new ImageMagick image. */

    magick_image = gdk_magick_new_image (width, height);
    if (magick_image == NULL)
    {
        return NULL;
    }

    /* Render the GdkImage into the ImageMagick image. */

    gdk_magick_render_from_gdk (magick_image, gdk_image);

    /* Return the new ImageMagick image. */
    
    return magick_image;
}

/*********************************************************************/
/**
   Convert an ImageMagick image to a GDK pixmap. The converted
   pixmap must later be freed by calling gdk_pixmap_unref.

   @param image ImageMagick image to convert
   @return The converted GDK pixmap
 **/

GdkPixmap *gdk_magick_to_gdkpixmap (Image *image)
{
    GdkPixmap *gdk_pixmap;
    GdkGC *gdk_gc;
    gint width, height, depth;

    width = (gint) image -> columns;
    height = (gint) image -> rows;
    depth = __gdk_magick_data.visual -> depth;

    /* Create a new pixmap which will hold the converted image. */

    gdk_pixmap = gdk_pixmap_new (NULL, width, height, depth);
    if (gdk_pixmap == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_PIXMAP);
        return NULL;
    }

    /* Create a graphics context to render the image into the pixmap. */
    
    gdk_gc = gdk_gc_new ((GdkWindow *) gdk_pixmap);
    if (gdk_gc == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_GC);
        if (gdk_pixmap != NULL) gdk_pixmap_unref (gdk_pixmap);
        return NULL;
    }
    
    /* Render the ImageMagick image into the GDK pixmap. */
    
    gdk_magick_render_to_gdk
    (
        (GdkDrawable *) gdk_pixmap,
        gdk_gc,
        image
    );

    /* Clean up. */

    if (gdk_gc != NULL) gdk_gc_unref (gdk_gc);
        
    return gdk_pixmap;
}

/*********************************************************************/
/**
   Convert a GDK pixmap to an ImageMagick image. The converted
   image must later be freed by calling DestroyImage.

   @param gdk_pixmap GDK pixmap to convert
   @return The converted ImageMagick image
 **/

Image *gdk_magick_from_gdkpixmap (GdkPixmap *gdk_pixmap)
{
    return gdk_magick_from_gdkdrawable ((GdkDrawable *) gdk_pixmap);
}

/*********************************************************************/
/**
   Convert an ImageMagick image to a monochrome GDK bitmap.
   The converted bitmap must later be freed by calling gdk_bitmap_unref.
   {\bf WARNING:} This function is currently unsupported.
  
   @param image ImageMagick image to convert
   @return The converted GDK bitmap
 **/

GdkBitmap *gdk_magick_to_gdkbitmap (Image *image)
{
    GdkBitmap *gdk_bitmap;
    GdkGC *gdk_gc;
    gint width, height;

    gdk_magick_stub ("This function is currently unsupported.");
    return NULL;
    
    width = (gint) image -> columns;
    height = (gint) image -> rows;

    /* Create a new bitmap which will hold the converted image. */

    gdk_bitmap = (GdkBitmap *) gdk_pixmap_new (NULL, width, height, 1);
    if (gdk_bitmap == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_BITMAP);
        return NULL;
    }

    /* Create a graphics context to render the image into the bitmap. */

    gdk_gc = gdk_gc_new ((GdkWindow *) gdk_bitmap);
    if (gdk_gc == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_GC);
        if (gdk_bitmap != NULL) gdk_bitmap_unref (gdk_bitmap);
        return NULL;
    }
    
    /* Render the ImageMagick image into the GDK bitmap. */
    
    gdk_magick_render_to_gdk
    (
        (GdkDrawable *) gdk_bitmap,
        gdk_gc,
        image
    );

    /* Clean up. */

    if (gdk_gc != NULL) gdk_gc_unref (gdk_gc);
        
    return gdk_bitmap;
}

/*********************************************************************/
/**
   Convert a GDK bitmap to an ImageMagick image. The converted image
   must later be freed by calling DestroyImage.

   @param gdk_bitmap GDK bitmap to convert
   @return The converted ImageMagick image
 **/

Image *gdk_magick_from_gdkbitmap (GdkBitmap *gdk_bitmap)
{
    GdkImage *gdk_image;
    Image *magick_image;
    gint x, y, width, height, depth;

    /*
     * Bitmaps are monochrome, and usually have a different visual
     * and colormap than pixmaps and images do. We need to handle
     * this as a special case.
     */
    
    /* Retrieve the GDK image from the GDK drawable. */

    gdk_window_get_geometry
    (
        (GdkWindow *) gdk_bitmap,
        &x,
        &y,
        &width,
        &height,
        &depth
    );
    gdk_image = gdk_image_get
    (
        (GdkWindow *) gdk_bitmap,
        0,
        0,
        width,
        height
    );

    /* Create a new ImageMagick image. */

    magick_image = gdk_magick_new_image (width, height);
    if (magick_image == NULL)
    {
        return NULL;
    }

    /* Render the GdkImage into the ImageMagick image. */

    gdk_magick_render_from_gdk_monochrome (magick_image, gdk_image);

    /* Return the new ImageMagick image. */
    
    return magick_image;
}

/*********************************************************************/
/**
   Convert an ImageMagick image to a GDK image. The converted image
   must later be freed by calling GdkDestroyImage.
   
   @param image ImageMagick image to convert
   @return The converted GDK image
 **/

GdkImage *gdk_magick_to_gdkimage (Image *image)
{
    GdkImage *gdk_image;
    GdkPixmap *gdk_pixmap;
    GdkGC *gdk_gc;
    gint width, height, depth;

    width = (gint) image -> columns;
    height = (gint) image -> rows;
    depth = __gdk_magick_data.visual -> depth;

    /*
     * If the visual type is direct-color or true-color, render the
     * ImageMagick image directly onto the GDK image, bypassing GdkRGB.
     */
    
    if ((__gdk_magick_data.visual -> type == GDK_VISUAL_DIRECT_COLOR)
        || (__gdk_magick_data.visual -> type == GDK_VISUAL_TRUE_COLOR))
    {
        gdk_image = gdk_image_new
        (
            GDK_IMAGE_NORMAL,
            __gdk_magick_data.visual,
            width,
            height
        );
        if (gdk_image == NULL)
        {
            gdk_magick_error (GDK_MAGICK_ERROR_GDK_IMAGE);
            return NULL;
        }
        gdk_magick_render_to_gdkimage_truecolor
        (
            gdk_image,
            image
        );
        return gdk_image;
    }

    /* Create a new pixmap which will hold the converted image. */

    gdk_pixmap = gdk_pixmap_new (NULL, width, height, depth);
    if (gdk_pixmap == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_PIXMAP);
        return NULL;
    }

    /* Create a graphics context to render the image into the pixmap. */
    
    gdk_gc = gdk_gc_new ((GdkWindow *) gdk_pixmap);
    if (gdk_gc == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_GC);
        if (gdk_pixmap != NULL) gdk_pixmap_unref (gdk_pixmap);
        return NULL;
    }
    
    /* Render the ImageMagick image into the GDK pixmap. */
    
    gdk_magick_render_to_gdk
    (
        (GdkDrawable *) gdk_pixmap,
        gdk_gc,
        image
    );

    /* Retrieve the GDK image from the GDK pixmap. */
    
    gdk_image = gdk_image_get ((GdkWindow *) gdk_pixmap, 0, 0, width, height);
    
    /* Clean up. */

    if (gdk_gc != NULL) gdk_gc_unref (gdk_gc);
    if (gdk_pixmap != NULL) gdk_pixmap_unref (gdk_pixmap);
    
    return gdk_image;
}

/*********************************************************************/
/**
   Convert a GDK image to an ImageMagick image. The converted
   image must later be freed by calling DestroyImage.

   @param gdk_image GDK image to convert
   @return The converted ImageMagick image
 **/

Image *gdk_magick_from_gdkimage (GdkImage *gdk_image)
{
    gint width, height, depth;
    Image *magick_image;
    
    width = (gint) gdk_image -> width;
    height = (gint) gdk_image -> height;
    depth = (gint) gdk_image -> depth;

    /* Create a new ImageMagick image. */

    magick_image = gdk_magick_new_image (width, height);
    if (magick_image == NULL)
    {
        return NULL;
    }

    /* Render the GdkImage into the ImageMagick image. */

    gdk_magick_render_from_gdk (magick_image, gdk_image);

    /* Return the new ImageMagick image. */
    
    return magick_image;
}
