/*
 *  This file is part of the GdkMagick library.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; see the file COPYING.LIB.  If not,
 *  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

/*
 *  gdkmagick_init.c: GdkMagick initialization routines
 */

#include "config.h"
#include <gdk_magick.h>
#include <gdk_magick_strings.h>
#include <gdk_magick_private.h>

/* Global variables that pertain to this file. */

static gboolean __gdk_magick_initialized = FALSE;

/*********************************************************************/
/**
   Initialize the GdkMagick library. This function
   initializes the library's internal data structures and creates a
   default visual and colormap that applications may use.

   {\bf NOTE:} This function must be called after calling
   gdk_init or gtk_init. This function takes the place of
   gdk_rgb_init in applications which use GdkRGB in addition to GdkMagick.
  
   @param argc Pointer to the number of command-line arguments
   @param argv Pointer to the list of command-line arguments
   @return {\tt TRUE} if the initialization was successful,
   {\tt FALSE} otherwise
 **/

gboolean gdk_magick_init (gint *argc, gchar **argv [])
{
    /* Ensure that gdk_magick_init hasn't already been called. */
    
    if (__gdk_magick_initialized)
    {
        gdk_magick_warning (GDK_MAGICK_WARNING_INITIALIZED);
        return TRUE;
    }

    __gdk_magick_initialized = TRUE;
    
    /*
     * Determine if the correct version of ImageMagick is being
     * used with this library.
     */

#ifdef MAGICK5
    if (MagickLibVersion < 0x0500)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_MAGICK4);
        return FALSE;
    }
#else
    if (MagickLibVersion >= 0x0500)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_MAGICK5);
        return FALSE;
    }
#endif /* MAGICK5 */
    
    /* Initialize all toolkits and data structures. */
    
    gdk_rgb_init ();
    __gdk_magick_data.visual = gdk_rgb_get_visual ();
    if (__gdk_magick_data.visual == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_VISUAL);
        return FALSE;
    }
    __gdk_magick_data.colormap = gdk_rgb_get_cmap ();
    if (__gdk_magick_data.colormap == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_COLORMAP);
        return FALSE;
    }
    __gdk_magick_data.alloc_color_list = NULL;
    __gdk_magick_data.alloc_color_tail = NULL;
    __gdk_magick_data.dither = TRUE;
    return TRUE;
}

/*********************************************************************/
/**
   Return the GDK visual used by the GdkMagick routines.
  
   @return The GDK visual used by the GdkMagick routines.
 **/

GdkVisual *gdk_magick_get_visual (void)
{
    return __gdk_magick_data.visual;
}

/*********************************************************************/
/**
   Return the GDK colormap used by the GdkMagick routines.
  
   @return The GDK colormap used by the GdkMagick routines.
 **/

GdkColormap *gdk_magick_get_colormap (void)
{
    return __gdk_magick_data.colormap;
}

/*********************************************************************/
/**
   Enable or disable dithering GDK objects
   when not enough colors are available in the GdkMagick colormap.
  
   @param dither {\tt TRUE} to enable image dithering,
   {\tt FALSE} to disable image dithering
 **/

void gdk_magick_set_dither (gboolean dither)
{
    __gdk_magick_data.dither = dither;
}
