/*
 *  This file is part of the GdkMagick library.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; see the file COPYING.LIB.  If not,
 *  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

/*
 *  gdkmagick_render.c: Core image rendering routines 
 */

#include "config.h"
#include <gdk_magick.h>
#include <gdk_magick_private.h>
#include <gdk_magick_strings.h>

/*********************************************************************/
/**
   Render an ImageMagick image into a GDK drawable. This function is
   essential for accessing an ImageMagick image using the GDK toolkit.
   This function uses the GdkRGB routines to render the image into the
   drawable. The drawable may be a window or pixmap. The drawable must
   be created with the same visual and colormap as the GdkMagick
   default visual and colormap.
   
   @param gdk_drawable GdkDrawable which will receive the rendered image
   @param gdk_gc GDK graphics context to use when rendering the image
   @param image ImageMagick Image structure to render
   @return {\tt TRUE} if the render was successful, {\tt FALSE} otherwise
 **/

gboolean gdk_magick_render_to_gdk
(
    GdkDrawable *gdk_drawable,
    GdkGC *gdk_gc,
    Image *image
)
{
    /* TODO: Handle transparencies properly... */

    /* Image variables */
    PixelPacket *packet;
    gint is_transformed;
    Image *transformed_image;
    gint rle_count;
    gint width, height, width_delta;
    gint quantum_depth;
    gint window_x, window_y, window_width, window_height, window_depth;
    
    /* RGB variables */
    guchar *rgb_data, *rgb_pointer;
    gint rgb_size, rgb_rowstride;
    guchar r, g, b;

    /* Loop variables */
    gint x, y;

    /* Initialize the image pointers to NULL. */
    
    transformed_image = NULL;

    /*
     * Determine the width, height, and RGB component depth of the image.
     * If the GDK drawable dimensions are smaller than the ImageMagick
     * image dimensions, the image will be cropped.
     */
    
    gdk_window_get_geometry
    (
        (GdkWindow *) gdk_drawable,
        &window_x,
        &window_y,
        &window_width,
        &window_height,
        &window_depth
    );
    width = GDK_MAGICK_MIN ((gint) image -> columns, window_width);
    height = GDK_MAGICK_MIN ((gint) image -> rows, window_height);
    quantum_depth = (gint) image -> depth;
    if ((image -> columns) > (window_width))
    {
        width_delta = (gint) ((image -> columns) - (window_width));
    }
    else
    {
        width_delta = 0;
    }
    
    /*
     * If the image is not in the RGB colorspace, make a copy of
     * the image and transform it into RGB colorspace.
     */
    
    is_transformed = 0;
    transformed_image = NULL;
    if (image -> colorspace != RGBColorspace)
    {
        is_transformed = 1;
        transformed_image = CloneImage (image, width, height, 1);    
        if (transformed_image == NULL)
        {
            gdk_magick_error (GDK_MAGICK_ERROR_RGB);
            return FALSE;
        }
        TransformRGBImage (transformed_image, RGBColorspace);
    }

    /* Create an RGB buffer to store the image pixels. */

    rgb_rowstride = (((width * 3) >> 2) + 1) << 2;
    rgb_size = rgb_rowstride * height;
    rgb_data = AllocateMemory (rgb_size * sizeof (guchar));
    if (rgb_data == NULL)
    {
        gdk_magick_error (GDK_MAGICK_ERROR_RGBDATA);
        if (transformed_image != NULL) DestroyImage (transformed_image);
        return FALSE;
    }

    /*
     * Determine whether to copy the original image or the RGB-transformed
     * image.
     */
    
    if (is_transformed)
    {
        packet = transformed_image -> pixels;
    }
    else
    {
        SyncImage (image);
        packet = image -> pixels;
    }

    /* Copy pixels from the RGB image into the RGB buffer data. */
    
    rle_count = 0;
    for (y = 0; y < height; y++)
    {
        if (!GDK_MAGICK_GET_PIXEL_CACHE (image, 0, y, width, 1))
        {
            /* TODO: Error case */
            break;
        }
        rgb_pointer = rgb_data + (y * rgb_rowstride);
        for (x = 0; x < width; x++)
        {
            GDK_MAGICK_RLE_WRAPPER
            (
                rle_count,
                (gint) packet -> length,

                {
                    /* Determine the RGB values of the current pixel. */
                       
                    if (quantum_depth == 16)
                    {
                        r = GDK_MAGICK_LOW_QUANTUM (packet -> red);
                        g = GDK_MAGICK_LOW_QUANTUM (packet -> green);
                        b = GDK_MAGICK_LOW_QUANTUM (packet -> blue);
                    }
                    else if (quantum_depth == 8)
                    {
                        r = (guchar) packet -> red;
                        g = (guchar) packet -> green;
                        b = (guchar) packet -> blue;
                    }
                    packet++;
                }
            )
            
            /* Draw the pixel into the RGB buffer. */
            
            *rgb_pointer = r;
            *(rgb_pointer + 1) = g;
            *(rgb_pointer + 2) = b;
            
            rgb_pointer += 3;
        }
        
        /*
         * Skip over the source image packets that won't be drawn
         * because the target GDK drawable is too narrow.
         */
        
        for (x = 0; x < width_delta; x++)
        {
            GDK_MAGICK_RLE_WRAPPER
            (
                rle_count,
                (gint) packet -> length,
                packet++;
            )    
        }
    }
    /* Destroy the transformed image at this point. */

    if (transformed_image != NULL) DestroyImage (transformed_image);
    
    /* Render the RGB buffer into the drawable. */

    gdk_draw_rgb_image
    (
        gdk_drawable,
        gdk_gc,
        0,
        0,
        width,
        height,
        __gdk_magick_data.dither ? GDK_RGB_DITHER_MAX : GDK_RGB_DITHER_NONE,
        rgb_data,
        rgb_rowstride
    );
    
    /* Destroy the RGB data. */

    if (rgb_data != NULL) FreeMemory (rgb_data);

    return TRUE;
}

/*********************************************************************/
/**
   Render a GDK image into an ImageMagick image. This function is
   essential for accessing a GDK image or drawable using the ImageMagick
   toolkit.

   @param image ImageMagick Image structure to render
   @param gdk_image GdkImage which will receive the rendered image
   @return {\tt TRUE} if the render was successful, {\tt FALSE} otherwise
 **/

gboolean gdk_magick_render_from_gdk (Image *image, GdkImage *gdk_image)
{
    GdkVisual *gdk_visual;
    GdkColormap *gdk_colormap;
    PixelPacket *packet;
    gint width, height;
    gint quantum_depth;
    guint16 r, g, b;
    gint x, y;
    guint32 pixel;

    /* Determine the visual and colormap to use for rendering the image. */

    gdk_visual = __gdk_magick_data.visual;
    gdk_colormap = __gdk_magick_data.colormap;
    
    /*
     * Determine the width, height, and RGB component depth of the image.
     * If the ImageMagick image dimensions are smaller than the
     * GDK drawable dimensions, the image will be cropped.
     */
    
    width = GDK_MAGICK_MIN ((gint) image -> columns, gdk_image -> width);
    height = GDK_MAGICK_MIN ((gint) image -> rows, gdk_image -> height);
    quantum_depth = (gint) image -> depth;
   
    /*
     * If the image is not in the RGB colorspace, transform it
     * into the RGB colorspace in place.
     *
     * TODO: We might want to consider retransforming the image
     * back into its original colorspace after we're done.
     */
    
    if (image -> colorspace != RGBColorspace)
    {
        TransformRGBImage (image, RGBColorspace);
    }

    /* Uncondense the image run-length packets into individual pixels. */

    GDK_MAGICK_UNCONDENSE (image);
        
    /* Copy the pixels from the GDK image to the ImageMagick image. */

    for (y = 0; y < height; y++)
    {
        packet = image -> pixels + (y * width);
        for (x = 0; x < width; x++)
        {
            /* Read the pixel from the source image. */
            
            pixel = gdk_image_get_pixel (gdk_image, x, y);

            /* Compute the RGB values based on the pixel value. */

            switch (gdk_visual -> type)
            {
                case GDK_VISUAL_DIRECT_COLOR:
                case GDK_VISUAL_TRUE_COLOR:
                    
                    r = GDK_MAGICK_FROM_GDK_PIXEL
                    (
                        pixel,
                        gdk_visual -> red_mask,
                        gdk_visual -> red_prec,
                        gdk_visual -> red_shift,
                        quantum_depth
                    );
                    g = GDK_MAGICK_FROM_GDK_PIXEL
                    (
                        pixel,
                        gdk_visual -> green_mask,
                        gdk_visual -> green_prec,
                        gdk_visual -> green_shift,
                        quantum_depth
                    );
                    b = GDK_MAGICK_FROM_GDK_PIXEL
                    (
                        pixel,
                        gdk_visual -> blue_mask,
                        gdk_visual -> blue_prec,
                        gdk_visual -> blue_shift,
                        quantum_depth
                    );
                    if (quantum_depth < gdk_visual -> bits_per_rgb)
                    {
                        r = GDK_MAGICK_LOW_QUANTUM (r);
                        g = GDK_MAGICK_LOW_QUANTUM (g);
                        b = GDK_MAGICK_LOW_QUANTUM (b);
                    }
                    else if (quantum_depth > gdk_visual -> bits_per_rgb)
                    {
                        r = GDK_MAGICK_HIGH_QUANTUM (r);
                        g = GDK_MAGICK_HIGH_QUANTUM (g);
                        b = GDK_MAGICK_HIGH_QUANTUM (b);
                    }
                    break;
                    
                case GDK_VISUAL_GRAYSCALE:
                case GDK_VISUAL_STATIC_GRAY:

                    r = GDK_MAGICK_GRAY_FROM_GDK_PIXEL
                    (
                        pixel,
                        gdk_visual -> depth,
                        quantum_depth
                    );
                    if (quantum_depth < gdk_visual -> bits_per_rgb)
                    {
                        r = GDK_MAGICK_LOW_QUANTUM (r);
                    }
                    else if (quantum_depth > gdk_visual -> bits_per_rgb)
                    {
                        r = GDK_MAGICK_HIGH_QUANTUM (r);
                    }
                    g = r;
                    b = r;
                    break;
                    
                case GDK_VISUAL_PSEUDO_COLOR:
                case GDK_VISUAL_STATIC_COLOR:
                    
                    r = (gdk_colormap -> colors [pixel]).red;
                    g = (gdk_colormap -> colors [pixel]).green;
                    b = (gdk_colormap -> colors [pixel]).blue;
                    if (quantum_depth == 8)
                    {
                        r = GDK_MAGICK_LOW_QUANTUM (r);
                        g = GDK_MAGICK_LOW_QUANTUM (g);
                        b = GDK_MAGICK_LOW_QUANTUM (b);
                    }
                    break;

                default:
                    /* TODO: Failed assertion here perhaps? */
                    break;       
            }
            
            /* Write the pixel to the target image. */
            /* TODO: What about pseudocolor ImageMagick images? */

            GDK_MAGICK_SET_PIXEL
            (
                (*packet),
                (Quantum) r,
                (Quantum) g,
                (Quantum) b,
                0,
                0,
                0
            );
            
            if (!GDK_MAGICK_SET_PIXEL_CACHE (image, 0, y, width, 1))
            {
                /* TODO: Error case */
                break;
            }
            
            packet++;
        }
    }

    /* Condense the new image into run-length packets. */

    GDK_MAGICK_CONDENSE (image);

    return TRUE;
}

/*********************************************************************/
/**
   Render a monochrome GDK image into an ImageMagick image. This
   function is essential for accessing a GDK two-color bitmap using
   the ImageMagick toolkit. This function renders independently of the
   GdkMagick default visual and colormap, since the default visual and
   colormap are not suitable for rendering monochrome images.

   @param image ImageMagick Image structure to render
   @param gdk_image GdkImage which will receive the rendered image
   @return {\tt TRUE} if the render was successful, {\tt FALSE} otherwise
 **/

gboolean gdk_magick_render_from_gdk_monochrome
(
    Image *image,
    GdkImage *gdk_image
)
{
    PixelPacket *packet;
    gint width, height;
    gint quantum_depth;
    guint16 r, g, b;
    gint x, y;
    guint32 pixel;

    /*
     * Determine the width, height, and RGB component depth of the image.
     * If the ImageMagick image dimensions are smaller than the
     * GDK drawable dimensions, the image will be cropped.
     */
    
    width = GDK_MAGICK_MIN ((gint) image -> columns, gdk_image -> width);
    height = GDK_MAGICK_MIN ((gint) image -> rows, gdk_image -> height);
    quantum_depth = (gint) image -> depth;
   
    /*
     * If the image is not in the RGB colorspace, transform it
     * into the RGB colorspace in place.
     *
     * TODO: We might want to consider retransforming the image
     * back into its original colorspace after we're done.
     */
    
    if (image -> colorspace != RGBColorspace)
    {
        TransformRGBImage (image, RGBColorspace);
    }

    /* Uncondense the image run-length packets into individual pixels. */

    GDK_MAGICK_UNCONDENSE (image);
        
    /* Copy the pixels from the GDK image to the ImageMagick image. */

    for (y = 0; y < height; y++)
    {
        packet = image -> pixels + (y * width);
        for (x = 0; x < width; x++)
        {
            /* Read the pixel from the source image. */
            
            pixel = gdk_image_get_pixel (gdk_image, x, y);

            /* Compute the RGB values based on the pixel value. */

            if (pixel != 0)
            {
                r = 0;
                g = 0;
                b = 0;
            }
            else if (quantum_depth == 8)
            {
                r = 0xFF;
                g = 0xFF;
                b = 0xFF;
            }
            else if (quantum_depth == 16)
            {
                r = 0xFFFF;
                g = 0xFFFF;
                b = 0xFFFF;
            }
            else
            {
                /* TODO: Otherwise, a failed assertion takes place... */
            }
            
            /* Write the pixel to the target image. */
            /* TODO: What about pseudocolor ImageMagick images? */

            GDK_MAGICK_SET_PIXEL
            (
                (*packet),
                (Quantum) r,
                (Quantum) g,
                (Quantum) b,
                0,
                0,
                0
            );
            
            if (!GDK_MAGICK_SET_PIXEL_CACHE (image, 0, y, width, 1))
            {
                /* TODO: Error case */
                break;
            }
            
            packet++;
        }
    }

    /* Condense the new image into run-length packets. */

    GDK_MAGICK_CONDENSE (image);

    return TRUE;
}

/*********************************************************************/
/**
   Convert an ImageMagick image into a true-color GDK image. This
   function bypasses GdkRGB and is is provided for speed optimization.

   @param gdk_image GdkImage which will receive the rendered image
   @param image ImageMagick Image structure to render
   @return {\tt TRUE} if the render was successful, {\tt FALSE} otherwise
 **/

gboolean gdk_magick_render_to_gdkimage_truecolor
(
    GdkImage *gdk_image,
    Image *image
)
{
    PixelPacket *packet;
    GdkVisual *gdk_visual;
    gint is_transformed;
    Image *transformed_image;
    gint width, height, width_delta;
    gint quantum_depth;
    gint x, y;
    guint32 gdk_pixel;
    gint rle_count;
    
    /* This only works for true color and direct color visuals... */

    gdk_visual = gdk_image -> visual;

    if ((gdk_visual -> type != GDK_VISUAL_DIRECT_COLOR)
        && (gdk_visual -> type != GDK_VISUAL_TRUE_COLOR))
    {
        return FALSE;
    }
    
    /* Initialize the image pointers to NULL. */
    
    transformed_image = NULL;

    /*
     * Determine the width, height, and RGB component depth of the image.
     * If the ImageMagick image dimensions are smaller than the
     * GDK drawable dimensions, the image will be cropped.
     */

    width = GDK_MAGICK_MIN ((gint) image -> columns, gdk_image -> width);
    height = GDK_MAGICK_MIN ((gint) image -> rows, gdk_image -> height);
    quantum_depth = (gint) image -> depth;
    if ((image -> columns) > (gdk_image -> width))
    {
        width_delta = (gint) ((image -> columns) - (gdk_image -> width));
    }
    else
    {
        width_delta = 0;
    }
    
    /*
     * If the image is not in the RGB colorspace, make a copy of
     * the image and transform it into RGB colorspace.
     */
    
    is_transformed = 0;
    transformed_image = NULL;
    if (image -> colorspace != RGBColorspace)
    {
        is_transformed = 1;
        transformed_image = CloneImage (image, width, height, 1);    
        if (transformed_image == NULL)
        {
            gdk_magick_error (GDK_MAGICK_ERROR_RGB);
            return FALSE;
        }
        TransformRGBImage (transformed_image, RGBColorspace);
    }
    
    if (is_transformed)
    {
        packet = transformed_image -> pixels;
    }
    else
    {
        SyncImage (image);
        packet = image -> pixels;
    }

    /* Copy pixels from the RGB image to the GDK image. */

    rle_count = 0;
    for (y = 0; y < height; y++)
    {
        if (!GDK_MAGICK_GET_PIXEL_CACHE (image, 0, y, width, 1))
        {
            /* TODO: Error case */
            break;
        }
        for (x = 0; x < width; x++)
        {
            GDK_MAGICK_RLE_WRAPPER
            (
                rle_count,
                (gint) packet -> length,
                
                /*
                 * Determine the value of the current pixel.
                 * This calculation only works for TrueColor visuals.
                 */
                
                gdk_pixel = GDK_MAGICK_TO_GDK_PIXEL
                (
                    packet -> red,
                    packet -> green,
                    packet -> blue,
                    gdk_visual -> red_prec,
                    gdk_visual -> green_prec,
                    gdk_visual -> blue_prec,
                    gdk_visual -> red_shift,
                    gdk_visual -> green_shift,
                    gdk_visual -> blue_shift,
                    quantum_depth
                );
            
                packet++;
            )
            
            /* Draw the pixel into the GDK image. */
            
            gdk_image_put_pixel (gdk_image, x, y, gdk_pixel);
        }

        /*
         * Skip over the source image packets that won't be drawn
         * because the target GdkImage is too narrow.
         */
        
        for (x = 0; x < width_delta; x++)
        {
            GDK_MAGICK_RLE_WRAPPER
            (
                rle_count,
                (gint) packet -> length,
                packet++;
            )
        }
    }
    
    /* Clean up. */
    
    if (transformed_image != NULL) DestroyImage (transformed_image);
    return TRUE;
}
