/*
 *  This file is part of the GdkMagick library.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; see the file COPYING.LIB.  If not,
 *  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

/*
 *  gdkmagick_version.c: Version management functions
 */

#include "config.h"
#include <gdk_magick.h>
#include <gdk_magick_private.h>
#include <gdk_magick_strings.h>

/*********************************************************************/
/**
   Return an integer representing the version number of the GdkMagick
   package. The format of the version number is a 32-bit integer with
   the fields {\tt 0xWWXXYYZZ}, where {\tt WW} represents the major
   version, {\tt XX} represents the minor version, {\tt YY} represents
   the sub-minor release number, and {\tt ZZ} represents the patch
   level.

   @return 32-bit version value
 **/

guint32 gdk_magick_get_version (void)
{
    return (GDKMAGICK_VERSION);
}

/*********************************************************************/
/**
   Determine if the running version of GdkMagick meets a minimum version
   requirement. This function is primarily used in the GdkMagick
   configuration macro.

   @param version_string string representing the required GdkMagick version
   @return {\tt TRUE} if the running GdkMagick version is greater than or
   equal to the required version, {\tt FALSE} otherwise.
 **/

gboolean gdk_magick_check_version (guchar *version_string)
{
    guint32 version_value;
    
    version_value = gdk_magick_string_to_version (version_string);

    if (version_value <= GDKMAGICK_VERSION)
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}


/*********************************************************************/
/**
   Convert a version string to an integer representing the version
   number of the GdkMagick
   package. The format of the version number is a 32-bit integer with
   the fields {\tt 0xWWXXYYZZ}, where {\tt WW} represents the major
   version, {\tt XX} represents the minor version, {\tt YY} represents
   the sub-minor release number, and {\tt ZZ} represents the patch
   level.

   @param version_string string representing the version number
   @return 32-bit version value
 **/

guint32 gdk_magick_string_to_version (guchar *version_string)
{
    guint major, minor, subminor, patch;
    guint32 version_value;
    gchar *buffer;
    gchar *start_num;
    gint length;
    gint state;
    gint i;
    
    /*
     * Extract the major, minor, subminor, and patch numbers from the
     * version string.
     */
    
    major = minor = subminor = patch = 0;
    length = strlen (version_string) + 1;
    buffer = g_malloc (sizeof (gchar) * length);
    strcpy (buffer, version_string);
    state = 0;
    start_num = buffer;
    for (i = 0; i < length; i++)
    {
        if ((buffer [i] == '.') || (buffer [i] == 0))
        {
            buffer [i] = 0;
            switch (state)
            {
                case 0:
                    major = atoi (start_num);
                    break;
                case 1:
                    minor = atoi (start_num);
                    break;
                case 2:
                    subminor = atoi (start_num);
                    break;
                case 3:
                    patch = atoi (start_num);
                    break;
                default:
                    break;
            }
            start_num = buffer + (i + 1);
            state++;
        }
    }
    g_free (buffer);

    /* Combine the individual version integers into a 32-bit version value. */

    version_value = (major << 24) | (minor << 16) | (subminor << 8) | patch;

    /* Return the version value. */
    
    return (version_value);
}
