/*
 * widgproc.c: GUI widget management routines
 */

#include "config.h"
#include <gdk_magick.h>
#include <gdk_magick_private.h>
#include <gtk/gtk.h>
#include "gdk_magick_test.h"

/*********************************************************************
 *
 * add_button: Add a button to a box.
 *
 * Parameters:
 *   box - the box where the button will be added
 *   object - object to connect the button click signal, or NULL to
 *     connect the button click signal to the button widget
 *   button_func - the button's callback function
 *   button_text - the text label to place within the button
 *
 * Return value:
 *   None.
 */

void add_button
(
    GtkWidget *box,
    GtkObject *object,
    GtkSignalFunc button_func,
    const gchar *button_text
)
{
    GtkWidget *button;
    
    button = gtk_button_new_with_label (button_text);
    gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);
    if (button_func != NULL)
    {
        if (object != NULL)
        {
            gtk_signal_connect_object
            (
                GTK_OBJECT (button),
                "clicked",
                GTK_SIGNAL_FUNC (button_func),
                GTK_OBJECT (object)
            );
        }
        else
        {
            gtk_signal_connect
            (
                GTK_OBJECT (button),
                "clicked",
                GTK_SIGNAL_FUNC (button_func),
                (gpointer) button_text
            );
        }
    }
    gtk_widget_show (button);
}

void add_fill_button
(
    GtkWidget *box,
    GtkObject *object,
    GtkSignalFunc button_func,
    const gchar *button_text
)
{
    GtkWidget *button;
    
    button = gtk_button_new_with_label (button_text);
    gtk_box_pack_start (GTK_BOX (box), button, TRUE, FALSE, 0);
    if (button_func != NULL)
    {
        if (object != NULL)
        {
            gtk_signal_connect_object
            (
                GTK_OBJECT (button),
                "clicked",
                GTK_SIGNAL_FUNC (button_func),
                GTK_OBJECT (object)
            );
        }
        else
        {
            gtk_signal_connect
            (
                GTK_OBJECT (button),
                "clicked",
                GTK_SIGNAL_FUNC (button_func),
                (gpointer) button_text
            );
        }
    }
    gtk_widget_show (button);
}

/*********************************************************************
 *
 * add_checkbox: Add a checkbox to a box.
 *
 * Parameters:
 *   box - the box where the checkbox will be added
 *   object - object to connect the toggle signal, or NULL to
 *     connect the toggle signal to the checkbox widget
 *   checkbox_func - the checkbox' callback function
 *   active - TRUE to set the checkbox to active by default,
 *     FALSE to leave the checkbox inactive
 *   checkbox_text - the text label to place within the checkbox
 *
 * Return value:
 *   None.
 */

void add_checkbox
(
    GtkWidget *box,
    GtkObject *object,
    GtkSignalFunc checkbox_func,
    gint active,
    const gchar *checkbox_text
)
{
    GtkWidget *checkbox;
    
    checkbox = gtk_check_button_new_with_label (checkbox_text);
    gtk_box_pack_start (GTK_BOX (box), checkbox, FALSE, FALSE, 0);

    gtk_toggle_button_set_active
    (
        GTK_TOGGLE_BUTTON (checkbox),
        active
    );
    
    if (checkbox_func != NULL)
    {
        if (object != NULL)
        {    
            gtk_signal_connect_object
            (
                GTK_OBJECT (checkbox),
                "toggled",
                GTK_SIGNAL_FUNC (checkbox_func),
                GTK_OBJECT (object)
            );
        }
        else
        {
            gtk_signal_connect
            (
                GTK_OBJECT (checkbox),
                "toggled",
                GTK_SIGNAL_FUNC (checkbox_func),
                NULL
            );
        }
    }
    gtk_widget_show (checkbox);
}

/*********************************************************************
 *
 * add_radio: Add a radio button a box.
 *
 * Parameters:
 *   box - the box where the radio button will be added
 *   object - object to connect the toggle signal, or NULL to
 *     connect the toggle signal to the radio button widget
 *   group - the group affected when the user selects the radio button
 *   radio_func - the radio button's callback function
 *   active - TRUE to set the radio button to active by default,
 *     FALSE to leave the radio button inactive
 *   radio_text - the text label to place within the radio button
 *
 * Return value:
 *   The group containing the newly created radio button
 */

GSList *add_radio
(
    GtkWidget *box,
    GtkObject *object,
    GSList *group,
    GtkSignalFunc radio_func,
    gint active,
    const gchar *radio_text
)
{
    GtkWidget *radio;
    
    radio = gtk_radio_button_new_with_label (group, radio_text);
    gtk_box_pack_start (GTK_BOX (box), radio, FALSE, FALSE, 0);
    
    gtk_toggle_button_set_active
    (
        GTK_TOGGLE_BUTTON (radio),
        active
    );
    
    if (radio_func != NULL)
    {
        if (object != NULL)
        {
            gtk_signal_connect_object
            (
                GTK_OBJECT (radio),
                "toggled",
                GTK_SIGNAL_FUNC (radio_func),
                GTK_OBJECT (object)
            );
        }
        else
        {
            gtk_signal_connect
            (
                GTK_OBJECT (radio),
                "toggled",
                GTK_SIGNAL_FUNC (radio_func),
                NULL
            );
        }
    }
    gtk_widget_show (radio);
    return (gtk_radio_button_group (GTK_RADIO_BUTTON (radio)));
}

/*********************************************************************
 *
 * add_checkbox: Add a menu item to a menu.
 *
 * Parameters:
 *   menu - the menu where the menu item will be added
 *   menu_item_func - the menu item's callback function
 *   item_text - the text label to place within the menu item
 *   data - data to pass to the callback function
 *
 * Return value:
 *   None.
 */

void add_menu_item
(
    GtkWidget *menu,
    GtkSignalFunc menu_item_func,
    const gchar *item_text,
    gpointer data
)
{
    GtkWidget *item;
    
    item = gtk_menu_item_new_with_label (item_text);
    gtk_signal_connect
    (
        GTK_OBJECT (item),
        "activate",
        menu_item_func,
        data
    );

    gtk_widget_show (item);
    gtk_menu_append (GTK_MENU (menu), item);
}

/*********************************************************************
 *
 * add_separator: Add a separator to a box.
 *
 * Parameters:
 *   box - the box where the separator will be added
 *
 * Return value:
 *   None.
 */

void add_separator (GtkWidget *box)
{
    GtkWidget *separator;
    
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box), separator, FALSE, FALSE, 5);
    gtk_widget_show (separator);
}

/*********************************************************************
 *
 * add_label: Add a text label to a box.
 *
 * Parameters:
 *   box - the box where the text label will be added
 *   label_text - the string to display within the text label
 *
 * Return value:
 *   None.
 */

void add_label (GtkWidget *box, const gchar *label_text)
{
    GtkWidget *label;
    
    label = gtk_label_new (label_text);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
    gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);
}

void add_fill_label (GtkWidget *box, const gchar *label_text)
{
    GtkWidget *label;
    
    label = gtk_label_new (label_text);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
    gtk_box_pack_start (GTK_BOX (box), label, TRUE, FALSE, 5);
    gtk_widget_show (label);
}
