/******************************************************************************
 JXHelpDirector.cc

	BASE CLASS = JXWindowDirector

	Copyright  1996 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXHelpDirector.h>
#include <JXWindow.h>
#include <JXHelpText.h>
#include <JXScrollbarSet.h>
#include <JXMenuBar.h>
#include <JXTextMenu.h>
#include <JXTextButton.h>
#include <jXGlobals.h>
#include <jAssert.h>

// File menu

static const JCharacter* kFileMenuTitleStr    = "File";
static const JCharacter* kFileMenuShortcutStr = "#F";

static const JCharacter* kMacFileMenuStr =
	"Table of Contents"
	"%l| Save text... %k Meta-S"
	"%l| Page setup... | Print... %k Meta-P"
	"%l| Preferences... | Save window size"
	"%l| Close %k Meta-W | Close all %k Meta-Shift-W";

static const JCharacter* kWinFileMenuStr =
	"Table of Contents"
	"%l| Save text... %h s %k Ctrl-S"
	"%l| Page setup... | Print... %h p %k Ctrl-P"
	"%l| Preferences... | Save window size"
	"%l| Close %h c %k Ctrl-W | Close all %h a %k Ctrl-Shift-W";

enum
{
	kShowTOCCmd = 1,
	kSaveTextCmd,
	kPageSetupCmd, kPrintCmd,
	kEditPrefsCmd, kSaveWindSizeCmd,
	kCloseCmd, kCloseAllCmd
};

/******************************************************************************
 Constructor

 ******************************************************************************/

JXHelpDirector::JXHelpDirector
	(
	const JCharacter*	title,
	const JCharacter*	text,
	const JXMenu::Style	menuStyle,
	JXPSPrinter*		printer
	)
	:
	JXWindowDirector(JXGetHelpManager())
{
	BuildWindow(title, text, menuStyle, printer);
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXHelpDirector::~JXHelpDirector()
{
}

/******************************************************************************
 BuildWindow (private)

 ******************************************************************************/

void
JXHelpDirector::BuildWindow
	(
	const JCharacter*	title,
	const JCharacter*	text,
	const JXMenu::Style	menuStyle,
	JXPSPrinter*		printer
	)
{
// begin JXLayout

    JXWindow* window = new JXWindow(this, 600,400, "");
    assert( window != NULL );
    SetWindow(window);

    JXScrollbarSet* scrollbarSet =
        new JXScrollbarSet(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 0,30, 600,370);
    assert( scrollbarSet != NULL );

    JXMenuBar* menuBar =
        new JXMenuBar(window,
                    JXWidget::kHElastic, JXWidget::kFixedTop, 0,0, 500,30);
    assert( menuBar != NULL );

    itsCloseButton =
        new JXTextButton("Close", window,
                    JXWidget::kFixedRight, JXWidget::kFixedTop, 550,0, 50,30);
    assert( itsCloseButton != NULL );
    itsCloseButton->SetShortcuts("^[");

    itsTOCButton =
        new JXTextButton("TOC", window,
                    JXWidget::kFixedRight, JXWidget::kFixedTop, 500,0, 50,30);
    assert( itsTOCButton != NULL );

// end JXLayout

	window->SetTitle(title);
	window->SetMinSize(100,100);

	ListenTo(itsCloseButton);

	itsFileMenu = menuBar->AppendTextMenu(kFileMenuTitleStr);
	if (menuStyle == JXMenu::kMacintoshStyle)
		{
		itsFileMenu->SetMenuItems(kMacFileMenuStr);
		}
	else
		{
		itsFileMenu->SetShortcuts(kFileMenuShortcutStr);
		itsFileMenu->SetMenuItems(kWinFileMenuStr);
		}
	itsFileMenu->SetUpdateAction(JXMenu::kDisableNone);
	ListenTo(itsFileMenu);

	itsText =
		new JXHelpText(text, scrollbarSet, scrollbarSet->GetScrollEnclosure(),
					   JXWidget::kHElastic, JXWidget::kVElastic, 0,0, 10,10);
	assert( itsText != NULL );
	itsText->FitToEnclosure();
	// for Copy, Select All
	itsText->AppendEditMenu(menuBar, menuStyle, kFalse, kFalse, kFalse, kFalse);
	itsText->ShouldAllowDragAndDrop(kTrue);
	itsText->UpdateScrollbars();
	itsText->SetPSPrinter(printer);

	if ((JXGetHelpManager())->HasTOC())
		{
		ListenTo(itsTOCButton);
		}
	else
		{
		itsTOCButton->Hide();
		menuBar->AdjustSize(itsTOCButton->GetFrameWidth(), 0);
		}
}

/******************************************************************************
 Receive (virtual protected)

 ******************************************************************************/

void
JXHelpDirector::Receive
	(
	JBroadcaster*	sender,
	const Message&	message
	)
{
	if (sender == itsFileMenu && message.Is(JXMenu::kNeedsUpdate))
		{
		UpdateFileMenu();
		}
	else if (sender == itsFileMenu && message.Is(JXMenu::kItemSelected))
		{
		const JXMenu::ItemSelected* selection =
			dynamic_cast(const JXMenu::ItemSelected*, &message);
		assert( selection != NULL );
		HandleFileMenu(selection->GetIndex());
		}

	else if (sender == itsTOCButton && message.Is(JXButton::kPushed))
		{
		(JXGetHelpManager())->ShowTOC();
		}
	else if (sender == itsCloseButton && message.Is(JXButton::kPushed))
		{
		Close();
		}

	else
		{
		JXWindowDirector::Receive(sender, message);
		}
}

/******************************************************************************
 UpdateFileMenu (private)

 ******************************************************************************/

void
JXHelpDirector::UpdateFileMenu()
{
	itsFileMenu->SetItemEnable(kShowTOCCmd, (JXGetHelpManager())->HasTOC());
}

/******************************************************************************
 HandleFileMenu (private)

 ******************************************************************************/

void
JXHelpDirector::HandleFileMenu
	(
	const JIndex index
	)
{
	if (index == kShowTOCCmd)
		{
		(JXGetHelpManager())->ShowTOC();
		}

	else if (index == kSaveTextCmd)
		{
		JChooseSaveFile* csf    = JGetChooseSaveFile();
		const JString& origName = (GetWindow())->GetTitle();
		JString fileName;
		if (csf->SaveFile("Save text as:", NULL, origName, &fileName))
			{
			itsText->WritePlainText(fileName, JTextEditor::kUNIXText);
			}
		}

	else if (index == kPageSetupCmd)
		{
		itsText->HandlePSPageSetup();
		}
	else if (index == kPrintCmd)
		{
		itsText->PrintPS();
		}

	else if (index == kEditPrefsCmd)
		{
		(JXGetHelpManager())->EditPrefs();
		}
	else if (index == kSaveWindSizeCmd)
		{
		(JXGetHelpManager())->SaveWindowPrefs(GetWindow());
		}

	else if (index == kCloseCmd)
		{
		Close();
		}
	else if (index == kCloseAllCmd)
		{
		(JXGetHelpManager())->CloseAll();
		}
}

/******************************************************************************
 ShowSubsection

 ******************************************************************************/

void
JXHelpDirector::ShowSubsection
	(
	const JCharacter* name
	)
{
	itsText->ShowSubsection(name);
}
