/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-2000 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef TWindow
#define TWindow TWindow

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/cursor.hh>
#include <toad/pointer.hh>

class TWindow;
typedef GSmartPointer<TWindow> PWindow;

#ifdef _TOAD_PRIVATE
#include <toad/region.hh>
#endif

#ifdef DEBUG
#define CHECK_REALIZED(txt) debug_check_realized(txt)
#else
#endif

#include <limits.h>
#define TPOS_PREVIOUS	INT_MIN
#define TSIZE_PREVIOUS INT_MIN

// TRowColumn
#define TS_HORIZONTAL 0
#define TS_VERTICAL		1

class TFocusManager;
class TPen;
class TPopup;
class TMenuBar;
class TToolTip;

// TX11CreateWindow is a data structure used in connection with 
// 'createX11Window'; it was invented while adding OpenGL support
#ifdef _TOAD_PRIVATE
struct TX11CreateWindow
{
  Display *display;
  Window parent;
  int x,y;
  unsigned width, height;
  unsigned border;      
  int depth;
  unsigned wclass;
  Visual *visual;
  unsigned long valuemask;
  XSetWindowAttributes *attributes;
};
#else
struct TX11CreateWindow;
#endif

class TWindow: 
	public TOADBase, public TSmartObject
{
		friend TOADBase;
		friend TPen;
		friend TFocusManager;

		TWindow& operator=(TWindow&);	// not allowed
		TWindow(const TWindow&);			// not allowed

	public:
		TWindow(TWindow *parent, const string &title);
		virtual ~TWindow();

		void Create();													// Create window & all childs
		void CreateModal();
		void Destroy();
		int Run(int argc=0, char** argv=NULL,char** envv=NULL)
			{return TOADBase::Run(this,argc,argv,envv);}

		// style flags
		//-------------
		unsigned taborder;					// taborder
		bool bShell:1;							// place window in the desktop window
		bool bPopup:1;
		bool bExplicitCreate:1;			// don't create window when parent is created
		bool bSaveUnder:1;					// suggest X11 to buffer the content of windows under this one
		bool bStaticFrame:1;				// don't let the window manager resize the window
		bool bBackingStore:1;				// suggest X11 to buffer the window contents to avoid paint event
		bool bNoBackground:1;				// don't let X11 paint a background
		bool bFocusManager:1;				// this window will act as a focus manager
		bool bNoFocus:1;						// reject focus even when keyUp or keyDown are implemented
		bool bFocusTraversal:1;			// receive focus during focus traversal (true)
		bool bDialogEditRequest:1;	// window will cooperate with the dialog editor
		bool bTabKey:1;							// `true' gets the tab key and disables traversal
		bool bDoubleBuffer:1;				// tell TPen to use double buffering with this window

		// X11 internal: use original X11 GC (needed for OpenGL window)
		bool bX11GC:1;							// use the Xlib default gc

		bool IsRealized(){return x11window!=0;}
		bool IsChildOf(TWindow *wnd);						// 'true', if window is child of 'wnd'

		void ClearWindow();
		void GrabMouse(ulong ulMouseMessages=TMMM_PREVIOUS,TWindow* confine_window=NULL, TCursor::EType type=TCursor::DEFAULT);
		void GrabPopupMouse(ulong ulMouseMessages=TMMM_PREVIOUS, TCursor::EType type=TCursor::DEFAULT);
		void UngrabMouse();
		void GetRootPos(int*,int*);
		
		void UpdateWindow(bool bEraseBackground=false);
		
		void Invalidate(bool clearbg=true);
		void Invalidate(int,int,int,int, bool clearbg=true);
		void Invalidate(const TRectangle&, bool clearbg=true);
		void Invalidate(const TRectangle *rect, bool clearbg=true) {
			Invalidate(*rect, clearbg);
		}
		void Invalidate(const TRegion&, bool bClrBG=true);
		void Invalidate(const TRegion *reg, bool clearbg=true) {
			Invalidate(*reg, clearbg);
		}
		void PaintNow();
		
		void RaiseWindow();
		void LowerWindow();
		
		void ScrollWindow(int x,int y, bool bClrBG=true);
		void ScrollRectangle(const TRectangle &rect, int x,int y, bool bClrBG=true);
		
		void SetOrigin(int x,int y);
		void ScrollTo(int x, int y);
		int OriginX() { return _dx; }
		int OriginY() { return _dy; }

		void SetPosition(int x,int y);
		void SetSize(int x,int y);
		void SetShape(int,int,int,int);
		void SetShape(const TRectangle &r){SetShape(r.x,r.y,r.w,r.h);}
		void SetShape(const TRectangle *r){SetShape(r->x,r->y,r->w,r->h);}
		void GetShape(TRectangle*);
		
		void SetMapped(bool);
		bool Mapped();

		void SetVisible(bool b) { SetMapped(b);	}		// obsolete
		bool Visible() { return Mapped(); }					// obsolete
		
		void SetIcon(TBitmap*);
		void SetCursor(TCursor::EType);
		// void SetToolTip(TToolTip*);							// implemented in tooltip.cc
		void SetToolTip(const string&);							// implemented in tooltip.cc
		
		void SetBackground(const TColor&);
    void SetBackground(byte r,byte g,byte b) {
       SetBackground(TColor(r,g,b));
    }
    void SetBackground(const TRGB &c) {
       SetBackground(TColor(c.r,c.g,c.b));
    }
		void SetBackground(TColor::ESystemColor c) {
			SetBackground(TColor(c));
		}
		void SetBackground(TColor::EColor16 c) {
			SetBackground(TColor(c));
		}
		void SetBackground(TBitmap*);
		void SetHasBackground(bool);
                                          
		// MouseMoveMessages
		static const ulong TMMM_NONE			=0x0000;
		static const ulong TMMM_FIRST			=0x0010;
		static const ulong TMMM_ALL				=0x0020;
		static const ulong TMMM_LBUTTON		=0x0040;
		static const ulong TMMM_MBUTTON		=0x0080;
		static const ulong TMMM_RBUTTON		=0x0100;
		static const ulong TMMM_ANYBUTTON	=0x0200;
		static const ulong TMMM_PREVIOUS	=0x03FF;
		void SetMouseMoveMessages(ulong);
		void ClrMouseMoveMessages(ulong);

		bool SetFocus();
		bool IsFocus();
		
		void AddChild(TWindow* child);
		void RemoveChild(TWindow *child);
		TWindow* Parent(){return parent;}
		TWindow* FirstChild();
		TWindow* LastChild();

		// Get the next/previous sibling of 'ptr' and return NULL when 'ptr'
		// points to the last/first child.
		static TWindow* NextSibling(TWindow *ptr);
		static TWindow* PrevSibling(TWindow *ptr);

		// don't you use these methods:
		void SetSuppressMessages(bool);
		bool IsSuppressMessages();

	protected:
		int _x, _y, _w, _h;				// window position and size
		int _b;										// border width
		int _dx, _dy;							// origin for TPen
		unsigned char _cursor;		// cursor type

	public:
		void SetBorder(bool b){_b=b?1:0;}
		int XPos() const {return _x;}
		int YPos() const {return _y;}
		int Width() const {return _w;}
		int Height() const {return _h;}
		void SetTitle(const string &title);
		const string& Title() const {return title;}
		
		#ifdef DEBUG
		void debug_check_realized(const char *txt);
		#endif

		// TOADs message methods
		virtual void adjust();								// adjust window to children
		virtual void resize();								// adjust children to window
		
		enum EChildNotify
		{
			TCHILD_TITLE, TCHILD_POSITION, TCHILD_RESIZE, TCHILD_ADD,
			TCHILD_REMOVE, TCHILD_CREATE, TCHILD_DESTROY, TCHILD_ICON,
			TCHILD_FOCUS, TCHILD_DIALOG_EDIT_REQUEST, TCHILD_BEFORE_ADD,
			TCHILD_BEFORE_CREATE
		};
		virtual void childNotify(TWindow*, EChildNotify);
		
	private:
		void _ChildNotify(EChildNotify type);
		
	public:
		virtual void closeRequest();					// the window should destroy itself
		virtual void focus();									// got/lost keyboard focus
		virtual void create();								// called before the window is created
		virtual void created();								// called after the window is created; add childs here etc.
		virtual void createX11Window(TX11CreateWindow*);
		virtual void handleX11Event();
		virtual void destroy();
		virtual void keyDown(TKey key, char *string, unsigned modifier);
		virtual void keyUp(TKey key, char *string, unsigned modifier);
		virtual void mouseMove(int x,int y, unsigned modifier);
		virtual void mouseEnter(int x,int y, unsigned modifier);
		virtual void mouseLeave(int x,int y, unsigned modifier);
		virtual void mouseLDown(int x,int y, unsigned modifier);
		virtual void mouseMDown(int x,int y, unsigned modifier);
		virtual void mouseRDown(int x,int y, unsigned modifier);
		virtual void mouseLUp(int x,int y, unsigned modifier);
		virtual void mouseMUp(int x,int y, unsigned modifier);
		virtual void mouseRUp(int x,int y, unsigned modifier);
		virtual void paint();
		virtual void saveYourself();						// the window should save itself

//		void StartDrag(TDragContext*);
//		void RegisterDrop(TDropContext*);
//		TDropContext* GetDropArea(int,int);

	#ifdef _TOAD_PRIVATE
	public:
	#else
	private:
	#endif
		TColor background;											// background color
		string title;														// title of the window
		TWindow* parent;
		TWindow* child;													// pointer to the first child
		TWindow* prev_sibling;
		TWindow* next_sibling;

		void _init();
		void _adjustW2C();
		void _create();
		void _destroy();
		void _cons(TWindow*, const char *);
	public:
		void _SetFocus(bool b);
	private:

		// internal flags
		bool bEraseBe4Paint:1;
		bool _bOwnsFocus:1;
		bool _visible:1;				// true when window should be visible
		/*
			bool _bWMResize:1;
			bool _bExplicitCreate:1;
			bool _bBeforeCreate:1;
			bool _bDeleteTitle:1;
			bool _bMMMMask:1;
			bool _bMMMModified:1;
			bool _bCreate:1;
			bool _bSuppressMsg:1;
		*/
		
		bool _bResizedBeforeCreate:1;
		bool _bToolTipAvailable:1;
		
		void _ProvidePaintRgn();
		
		// old style, have to rewrite this:
		// ATTENTION: IFLAG_MMM_MASK & IFLAG_MMM_MODIFIED will be difficult to remove!!!
		static const ulong IFLAG_WMRESIZE					=0x0001;
		static const ulong IFLAG_EXPLICIT_CREATE	=0x0002;
		static const ulong IFLAG_BEFORE_CREATE		=0x0004;
		static const ulong IFLAG_DELETE_TITLE			=0x0008;
		static const ulong IFLAG_MMM_MASK					=0x07F0;
		static const ulong IFLAG_MMM_MODIFIED			=0x0400;
		static const ulong IFLAG_CREATE						=0x0800;
		static const ulong IFLAG_SUPPRESSMSG			=0x1000;
		static const ulong IFLAG_CHILD_NOTIFY			=0x2000;
		
		ulong iflags;
		void set_iflag(ulong f){iflags|=f;}
		void clr_iflag(ulong f){iflags&=~f;}
		bool is_iflag(ulong f){return iflags&f?true:false;}
		
		long build_eventmask();
		long build_mouse_eventmask();

		#ifdef DEBUG
		static const unsigned DFLAG_WMINIT		=	1;
		static const unsigned DFLAG_WMADJUST	=	2;
		unsigned debug_flags;
		#endif

	private:
		class TPaintRegion;
		static bool _HavePaintEvents();
		static void _DispatchPaintEvent();
		TPaintRegion *paint_rgn;
		
	public:
		TRegion* UpdateRegion();
	private:

	#ifdef _TOAD_PRIVATE
	public:
	#endif
		_TOAD_WINDOW x11window;
};

#ifdef _TOAD_PRIVATE
class TWindow::TPaintRegion:
	public TRegion
{
	public:
		TWindow *wnd;
};
#endif

// these should've been in `toadbase.hh' but this confuses the include
// hierachy
#include <toad/connect.hh>
#include <toad/messagebox.hh>

#endif
