// $Id: Dialog.cpp,v 1.2 1997/04/17 03:19:44 jharris Exp $

#include "Dialog.h"

#include "GenerateDialog.h"
#include "DlgParser.h"

#include "DialogWnd.h"
#include "ProjectWnd.h"

Dialog::Dialog( ProjectWnd &prj, const QString &name )
  : DlgObject( prj, name ),  
    itsWindowType( WT_Dialog ),
    isDialogModal( TRUE )
{
  itsWindow = new DialogWnd( this, &prj, "" );
  if( ! itsWindow )
    fatal( "Failed to new a Dialog..." );

  itsLayout = itsWindow->GetLayout();

  /* Create three popupmenus for the List:
   * 1. Not in a project
   * 2. In a project, not hidden
   * 3. In a project, hidden
   */
  unboundPopup = new QPopupMenu;
  unboundPopup->insertItem( "Save", this, 
                              SLOT(Save()) );
  unboundPopup->insertItem( "Save As...", this, 
                              SLOT(SaveAs()) );
  unboundPopup->insertItem( "Close", this, 
                              SLOT(Hide()) );
  unboundPopup->insertSeparator();
  unboundPopup->insertItem( "Raise", this, 
                              SLOT(Raise()) );

  unboundHiddenPopup = new QPopupMenu;
  unboundHiddenPopup->insertItem( "Show", this, SLOT(Show()) );

  boundPopup = new QPopupMenu;
  boundPopup->insertItem( "Save", this, SLOT(Save()) );
  boundPopup->insertItem( "Save As...", this, SLOT(SaveAs()) );
  boundPopup->insertSeparator();
  boundPopup->insertItem( "Raise", this, SLOT(Raise()) );
  boundPopup->insertSeparator();
  boundPopup->insertItem( "Remove", this, SLOT(Remove()) );

  // Popupmenu for closed dialog entries in projects
  boundHiddenPopup = new QPopupMenu;
  boundHiddenPopup->insertItem( "Show", this, SLOT(Show()) );
  boundHiddenPopup->insertSeparator();
  boundHiddenPopup->insertItem( "Remove", this, SLOT(Remove()) );

  status = DlgObject_Init;
}

Dialog::~Dialog()
{
  delete itsWindow;
}

void Dialog::GetGeneratedFiles( QString &src, 
                                QString &hdr, 
                                QString &mhdr )
{
  src = itsClassSource + " " + itsDataSource;
  hdr = itsClassHeader + " " + itsDataHeader;
  mhdr = hdr;
}

bool Dialog::Load()
{
  if( !itsFilename.isEmpty() )
  {
    itsParser = new DlgParser( itsFilename );

    connect( itsParser, SIGNAL(DlgKeyValue(QString &,QString &)), 
             SLOT(RestoreDialog(QString &,QString &)) );

    connect( itsParser, SIGNAL(NewLayout()), 
             SLOT(RestoreLayout()) );

    // And start the action
    itsParser->StartParsing();

    // That's it for the Parser object
    delete itsParser;    

    itsParser = NULL;

    // set to the minimum size
    itsWindow->resize( 0,0 );
  }

  status = DlgObject_Loaded;
  return TRUE;
}

void Dialog::RestoreDialog
(
 QString&	key,
 QString&	value
 )
{
  if( key == "ClassHeader" )
  {
    itsClassHeader = value;
  }
  else if( key == "ClassSource" )
  {
    itsClassSource = value;
  }
  else if( key == "ClassName" )
  {
    itsClassName = value;
  }
  else if( key == "DataHeader" )
  {
    itsDataHeader = value;
  }
  else if( key == "DataSource" )
  {
    itsDataSource = value;
  }
  else if( key == "DataName" )
  {
    itsDataName = value;
  }
  else if( key == "DataVariableName" )
  {
    itsDataVariableName = value;
  }
  else if( key == "WindowBaseClass" )
  {
    if( value == "QDialog" )
    {
	    itsWindowType = WT_Dialog;
    }
    else if( value == "QTabDialog" )
    {
	    itsWindowType = WT_TabDialog;
    }
    else if( value == "QWidget" )
    {
	    itsWindowType = WT_Widget;
    }
    else if( value == "QFrame" )
    {
	    itsWindowType = WT_Frame;
    }
  }
  else if( key == "IsModal" )
  {
    SetIsDialogModal( (value == "TRUE") );
  }
  else if( key == "WindowCaption" )
  {
    SetCaption( value );
  } 
  else
  {
    // do some error msg
  }
}

void Dialog::RestoreLayout()
{
  itsLayout->RestoreContents( itsParser );
}

bool Dialog::Save( )
{
  QFile file( itsFilename );
  if( !file.open( IO_WriteOnly ) )
  {
    // error msg
    return FALSE;
  }
    
  QTextStream stream( &file );
    
  stream << "DlgEdit:v1.2:Dialog:\n";
  stream << "Dialog {\n";
  stream << "\tClassHeader {" << itsClassHeader << "}\n";
  stream << "\tClassSource {" << itsClassSource << "}\n";
  stream << "\tClassName {" << itsClassName << "}\n";
  stream << "\tDataHeader {" << itsDataHeader << "}\n";
  stream << "\tDataSource {" << itsDataSource << "}\n";
  stream << "\tDataName {" << itsDataName << "}\n";

  stream << "\tWindowBaseClass {" << GetWindowStr() << "}\n";

  if( (itsWindowType == WT_Dialog) || (itsWindowType == WT_TabDialog) )
  {
    stream << "\tIsModal {";
    SaveBool( stream, isDialogModal );
    stream << "}\n";
  }

  if( !itsCaption.isEmpty() )
  {
    stream << "\tWindowCaption {" << itsCaption << "}\n";
  }

  stream << "}\n";

  itsLayout->SaveContents( stream );

  status = DlgObject_Saved;
  emit Saved( this );

  return TRUE;
}

bool Dialog::Generate( bool force )
{
  GenerateDialog dlg( itsFilename, itsClassName,
		      itsClassHeader, itsClassSource,
                      itsDataHeader, itsDataSource, itsWindow );

  if( force || dlg.exec() )
  {
    if( dlg.GenerateSource( itsClassHeader, itsClassSource ) )
    {
      GenerateSource();
    }
	
    if( dlg.GenerateData( itsDataHeader, itsDataSource ) )
    {
	    GenerateData();
    }	
  }

  status = DlgObject_Generated;
  return TRUE;
}

void Dialog::GenerateSource( void )
{
    if( itsClassHeader.isEmpty() )
    {
	QMessageBox::message( "Error", "Dialog class header file name "
			      "not specified" );
	return;
    }
    
    if( itsClassSource.isEmpty() )
    {
	QMessageBox::message( "Error", "Dialog class source file name "
			      "not specified" );
	return;
    }

    if( itsDataHeader.isEmpty() )
    {
	QMessageBox::message( "Error", "Dialog data class header file name "
			      "not specified" );
	return;
    }
    
    // generate header file
    QFile headerFile( itsClassHeader );
    if( !headerFile.open( IO_WriteOnly ) )
    {
	// error msg
	return;
    }
    
    QTextStream headerStream( &headerFile );
    
    OutputDlgeditHeader( headerStream, itsClassHeader, FALSE );
    GenerateSourceHeader( headerStream );
    
    headerFile.close();
    
  // generate source file
    QFile sourceFile( itsClassSource );
    if( !sourceFile.open( IO_WriteOnly ) )
    {
	// error msg
	return;
    }
    
    QTextStream sourceStream( &sourceFile );
    
    OutputDlgeditHeader( sourceStream, itsClassSource, FALSE );
    GenerateSourceSource( sourceStream );
    
    sourceFile.close();
}


void Dialog::GenerateSourceHeader
(
    QTextStream&	stream
)
{
    stream << "#ifndef " << itsClassName << "_included\n";
    stream << "#define " << itsClassName << "_included\n";
    stream << '\n';
    stream << "#include \"" << itsDataHeader << "\"\n";
    
    stream << '\n';
    stream << "class " << itsClassName << " : public " << itsDataName << '\n';
    
    stream << "{\n";
    stream << "    Q_OBJECT\n";
    stream << '\n';
    stream << "public:\n";
    stream << '\n';
    stream << "    " << itsClassName << '\n';
    stream << "    (\n";
    stream << "        QWidget* parent = NULL,\n";
    stream << "        const char* name = NULL\n";
    stream << "    );\n";
    stream << '\n';
    stream << "    virtual ~" << itsClassName << "();\n";
    stream << '\n';

    stream << "};\n";

    stream << "#endif // " << itsClassName << "_included\n";
}


void Dialog::GenerateSourceSource
(
    QTextStream&	stream
)
{
    stream << "#include \"" << itsClassHeader << "\"\n";

    stream << '\n';
    stream << "#define Inherited " << itsDataName << "\n";
    stream << '\n';
    stream << itsClassName << "::" << itsClassName << '\n';
    stream << "(\n";
    stream << "\tQWidget* parent,\n";
    stream << "\tconst char* name\n";
    stream << ")\n";
    stream << "\t:\n";
    stream << "\tInherited( parent, name )\n";
    stream << "{\n";
    
    if( !itsCaption.isEmpty() )
    {
	stream << "\tsetCaption( \"" << itsCaption << "\" );\n";
    }
    
    stream << "}\n";
    stream << "\n\n";
    stream << itsClassName << "::~" << itsClassName << "()\n";
    stream << "{\n";
    stream << "}\n";
}


void Dialog::GenerateData( void )
{
    if( itsDataHeader.isEmpty() )
    {
	QMessageBox::message( "Error", "Dialog data header file name not "
			      "specified" );
	return;
    }
    
    if( itsDataSource.isEmpty() )
    {
	QMessageBox::message( "Error", "Dialog data source file name not "
			      "specified" );
	return;
    }
    
    // generate header file
    QFile headerFile( itsDataHeader );
    if( !headerFile.open( IO_WriteOnly ) )
    {
	// error msg
	return;
    }
    
    QTextStream headerStream( &headerFile );
    
    OutputDlgeditHeader( headerStream, itsDataHeader, TRUE );
    itsLayout->GenerateDataHeader( headerStream, itsDataName,
				   GetWindowStr(), GetWindowHeader() );
    
    headerFile.close();
    
  // generate source file
    QFile sourceFile( itsDataSource );
    if( !sourceFile.open( IO_WriteOnly ) )
    {
	// error msg
	return;
    }
    
    QTextStream sourceStream( &sourceFile );
    
    bool modalWindow = ( ( (itsWindowType == WT_Dialog) || 
			   (itsWindowType == WT_TabDialog) ) &&
			 isDialogModal );

    OutputDlgeditHeader( sourceStream, itsDataSource, TRUE );
    itsLayout->GenerateDataSource( sourceStream, itsDataHeader, 	
				   itsDataName, GetWindowStr(),
				   modalWindow );
    
    sourceFile.close();
}


void Dialog::OutputDlgeditHeader
(
    QTextStream&	stream,
    const QString&	filename,
    bool		editWarning
)
{
    stream << "/**********************************************************************\n\n";
    stream << "\t--- Dlgedit generated file ---\n\n";
    stream << "\tFile: " << filename << '\n';
    stream << "\tLast generated: " << 
	QDateTime::currentDateTime().toString() << '\n';
    stream << '\n';
    
    if( editWarning )
    {
	stream << "\tDO NOT EDIT!!!  This file will be automatically\n";
	stream << "\tregenerated by dlgedit.  All changes will be lost.\n\n";
    }
    
    stream << " *********************************************************************/\n\n";
}


void Dialog::Hide()
{
  itsWindow->hide();
  status = DlgObject_Hidden;
}

void Dialog::Show()
{
  itsWindow->show();
  status = DlgObject_Shown;
}

void Dialog::Raise()
{
  itsWindow->raise();
}

bool Dialog::EnterProject( const Project *prj )
{
  return DlgObject::EnterProject( prj );
}

bool Dialog::LeaveProject( const Project *prj )
{
  return DlgObject::LeaveProject( prj );
}

void Dialog::CPAvailable( bool av )
{
    // propagate signal
    itsWindow->CPAvailable( av );
}

void Dialog::SetName( const QString &dlgFile, bool change )
{
  DlgObject::SetName( dlgFile, change );
  itsWindow->setCaption( "Qt Dialog Editor : " + dlgFile );
}

const QString Dialog::GetWindowStr( void ) const
{
    QString str;
    
    switch( itsWindowType )
    {
	case WT_Dialog:
	{
	    str = "QDialog";
	    break;
	}
	case WT_TabDialog:
	{
	    str = "QTabDialog";
	    break;
	}
	case WT_Widget:
	{
	    str = "QWidget";
	    break;
	}
	case WT_Frame:
	{
	    str = "QFrame";
	    break;
	}
    }
    
    return str;
}


const QString Dialog::GetWindowHeader( void ) const
{
    QString str;
    
    switch( itsWindowType )
    {
	case WT_Dialog:
	{
	    str = "<qdialog.h>";
	    break;
	}
	case WT_TabDialog:
	{
	    str = "<qtabdlg.h>";
	    break;
	}
	case WT_Widget:
	{
	    str = "<qwidget.h>";
	    break;
	}
	case WT_Frame:
	{
	    str = "<qframe.h>";
	    break;
	}
    }
    
    return str;
}

void Dialog::CutSelectedWidgets()
{
  QTextStream stream( itsProjectWnd.GetCPBuffer(), IO_WriteOnly );

  stream << "DlgEdit:v1.2:Dialog:\n";

  itsLayout->CutSelected( stream );

  emit CPDataSet( this );
  emit Changed( this );
}

void Dialog::CopySelectedWidgets()
{
  QTextStream stream( itsProjectWnd.GetCPBuffer(), IO_WriteOnly );

  stream << "DlgEdit:v1.2:Dialog:\n";

  itsLayout->CopySelected( stream );

  emit CPDataSet( this );
}

void Dialog::PasteSelectedWidgets()
{
  QString fileName = itsProjectWnd.GetCPFileName();

  if( ! fileName.isEmpty() )
  {
    itsParser = new DlgParser( fileName );

    itsLayout->AutoSelect( TRUE );

    // Connect the signals which are interesting for us
    connect( itsParser, SIGNAL(NewLayout()), 
             SLOT(RestoreLayout()) );

    // And start the action
    itsParser->StartParsing();

    // That's it for the Parser object
    delete itsParser;    

    itsLayout->AutoSelect( FALSE );

    itsParser = NULL;
    emit Changed( this );
  }
}

void Dialog::LayoutChanged()
{
  emit Changed( this );
  status = DlgObject_Changed;
}
