// $Id: WidgetLayout.cpp,v 1.2 1997/04/17 03:19:51 jharris Exp $

#include <qapp.h>
#include <qpainter.h>
#include <qmsgbox.h>
#include <qfile.h>
#include <qstrlist.h>

#include "WidgetLayout.h"
#include "DialogWnd.h"
#include "GuiTools.h"

#include "widget/DlgButton.h"
#include "widget/DlgButtonGroup.h"
#include "widget/DlgCheckBox.h"
#include "widget/DlgComboBox.h"
#include "widget/DlgFrame.h"
#include "widget/DlgGroupBox.h"
#include "widget/DlgLCDNumber.h"
#include "widget/DlgLabel.h"
#include "widget/DlgLineEdit.h"
#include "widget/DlgListBox.h"
#include "widget/DlgMenuBar.h"
#include "widget/DlgMultiLineEdit.h"
#include "widget/DlgPushButton.h"
#include "widget/DlgRadioButton.h"
#include "widget/DlgScrollBar.h"
#include "widget/DlgUser.h"

#include "GridDialog.h"
#include "DlgParser.h"

#define Inherited QObject

int WidgetLayout::layoutNumber = 1;

WidgetLayout::WidgetLayout
(
 DialogWnd*	parent,
    const char*	name
 )
    :
    Inherited( (QObject *)parent, name ),
    itsParent( parent ),
    itsDlgEditWnd( NULL ),
    isDragging( FALSE),
    itsDragSquare( none ),
  itsParser( NULL ),
  autoSelect( FALSE )
{
    itsCursors[ topLeft		] = &sizeFDiagCursor;
    itsCursors[ left		] = &sizeHorCursor;
    itsCursors[ bottomLeft	] = &sizeBDiagCursor;
    itsCursors[ bottom		] = &sizeVerCursor;
    itsCursors[ bottomRight 	] = &sizeFDiagCursor;
    itsCursors[ right		] = &sizeHorCursor;
    itsCursors[ topRight	] = &sizeBDiagCursor;
    itsCursors[ top		] = &sizeVerCursor;	
    itsCursors[ none		] = &sizeAllCursor;

    itsSelectedWidget.setAutoDelete( TRUE );

    itsNoMoveWidgets.setAutoDelete( FALSE );
    itsDlgWidgets.setAutoDelete( TRUE );
    itsZOrdering.setAutoDelete( FALSE );

    // Generate a name
    QString tmp;
    tmp.setNum( layoutNumber );

    ++layoutNumber;

    itsWidgetPopup = new QPopupMenu;
    itsWidgetPopup->insertItem( "Raise", this, SLOT(RaiseSelectedWidget()) );
    itsWidgetPopup->insertItem( "Lower", this, SLOT(LowerSelectedWidget()) );
    itsWidgetPopup->insertSeparator();
    itsWidgetPopup->insertItem( "Delete", this, SLOT(DeleteSelectedWidget()) );
    itsWidgetPopup->insertSeparator();
    itsWidgetPopup->insertItem( "Properties", this,	
				SLOT(SelectedWidgetProperties()) );
}


WidgetLayout::~WidgetLayout()
{
}


DlgWidget* WidgetLayout::AddWidget
(
    DlgWidget::Type	type
 )
{
    DlgWidget* newWidget = DlgWidget::NewDlgWidget( itsDlgEditWnd, type );

    bool widgetOnTop = TRUE;

    if( type == DlgWidget::DW_ButtonGroup ||
	type == DlgWidget::DW_GroupBox )
    {
	widgetOnTop = FALSE;
    }

    if( widgetOnTop )
    {
	itsZOrdering.append( newWidget );
    }
    else
    {
	itsZOrdering.insert( 0, newWidget );
    }

    QWidget* widget = newWidget->GetWidget();
    QString name = widget->name();

    ConnectToWidget( widget );

    itsDlgWidgets.insert( name, newWidget );
    
    widget->show();
    SetDragging( widget );

  if( autoSelect )
    AddToSelection( widget );

    emit Changed();

    return newWidget;
}
	


void WidgetLayout::SaveContents
(
    QTextStream&	stream
 )
{
    QRect rect = itsDlgEditWnd->geometry();

    stream << "WidgetLayout {\n";
    stream << "\t\tRect {" << rect << "}\n";
    stream << "\t\tGrid {" << itsDlgEditWnd->GetGrid() << "}\n";

    QListIterator< DlgWidget > widgetIter( itsZOrdering );
    DlgWidget* dlgWidget;
    
    while( (dlgWidget = widgetIter()) )
    {
	dlgWidget->SaveContents( stream );
    }
    stream << "}" << endl;
}


void WidgetLayout::ResetContents( void )
{
    //itsDlgWidgets.setAutoDelete( TRUE );
    itsDlgWidgets.clear();
    //itsDlgWidgets.setAutoDelete( FALSE );
    
    itsZOrdering.clear();
    itsNoMoveWidgets.clear();
}



void WidgetLayout::RestoreContents
(
 DlgParser *theParser
 )
{
  // Remember the Parser 
  itsParser = theParser;
    
  // Connecting the Parser
  connect( itsParser, SIGNAL(NewWidget(QString &)), 
           SLOT(RestoreNewWidget(QString &)));
  connect( itsParser, SIGNAL(WidgetKeyValue(QString &,QString &)), 
           SLOT(RestoreWidgetKeyValue(QString &,QString &)) );
  connect( itsParser, SIGNAL(EndWidget()), 
           SLOT(RestoreEndWidget()) );

  connect( itsParser, SIGNAL(LayoutKeyValue(QString &,QString &)), 
           SLOT(RestoreLayoutKeyValue(QString &,QString &)) );
  connect( itsParser, SIGNAL(EndLayout()), 
           SLOT(ContentsRestored()) );
	
}

/* Slot, called when all Widgets for this layout 
 * have been read
 *
 * Undo all connections done by RestoreContents
 */
void WidgetLayout::ContentsRestored()
{
  if( ! itsParser )
    return;

  itsParser->disconnect( this );

  itsParser = NULL;
}

void WidgetLayout::RestoreNewWidget( QString &widgetName )
{
    // Create a new Widget
    widgetInConstruction = DlgWidget::NewDlgWidget( itsDlgEditWnd, 
						    widgetName );
	    
    // No Widget => Error in File, stop action
    if( ! widgetInConstruction )
    {
	itsParser->StopParsing();
	return;
    }	
    else
    {
	ConnectToWidget( widgetInConstruction->GetWidget() );
    }
}

void WidgetLayout::RestoreLayoutKeyValue( QString &key, QString &value )
{
    if( key == "Rect" )
    {
	QTextStream read( value, IO_ReadOnly );
	QRect aRect;
	read >> aRect;
	itsDlgEditWnd->setFixedSize( aRect.width(), aRect.height() );
	itsDlgEditWnd->move( aRect.left(), aRect.top() );
	emit WidgetWndChanged();
    } else if( key == "Grid" )
    {
	itsDlgEditWnd->SetGrid( value.toInt() );
    }
    else
    {
	fprintf( stderr, "Unknown key value Pair: < %s : %s >\n",
		 (const char *)key, (const char *) value );
    }
}

void WidgetLayout::RestoreWidgetKeyValue( QString &key, QString &value )
{
  // Just send the values to the widget
  if( ! widgetInConstruction->RestoreKeyValue( key, value ) )
  {
    fprintf( stderr, "Unknown key value Pair: < %s : %s >\n",
             (const char *)key, (const char *) value );
  }

  // The Name token is of special interest for us
  if( key == "Name" && widgetInConstruction )
    itsDlgWidgets.insert( value, widgetInConstruction );
}

void WidgetLayout::RestoreEndWidget( )
{
  if( widgetInConstruction )
  {
    widgetInConstruction->GetWidget()->show();
    itsZOrdering.append( widgetInConstruction );
    if( autoSelect )
      AddToSelection( widgetInConstruction->GetWidget() );
  }
}


void WidgetLayout::GenerateDataHeader
(
    QTextStream&	stream,
    const QString&	dataName,
    const QString&	dataBaseName,
    const QString&	dataBaseHeader
 )
{
    stream << "#ifndef " << dataName << "_included\n";
    stream << "#define " << dataName << "_included\n";
    stream << '\n';

    // find the list of all the header's we will need
    QListIterator< DlgWidget > iter( itsZOrdering );
    DlgWidget* widget;
    QIntDict<QString> headerDict;
    DlgWidget::Type type;

    headerDict.setAutoDelete( TRUE );

    // create a dictionary of header files that are needed for the mapped
    // variables, without duplicates
    while( (widget = iter()) )
    {
	type = widget->GetType();
	if( !headerDict[ type ] && widget->IsVariableMapped() )
	{
	    headerDict.insert( type, 
			       new QString( widget->GetHeaderFilename() ) );
	}
    }
    
    QIntDictIterator< QString > headerIter( headerDict );
    QString* header;
    
    stream << "#include " << dataBaseHeader << '\n';

    while( (header = headerIter()) )
    {
	stream << "#include " << *header << '\n';
    }
    
    stream << '\n';
    stream << "class " << dataName << " : public " << dataBaseName << '\n';
    stream << "{\n";
    stream << "    Q_OBJECT\n";
    stream << '\n';
    stream << "public:\n";
    stream << '\n';
    stream << "    " << dataName << '\n';
    stream << "    (\n";
    stream << "        QWidget* parent = NULL,\n";
    stream << "        const char* name = NULL\n";
    stream << "    );\n";
    stream << '\n';
    stream << "    virtual ~" << dataName << "();\n";
    stream << "\n";

    // define slots
    
    QDict<QString> publicSlots, protectedSlots;
    SignalConnection* sig;
    QString slot;
    const QStrList& qdialogSlots = GuiTools::GetQDialogSlots();
    
    publicSlots.setAutoDelete( TRUE );
    protectedSlots.setAutoDelete( TRUE );

    // scan through all the widgets and their slots, placing the slots into
    // the correct dictionary for the scope making sure there are no 
    // duplicates
    iter.toFirst();
    while( (widget = iter()) )
    {
	QListIterator<SignalConnection> 
	    signalIter( widget->GetConnectedSignalList() );
	
	while( (sig = signalIter()) )
	{
	    slot = sig->GetSlot();

	    // if the base is a dialog and the slot is defined by the QDialog
	    // class, skip it
	    if( dataBaseName != "QDialog" || !qdialogSlots.contains( slot ) )
	    {
		switch( sig->GetScope() )
		{
		    case SignalConnection::Protected:
		    {
			if( !protectedSlots[ slot ] )
			{
			    protectedSlots.insert( slot, new QString( slot ) );
			}
			break;
		    }
		    case SignalConnection::Public:
		    {
			if( !publicSlots[ slot ] )
			{
			    publicSlots.insert( slot, new QString( slot ) );
			}
			break;
		    }
		}
	    }
	}
    }
    
    QDictIterator<QString> publicIter( publicSlots );
    QString* slotStr;

    stream << "public slots:\n\n";
    while( (slotStr = publicIter()) )
    {
	stream << "    virtual void " << *slotStr << ";\n";
    }
    stream << '\n';

    QDictIterator<QString> protectedIter( protectedSlots );
    
    stream << "protected slots:\n\n";
    while( (slotStr = protectedIter()) )
    {
	// if the user declared a slot to be both protected and private,
	// make it public
	if( !publicSlots[ *slotStr ] )
	{
	    stream << "    virtual void " << *slotStr << ";\n";
	}
    }
    stream << '\n';    

    // declare mapped variables
    
    stream << "protected:\n";

    iter.toFirst();
    while( (widget = iter()) )
    {
	if( widget->IsVariableMapped() )
	{
	    stream << "    " << widget->GetWidgetClass() << "* " <<
		widget->GetVariableName() << ";\n";
	}
    }
    
    stream << '\n';
    stream << "};\n";

    stream << '\n';
    stream << "#endif // " << dataName << "_included\n";
}



void WidgetLayout::GenerateDataSource
(
    QTextStream&	stream,
    const QString&	dataHeader,
    const QString&	dataName,
    const QString&	dataBaseName,
    bool		modalWindow
 )
{
    stream << "#include \"" << dataHeader << "\"\n";
    stream << '\n';
    stream << "#define Inherited " << dataBaseName << "\n";
    stream << "\n";

    QIntDict<QString> nonMappedClasses;
    QListIterator<DlgWidget> widgetIter( itsZOrdering );
    DlgWidget* widget;
    DlgWidget::Type type;

    nonMappedClasses.setAutoDelete( TRUE );

    // create a dictionary of the header files needed for the widgets that
    // have not been mapped
    while( (widget = widgetIter()) )
    {
	type = widget->GetType();
	if( !nonMappedClasses[ type ] && !widget->IsVariableMapped() )
	{
	    nonMappedClasses.insert( 
		type,
		new QString( widget->GetHeaderFilename() ) );
	}
    }
    
    QIntDictIterator<QString> headerIter( nonMappedClasses );
    QString* header;
    
    while( (header = headerIter()) )
    {
	stream << "#include " << *header << '\n';
    }
    
    stream << '\n';
    stream << dataName << "::" << dataName << '\n';
    stream << "(\n";
    stream << "\tQWidget* parent,\n";
    stream << "\tconst char* name\n";
    stream << ")\n";
    stream << "\t:\n";
    
    if( modalWindow )
    {
	stream << "\tInherited( parent, name, TRUE )\n";
    }
    else
    {
	stream << "\tInherited( parent, name )\n";
    }

    stream << "{\n";
    
    // initialize widgets, do the button groups last to make sure the 
    // widgets inserted are constructed
    QString varName;
    QIntDict<QString> tmpWidgetNameDict;

    tmpWidgetNameDict.setAutoDelete( TRUE );
    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {	
	if( widget->GetType() != DlgWidget::DW_ButtonGroup )
	{
	    GenerateWidgetSource( widget, tmpWidgetNameDict, stream );
	}
    }

    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {	
	if( widget->GetType() == DlgWidget::DW_ButtonGroup )
	{
	    GenerateWidgetSource( widget, tmpWidgetNameDict, stream );
	}
    }
    

    // set width and height
    stream << "\tresize( " << itsDlgEditWnd->width() << ", " <<
	itsDlgEditWnd->height() << " );\n";
    
    stream << "}\n";
    stream << "\n\n";

    // define the descructor

    stream << dataName << "::~" << dataName << "()\n";
    stream << "{\n";
    stream << "}\n";
    
    // define all the slots

    QDict<SignalConnection> slotDict;
    SignalConnection* sig;
    QString slot;
    const QStrList& qdialogSlots = GuiTools::GetQDialogSlots();

    slotDict.setAutoDelete( FALSE );
    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {
	QListIterator<SignalConnection> 
	    signalIter( widget->GetConnectedSignalList() );

	while( (sig = signalIter()) )
	{
	    slot = sig->GetSlot();

	    // If the base class is QDialog, and the slot is defined by the 
	    // QDialog class, skip it
	    if( dataBaseName != "QDialog" || !qdialogSlots.contains(slot) )
	    {
		if( !slotDict[ slot ] )
		{
		    slotDict.insert( slot, sig );
		
		    stream << "void " << dataName << "::" << slot << '\n';
		    stream << "{\n";
		    stream << "}\n";
		}
	    }
	}
    }
}


void WidgetLayout::GenerateWidgetSource
(
    DlgWidget*		widget,
    QIntDict<QString>&	nameDict,
    QTextStream&	stream
)
{
    QString varName;
    DlgWidget::Type type;
    
    // construct a new widget
    if( !widget->IsVariableMapped() )
    {
	type = widget->GetType();
	if( !nameDict[ type ] )
	{
	    varName = "tmp" + widget->GetWidgetClass();
	    nameDict.insert( type, new QString( varName ) );

	    // declare new variable
	    stream << '\t' << widget->GetWidgetClass() << "* " << 
		varName << ";\n";
	}
	else
	{
	    varName = *nameDict[ type ];
	}
    }
    else
    {
	varName = widget->GetVariableName();
	    
	// raise widget so it doesn't get hidden under anything
	//stream << '\t' << varName << widget->Dot() << "raise();\n";
    }
	
    // 'new' variable
    widget->GenerateCtor( stream, varName );

    varName = "\t" + varName + widget->Dot();
    widget->GenerateSource( stream, varName, itsZOrdering );

    stream << '\n';
}



void WidgetLayout::SetDragging
(
    QWidget*	widget
 )
{
    ClearSelection();
    AddToSelection( widget );
    ProcessResize( widget );
    
    itsDragSquare = none;
    
    UpdateCursor( widget );
    
    itsWidgetClick.setX( widget->width()/2 );
    itsWidgetClick.setY( widget->height()/2 );
    
    widget->move( 0,0 );
}


void WidgetLayout::ConnectToWidget( QWidget* widget )
{
    connect( widget, SIGNAL(MousePress(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMousePress(QWidget*,QMouseEvent*)) );
    connect( widget, SIGNAL( MouseMove( QWidget*, QMouseEvent* ) ),
	     SLOT( ProcessMouseMove( QWidget*, QMouseEvent* ) ) );
    connect( widget, SIGNAL( MouseRelease() ),
	     SLOT( ProcessMouseRelease() ) );
    connect( widget, SIGNAL( Paint( QWidget* ) ),
	     SLOT( ProcessPaint( QWidget* ) ) );
    connect( widget, SIGNAL( Resize( QWidget* ) ),
	     SLOT( ProcessResize( QWidget* ) ) );
}


void WidgetLayout::AlignTop()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int top = sel->widget->geometry().top();
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->move( widget->x(), top );	
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignBottom()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int bottom = sel->widget->geometry().bottom();
	QRect rect;
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		rect = widget->geometry();
		rect.moveBottomRight( QPoint( rect.right(), bottom ) );
		widget->setGeometry( rect );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignLeft()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int left = sel->widget->geometry().left();
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->move( left, widget->y() );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignRight()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int right = sel->widget->geometry().right();
	QRect rect;
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		rect = widget->geometry();
		rect.moveBottomRight( QPoint( right, rect.bottom() ) );
		widget->setGeometry( rect );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeHorizontal()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int width = sel->widget->width();
	QWidget* widget;
	
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;

	    // make sure you don't size one that isn't supposed to be sized
	    // no move == no size
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->setFixedSize( width, widget->height() );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeVertical()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int height = sel->widget->height();
	QWidget* widget;
	
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;

	    // make sure you don't size one that isn't supposed to be sized
	    // no move == no size
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->setFixedSize( widget->width(), height );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeBothHV()
{
    SizeHorizontal();
    SizeVertical();
}


void WidgetLayout::ProcessMousePress
(
    QWidget* 		widget,
    QMouseEvent*	event
 )
{
    if( ! enabled )
	return;

    switch( event->button() )
    {
	case LeftButton:
	{
	    if( event->state() & ControlButton )
	    {
		if( IsSelected( widget ) )
		{
		    RemoveFromSelection( widget );
		}
		else
		{
		    AddToSelection( widget );
		}
	    } else 
	    {
		ClearSelection();
		AddToSelection( widget );
	    }

	    int i;
	    itsDragSquare = none;
	    QPoint pos = event->pos();
	    Selected *sel = itsSelectedWidget.first();
	    
	    if( itsSelectedWidget.count() == 1 )
	    {
		for( i = 0; (i < 8) && (itsDragSquare == none); i++ )
		{
		    if( sel->itsResizeSquares[i].contains( pos ) )
		    {
			itsDragSquare = (enum ResizeSquare)i;
		    }
		}
	    }
	    UpdateCursor( widget );
	
	    itsWidgetClick = event->pos();
	    isDragging = TRUE;	
	
	    break;
	}
	
	case MidButton:
	{
	    ClearSelection();
      emit WidgetDeselected();
	    break;
	}
		
	case RightButton:
	{
	    ClearSelection();
	    AddToSelection( widget );

	    DlgWidget* dlgWidget = GetDlgWidget( widget );
	    if( dlgWidget )
	    { 
		itsWidgetPopup->popup( QCursor::pos() );
	    }
	    
	    break;
	}
    }
}


void WidgetLayout::ProcessMouseMove
(
    QWidget*	 	widget,
    QMouseEvent*	event
)
{
    if( ! enabled )
	return;

    if( isDragging && 
	itsSelectedWidget.count() > 1 && 
	itsDragSquare == none )              // Moving only
    {
	QPoint oldWidgetPos = widget->pos();
	QRect  newWidgetRect = CalculateGeometryRect( widget, event->pos() );
	QPoint newWidgetPos = newWidgetRect.topLeft();
	bool   moveOKForAll = TRUE;
	int    maxWidth  = itsDlgEditWnd->width();
	int    maxHeight = itsDlgEditWnd->height();

	if( oldWidgetPos != newWidgetPos )
	{
	    QPoint delta = newWidgetPos - oldWidgetPos;
	    QListIterator<Selected> iter( itsSelectedWidget );
	    Selected *sel;

	    while( (sel = iter()) && moveOKForAll )
	    {
		// don't move one that's not allowed
		if( itsNoMoveWidgets[ sel->widget->name() ] )
		{
		    moveOKForAll = FALSE;
		}
		else
		{
		    if( sel->widget == widget )
		    {
			sel->newPos = newWidgetPos;
		    }
		    else
		    {
			sel->newPos = sel->widget->pos() + delta;
			int x = sel->newPos.x();
			int y = sel->newPos.y();

			if( x < 0 || x+sel->widget->width() > maxWidth ||
			    y < 0 || y+sel->widget->height() > maxHeight )
			{
			    moveOKForAll = FALSE;
			}
		    }
		}
	    }

	    if( moveOKForAll )
	    {
		iter.toFirst();

		while( (sel = iter()) )
		{
		    sel->widget->move( sel->newPos );
		}
	  
		WidgetWndChanged();
		WidgetGeometryChange( widget );
		emit Changed();
	    }      
	}
    } 
    else if( isDragging && itsSelectedWidget.first() &&
	     itsSelectedWidget.first()->widget == widget) 
    {
	// don't move the top left corner of a non movable widget
	if( ( itsNoMoveWidgets[ widget->name() ] == NULL ) ||
	    ( (itsDragSquare == right) || (itsDragSquare == bottomRight) ||
	      (itsDragSquare == bottom) ) )
	{
	    QRect rect = CalculateGeometryRect( widget, event->pos() );
	    widget->setFixedSize( rect.width(), rect.height() );
	    widget->move( rect.left(), rect.top() );

	    // currently only the widget frame is not moveable
	    if( itsNoMoveWidgets[ widget->name() ] != NULL )
	    {
		emit WidgetWndChanged();
	    }
	
	    WidgetGeometryChange( widget );
	    emit Changed();
	}
    }

    UpdateCursor( widget );
}

void WidgetLayout::ProcessMouseRelease()
{
  if( ! enabled )
    return;
  
  isDragging = FALSE;
  
}


void WidgetLayout::ProcessPaint
(
    QWidget*		//widget
)
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;
    
    while( (sel = iter()) )
    {
	QPainter	painter;	
	QBrush		brush( sel->widget->foregroundColor(), Dense4Pattern );
	
	painter.begin( sel->widget );
	painter.setPen( QPen( sel->widget->foregroundColor(), 3, DotLine ) );
	
	// draw border around widget
	painter.drawRect( painter.window() );

	// draw rects to resize
	int i;
	for( i=0; i < 8; i++ )
	{
	    painter.fillRect( sel->itsResizeSquares[i], brush );
	}
	
	painter.end();
    }
}
	

void WidgetLayout::ProcessResize
(
    QWidget* widget
 )
{
  if( ! enabled )
    return;

  Selected findSel( widget );
  if( itsSelectedWidget.find( &findSel ) != -1 )
    {	
    Selected *sel = itsSelectedWidget.current();
	QRect	rect = widget->geometry();
	QRect	sq( 0,0, 7,7 );
	QRect	null( 0,0, 0,0 );
	bool	moveable = (itsNoMoveWidgets[ widget->name() ]  == NULL );
    
	rect.moveTopLeft( QPoint(0,0) );

	sq.moveTopLeft( rect.topLeft());
    sel->itsResizeSquares[ topLeft	] = moveable ? sq : null;

	sq.moveBy( 0, rect.height()/2 - sq.height()/2 );
    sel->itsResizeSquares[ left		] = moveable ? sq : null;

	sq.moveBottomLeft( rect.bottomLeft() );
    sel->itsResizeSquares[ bottomLeft	] = moveable ? sq : null;

	sq.moveBy( rect.width()/2 - sq.width()/2, 0 );
    sel->itsResizeSquares[ bottom	] = sq;

	sq.moveBottomRight( rect.bottomRight() );
    sel->itsResizeSquares[ bottomRight	] = sq;

	sq.moveBy( 0, -(rect.height()/2 - sq.height()/2 ) );
    sel->itsResizeSquares[ right		] = sq;

	sq.moveTopRight( rect.topRight() );
    sel->itsResizeSquares[ topRight	] = moveable ? sq : null;

	sq.moveBy( -(rect.width()/2 - sq.width()/2), 0 );
    sel->itsResizeSquares[ top		] = moveable ? sq : null;
    }
}
    

void WidgetLayout::RaiseSelectedWidget()
{
  if( ! enabled )
    return;

  if( itsSelectedWidget.count() == 1 )
  {
    itsSelectedWidget.first()->widget->raise();
    
    DlgWidget* dlgWidget = GetDlgWidget( itsSelectedWidget.first()->widget );
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	    itsZOrdering.append( dlgWidget );
    }
    
    emit Changed();
  }
}


void WidgetLayout::LowerSelectedWidget()
{
  if( ! enabled )
    return;

  if( itsSelectedWidget.count() == 1 )
  {    
    itsSelectedWidget.first()->widget->lower();

    DlgWidget* dlgWidget = GetDlgWidget( itsSelectedWidget.first()->widget );
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	    itsZOrdering.insert( 0, dlgWidget );
    }
    emit Changed();
  }
}


void WidgetLayout::DeleteSelectedWidget()
{
    if( ! enabled )
	return;

    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;

    while( (sel = iter()) )
    {
	DlgWidget* dlgWidget = GetDlgWidget( sel->widget );
	if( dlgWidget )
	{
	    itsZOrdering.removeRef( dlgWidget );
	
	    QString name = sel->widget->name();

	    // remove will delete dlgWidget but not the actual widget
	    itsDlgWidgets.remove( name ); 
	    delete sel->widget;
	
	    itsNoMoveWidgets.remove( name );

	    emit Changed();
	}
    }
    itsSelectedWidget.clear();
}

void WidgetLayout::SelectedWidgetProperties()
{
    if( ! enabled )
	return;

    if( itsSelectedWidget.count() > 1 )
	return;

    Selected *sel = itsSelectedWidget.first();
    DlgWidget* dlgWidget;

    if( sel && (dlgWidget = GetDlgWidget(sel->widget)) )
    {
	// Remember the name first
	QString origName( sel->widget->name() );

	if( dlgWidget->DoPropertyDialog( itsDlgEditWnd, "Properties" ) == 
	    QDialog::Accepted )
	{
	    emit Changed();

	    // Check the name and register if neccesary
	    QString newName( sel->widget->name() );

	    // user might have changed the widget's name or variable mapping
	    emit WidgetSelected( newName, dlgWidget->GetVariableName() );

	    if( newName != origName )
	    {
		if( itsDlgWidgets[newName] )
		{
		    QMessageBox::message( "Name conflict...",
					  "The new widget name is already in use\n"
					  "The old name is restored !",
					  "OK" );
		    sel->widget->setName( origName );
		} else
		{
		    dlgWidget = itsDlgWidgets.take( origName );
		    itsDlgWidgets.insert( newName, dlgWidget );
		}
	    }
	}
    }
    else // assume getting properties on the whole dialog
    {
	if( itsParent->DoPropertyDialog() == QDialog::Accepted )
	{
	    emit Changed();
	}
    }
}

void WidgetLayout::OptionsGrid( )
{
	GridDialog gridDialog( NULL, "GridDialog", itsDlgEditWnd->GetGrid() );
	if( gridDialog.exec() == QDialog::Accepted )
  {
		itsDlgEditWnd->SetGrid( gridDialog.GetNewGrid() );
    emit Changed();
  }
}

void WidgetLayout::EnableLayout()
{

  if( ! itsDlgEditWnd )
    return;

  connect( itsDlgEditWnd, 
           SIGNAL(MousePress(QWidget*,QMouseEvent*)),
           SLOT(ProcessMousePress(QWidget*,QMouseEvent*)) );
  connect( itsDlgEditWnd, 
           SIGNAL(MouseMove(QWidget*,QMouseEvent*)),
           SLOT(ProcessMouseMove(QWidget*,QMouseEvent*)) );
  connect( itsDlgEditWnd, 
           SIGNAL(MouseRelease()), 
           SLOT(ProcessMouseRelease()) );
  connect( itsDlgEditWnd, 
           SIGNAL(Paint(QWidget*)),
           SLOT(ProcessPaint(QWidget*)) );
  connect( itsDlgEditWnd, 
           SIGNAL(Resize(QWidget*)),
           SLOT(ProcessResize(QWidget*)) );

  enabled = TRUE;
}
 
void WidgetLayout::DisableLayout()
{
  if( ! itsDlgEditWnd )
    return;

  itsDlgEditWnd->disconnect( this );

  enabled = FALSE;

  ClearSelection();
}

void WidgetLayout::SetSelected
(
    QWidget*	widget,
    bool	selected
 )
{
  if( widget )
  {
      widget->setMouseTracking( selected );
    
      widget->update();

      if( selected )
      {
	  WidgetGeometryChange( widget );

	  QString varName;
	  DlgWidget* dlgWidget = itsDlgWidgets[ widget->name() ];
	  if( dlgWidget )
	  {
	      varName = dlgWidget->GetVariableName();
	  }

	  emit WidgetSelected( widget->name(), varName );
      }
  }
}

void WidgetLayout::ClearSelection
(
)
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;
    
    while( (sel = iter()) )
    {
	SetSelected( sel->widget, FALSE );
	sel->widget->setCursor( arrowCursor );
    }

    itsSelectedWidget.clear();
}

void WidgetLayout::AddToSelection
(
    QWidget *widget
)
{
    if( ! IsSelected( widget ) )
    {
	itsSelectedWidget.append( new Selected( widget ) );
    
	SetSelected( widget, TRUE );
	ProcessResize( widget );
    }
}

void WidgetLayout::RemoveFromSelection
(
    QWidget *widget
)
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected *sel;

    while( (sel = iter()) && (sel->widget != widget) );

    if( sel )
    {
	// SetSelected( sel->widget, FALSE );
	//sel->widget->setCursor( arrowCursor );
	itsSelectedWidget.remove( sel );
	SetSelected( widget, FALSE );
	ProcessResize( widget );
    }
}


bool WidgetLayout::IsSelected( QWidget*	widget )
{
    Selected sel( widget );
    return itsSelectedWidget.contains( &sel );
}


void WidgetLayout::UpdateCursor
(
    QWidget*	widget
 )
{
    if( isDragging )
    {
    }
  else 
    {
    if( IsSelected( widget ) )
    {
      Selected *sel = itsSelectedWidget.first();
	QPoint  cursorPos = QCursor::pos();	
	cursorPos = widget->mapFromParent( cursorPos );
	cursorPos = widget->mapFromGlobal( QCursor::pos() );

	ResizeSquare rs = none;
	int i;
	
	for( i = 0; (i < 8) && (rs == none); i++ )
	{
        if( sel->itsResizeSquares[i].contains( cursorPos ) )
	    {
		rs = (ResizeSquare) i;
	    }
	}
	
	widget->setCursor( *itsCursors[rs] );
    }
    else
    {
	widget->setCursor( arrowCursor );
    }
  }
}


void WidgetLayout::WidgetGeometryChange( QWidget* widget )
{
    if( itsNoMoveWidgets[ widget->name() ] == NULL )
    {
	emit WidgetGeometryChanged( widget->x(), widget->y(),
				    widget->width(), widget->height() );
    }
    else
    {
	emit WidgetGeometryChanged( 0, 0, 
				    widget->width(), widget->height() );
    }
}


QRect WidgetLayout::CalculateGeometryRect
(
    QWidget*	widget,
    QPoint	pos
 )
{
    QPoint eventPos = widget->mapToParent( pos );
    QRect rect;
    QRect widgetRect = widget->geometry();
    int minWidth = 10; //widget->minimumSize().width();
    int minHeight = 10; //widget->minimumSize().height();
	int grid = itsDlgEditWnd->GetGrid();
	int width, height;

    switch( itsDragSquare )
    {
	case none: // move widget	
	{
	    rect = widgetRect;
			QPoint gridPoint = eventPos - itsWidgetClick;
			if( grid > 1 )
			{
				gridPoint /= grid;
				gridPoint *= grid;
			}
	    rect.moveTopLeft( gridPoint );

	    QRect parentRect = widget->parentWidget()->geometry();

	    if( rect.left() < 0 )
	    {
		rect.moveTopLeft( QPoint(0,rect.top()) );
	    }
	    
	    if( rect.top() < 0 )
	    {
		rect.moveTopLeft( QPoint(rect.left(),0) );
	    }
	    
	    if( rect.right() > parentRect.width() )
	    {
		rect.moveBottomRight(QPoint(parentRect.width(),rect.bottom()));
	    }
	    
	    if( rect.bottom() > parentRect.height() )
	    {
		rect.moveBottomRight(QPoint(rect.right(),parentRect.height()));
	    }
	    
	    break;
	}

	case topLeft:
	{
			width = widgetRect.right() - eventPos.x();
			height = widgetRect.bottom() - eventPos.y();

			if( grid > 1 )
			{
				width /= grid; width *= grid;
				height /= grid; height *= grid;
			}
	    rect.setWidth( QMAX( width, minWidth) );
	    rect.setHeight( QMAX( height, minHeight) );
	    rect.moveBottomRight( widgetRect.bottomRight() );
	    
	    break;
	}
	    
	case left:
	{
			width = widgetRect.right() - eventPos.x();

			if( grid > 1 )
			{
				width /= grid; width *= grid;
			}

			rect = widgetRect;
	    rect.setWidth( QMAX(width, minWidth) );
	    rect.moveTopRight( widgetRect.topRight() );

	    break;
	}
	    
	case bottomLeft:
	{
			width = widgetRect.right() - eventPos.x();
			height = eventPos.y() - widgetRect.y();

			if( grid > 1 )
			{
				width /= grid; width *= grid;
				height /= grid; height *= grid;
			}

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height,	minHeight) );
	    rect.moveTopRight( widgetRect.topRight() );
	    
	    break;
	}

	case bottom:
	{
			height = eventPos.y() - widgetRect.y();

			if( grid > 1 )
			{
				height /= grid; height *= grid;
			}

	    rect = widgetRect;
	    rect.setHeight( QMAX( height, minHeight ) );

	    break;
	}
	    
	case bottomRight:
	{
			width = eventPos.x() - widgetRect.x();
			height = eventPos.y() - widgetRect.y(); 

			if( grid > 1 )
			{
				width /= grid; width *= grid;
				height /= grid; height *= grid;
			}

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height,	minHeight) );
	    rect.moveTopLeft( widgetRect.topLeft() );
	    
	    break;
	}
	    
	case right:
	{
			width =  eventPos.x() - widgetRect.x();

			if( grid > 1 )
			{
				width /= grid; width *= grid;
			}

	    rect = widgetRect;
	    rect.setWidth( QMAX( width, minWidth ) );

	    break;
	}
	    
	case topRight:
	{
			width = eventPos.x() - widgetRect.x();
			height = widgetRect.bottom() - eventPos.y();

			if( grid > 1 )
			{
				width /= grid; width *= grid;
				height /= grid; height *= grid;
			}

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height, minHeight) );
	    rect.moveBottomLeft( widgetRect.bottomLeft() );

	    break;
	}
	    
	    
	case top:
	{
			height = widgetRect.bottom() - eventPos.y();
			
			if( grid > 1 )
			{
				height /= grid; height *= grid;
			}

	    rect = widgetRect;
	    rect.setHeight( QMAX(height, minHeight) );
	    rect.moveBottomLeft( widgetRect.bottomLeft() );

	    break;
	}
    }

    QRect parentRect = widget->parentWidget()->geometry();

    if( itsDragSquare == none )
    {
	if( rect.left() < 0 )
	{
	    rect.moveTopLeft( QPoint(0,rect.top()) );
	}
	    
	if( rect.top() < 0 )
	{
	    rect.moveTopLeft( QPoint(rect.left(),0) );
	}
	    
	if( rect.right() > parentRect.width() )
	{
	    rect.moveBottomRight(QPoint(parentRect.width(),rect.bottom()));
	}
	    
	if( rect.bottom() > parentRect.height() )
	{
	    rect.moveBottomRight(QPoint(rect.right(),parentRect.height()));
	}
    }
    else
    {	    
	parentRect.moveTopLeft( QPoint(0,0) );

	if( rect.left() < parentRect.left() )
	{
	    rect.setLeft( parentRect.left() );
	}

	if( rect.top() < parentRect.top() )
	{
	    rect.setTop( parentRect.top() );
	}

	if( rect.right() > parentRect.right() )
	{
	    rect.setRight( parentRect.right() );
	}
    
	if( rect.bottom() > parentRect.bottom() )
	{
	    rect.setBottom( parentRect.bottom() );
	}
    }
    

    return rect;
}


bool WidgetLayout::CopySelected( QTextStream &stream )
{
    if( itsSelectedWidget.isEmpty() )
	return FALSE;

    stream << "WidgetLayout {\n";

    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;

    while( (sel = iter()) )
    {
	DlgWidget *dlgWidget = GetDlgWidget( sel->widget );
	if( ! dlgWidget )
	    continue;

	dlgWidget->SaveContents( stream );
    }
    stream << "}" << endl;

    return TRUE;
}


bool WidgetLayout::CutSelected( QTextStream &stream )
{
  if( CopySelected( stream ) )
  {
    DeleteSelectedWidget();
    return TRUE;
  }

  return FALSE;
}


void WidgetLayout::RaiseDlgWidget
(
    const DlgWidget*	dlgWidget
)
{
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	itsZOrdering.append( dlgWidget );
    }
}


void WidgetLayout::LowerDlgWidget
(
    const DlgWidget*	dlgWidget
)
{
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	itsZOrdering.insert( 0, dlgWidget );
    }
}

    
QWidget* WidgetLayout::GetSelectedWidget( void )
{
    QWidget* widget = NULL;

    if( itsSelectedWidget.count() == 1 )
    {
	widget = itsSelectedWidget.first()->widget;
    }

    return widget;
}


DlgWidget* WidgetLayout::GetDlgWidget
(
    const QWidget*	widget 
)
{
    DlgWidget* dlgWidget = NULL;

    if( widget )
    {
	QString name( widget->name() );
	dlgWidget = itsDlgWidgets[ name ];	
    }

    return dlgWidget;
}


DlgWidget* WidgetLayout::GetSelectedDlgWidget( void )
{
    DlgWidget* dlgWidget = NULL;

    if( itsSelectedWidget.count() == 1 )
    {
	dlgWidget =  GetDlgWidget( itsSelectedWidget.first()->widget );
    }
    
    return dlgWidget;
}


