#ifndef WidgetLayout_included
#define WidgetLayout_included

#include <qobject.h>
#include <qwidget.h>
#include <qlist.h>
#include <qdict.h>
#include <qintdict.h>
#include <qrect.h>
#include <qcursor.h>
#include <qpopmenu.h>

#include "widget/DlgWidget.h"

class DlgParser;
class DialogWnd;
class WidgetFrame;

struct Selected
{
  Selected( QWidget *w )
  { widget = w; };

  QWidget *widget;
  QPoint   newPos;

  QRect		itsResizeSquares[8];
  // squares to grab and drag in the widget's coordinates for the
  // selected widget
};

class SelectList : public QList<Selected>
{
protected:
int compareItems( GCI i1, GCI i2 )
{ return (((const Selected*)i1)->widget != 
          ((const Selected*)i2)->widget ); };
};

class WidgetLayout : public QObject
{
  Q_OBJECT
    
public: // methods
    
  WidgetLayout
  (
   DialogWnd*	parent = NULL,
   const char*	name = NULL
   );


    virtual ~WidgetLayout();
    

  DlgWidget* AddWidget
  ( 	
   DlgWidget::Type	type
   );
    
  void SetDragging
  (
   QWidget*	widget
   );

  void SetNoMove
  (
   QWidget*	widget
   );
  // don't allow user to move top left corner

  QWidget* GetSelectedWidget( void );

  DlgWidget *GetSelectedDlgWidget();

  void SetDlgEditWnd
  (
   WidgetFrame*	editWnd
   );
  // set the window used to hold the widgets

  WidgetFrame *GetDlgEditWnd() const;
  // retrieve the window used to hold the widgets

  void SaveContents
  (
   QTextStream&
   );
  // save contents, including the widgets

  void ResetContents( void );
  // reset the contents of the layout

  void RestoreContents
  (
   DlgParser *
   );
  // restore contents, creating the widgets

  void GenerateDataHeader
  (
      QTextStream&	stream,
      const QString&	dataName,
      const QString&	dataBaseName,
      const QString&	dataBaseHeader
   );
  // generate the dialog data header file

  void GenerateDataSource
  (
      QTextStream&	stream,
      const QString&	dataHeader,
      const QString&	dataName,
      const QString&	dataBaseName,
      bool		modalWindow
   );
  // generate the dialog data source file

    void GenerateWidgetSource
    (
	DlgWidget*		widget,
	QIntDict<QString>&	nameDict,	
	QTextStream&		stream
    );
    // generate the source for one widget

  void RaiseDlgWidget
  (
   const DlgWidget*	dlgWidget
   );
  // raise the dlg widget in the z ordering

  void LowerDlgWidget
  (
   const DlgWidget*	dlgWidget
   );
  // lower the dlg widget in the z ordering
    
  void EnableLayout();
  // Enable interaction

  void DisableLayout();
  // Disable interaction

  void AutoSelect( bool = TRUE );
  // Automatically select all new widgets

  bool CopySelected( QTextStream & );
  bool CutSelected( QTextStream & );

signals:
  void Changed();

    void WidgetWndChanged();

    void WidgetGeometryChanged( int, int, int, int );
    // emit when a widget's geometry has changed with its x and y coordinate
    // and width and height

    void WidgetSelected( const QString&, const QString& );
    // emit when a widget is selected with the widget's name and 
    //mapped variable
    
  void WidgetDeselected();
  // emit when all widgets where deselected (Mouse button 2)
    
  public slots: // methods

  // Process events on the widgets

  void ProcessMousePress
  (
   QWidget* 	widget,
   QMouseEvent*	event
   );
    
  void ProcessMouseMove
  (	
   QWidget*	widget,
   QMouseEvent*	event
   );
    
  void ProcessMouseRelease();
    
  void ProcessPaint
  (	
   QWidget*	widget
   );
    
  void ProcessResize
  (
   QWidget*	widget
   );

  void RaiseSelectedWidget();

  void LowerSelectedWidget();

  void DeleteSelectedWidget();
    
  void SelectedWidgetProperties();
  void OptionsGrid();

  void ContentsRestored();
  // All Widgets are restored, disconnect from Parser

    void AlignTop();
    // align selected widgets to top of first widget

    void AlignBottom();
    // align selected widgets to bottom of first widget

    void AlignLeft();
    // align selected widgets to left of first widget

    void AlignRight();
    // align selected widgets to right of first widget

    void SizeHorizontal();
    // give selected widgets the same width as the first widget

    void SizeVertical();
    // give selected widgets the same height as the first widget

    void SizeBothHV();
    // give selected widgets the same width and height as the first widget

  private slots: // methods
  void RestoreNewWidget
  ( 
   QString &widgetName 
   );
  // Restore a new Widget

  void RestoreLayoutKeyValue
  ( 
   QString &key, 
   QString &value 
   );        
  // Restore a single key/value pair for the Layout 

  void RestoreWidgetKeyValue
  ( 
   QString &key, 
   QString &value 
   );        
  // Restore a single key/value pair for a Widget

  void RestoreEndWidget();        
  // Constructing a new widget is finished

private: // methods

  QRect CalculateGeometryRect
  (
   QWidget*	widget,
   QPoint		pos
   );
  // calculate the new geometry rect for the widget


  DlgWidget* GetDlgWidget
  (
   const QWidget*	widget 
   );
  // given a widget, return its DlgWidget
    
  void SetSelected
  (
   QWidget*	widget,
   bool 		selected
   );

  void ClearSelection();

  void AddToSelection( QWidget * );

  void RemoveFromSelection( QWidget * );

    bool IsSelected( QWidget*	widget );


  void UpdateCursor
  (
   QWidget*	widget
   );

    void WidgetGeometryChange( QWidget* widget );
    // process a change in widget geometry

    void ConnectToWidget( QWidget* widget );
    // connect the necessary events from the widget to the slots for 
    // manipulating the widget
        

private: // attributes

  enum ResizeSquare
  {
    topLeft 	= 0,
    left 		= 1,
    bottomLeft	= 2,
    bottom		= 3,
    bottomRight	= 4,
    right		= 5,
    topRight	= 6,
    top		= 7,
    none		= 8
  };

  enum RestoreAttr
  {
    APPLICATION,
    LAYOUT,
    WIDGET
  };

  RestoreAttr attrType;
  // The type of the actual read attribute
  
  DialogWnd *itsParent;
  // The application
  
  WidgetFrame*		itsDlgEditWnd;
  // window on which the widgets are created and manipulated
    
    QPopupMenu* itsWidgetPopup;
    // popup menu for widgets
    
  SelectList itsSelectedWidget;
  // List of actual selected Widgets

  QRect		itsResizeSquares[8];
  // squares to grab and drag in the widget's coordinates for the
  // selected widget

  QPoint		itsWidgetClick;
  bool		isDragging;
  ResizeSquare	itsDragSquare;

  const QCursor*	itsCursors[9];

  QDict< DlgWidget >	itsDlgWidgets;
  // dictionary of dlgwidgets indexed by the name of the widget

  QDict< QWidget >	itsNoMoveWidgets;
  // widgets whose top left corner doesn't move

  QList< DlgWidget >	itsZOrdering;
  // z ordering of the widgets on the screen.  index 0 is the bottommost
  // widget

  DlgParser *itsParser;
  // The Parser which will be used to read the dilaog file

  DlgWidget *widgetInConstruction;
  // The actual Widget which is just build

  bool enabled;
  // Should this Layout react on signals from the outer space ?

  bool autoSelect;
  // Automatically select all new widgets

  static int layoutNumber;
  // Numbering all the Layouts to allow automatic name generation
};


inline WidgetFrame *WidgetLayout::GetDlgEditWnd() const
{
   return itsDlgEditWnd;
}

inline void WidgetLayout::SetNoMove
(
    QWidget*	widget
)
{
    itsNoMoveWidgets.insert( widget->name(), widget );
}


inline void WidgetLayout::SetDlgEditWnd
(
    WidgetFrame*	editWnd
)
{
    itsDlgEditWnd = editWnd;
}


inline void WidgetLayout::AutoSelect( bool aS )
{
  autoSelect = aS;
}

#endif // WidgetLayout_included



