#include <qregexp.h>

#include "FontSelectDialog.h"

#include "GuiTools.h"

#define Inherited FontSelectDialogData

FontSelectDialog::FontSelectDialog
(
    const QFont& font,
    QWidget* parent,
    const char* name
)
    :
    Inherited( parent, name )
{
    setCaption( "Font Selector" );

    itsSampleText->setFont( font );

    // set family
    int i, index = -1;
    QString str = font.family();
    
    for( i = 0; (i < itsFamily->count()) && (index == -1); i++ )
    {
	if( itsFamily->text( i ) == str )
	{
	    index = i;
	}
    }

    if( index == -1 )
    {
	index = itsFamily->count();
	itsFamily->insertItem( str, index );
    }
    itsFamily->setCurrentItem( index );
    
    // set size
    index = -1;
    str.setNum( font.pointSize() );
    for( i = 0; (i < itsPointSize->count()) && (index == -1); i++ )
    {
	if( itsPointSize->text( i ) == str )
	{
	    index = i;
	}
    }

    if( index == -1 )
    {
	index = itsPointSize->count();
	itsPointSize->insertItem( str, index );
    }
    itsPointSize->setCurrentItem( index );

    // set weight
    index = -1;
    str = GuiTools::QFontWeightStr( font.weight() );
    QRegExp rexp( "^" + str );
    for( i = 0; (i < itsWeight->count()) && (index == -1); i++ )
    {
	if( rexp.match( itsWeight->text( i ) ) != -1 )
	{
	    index = i;
	}
    }
    
    if( index == -1 )
    {
	index = itsWeight->count();
	itsWeight->insertItem( str, index );
    }
    itsWeight->setCurrentItem( index );

    isItalic->setChecked( font.italic() );
    isUnderlined->setChecked( font.underline() );
    isStrikeout->setChecked( font.strikeOut() );
    isFixedPitch->setChecked( font.fixedPitch() );

    connect( itsFamily, SIGNAL(activated(const char*)),
	     SLOT(FamilyChanged(const char*)) );
    connect( itsPointSize, SIGNAL(activated(const char*)), 	
	     SLOT(PointSizeChanged(const char*)) );
    connect( itsWeight, SIGNAL(activated(const char*)),
	     SLOT(WeightChanged(const char*)) );
    connect( itsStyleHint, SIGNAL(activated(const char*)),
	     SLOT(StyleHintChanged(const char*)) );
    connect( itsCharSet, SIGNAL(activated(const char*)),
	     SLOT(CharSetChanged(const char*)) );
    connect( isItalic, SIGNAL(toggled(bool)), 
	     SLOT(ItalicToggled(bool)) );
    connect( isUnderlined, SIGNAL(toggled(bool)), 
	     SLOT(UnderlinedToggled(bool)) );
    connect( isStrikeout, SIGNAL(toggled(bool)), 
	     SLOT(StrikeoutToggled(bool)) );
    connect( isFixedPitch, SIGNAL(toggled(bool)), 
	     SLOT(FixedPitchToggled(bool)) );

    connect( itsOkButton, SIGNAL(clicked()), SLOT(accept()) );
    connect( itsCancelButton, SIGNAL(clicked()), SLOT(reject()) );
}


FontSelectDialog::~FontSelectDialog()
{
}


void FontSelectDialog::FamilyChanged( const char*	family )
{
    QFont font = itsSampleText->font();
    font.setFamily( family );
    itsSampleText->setFont( font );
}


void FontSelectDialog::PointSizeChanged( const char*	size )
{
    QString str = size;
    QFont font = itsSampleText->font();
    
    font.setPointSize( str.toInt() );
    itsSampleText->setFont( font );
}


void FontSelectDialog::WeightChanged( const char*	weight )
{
    QRegExp rexp( "([0-9]+)" );
    int len;
    int pos = rexp.match( weight, 0, &len );
    QString str = weight;
    
    if( pos != -1 )
    {
	str = str.mid( pos + 1, len - 2 );
    }
    
    QFont font = itsSampleText->font();
    font.setWeight( str.toInt() );
    itsSampleText->setFont( font );
}

    
void FontSelectDialog::StyleHintChanged( const char*	hint )
{
    QFont font = itsSampleText->font();
    font.setStyleHint( GuiTools::QFontStyleHint( hint ) );
    itsSampleText->setFont( font );
}


void FontSelectDialog::CharSetChanged( const char*	set )
{
    QFont font = itsSampleText->font();
    font.setCharSet( GuiTools::QFontCharSet( set ) );
    itsSampleText->setFont( font );
}


void FontSelectDialog::ItalicToggled( bool	clicked )
{
    QFont font = itsSampleText->font();
    font.setItalic( clicked );
    itsSampleText->setFont( font );
}


void FontSelectDialog::UnderlinedToggled( bool	clicked )
{
    QFont font = itsSampleText->font();
    font.setUnderline( clicked );
    itsSampleText->setFont( font );
}


void FontSelectDialog::StrikeoutToggled( bool	clicked )
{
    QFont font = itsSampleText->font();
    font.setStrikeOut( clicked );
    itsSampleText->setFont( font );
}


void FontSelectDialog::FixedPitchToggled( bool	clicked )
{
    QFont font = itsSampleText->font();
    font.setFixedPitch( clicked );
    itsSampleText->setFont( font );
}


