#ifndef DlgWidget_included
#define DlgWidget_included

#include <qtstream.h>
#include <qwidget.h>
#include <qlist.h>
#include <qlistbox.h>
#include <qstring.h>
#include <qstrlist.h>

#include "widget/SignalConnection.h"

class DlgWidget
{
    
public:

    enum Type
    {
	DW_Button,
	DW_ButtonGroup,
	DW_CheckBox,
	DW_ComboBox,
	DW_Frame,
	DW_GroupBox,
	DW_LCDNumber,
	DW_Label,
	DW_LineEdit,
	DW_ListBox,
	DW_MenuBar,
	DW_MultiLineEdit,
	DW_PushButton,
	DW_RadioButton,
	DW_ScrollBar,
	DW_Slider,
	DW_User,
	DW_None = -1
    };
    
public:

    DlgWidget
    (
	Type		type
    );

    virtual ~DlgWidget();

    static DlgWidget *NewDlgWidget( QWidget *, 
				    DlgWidget::Type, 
				    const QString * = NULL );
    // Construct a new subclass using DlgWidget::Type

    static DlgWidget *NewDlgWidget( QWidget *,
				    const QString&,
				    const QString* = NULL );
    // Construct a new subclass using a Widgetname
    
    virtual QWidget* GetWidget( void ) = 0;
    // return the widget's widget

    virtual void SaveContents
    (
	QTextStream&	stream
    );
    // save the contents of the widget

    virtual int DoPropertyDialog( QWidget *parent, char *name ) = 0;
    // Do the property dialoges

    virtual bool RestoreKeyValue( QString &key, QString &value );
    // Restore a single key/value pair

    void SaveQt
    (
        QTextStream &,
        const QWidget *
    );
    // Used by original Qt subclasses
 
    bool RestoreQt
    (
        const QString &,
        const QString &,
        QWidget *
    );
    // Used by original Qt subclasses
 
    void SetVariableMapping
    (
	const QString&	variableName
    );
    // if variableName != "", then the widget is mapped to a variable

    bool IsVariableMapped() const;
    // is the widget mapped to a variable

    const QString& GetVariableName() const;
    // return the name of the variable, or "" if not mapped

    Type GetType() const;
    // return the type of the widget

    virtual QString GetHeaderFilename() const = 0;
    // return the header filename for the widget
    
    virtual QString GetWidgetClass() const = 0;
    // return the class name for the widget

    const char* Dot() const;
    // return a '->' since all variables are allocated dynamically

    virtual void GenerateSource
    (
	QTextStream&		stream,
	const QString&		varName,
	const QList<DlgWidget>&	zOrdering
    );
    // call GenSrc for the appropriate widget base classes. varName already
    // has a leading tab and the correct dereferencer

    virtual void GenerateCtor
    (
	QTextStream&		stream,
	const QString&		varName
    );
    // output a constructor for the widget.  The
    // ctor for leaves the output at the start of the next line

    static void GenSrc
    (
	DlgWidget&		dlgWidget,
	QTextStream&		stream,
	const QString&		varName,
	const QList<DlgWidget>&	zOrdering
    );
    // static method to actually generate the source

    QList<SignalConnection>& GetConnectedSignalList( void );
    // return a reference to the list of signals

    void GetSignalList( QStrList& signalList ) const;
    // return a list of the names of the signals that can be connected
    // to this widget


protected: //Methods
  QString CreateName( QString, const char *, int & );
  // Generates a unique WidgetName based on the Class name, a user-
  // provided name (no name generation) and a class specific number

    virtual void AddSignalsToList( QStrList&	signalList ) const;
    // add signals specific to a particular widget type to the list
  
private:

    QString	itsVariableName;
    
    Type	itsType;

    QList<SignalConnection>	itsConnectedSignalList;
    
};


inline DlgWidget::Type DlgWidget::GetType() const
{
    return itsType;
}


inline bool DlgWidget::IsVariableMapped() const
{
    return !itsVariableName.isEmpty();
}


inline const QString& DlgWidget::GetVariableName() const
{
    return itsVariableName;
}


inline void DlgWidget::SetVariableMapping
(
    const QString&	variableName
)
{
    itsVariableName = variableName;
}


inline void DlgWidget::GenerateSource
(
    QTextStream&		stream,
    const QString&		varName,
    const QList<DlgWidget>&	zOrdering
)
{
    GenSrc( *this, stream, varName, zOrdering );
}


inline QList<SignalConnection>& DlgWidget::GetConnectedSignalList( void )
{
    return itsConnectedSignalList;
}


QTextStream& operator <<
(
    QTextStream&	ostream,
    const QRect&	rect
);


QTextStream& operator >>
(
    QTextStream&	istream,
    QRect&		rect
);


// write palette to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QPalette&	palette
);


// read palette from stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QPalette&		palette
);


// write color group to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QColorGroup&	group
);


// read color group from a text stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QColorGroup&	group
);


// write font to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QFont&	font
);


// read a font from the text stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QFont&		font
);


void SaveBool
(
    QTextStream&	ostream,
    const bool& 
);

void RestoreBool
(
    QTextStream&	istream,
    bool &
);


#endif // DlgWidget_included





