
#include <stdio.h>
#include <stdlib.h>

#include "SDL.h"
#include "SDLttf.h"

/* Macro to convert a character to a Unicode value -- assume already Unicode */
#define UNICODE(c)	c

static TT_Engine engine;

int TTF_Init(void)
{
	int error;

	error = TT_Init_FreeType(&engine);
	if ( error ) {
		SDL_SetError("Couldn't init FreeType engine");
		return(-1);
	}
	return(0);
}

TTF_Font *TTF_OpenFont(char *file, int ptsize)
{
	TTF_Font *font;
	TT_Instance_Metrics imetrics;
	int i, n;
	unsigned short platform, encoding;
	int error;

	font = (TTF_Font *)malloc(sizeof(*font));
	if ( font == NULL ) {
		SDL_SetError("Out of memory");
		return(NULL);
	}
	memset(font, 0, sizeof(*font));

	/* Open the font and create ancillary data */
	error = TT_Open_Face(engine, file, &font->face);
	if ( error ) {
		SDL_SetError("Couldn't load font file");
		free(font);
		return(NULL);
	}
	error = TT_New_Glyph(font->face, &font->glyph);
	if ( error ) {
		SDL_SetError("Couldn't create glyph container");
		TTF_CloseFont(font);
		return(NULL);
	}
	error = TT_New_Instance(font->face, &font->inst);
	if ( error ) {
		SDL_SetError("Couldn't create font instance");
		TTF_CloseFont(font);
		return(NULL);
	}

	/* Set the display resolution */
	error = TT_Set_Instance_Resolutions(font->inst, 96, 96);
	if ( error ) {
		SDL_SetError("Couldn't set font resolution");
		TTF_CloseFont(font);
		return(NULL);
	}
	error = TT_Set_Instance_CharSize(font->inst, ptsize*64);
	if ( error ) {
		SDL_SetError("Couldn't set font size");
		TTF_CloseFont(font);
		return(NULL);
	}

	/* Get a Unicode mapping for this font */
	n = TT_Get_CharMap_Count(font->face);
	for ( i=0; i<n; ++i ) {
		TT_Get_CharMap_ID(font->face, i, &platform, &encoding);
		if ( (platform == 3 && encoding == 1) ||
		     (platform == 0 && encoding == 0) ) {
			TT_Get_CharMap(font->face, i, &font->map);
			break;
		}
	}
	if ( i == n ) {
		SDL_SetError("Font doesn't have a Unicode mapping");
		TTF_CloseFont(font);
		return(NULL);
	}

	/* Create a scratch pixmap for this font */
	TT_Get_Instance_Metrics(font->inst, &imetrics);
	font->scratch.rows = imetrics.y_ppem + 32;
	font->scratch.width = imetrics.x_ppem + 32;
	font->scratch.flow = TT_Flow_Up;
	font->scratch.cols = (font->scratch.width+3) & -4;
	font->scratch.size = font->scratch.rows*font->scratch.cols;
	font->scratch.bitmap = malloc((int)font->scratch.size);
	if ( font->scratch.bitmap == NULL ) {
		SDL_SetError("Out of memory");
		TTF_CloseFont(font);
		return(NULL);
	}
	return(font);
}

int TTF_FontHeight(TTF_Font *font)
{
	return(font->scratch.rows - 32);
}

void TTF_CloseFont(TTF_Font *font)
{
	TT_Close_Face(font->face);
	if ( font->scratch.bitmap ) {
		free(font->scratch.bitmap);
	}
	free(font);
}

/* The rendering is a three step process:

   Step 1.  Find the bounding box and maximum drop below baseline
   Step 2.  Render a glyph.
   Step 3.  OR the glyph into full surface at offset maxdrop + miny;
*/
SDL_Surface *TTF_RenderText(TTF_Font *font,
				char *text, SDL_Color fg, SDL_Color bg)
{
	SDL_Surface *textbuf;
	char *ch;
	int index;
	TT_Glyph_Metrics metrics;
	TT_F26Dot6 xoff, z, minx, miny, maxx, maxy;
	int descent;
	SDL_Palette *palette;
	int rdiff, gdiff, bdiff;
	Uint16 x, w, h;
	Uint8 *srcbuf;
	Uint16 srcpitch;
	Uint8 *dstbuf;
	Uint16 dstpitch;
	TT_Error error;

	/* Initialize everything to 0 */
	minx = miny = maxx = maxy = 0;

	/* Load each character and sum it's bounding box */
	xoff = 0;
	for ( ch=text; *ch; ++ch ) {
		index = TT_Char_Index(font->map, UNICODE(*ch));
		if ( ! (error=TT_Load_Glyph(font->inst, font->glyph,
						index, TTLOAD_DEFAULT)) ) {
			TT_Get_Glyph_Metrics(font->glyph, &metrics);

			z = xoff + metrics.bbox.xMin;
			if ( minx > z )
				minx = z;

			z = xoff + metrics.bbox.xMax;
			if ( maxx < z )
				maxx = z;

			z = metrics.bbox.yMin;
			if ( miny > z )
				miny = z;

			z = metrics.bbox.yMax;
			if ( maxy < z )
				maxy = z;

			xoff += metrics.advance & -64;
		}
	}

	/* Fill the bounds rectangle */
	w = (maxx-minx)/64;
	h = (maxy-miny)/64;
	descent = -miny/64;
#ifdef TEXT_DEBUG
fprintf(stderr, "Text block: %dx%d with %d descent\n", w, h, descent);
#endif
	/* Allocate an 8-bit pixmap */
	textbuf = SDL_AllocSurface(SDL_SWSURFACE, w, h, 8, 0, 0, 0, 0);
	if ( textbuf == NULL ) {
		return(NULL);
	}

	/* Fill the palette with 5 levels of shading from bg to fg */
	palette = textbuf->format->palette;
	rdiff = fg.r - bg.r;
	gdiff = fg.g - bg.g;
	bdiff = fg.b - bg.b;
	for ( index=0; index<5; ++index ) {
		palette->colors[index].r = bg.r + (index*rdiff)/4;
		palette->colors[index].g = bg.g + (index*gdiff)/4;
		palette->colors[index].b = bg.b + (index*bdiff)/4;
	}
	/* The other 3 levels are used as overflow when ORing pixels */
	for ( ; index<8; ++index ) {
		palette->colors[index] = palette->colors[4];
	}

	/* Load and render eadh character */
	xoff = 0;
	for ( ch=text; *ch; ++ch ) {
		index = TT_Char_Index(font->map, UNICODE(*ch));
		if ( ! (error=TT_Load_Glyph(font->inst, font->glyph,
						index, TTLOAD_DEFAULT)) ) {

			/* Get the bounding box and glyph pixmap */
			TT_Get_Glyph_Metrics(font->glyph, &metrics);
			minx = metrics.bbox.xMin;
			maxx = metrics.bbox.xMax;
			miny = metrics.bbox.yMin;
			maxy = metrics.bbox.yMax;
#ifdef TEXT_DEBUG
fprintf(stderr, "Rendering '%c'  %dx%d minx = %d, miny = %d at x = %d\n",
	*ch, (maxx-minx)/64, (maxy-miny)/64, minx/64, miny/64, xoff/64);
#endif
			memset(font->scratch.bitmap, 0, font->scratch.size);
			TT_Get_Glyph_Pixmap(font->glyph,
						&font->scratch, -minx, -miny);

			/* OR the pixmap into the text surface */
			w = (maxx-minx)/64;
			if ( w > font->scratch.cols )
				w = font->scratch.cols;
			if ( (xoff == 0) && minx < 0 )
				minx = 0;
			h = (maxy-miny)/64;
			if ( h > font->scratch.rows )
				h = font->scratch.rows;
			srcbuf = (Uint8 *)font->scratch.bitmap;
			srcpitch = font->scratch.cols - w;
			dstbuf = (Uint8 *)textbuf->pixels +
				(textbuf->h-descent-miny/64-1)*textbuf->pitch +
								(xoff+minx)/64;
			dstpitch = textbuf->pitch + w;
			while ( h-- ) {
				for ( x = 0; x < w; ++x )
					*dstbuf++ |= *srcbuf++;
				srcbuf += srcpitch;
				dstbuf -= dstpitch;
			}
			xoff += metrics.advance & -64;
		}
	}
	return(textbuf);
}

void TTF_Quit(void)
{
	TT_Done_FreeType(engine);
}
