/*
 * X-Crillion 1.1 : A nice ballgame based on the SDL library
 * Copyright(C) 1999 by Mike Krueger (mkrueger@inf.fu-berlin.de) 
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL.h>
#include <SDL_audio.h>
#include "SDLttf.h"
#include <time.h>
#include <mixer.h>
#include "level.h"
#include "action.h"

#define A_NONE   0
#define A_LEFT   1
#define A_RIGHT  2
#define A_UP     4
#define A_DOWN   8

SDL_Surface *screen, *out;
SDL_Surface *killed;

#define FREE_MUSIC(x) if (x != NULL) { Mix_FreeMusic(x); x = NULL; }
#define FREE_CHUNK(x) if (x != NULL) { Mix_FreeChunk(x); x = NULL; }

static int audio_open = 0;
static Mix_Music *title_tune = NULL, *endgame_tune = NULL;
static Mix_Music *game_tune[4];
static Mix_Chunk *bounce = NULL, *crunch = NULL, *color = NULL, *dead = NULL;
static Mix_Chunk *sand = NULL, *move = NULL;

int curlevel = 0;
int timer = 5 * 60;
TTF_Font *font;

SDL_Event event;
SDL_Rect dest, dst;
int x, y, a, done = 0, curtile = 1, lives = 5, score = 0;
char filename[80], outtxt[90];
unsigned int audio_rate;
unsigned short audio_format;
int audio_channels, old_level = -1;
int i, cheat = 0;
SDL_Color white = { 0xFF, 0xFF, 0xFF, 0 };
SDL_Color black = { 0x00, 0x00, 0x00, 0 };

Levelstruct level;
Ballstruct ball;
TileSetstruct tileset;

int numpassword = 18;
char *passwd[] =  { "first", "realaty", "eternal", "death", "valley", "master", "orion", "blaster", "identify", "realm", "darkness",
                    "phobia", "fluid",  "bubble",  "star",  "book",   "wood", "milk"};
char curpasswd[100];
SDL_Surface *level_layer[15][32];

char stard[1000];
int starx[1000], stary[1000], starz[1000];
int sx1, sy1, sx2, sy2;
int bp = 2;
unsigned int flags = SDL_FULLSCREEN | SDL_SWSURFACE;

void init_bgfx (int x1, int y1, int x2, int y2)
{
	int x;
	for (x = 0; x < 1000; x++) {
		starx[x] = x1 + rand () % (x2 - x1);
		stary[x] = y1 + rand () % (y2 - y1);
		starz[x] = 1 + rand () % 255;
		stard[x] = 0;
	}
	sx1 = x1;
	sy1 = y1;
	sx2 = x2;
	sy2 = y2;
}

unsigned short getrgb (char r, char g, char b)
{
	return (((r & 31) << 11) + ((g & 31) << 6) + (b & 31));
}

void bgfx (int fx, SDL_Surface * screen)
{
	int x, y, z, ax, ay;
	Uint16 *ptr;
	static int counter = 0;
	counter++;

	if (SDL_MUSTLOCK (screen) && SDL_LockSurface (screen) < 0)
		return; /* return if lock failed */

	switch (fx) {
		case 0:
			for (x = 0; x < 500; x++) {
				if (stard[x]) {
					ptr = (Uint16 *) (((Uint8 *) screen->pixels) + stary[x] * screen->pitch + starx[x] * bp);
					*ptr = 0;
					stard[x] = 0;
					SDL_UpdateRect (screen, starx[x], stary[x], 1, 1);
				}
				if (x < 100 && counter % 4 == 0)
						starx[x]++;
					else if (x >= 100 && x < 200 && counter % 3 == 0)
						starx[x]++;
					else if (x >= 200 && x < 300 && counter % 2 == 0)
						starx[x]++;
					else if (x >= 300 && x < 400)
						starx[x]++;
				else starx[x] += 2;
				if (starx[x] > sx2)
					starx[x] = sx1;
				ptr = (Uint16 *) (((Uint8 *) screen->pixels) + stary[x] * screen->pitch + starx[x] * bp);
				if (*ptr == 0) {
					if (x < 100)
							*ptr = getrgb (10, 10, 10);
						else if (x < 200)
							*ptr = getrgb (15, 15, 15);
						else if (x < 300)
						*ptr = getrgb (20, 20, 20);
						else if (x < 400)
							*ptr = getrgb (25, 25, 25);
						else *ptr = getrgb (31, 31, 31);
					stard[x] = 1;
					SDL_UpdateRect (screen, starx[x], stary[x], 1, 1);
				}
				}
		break;
		case 1:
			z = (sx2 - sx1) / 2;
			for (x = 0; x < 1000; x++) {
				if (stard[x]) {
					ax = z + ((starx[x] - z) << 7) / starz[x];
					ay = z + ((stary[x] - z) << 7) / starz[x];
					ptr = (Uint16 *) (((Uint8 *) screen->pixels) + ay * screen->pitch + ax * bp);
					*ptr = 0;
					stard[x] = 0;
					SDL_UpdateRect (screen, ax, ay, 1, 1);
				}
				starz[x]--;
				if (starz[x] <= 0)
					starz[x] = 255;
				ax = z + ((starx[x] - z) << 7) / starz[x];
				ay = z + ((stary[x] - z) << 7) / starz[x];
		
				if (ax > sx1 && ay > sy1 && ax < sx2 && ay < sy2) {
					ptr = (Uint16 *) (((Uint8 *) screen->pixels) + ay * screen->pitch + ax * bp);
					if (*ptr == 0) {
						y = (255 - starz[x]);
						*ptr = getrgb (y / 8, y / 8, 0);
						stard[x] = 1;
						SDL_UpdateRect (screen, ax, ay, 1, 1);
					}
				}
			}
		break;
		case 2:
			for (x = 0; x < 500; x++) {
				if (stard[x]) {
					ptr = (Uint16 *) (((Uint8 *) screen->pixels) + stary[x] * screen->pitch + starx[x] * bp);
					*ptr = 0;
					stard[x] = 0;
					SDL_UpdateRect (screen, starx[x], stary[x], 1, 1);
				}
				if (x < 100 && counter % 4 == 0)
						stary[x]++;
					else if (x >= 100 && x < 200 && counter % 3 == 0)
						stary[x]++;
					else if (x >= 200 && x < 300 && counter % 2 == 0)
						stary[x]++;
					else if (x >= 300 && x < 400)
						stary[x]++;
					else stary[x] += 2;

				if (stary[x] > sy2)
					stary[x] = sy1;
				ptr = (Uint16 *) (((Uint8 *) screen->pixels) + stary[x] * screen->pitch + starx[x] * bp);
			
				if (*ptr == 0) {
					if (x < 100)
							*ptr = getrgb (0, 10, 10);
						else if (x < 200)
							*ptr = getrgb (0, 15, 15);
						else if (x < 300)
							*ptr = getrgb (0, 20, 20);
						else if (x < 400)
							*ptr = getrgb (0, 25, 25);
						else *ptr = getrgb (0, 31, 31);
					stard[x] = 1;
					SDL_UpdateRect (screen, starx[x], stary[x], 1, 1);
				}
			}
		break;
	}
	if (SDL_MUSTLOCK (screen))
		SDL_UnlockSurface (screen);
}


int get_password_level ()
{
	int curlev = 0, x;

	for (x = 0; x < numpassword; x++)
		if (!strcmp (curpasswd, passwd[x]))
			curlev = x;
	return curlev;
}

extern void WaitFrame (void);

void CleanUp (void)
{
	int x;

	free_Tileset (&tileset);
	free_Ballstruct (&ball);

	if (audio_open) {
		audio_open = 0;
		Mix_CloseAudio();
		
		FREE_MUSIC(title_tune);
		FREE_MUSIC(endgame_tune);
		
		FREE_CHUNK(bounce);
		FREE_CHUNK(crunch);
		FREE_CHUNK(color);
		FREE_CHUNK(dead);

		FREE_CHUNK(sand);
		FREE_CHUNK(move);		
		
		for (x = 0; x < 4; x++)
			FREE_MUSIC(game_tune[x]);
	}
	
	if (killed)
		SDL_FreeSurface(killed);

	TTF_CloseFont (font);
	if (audio_open)
		Mix_CloseAudio();

	TTF_Quit ();
	SDL_Quit ();
}


void draw_level (TileSetstruct * tileset)
{
	SDL_Rect dest;
	int x, y;

	for (y = 0; y < 15; y++)
	for (x = 0; x < 32; x++) {
		dest.x = x * TILE_WIDTH;
		dest.y = 24 + y * TILE_HEIGHT;
		dest.w = tileset->tile[level.level[y][x].stone_type]->w;
		dest.h = tileset->tile[level.level[y][x].stone_type]->h;
		if (level.level[y][x].stone_type == 0)
		    SDL_BlitSurface (tileset->background, &dest, screen, &dest);
		else
			SDL_BlitSurface (tileset->tile[level.level[y][x].stone_type], NULL, screen, &dest);
	}
}

void del_level_layer ()
{
	int x, y;

	for (y = 0; y < 15; y++)
	for (x = 0; x < 32; x++) {
		if (level_layer[y][x] != NULL)
			SDL_FreeSurface (level_layer[y][x]);
		level_layer[y][x] = NULL;
	}
}

void update_level (TileSetstruct * tileset)
{
	SDL_Rect dest;

	int x, y;

	for (y = 0; y < 15; y++)
	for (x = 0; x < 32; x++) {
		dest.x = x * TILE_WIDTH;
		dest.y = 24 + y * TILE_HEIGHT;
		dest.w = tileset->tile[level.level[y][x].stone_type]->w;
		dest.h = tileset->tile[level.level[y][x].stone_type]->h;

		if (level.level[y][x].stone_type >= 25 && level.level[y][x].stone_type <= 29) {
			if (level_layer[y][x] == NULL) {
				level_layer[y][x] = SDL_DisplayFormat (tileset->tile[0]);
				SDL_BlitSurface (tileset->background, &dest, level_layer[y][x], NULL);
				SDL_BlitSurface (tileset->tile[level.level[y][x].stone_type], NULL, level_layer[y][x], NULL);
			}
			SDL_BlitSurface (level_layer[y][x], NULL, screen, &dest);
			SDL_UpdateRects (screen, 1, &dest);
		}
	}
}


int load_newlevel (int num, Levelstruct * level)
{
	char filename[60];
	del_level_layer ();
	remove_all_actions ();
	sprintf (filename, INSTALL_PATH "level/level%d.dat", num);
	load_level (filename, level);
	draw_level (&tileset);
	ball.color = level->Ball_Color;
	ball.x = level->Ball_x;
	ball.y = level->Ball_y;
	if (level->Ball_Direction)
		ball.speedy = 2;
	else
		ball.speedy = -2;
	ball.painted = 0;
	timer = level->Minutes * 60 + level->Seconds;
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	return 1;
}

int move_stone (int x1, int y1, int x2, int y2)
{
	if (level.level[y2][x2].stone_type == 0) {
		add_action (DO_MOVE, x1, y1, x2, y2);
		score += 10;
		return 1;
	}
	return 0;
}

void kill_crillion ()
{
	SDL_Rect dest;

	if (audio_open)
		Mix_PlayChannel (-1, dead, 0);
	
	dest.x = (800 - killed->w) / 2;
	dest.y = (600 - killed->h) / 2;
	dest.w = killed->w;
	dest.h = killed->h;
	remove_all_actions ();
	SDL_BlitSurface (killed, NULL, screen, &dest);
	SDL_UpdateRects (screen, 1, &dest);
	for (a = 0; a < 100; a++) /* TODO: better timer for the next SDL project */
		WaitFrame ();
	load_newlevel (curlevel - 1, &level);
	lives--;
}

int ball_action (int y, int x, int direction)
{
	int sound = 0;

	if (level.level[y][x].stone_type == 0)
		return 0;

	if (level.level[y][x].stone_type == 20 + ball.color) { /*  Spiders */
		add_action (DO_KILLSTONE, x, y, 0, 0);
		if (audio_open)
			Mix_PlayChannel (-1, crunch, 0);
		score += 15;
		sound = 1;
	}
	if (level.level[y][x].stone_type == ball.color + 30) { /* Skorpions */
		add_action (DO_MORPH, x, y, level.level[y][x].stone_type, level.level[y][x].stone_type - 10);
		if (audio_open)
			Mix_PlayChannel (-1, crunch, 0);
		score += 20;
		sound = 1;
	}
	
	if (level.level[y][x].stone_type >= 40 && level.level[y][x].stone_type <= 44 && level.level[y][x].stone_type != ball.color + 40) { /* Color changer */
		ball.color = level.level[y][x].stone_type - 40;
		if (audio_open)
			Mix_PlayChannel (-1, color, 0);
		if (score >= 5)
			score -= 5;
		sound = 1;
	}
	
	if (level.level[y][x].stone_type == 60 + ball.color) { /* Sandstone */
		add_action (DO_REMOVE, x, y, level.level[y][x].stone_type, 0);
		level.level[y][x].stone_type = 0;
		if (audio_open)	
			Mix_PlayChannel (-1, sand, 0);
		score += 5;
		sound = 1;
	}
	if (level.level[y][x].stone_type == 50 + ball.color) { /* EGG */
		switch (direction) {
			case A_UP:
				if (move_stone (x, y, x, y - 1)) {
					if (audio_open)
						Mix_PlayChannel (-1, move, 0);
					sound = 1;
				}
			break;
			case A_DOWN:
				if (move_stone (x, y, x, y + 1)) {
					if (audio_open)
						Mix_PlayChannel (-1, move, 0);
					sound = 1;
				}
			break;
			case A_LEFT:
				if (move_stone (x, y, x - 1, y)) {
					if (audio_open)
						Mix_PlayChannel (-1, move, 0);
					sound = 1;
				}
			break;
			case A_RIGHT:
				if (move_stone (x, y, x + 1, y)) {
					if (audio_open)
						Mix_PlayChannel (-1, move, 0);
					sound = 1;
				}
			break;
		}
	}
	if (level.level[y][x].stone_type == 75 || level.level[y][x].stone_type == 70 + ball.color) {
		kill_crillion();
		sound = 1;
	}
	
	if (!sound && audio_open)
		Mix_PlayChannel (-1, bounce, 0);
  return 1;
}


void WaitFrame (void)
{
	static int next_tick = 0;
	int this_tick;
	static int time = 0;
	if ((time % 55) == 0)
		timer--;
	time++;
	/* Wait for the next frame */
	this_tick = SDL_GetTicks ();
	if (this_tick <= next_tick)
		SDL_Delay (next_tick - this_tick);
	next_tick = this_tick + (1500 / 50);
}

int level_won ()
{
	int x, y;

	for (y = 0; y < 15; y++)
	for (x = 0; x < 32; x++)
		if ((level.level[y][x].stone_type >= 20 &&  level.level[y][x].stone_type <= 24) || /* spiders   */
		    (level.level[y][x].stone_type >= 30 && level.level[y][x].stone_type <= 34))    /* skorpions */
			return 0;
	return 1;
}

void init ()
{
	/* Initialize variables  */
	audio_rate = 44200;
	audio_format = AUDIO_S16;
	audio_channels = 2;

	/* Initialize SDL */
	if (SDL_Init (SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0) {
		fprintf (stderr, "Couldn't initialize SDL: %s\n", SDL_GetError ());
		exit (2);
	}
	/* TTF engine */
	if (TTF_Init () < 0) {
		fprintf (stderr, "Couldn't initialize SDL: %s\n", SDL_GetError ());
		SDL_Quit();
		exit (2);
	}

	/* audio,mixer init */
	if (Mix_OpenAudio (audio_rate, audio_format, audio_channels, 4096) < 0) {
		fprintf (stderr, "Couldn't open audio: %s\n", SDL_GetError ());
		audio_open = 0;
	} else {
		Mix_QuerySpec (&audio_rate, &audio_format, &audio_channels);
		printf ("Opened audio at %d Hz %d bit %s\n", audio_rate, (audio_format & 0xFF), (audio_channels > 1) ? "stereo" : "mono");
		audio_open = 1;
	}

	/* Load Font */
	font = TTF_OpenFont (INSTALL_PATH "font/exlogo.ttf", 14);

	/* Set a 800x600x32 video mode */
	screen = SDL_SetVideoMode (800, 600, 16, flags);

	if (screen == NULL) {
		fprintf (stderr, "Couldn't set 800x600x32 video mode: %s\n", SDL_GetError ());
		CleanUp();
		exit (2);
	}
}


int check_border (int y, int x, Ballstruct * ball, Levelstruct * level)
{
	if (level->level[y][x].stone_type > 0 && level->level[y][x].stone_type != 25 + ball->color) {
		if (level->level[y][x].stone_type == 60 + ball->color) { /* sandstone */
			ball_action (y, x, A_NONE);
			return 0;
		}
		return 1;
	}
	return 0;
}

void move_ball (Ballstruct * ball, Levelstruct * level)
{
	int yup, ydown;
	int xleft, xright;

	yup = (ball->y + ball->speedy) / TILE_HEIGHT;
	ydown = (ball->y + ball->speedy + ball->h) / TILE_HEIGHT;
	xleft = ball->x / TILE_WIDTH;
	xright = (ball->x + ball->w) / TILE_WIDTH;

	if (check_border (yup, xleft, ball, level)) {
		if (!ball_action (yup, (ball->x + ball->w / 2) / TILE_WIDTH, A_UP))
		ball_action (yup, xleft, A_UP);
		ball->speedy *= -1;
		ball->y += ball->speedy;
	} else if (check_border (ydown, xleft, ball, level)) {
		if (!ball_action (ydown, (ball->x + ball->w / 2) / TILE_WIDTH, A_DOWN))
			ball_action (ydown, xleft, A_DOWN);
		ball->speedy *= -1;
		ball->y += ball->speedy;
	} else if (check_border (yup, xright, ball, level)) {
		if (!ball_action (yup, (ball->x + ball->w + ball->w / 2) / TILE_WIDTH, A_UP))
			ball_action (yup, xright, A_UP);
		ball->speedy *= -1;
		ball->y += ball->speedy;
	} else if (check_border (ydown, xright, ball, level)) {
		if (!ball_action (ydown, (ball->x + ball->w + ball->w / 2) / TILE_WIDTH, A_DOWN))
			ball_action (ydown, xright, A_DOWN);
		ball->speedy *= -1;
		ball->y += ball->speedy;
	}
	ball->y += ball->speedy;
	yup = (ball->y) / TILE_HEIGHT;
	ydown = (ball->y + ball->h) / TILE_HEIGHT;
	xleft = (ball->x + ball->speedx) / TILE_WIDTH;
	xright = (ball->x + ball->speedx + ball->w) / TILE_WIDTH;
	if (check_border (yup, xleft, ball, level)) {
		ball_action (yup, xleft, A_LEFT);
		ball->speedx *= -1;
		ball->x += ball->speedx;
	} else if (check_border (ydown, xleft, ball, level)) {
		ball_action (ydown, xleft, A_LEFT);
		ball->speedx *= -1;
		ball->x += ball->speedx;
	} else if (check_border (yup, xright, ball, level)) {
		ball_action (yup, xright, A_RIGHT);
		ball->speedx *= -1;
		ball->x += ball->speedx;
	} else if (check_border (ydown, xright, ball, level)) {
		ball_action (ydown, xright, A_RIGHT);
		ball->speedx *= -1;
		ball->x += ball->speedx;
	}
	ball->x += ball->speedx;
}

void load_media ()
{
	SDL_Surface *temp;
	/* Load the music file */
	if (audio_open) {
		bounce = Mix_LoadWAV (INSTALL_PATH "mus/17.wav");
		crunch = Mix_LoadWAV (INSTALL_PATH "mus/18.wav");
		color  = Mix_LoadWAV (INSTALL_PATH "mus/19.wav");
		dead   = Mix_LoadWAV (INSTALL_PATH "mus/20.wav");
		sand   = Mix_LoadWAV (INSTALL_PATH "mus/21.wav");
		move   = Mix_LoadWAV (INSTALL_PATH "mus/22.wav");
	
		title_tune   = Mix_LoadMUS (INSTALL_PATH "mus/space_de.mod");	
		endgame_tune = Mix_LoadMUS (INSTALL_PATH "mus/planets.mod");
		game_tune[0] = Mix_LoadMUS (INSTALL_PATH "mus/ball_1.mod");
		game_tune[1] = Mix_LoadMUS (INSTALL_PATH "mus/ball_2.mod");
		game_tune[2] = Mix_LoadMUS (INSTALL_PATH "mus/ball_3.mod");
		game_tune[3] = Mix_LoadMUS (INSTALL_PATH "mus/ball_4.mod");
	}

	/* Load the picture */
	temp = SDL_LoadBMP (INSTALL_PATH "art/killed.bmp");
	SDL_SetColorKey (temp, SDL_SRCCOLORKEY, 0);
	killed = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	load_tileset (0, &tileset);
	init_Ballstruct (&ball);
}

void paint_ball (Ballstruct * ball)
{
	SDL_Rect old;
	static SDL_Rect dest;

	/* clear old ball */
	if (ball->painted)
		SDL_BlitSurface (ball->back, NULL, screen, &dest);
	memcpy (&old, &dest, sizeof (SDL_Rect));
	dest.x = ball->x;
	dest.y = 24 + ball->y;
	dest.w = ball->w;
	dest.h = ball->h;
	
	/* get ball background */
	SDL_BlitSurface (tileset.background, &dest, ball->back, NULL);

	/* draw new ball */
	SDL_BlitSurface (ball->color_img[ball->color], NULL, screen, &dest);

	/* Update Screen */
	if (ball->painted)
		SDL_UpdateRects (screen, 1, &old);
	SDL_UpdateRects (screen, 1, &dest);
	ball->painted = 1;
}

void outro ()
{
	SDL_Color blue   = { 0x00, 0x00, 0xEE, 0 };
	SDL_Color red    = { 0xFF, 0x00, 0x00, 0 };
	SDL_Color yellow = { 0x00, 0xFF, 0xFF, 0 };

	if (audio_open)
		Mix_HaltMusic ();
	out = SDL_LoadBMP (INSTALL_PATH "art/outro.bmp");
	SDL_BlitSurface (out, NULL, screen, NULL);
	SDL_FreeSurface (out);

	sprintf (outtxt, "THE END ?");
	out = TTF_RenderText (font, outtxt, blue, black);
	dst.x = 260;
	dst.y = 150;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);

	sprintf (outtxt, "You played all levels of crillion (did you?) and now ");
	out = TTF_RenderText (font, outtxt, yellow, black);
	dst.x = 50;
	dst.y = 200;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);

	sprintf (outtxt, "you can be called a crillion master.");
	out = TTF_RenderText (font, outtxt, yellow, black);
	dst.x = 50;
	dst.y = 225;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);

	sprintf (outtxt, "You have succeeded where many before you failed.");
	out = TTF_RenderText (font, outtxt, yellow, black);
	dst.x = 50;
	dst.y = 250;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);
	sprintf (outtxt, "thanx fly out to :");
	out = TTF_RenderText (font, outtxt, red, black);
	dst.x = 50;
	dst.y = 500;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);

	sprintf (outtxt, "The guys from the SDL Project (keep on the good work),");
	out = TTF_RenderText (font, outtxt, red, black);
	dst.x = 50;
	dst.y = 525;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);

	sprintf (outtxt, "www.freeimages.com, to the guys who created the soundtrack");
	out = TTF_RenderText (font, outtxt, red, black);
	dst.x = 50;
	dst.y = 550;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);
	
	sprintf (outtxt, "and the forgotton rest ...");
	out = TTF_RenderText (font, outtxt, red, black);
	dst.x = 50;
	dst.y = 575;
	dst.w = out->w;
	dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	do {
		if (!Mix_PlayingMusic () && audio_open) {
			Mix_PlayMusic (endgame_tune);
			Mix_VolumeMusic (5);
		}

		if (SDL_PollEvent (&event))
		switch (event.type) {
			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_ESCAPE)
					done = 1;
			break;
		}
	} while (!done);
}

void intro ()
{
	int t = 0;
	int done = 0;
	out = SDL_LoadBMP (INSTALL_PATH "art/intro.bmp");
	SDL_BlitSurface (out, NULL, screen, NULL);
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	SDL_FreeSurface (out);
	init_bgfx (0, 0, 800, 600);
	do {
		if (t < 500)
			bgfx (0, screen);
		t++;
		if (t == 500) {
			out = SDL_LoadBMP (INSTALL_PATH "art/screen01.bmp");
			SDL_BlitSurface (out, NULL, screen, NULL);
			SDL_UpdateRect (screen, 0, 0, 0, 0);
			SDL_FreeSurface (out);
		}
		if (t == 1000) {
			out = SDL_LoadBMP (INSTALL_PATH "art/screen02.bmp");
			SDL_BlitSurface (out, NULL, screen, NULL);
			SDL_UpdateRect (screen, 0, 0, 0, 0);
			SDL_FreeSurface (out);
		}
		if (t >= 1500) {
			out = SDL_LoadBMP (INSTALL_PATH "art/intro.bmp");
			SDL_BlitSurface (out, NULL, screen, NULL);
			SDL_UpdateRect (screen, 0, 0, 0, 0);
			SDL_FreeSurface (out);
			t = 0;
		}
		WaitFrame ();
		if (!Mix_PlayingMusic () && audio_open) {
			Mix_PlayMusic (title_tune);
			Mix_VolumeMusic (5);
		}
		
		if (SDL_PollEvent (&event)) {
			switch (event.type) {
				case SDL_KEYDOWN:
					done = 1;
				break;
			}
		}
	} while (!done);
}

void end_level ()
{
	SDL_Surface *level, *out;
	SDL_Rect dest;
	Mix_Chunk *bing = NULL;

	if (audio_open)
		bing = Mix_LoadWAV (INSTALL_PATH "mus/bbing.wav");

	out = SDL_LoadBMP (INSTALL_PATH "art/end_level.bmp");
	SDL_SetColorKey (out, SDL_SRCCOLORKEY, 0);
	level = SDL_DisplayFormat (out);
	
	dest.x = (800 - out->w) / 2;
	dest.y = (600 - out->h) / 2;
	dest.w = out->w;
	dest.h = out->h;
	SDL_BlitSurface (level, NULL, screen, &dest);

	SDL_UpdateRects (screen, 1, &dest);

	SDL_FreeSurface (out);
	SDL_FreeSurface (level);

	do {
		timer -= 5;
		if (timer > 0)
			score += 5;
		if (timer < 0)
			timer = 0;
		if (audio_open)
			Mix_PlayChannel (-1, bing, 0);
		WaitFrame ();
		sprintf (outtxt, "SCORE: %06d  ", score);
		out = TTF_RenderText (font, outtxt, white, black);
		dst.x = 300; dst.y = 4;
		dst.w = out->w; dst.h = out->h;
		SDL_BlitSurface (out, NULL, screen, &dst);
		SDL_UpdateRects (screen, 1, &dst);
		SDL_FreeSurface (out);
		sprintf (outtxt, "TIME LEFT : %02d:%02d  ", timer / 60, timer % 60);
		out = TTF_RenderText (font, outtxt, white, black);
		dst.x = 600; dst.y = 1;
		dst.w = out->w; dst.h = out->h;
		SDL_BlitSurface (out, NULL, screen, &dst);
		SDL_UpdateRects (screen, 1, &dst);
		SDL_FreeSurface (out);
	} while (timer > 0);
	FREE_CHUNK (bing);
}

int manage_score ()
{
	SDL_Surface *out;
	SDL_Rect old;
	char outtxt[100], code[100], codetxt[100];
	char name[10][20], level[10], fade[10];
	int render = 0, first = 0, x, points[10], num;
	FILE *fp;
	SDL_Color blue = { 0x00, 0x00, 0xEE, 0 };
	SDL_Color red = { 0xFF, 0x00, 0x00, 0 };

	fp = fopen (INSTALL_PATH "highscore.crl", "rb");
	for (x = 0; x < 10; x++) {
		fread (name[x], 20, 1, fp);
		fread (&points[x], 4, 1, fp);
		level[x] = fgetc (fp);
		fade[x] = 0;
	}
	fclose (fp);
	for (x = 0; x < 10; x++)
		if (points[x] <= score)
			break;
	num = x;
	if (points[num] <= score && num < 10) {
		sprintf (outtxt, "you made a highscore (#%d) ", num);
		out = TTF_RenderText (font, outtxt, red, black);
		dst.x = (800 - out->w) / 2;
		dst.y = 430;
		dst.w = out->w;
		dst.h = out->h;
		SDL_BlitSurface (out, NULL, screen, &dst);
		SDL_UpdateRects (screen, 1, &dst);
		SDL_FreeSurface (out);
		sprintf (outtxt, "Enter your name ");
		out = TTF_RenderText (font, outtxt, red, black);
		dst.x = (800 - out->w) / 2; dst.y = 450;
		dst.w = out->w; dst.h = out->h;
		SDL_BlitSurface (out, NULL, screen, &dst);
		SDL_UpdateRects (screen, 1, &dst);
		SDL_FreeSurface (out);
		code[0] = 0;
		codetxt[0] = 0;
		outtxt[0] = 0;
		while (1) {
			if (!Mix_PlayingMusic () && audio_open) {
				Mix_PlayMusic (endgame_tune);
				Mix_VolumeMusic (5);
			}
			if (render) {
				if (first && strlen (codetxt) > 0) {
					out = TTF_RenderText (font, codetxt, black, black);
					SDL_BlitSurface (out, NULL, screen, &dst);
					SDL_FreeSurface (out);
					memcpy (&old, &dst, sizeof (SDL_Rect));
				}
				sprintf (codetxt, "%s", code);
				if (strlen (code) > 0) {
					out = TTF_RenderText (font, codetxt, blue, black);
					dst.x = (800 - out->w) / 2;
					dst.y = 470;
					dst.w = out->w;
					dst.h = out->h;
					SDL_BlitSurface (out, NULL, screen, &dst);
					SDL_UpdateRects (screen, 1, &dst);
					SDL_FreeSurface (out);
				}
				if (first)
					SDL_UpdateRects (screen, 1, &old);
				first = 1;
				render = 0;
			}
			if (SDL_PollEvent (&event)) {
				if (event.type == SDL_KEYDOWN) {
					switch (event.key.keysym.sym) {
						case SDLK_RETURN:
							for (y = 10; y > num; y--) {
								memcpy (name[y], name[y - 1], 20);
								points[y] = points[y - 1];
								level[y] = level[y - 1];
							}
							memcpy (name[num], code, 20);
							points[num] = score;
							level[num] = curlevel;
							fp = fopen (INSTALL_PATH "highscore.crl", "wb");
							for (x = 0; x < 10; x++) {
								fwrite (name[x], 20, 1, fp);
								fwrite (&points[x], 4, 1, fp);
								fputc (level[x], fp);
							}
							fclose (fp);
							return 1;
						break;
						case SDLK_BACKSPACE:
							if (strlen (code) > 0) {
								code[strlen (code) - 1] = 0;
								render = 1;
							}
						break;
						default:
							if (strlen (code) < 19)	{
								sprintf (code, "%s%c", code, event.key.keysym.sym);
								render = 1;
							}
					}
				}
			}
		}
	}
	return 0;
}


void play_game ()
{
	int otime = -1, olives = -1, oscore = -1, olevel = -1;
	int leveljump = 0;

	done = 0;
	lives = 5;
	score = 0;
	SDL_FillRect (screen, NULL, 0);

	load_newlevel (curlevel++, &level);
	
	while (!done) { /* GAME MAIN LOOP */
		paint_ball (&ball);
		update_level (&tileset);
		do_action (&level, screen, &tileset);
		WaitFrame();
		if (!Mix_PlayingMusic () && audio_open) {
			Mix_PlayMusic (game_tune[rand () % 4]);
			Mix_VolumeMusic (3);
		}
		/* Update Status */
		if (olives != lives) {
			printf (outtxt, "LIVES: %02d  ", lives);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = 1; dst.y = 4;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			olives = lives;
		}
		if (curlevel != olevel) {
			sprintf (outtxt, "LEVEL: %02d  ", curlevel);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = 120; dst.y = 4;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			olevel = curlevel;
		}
		if (score != oscore) {
			sprintf (outtxt, "SCORE: %06d  ", score);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = 300; dst.y = 4;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			oscore = score;
		}
		if (otime != timer) {
			sprintf (outtxt, "TIME LEFT : %02d:%02d  ", timer / 60, timer % 60);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = 600; dst.y = 1;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			otime = timer;
		}
		/* Check Time Out */
		if (timer <= 0)
			kill_crillion ();
		
		if (level_won () || leveljump) {
			end_level ();
			if (numpassword == curlevel) {
				outro ();
				done = 1;
			} else load_newlevel (curlevel++, &level);
			leveljump = 0;
		}
		move_ball (&ball, &level);
		
		/* Handle Events */
		if (SDL_PollEvent (&event)) {
			switch (event.type) {
				case SDL_KEYUP:
					ball.speedx = 0;
				break;
				case SDL_KEYDOWN:
					if (event.key.keysym.sym == SDLK_F1)
						SDL_SaveBMP (screen, "dump.bmp");

					if (cheat) {
						if (event.key.keysym.sym == SDLK_F2)
							leveljump = 1;
						if (event.key.keysym.sym == SDLK_F5)
							ball.color = 0;
						if (event.key.keysym.sym == SDLK_F6)
							ball.color = 1;
						if (event.key.keysym.sym == SDLK_F7)
							ball.color = 2;
						if (event.key.keysym.sym == SDLK_F8)
							ball.color = 3;
						if (event.key.keysym.sym == SDLK_F9)
							ball.color = 4;
					}
					if (event.key.keysym.sym == SDLK_F10)
						cheat = 1;
					if (event.key.keysym.sym == SDLK_ESCAPE)
						done = 1;
					if (event.key.keysym.sym == SDLK_LEFT)
						ball.speedx = -1;
					if (event.key.keysym.sym == SDLK_RIGHT)
						ball.speedx = +1;
					if (event.key.keysym.sym == SDLK_TAB)
						kill_crillion ();
				break;
				case SDL_QUIT:
					done = 1;
				break;
			}
		}

		if (lives < 0) {
			if (audio_open)
				Mix_HaltMusic ();
			out = SDL_LoadBMP (INSTALL_PATH "art/end_game.bmp");
			SDL_BlitSurface (out, NULL, screen, NULL);
			SDL_FreeSurface (out);
			sprintf (outtxt, "IN LEVEL %d", curlevel);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = (800 - out->w) / 2; dst.y = 350;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_FreeSurface (out);
			sprintf (outtxt, "with %d points ", score);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = (800 - out->w) / 2; dst.y = 375;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_FreeSurface (out);
			sprintf (outtxt, "levelpassword is %s", passwd[curlevel - 1]);
			out = TTF_RenderText (font, outtxt, white, black);
			dst.x = (800 - out->w) / 2; dst.y = 400;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_FreeSurface (out);
			SDL_UpdateRect (screen, 0, 0, 0, 0);
			if (!manage_score ())
				do {
					if (!Mix_PlayingMusic () && audio_open) {
						Mix_PlayMusic (endgame_tune);
						Mix_VolumeMusic (5);
					}
					if (SDL_PollEvent (&event))
					switch (event.type) {
						case SDL_KEYDOWN:
							if (event.key.keysym.sym == SDLK_ESCAPE)
								done = 1;
						 break;
					}
				} while (!done);
			done = 1;
		}
	}
}

void enter_code ()
{
	SDL_Surface *enter_code = NULL, *out = NULL;
	SDL_Rect dst, old;
	char code[100], codetxt[100];
	int render = 0, first = 0;
	memset (code, 0, 100);

	done = 0;
	enter_code = SDL_LoadBMP (INSTALL_PATH "art/code.bmp");
	SDL_BlitSurface (enter_code, NULL, screen, NULL);
	SDL_FreeSurface (enter_code);
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	init_bgfx (310, 130, 800, 600);
	do {
		if (render) {
			if (first && strlen (codetxt) > 0) {
				out = TTF_RenderText (font, codetxt, black, black);
				SDL_BlitSurface (out, NULL, screen, &dst);
				SDL_FreeSurface (out);
				memcpy (&old, &dst, sizeof (SDL_Rect));
			}
			sprintf (codetxt, "%s", code);
			if (strlen (code) > 0) {
				out = TTF_RenderText (font, codetxt, white, black);
				dst.x = 300 + (500 - out->w) / 2; dst.y = 300;
				dst.w = out->w; dst.h = out->h;
				SDL_BlitSurface (out, NULL, screen, &dst);
				SDL_UpdateRects (screen, 1, &dst);
				SDL_FreeSurface (out);
			}
			if (first)
				SDL_UpdateRects (screen, 1, &old);
			first = 1; render = 0;
		}
		bgfx (2, screen);
		WaitFrame ();
		if (SDL_PollEvent (&event)) {
			switch (event.type) {
				case SDL_KEYDOWN:
					switch (event.key.keysym.sym) {
						case SDLK_ESCAPE:
							done = 1;
						break;
						case SDLK_RETURN:
							sprintf (curpasswd, "%s", code);
							done = 1;
						break;
						case SDLK_BACKSPACE:
							if (strlen (code) > 0) {
								code[strlen (code) - 1] = 0;
								render = 1;
							}
						break;
						default:
							if (strlen (code) < 20) {
								sprintf (code, "%s%c", code, event.key.keysym.sym);
								render = 1;
							}
					}
				break;
			}
		}
	} while (!done);
}

void high_score ()
{
	SDL_Surface *enter_code = NULL, *out = NULL;
	SDL_Rect dst;
	SDL_Color blue = { 0x00, 0x00, 0xEE, 0 };
	SDL_Color gray[10] = {
		{0x66, 0x66, 0x66, 0},
		{0x77, 0x77, 0x77, 0},
		{0x88, 0x88, 0x88, 0},
		{0x99, 0x99, 0x99, 0},
		{0xAA, 0xAA, 0xAA, 0},
		{0xBB, 0xBB, 0xBB, 0},
		{0xCC, 0xCC, 0xCC, 0},
		{0xDD, 0xDD, 0xDD, 0},
		{0xEE, 0xEE, 0xEE, 0},
		{0xFF, 0xFF, 0xFF, 0} };

	char text[100];
	char name[10][20], level[10];
	int x, points[10], fade[10], t;
	FILE *fp;

	fp = fopen (INSTALL_PATH "highscore.crl", "rb");
	for (x = 0; x < 10; x++) {
		fread (name[x], 20, 1, fp);
		fread (&points[x], 4, 1, fp);
		level[x] = fgetc (fp);
		fade[x] = 0;
	}
	fclose (fp);

	done = 0;
	enter_code = SDL_LoadBMP (INSTALL_PATH "art/highscore.bmp");
	SDL_BlitSurface (enter_code, NULL, screen, NULL);
	SDL_FreeSurface (enter_code);

	sprintf (text, "NAME               SCORE      LEVEL");
	out = TTF_RenderText (font, text, blue, black);
	dst.x = 200; dst.y = 270;
	dst.w = out->w; dst.h = out->h;
	SDL_BlitSurface (out, NULL, screen, &dst);
	SDL_FreeSurface (out);
	t = 1;
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	init_bgfx (0, 260, 800, 600);

	do {
		if (t < 100) {
			x = t++ / 10;
			sprintf (text, "%-20s", name[x]);
			out = TTF_RenderText (font, text, gray[fade[x]], black);
			dst.x = 200; dst.y = 300 + x * 18;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);

			sprintf (text, "%06d", points[x]);
			out = TTF_RenderText (font, text, gray[fade[x]], black);
			dst.x = 410; dst.y = 300 + x * 18;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			
			sprintf (text, "%02d", level[x]);
			out = TTF_RenderText (font, text, gray[fade[x]], black);
			dst.x = 560; dst.y = 300 + x * 18;
			dst.w = out->w; dst.h = out->h;
			SDL_BlitSurface (out, NULL, screen, &dst);
			SDL_UpdateRects (screen, 1, &dst);
			SDL_FreeSurface (out);
			fade[x]++;
		}
		bgfx (2, screen);
		WaitFrame();
		if (SDL_PollEvent (&event)) {
			switch (event.type) {
				case SDL_KEYDOWN:
					if (event.key.keysym.sym == SDLK_ESCAPE)
						done = 1;
				break;
			}
		}
	} while (!done);
}

void main_menu ()
{
	SDL_Surface *title_screen, *temp;
	SDL_Surface *menu[4][2];
	SDL_Rect dest;
	Mix_Chunk *wargame = NULL;
	int done = 0;
	int k = 0, x, count = 0;
	int redraw = 1;
	
	if (audio_open)
		wargame = Mix_LoadWAV (INSTALL_PATH "mus/wargame2.wav");

	title_screen = SDL_LoadBMP (INSTALL_PATH "art/main.bmp");
	temp = SDL_LoadBMP (INSTALL_PATH "art/start_1.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[0][0] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/start_2.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[0][1] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/enter_1.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[1][0] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/enter_2.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[1][1] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/highscore_1.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[2][0] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/highscore_2.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[2][1] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);
	
	temp = SDL_LoadBMP (INSTALL_PATH "art/quit_1.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[3][0] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);

	temp = SDL_LoadBMP (INSTALL_PATH "art/quit_2.bmp");
	SDL_SetColorKey (temp, (SDL_SRCCOLORKEY | SDL_RLEACCEL), *(Uint32 *) temp);
	menu[3][1] = SDL_DisplayFormat (temp);
	SDL_FreeSurface (temp);
	count = 1000;

	do {
	
		if (count++ > 500) {
			SDL_ShowCursor (0);
			intro ();
			init_bgfx (0, 0, 800, 600);
			SDL_BlitSurface (title_screen, NULL, screen, NULL);
			SDL_UpdateRect (screen, 0, 0, 0, 0);
			count = 0;
			redraw = 1;
		}
		bgfx (1, screen);
		WaitFrame ();
	
		if (audio_open) {
			if (rand () % 1000 == 40)
				Mix_PlayChannel (-1, wargame, 0);
			if (!Mix_PlayingMusic ()) {
				Mix_PlayMusic (title_tune);
				Mix_VolumeMusic (5);
			}
		}
		if (redraw) {
			for (x = 0; x < 4; x++) {
				dest.x = (800 - menu[x][0]->w) / 2;
				dest.y = 150 + x * 85;
				dest.w = menu[x][0]->w;
				dest.h = menu[x][0]->h;
				if (k == x)
					SDL_BlitSurface (menu[x][0], NULL, screen, &dest);
				else	
					SDL_BlitSurface (menu[x][1], NULL, screen, &dest);
				SDL_UpdateRects (screen, 1, &dest);
			}
			redraw = 0;
		}
		
		if (SDL_PollEvent (&event)) {
			switch (event.type) {
				case SDL_KEYDOWN:
					count = 0;
					if (event.key.keysym.sym == SDLK_ESCAPE)
						done = 1;
					if (event.key.keysym.sym == SDLK_UP && k > 0) {
						k--;redraw = 1;
					}
					if (event.key.keysym.sym == SDLK_DOWN && k < 3) {
						k++;redraw = 1;
					}
					if (event.key.keysym.sym == SDLK_RETURN) {
						if (k == 0) {
							curlevel = get_password_level ();
							if (audio_open)
								Mix_HaltMusic ();
							play_game ();
							if (audio_open)
								Mix_HaltMusic ();
							init_bgfx (0, 0, 800, 600);
							 SDL_BlitSurface (title_screen, NULL, screen, NULL);
							 SDL_UpdateRect (screen, 0, 0, 0, 0);
							 redraw = 1;
						}
						if (k == 1)   {
							enter_code ();
							init_bgfx (0, 0, 800, 600);
							SDL_BlitSurface (title_screen, NULL, screen, NULL);
							SDL_UpdateRect (screen, 0, 0, 0, 0);
							redraw = 1;
						}
						if (k == 2) {
							high_score ();
							init_bgfx (0, 0, 800, 600);
							SDL_BlitSurface (title_screen, NULL, screen, NULL);
							SDL_UpdateRect (screen, 0, 0, 0, 0);
							redraw = 1;
						}
						if (k == 3)
							done = 1;
					}
				break;
				case SDL_QUIT:
					done = 1;
				break;
			}
		}
	} while (!done);

	SDL_FreeSurface (title_screen);

	for (y = 0; y < 4; y++)
	for (x = 0; x < 2; x++)
		SDL_FreeSurface (menu[y][x]);

	FREE_CHUNK (wargame);
}

int main (int argc, char *argv[])
{
	int x;
	for (x=1;x<argc;x++) {
		if (strcmp(argv[x], "?") == 0      || strcmp(argv[x], "-?") == 0  || strcmp(argv[x], "-h") == 0 ||
		    strcmp(argv[x], "--help") == 0 || strcmp(argv[x], "--h") == 0) {
		    	printf("usage :\n");
		    	printf("         xcrillion [OPTION] \n\n");
		    	printf("         -w, --window - run in window (instead of fullscreen)\n");
		    	printf("         -h, --help   - this help\n\n");		    	
		    	return 0;
		    } else
		if (strcmp(argv[x], "-w") == 0 ||
		    strcmp(argv[x], "--window") == 0)
		    flags = SDL_SWSURFACE;
		      else {
		      printf("unknown option, try --help\n");
		      return 0;
		}
	}
	init ();
	load_media ();
	main_menu ();
	CleanUp ();
	return 0;
}
