#include <unistd.h>

#include <qmessagebox.h>
#include <qstring.h>
#include <qevent.h>
#include <qlistbox.h>
#include <qtabdialog.h>
#include <qdir.h>
#include <qfile.h>
#include <qstrlist.h>

#include <kapp.h>
#include <ktoolbar.h>
#include <kconfig.h>
#include <kmenubar.h>
#include <kiconloader.h>
#include <knewpanner.h>

#include "../config.h"
#include "kxglobal.h"
#include "kxmainwin.h"
#include "kxentrylist.h"
#include "kxviewport.h"
#include "kxaccpre.h"
#include "kxngdialog.h"
#include "kxgenopts.h"
#include "kxexpire.h"
#include "kxcomposer.h"
#include "kfileio.h"

#include "kxmainwin.moc"

static QString aboutText;

// Define status item codes
#define ID_MSGTEXT 10

KXmainwin::KXmainwin()
{
	aboutText.sprintf(
	"kexpress [%s]\n\nby Peter Wichert <peter.wichert@t-online.de>\n\nThis program was inspired by and contains code of\n\"kmail\" by Stefan Taferner\nand\n\"krn\" by Roberto Alsina\n\nIt is covered by the GPL.",VERSION);

  initMenuBar();
  initToolBar();
  initStatusBar();

	setMinimumSize(400,300);

  mVertPanner = new KNewPanner(this, "vertPanner", KNewPanner::Vertical,
	  KNewPanner::Absolute);

  mVertPanner->resize(size());
	setView(mVertPanner);

  mHorizPanner = new KNewPanner(mVertPanner, "horizPanner",
		KNewPanner::Horizontal, KNewPanner::Absolute);

	gEntryList = new KXEntryList(mVertPanner,"Folder view");
  gHeaderList = new KXHeaderList(mHorizPanner, "Header view");
  gViewPort = new KXViewPort(mHorizPanner, "Message view");

	readConfig();

	mVertPanner->activate(gEntryList,mHorizPanner);
	mHorizPanner->activate(gHeaderList,gViewPort);

  mVertPanner->setAbsSeparatorPos(mVertPannerSep);
	mHorizPanner->setAbsSeparatorPos(mHorizPannerSep);
}

KXmainwin::~KXmainwin()
{
	writeConfig();
}
	
void KXmainwin::closeEvent(QCloseEvent *e)
{
	cleanup();
	e->accept();
}

void KXmainwin::readConfig(void)
{
	KConfig *config = app->getConfig();
	QString str;
	int w,h;

	config->setGroup("Geometry");

	str = config->readEntry("MainWin", "600,600");
	if (!str.isEmpty() && str.find(',')>=0)
	{
		sscanf(str,"%d,%d",&w,&h);
		resize(w,h);
	}

	str = config->readEntry("Panners", "100,100");
	if (!str.isEmpty() && str.find(',')>=0)
		sscanf(str,"%d,%d",&mVertPannerSep,&mHorizPannerSep);
	else
		mHorizPannerSep = mVertPannerSep = 100;
}

void KXmainwin::writeConfig(void)
{
  QString s(32);
  KConfig *config = app->getConfig();
  QRect r = geometry();

	setMsg("writing Config");

  config->setGroup("Geometry");

  s.sprintf("%i,%i", r.width(), r.height());
  config->writeEntry("MainWin", s);

  s.sprintf("%i,%i", mVertPanner->separatorPos(),
      mHorizPanner->separatorPos());
  config->writeEntry("Panners", s);   

	config->setGroup("Columns");

	// this has to be changed if the Col-Order changes by Drag'nDrop
	// to the new order
	config->writeEntry("ColIndex","0,1,2,3,4");

	// writes the TabListViews Columnwidth
	gHeaderList->writeConfig();
}

void KXmainwin::initMenuBar()
{
  QPopupMenu *file = new QPopupMenu;
	file->insertItem( klocale->translate("Send pending"), this,
		SLOT(slotSendPending()));
	file->insertItem( klocale->translate("Delete pending"), this,
		SLOT(slotDelPending()));
  file->insertSeparator();
	file->insertItem( klocale->translate("Exit"), this,
		SLOT(slotExit()));

  QPopupMenu *compose = new QPopupMenu;
	compose->insertItem( klocale->translate("New Message..."), this,
		SLOT(slotComposer()));
	compose->insertItem( klocale->translate("Reply..."), this,
		SLOT(slotReply()));

  QPopupMenu *options = new QPopupMenu;
	options->insertItem( klocale->translate("Accounts..."), this,
		SLOT(slotAccount()));
	options->insertItem( klocale->translate("General Options..."), this,
		SLOT(slotGenOpts()));
	options->insertItem( klocale->translate("Newsgroups..."), this,
		SLOT(slotNewsgroups()));

  QPopupMenu *help = new QPopupMenu;
	help = app->getHelpMenu(TRUE, aboutText);

  menu = new KMenuBar( this );
  CHECK_PTR( menu );
  menu->insertItem( klocale->translate("&File"), file );
  menu->insertSeparator();
  menu->insertItem( klocale->translate("&Compose"), compose );
  menu->insertSeparator();
  menu->insertItem( klocale->translate("&Options"), options );
  menu->insertSeparator();
  menu->insertItem( klocale->translate("&Help"), help );
  menu->insertSeparator();
  menu->show();

  setMenu(menu);
}

void KXmainwin::initToolBar()
{
  KIconLoader *loader = kapp->getIconLoader();
  toolbar = new KToolBar(this);

  toolbar->insertButton(loader->loadIcon("exchg.xpm"),0,
			SIGNAL(clicked()),this,
			SLOT(slotExchg()),TRUE,i18n("Send and receive"));

	toolbar->insertSeparator();

  addToolBar(toolbar);
}

void KXmainwin::initStatusBar()
{
  statusbar= new KStatusBar(this);

  statusbar->insertItem(klocale->translate("Welcome to kexpress"), ID_MSGTEXT);
  statusbar->show();
  setStatusBar(statusbar);
}

void KXmainwin::setMsg(const char *text)
{
  statusbar->changeItem(text, ID_MSGTEXT);
	app->flushX();
	app->processEvents(100);
}

void KXmainwin::setTmpMsg(const char *text,int time)
{
  statusbar->message(text,time);
}

void KXmainwin::slotAccount()
{
	QDialog *dlg;

	dlg = new KXAccPre(this);
	dlg->exec();
}

void KXmainwin::slotGenTabOk()
{
  KConfig *config = app->getConfig();

	debug("writing general options");

	config->setGroup("General");
	
	opts->writeConfig(config);

	config->sync();
}

void KXmainwin::slotGenOpts()
{
  KConfig *config = app->getConfig();
	QTabDialog *tab = new QTabDialog(toplevel,0,TRUE);

	config->setGroup("General");

	opts = new KXGenOpts(config,tab);
	tab->addTab(opts,klocale->translate("General"));

	tab->setOKButton(klocale->translate("Ok"));
	tab->setCancelButton(klocale->translate("Cancel"));
	tab->setCaption(klocale->translate("General Options"));
	tab->resize(500,500);

	connect(tab, SIGNAL(applyButtonPressed()), this, SLOT(slotGenTabOk()));

	tab->show();
}

void KXmainwin::slotNewsgroups()
{
	QDialog *dlg;
	KXEntry *entry;

	if (gEntryList->currentItem() < 0)
		return;

	entry = gEntryList->getEntry(gEntryList->currentItem());

	dlg = new KXNgDialog(entry,this);
	dlg->exec();
}

void KXmainwin::slotExit()
{
	cleanup();
	app->quit();
}

void KXmainwin::slotExchg()
{
	doSendPending(FALSE);
	gEntryList->doFolderSelected();
}

void KXmainwin::enableCommand(int id) //SLOT
{
  toolbar->setItemEnabled(id, true);
  menu->setItemEnabled(id, true);
}

void KXmainwin::disableCommand(int id) // SLOT
{
  toolbar->setItemEnabled(id, false);
  menu->setItemEnabled(id, false);
}

void KXmainwin::cleanup()
{
	writeConfig();
	app->getConfig()->sync();

	doExpire();
}

void KXmainwin::slotDelPending()
{
  if (gEntryList->selEntry == (KXEntry *) 0) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("You must choose a server first"),
      klocale->translate("Ok"));
    return;
  }   

	QDir spoolDir(gEntryList->selEntry->spoolPath->data());
	QStrList *idList = (QStrList *) spoolDir.entryList();
	QString boxMsg;

	if (idList->count() == 2) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("There are no pending messages"),
      klocale->translate("Ok"));
		return;
	}

	boxMsg.sprintf("Do you really want to delete %d pending message(s)?",idList->count()-2);
	
	if (QMessageBox::information(this,klocale->translate("Delete"),
	 klocale->translate(boxMsg),
		klocale->translate("Yes"), klocale->translate("No"), 0, 1)) {
	 return;
	}     

	char *ind;

	for (ind = idList->first();ind != 0;ind = idList->next()) {
		if (ind[0] == '.')
			continue;
		unlink(spoolDir.absFilePath(ind,TRUE).data());
	}
}

void KXmainwin::slotSendPending()
{
	doSendPending(TRUE);
}

void KXmainwin::doSendPending(bool verbose = TRUE)
{
	int count = 0;
	int good = 0;
	int bad = 0;

  if (gEntryList->selEntry == (KXEntry *) 0) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("You must choose a server first"),
      klocale->translate("Ok"));
    return;
  }   

	QDir spoolDir(gEntryList->selEntry->spoolPath->data());
	QStrList *idList = (QStrList *) spoolDir.entryList();
	char *ind;
	QString msgToSend;
	QString statusMsg;
	int res;

	if (idList->count() == 2) {
		if (verbose) {
			QMessageBox::information(this, klocale->translate("Composer"),
				klocale->translate("There are no pending messages"),
				klocale->translate("Ok"));
		}
		return;
	}

	for (ind = idList->first();ind != 0;ind = idList->next()) {
		if (ind[0] == '.')
			continue;

		debug("sending %s",spoolDir.absFilePath(ind,TRUE).data());

		statusMsg.sprintf("sending %s",ind);
		setMsg(statusMsg.data());

		msgToSend = kFileToString(spoolDir.absFilePath(ind,TRUE));

		res = gNNTP->sendMsg(gEntryList->selEntry,msgToSend);

		if (res < 0) {
//			QMessageBox::information(this, klocale->translate("Composer"),
//				klocale->translate("Your message(s) could not be send and keep spooled now.\nTry again later"),
//				klocale->translate("Ok"));
//			return;
				bad++;
		}
		else {
			unlink(spoolDir.absFilePath(ind,TRUE).data());
			good++;
		}

		count++;
	}

	setMsg("Ready");

	QString boxMess;
	boxMess.sprintf("%d message(s) (of %d) have been successfully sent",good,count);
	if (bad > 0)
		boxMess.append("\nThe not-sent messages keep spooled, try again later");

	if (verbose || bad > 0) {
		QMessageBox::information(this, klocale->translate("Composer"),
			klocale->translate(boxMess), klocale->translate("Ok"));
	}
}

void KXmainwin::slotComposer()
{
	KXComposer *dlg;
  KConfig *config = app->getConfig();

  if (gEntryList->selEntry == (KXEntry *) 0) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("You must choose a server first"),
      klocale->translate("Ok"));
    return;
  }   

  if (gEntryList->selEntry->fullName.isEmpty() ||
			gEntryList->selEntry->emailAddress.isEmpty()) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("Your Name/eMail-adress is not set"),
      klocale->translate("Ok"));
    return;
  }   

	KXMessage *mess = new KXMessage();
	QString sigFile;
	QString sig;
	QDir home;

	mess->genMessageId();
	mess->setNewsgrp(gEntryList->selectedGroup());

	config->setGroup("General");

	if (config->readBoolEntry("IncludeSig", TRUE)) {
		sigFile.sprintf("%s/%s",home.homeDirPath().data(),".signature");
		sig = kFileToString(sigFile,TRUE,FALSE);

		if (!sig.isEmpty()) {
			sig.prepend("\n-- \n");
			mess->setBody(sig.data());
		}
	}

	dlg = new KXComposer(mess);
	dlg->show();
}

void KXmainwin::slotReply()
{
  KConfig *config = app->getConfig();

	int item = gHeaderList->currentItem();

	if (item < 0) {
		debug("nothing to reply to");
		return;
	}

	QString id(*gHeaderList->artAt(item)->ID);
	QString a(*gNNTP->article(id));

	if (a.isEmpty()) {
		return;
	}

  if (gEntryList->selEntry->fullName.isEmpty() ||
			gEntryList->selEntry->emailAddress.isEmpty()) {
    QMessageBox::information(this, klocale->translate("Composer"),
      klocale->translate("Your Name/eMail-adress is not set"),
      klocale->translate("Ok"));
    return;
  }   

	KXComposer *dlg;
	QString refs;

	KXMessage *mess = new KXMessage();
	KXMessage *repMess = new KXMessage();
	repMess->fromString(a);
//	mess->fromString(a);

	mess->genMessageId();
	mess->setDateToday();

	if (!repMess->references().isEmpty()) {
      refs.append(repMess->references());
      refs.append(" ");
	}

	refs.append(repMess->id());
	mess->setReferences(refs);

	QString sub = repMess->subject();
	if (sub.left(4) != QString("Re: "))
		sub.prepend("Re: ");
	mess->setSubject(sub);

	mess->setNewsgrp(repMess->groups());
	mess->setFollowupto(repMess->followupto());

	QString repHeader;
	repHeader = repMess->from() + " wrote:\n";
	config->setGroup("General");
//	mess->setBody(repMess->body());
	mess->setReplyBody(repMess,repHeader,config->readEntry("ReplyPrefix", ">"));

	dlg = new KXComposer(mess);
	dlg->show(); 
}
