/*
    oe5_2mbox.cxx
    
    Copyright (C) 6-2000  Hans Dijkema 
    See README and COPYING for more information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <oe5_2mbox.hxx>

///////////////////////////////////////////////////////////////////////////
// Some little types 
///////////////////////////////////////////////////////////////////////////

typedef enum { RDMSG, EOFMSG } state_t;

///////////////////////////////////////////////////////////////////////////
// We want to log without keeping buffers
///////////////////////////////////////////////////////////////////////////

#define OUTNOW(log)	fseek(log,0,SEEK_END)

///////////////////////////////////////////////////////////////////////////
// Static function definitions used by the convert function
///////////////////////////////////////////////////////////////////////////

static unsigned long ReadLong(FILE *in,int *err);
static unsigned long ReadTripple(FILE *in,int *err);
static void WriteSkip0x0D(unsigned char *buf,int size,int n,FILE *out,int *err);
static void forward(FILE *in,int longs,int *err);

static unsigned long *GetMessageStarts(FILE *in,FILE *log,int *err);
static void ConvertMessage(FILE *in,FILE *out,unsigned long start,
                           FILE *log,int *err
                          );
static void FreeMessageStarts(unsigned long *msgstarts,FILE *log,int *err);

///////////////////////////////////////////////////////////////////////////
// Static function definitions used by the convert function
///////////////////////////////////////////////////////////////////////////

oe5_2mbox::oe5_2mbox(FILE *_in,FILE *_out,FILE *_log)
{
  in=_in;
  out=_out;
  log=_log;
}

oe5_2mbox::~oe5_2mbox()
{}


int oe5_2mbox::convert(void)
//
// convert(), converts the Outlook Express files.
// ==0,  Everything seems to work out.
// !=0,  We think we have a problem. Look in the Log file.
//
{
unsigned long *msgstarts;
int i,_err=0,*err=&_err;

  if (log!=NULL) {
    fprintf(log,"Reading message starts\n");OUTNOW(log);
  }

  msgstarts=GetMessageStarts(in,log,err);

  for(i=0;msgstarts[i]!=0;i++) {
    if (log!=NULL) { 
       fprintf(stderr,"Converting message at %08x\n",msgstarts[i]);OUTNOW(log);
    }
    ConvertMessage(in,out,msgstarts[i],log,err);
  }

  FreeMessageStarts(msgstarts,log,err);

return _err;
}

///////////////////////////////////////////////////////////////////////////
// The real conversion is standard C
///////////////////////////////////////////////////////////////////////////

void ConvertMessage(FILE *in,FILE *out,unsigned long start,FILE *log,int *err) 
{
unsigned long word,offset,max_blocklen,blocklen,next_offset;
unsigned char *buf=(unsigned char *) malloc(10);
state_t s;
int   FirstBlock=1;

   // This is for the mbox format (also used by kmail) 
   // to start the message.

   fprintf(out,"\n");
   fprintf(out,"From ahriman@material.world Fri Jun 23 05:39:56 EDT 2000\n"); 

   fseek(in,start,SEEK_SET);
   offset=start;
   word=ReadLong(in,err);

   if (word!=offset) {
     err[0]=OE5_ERR_WRONGLINK;
     if (log!=NULL) {
       fprintf(log,"HEY! This should not happen, can't read message here!\n");
       OUTNOW(log);
     }
   }
   else {
       max_blocklen=ReadLong(in,err);
       blocklen=ReadLong(in,err);
       next_offset=ReadLong(in,err);

       /* start reading message */

       s=RDMSG;
       while (s!=EOFMSG) {

         // Read linked blocks

         if (blocklen!=0) {
           buf=(unsigned char *) realloc(buf,blocklen);
           fread(buf,blocklen,1,in);
           WriteSkip0x0D(buf,blocklen,1,out,err);
         }

         // seek to next block and read next long 

	 fseek(in,next_offset,SEEK_SET);
	 word=ReadLong(in,err);

         if (word==next_offset && next_offset!=0) {
           max_blocklen=ReadLong(in,err);
           blocklen=ReadLong(in,err);
           next_offset=ReadLong(in,err);
         }
         else if (next_offset==0) {
           s=EOFMSG;
         }
	 else {
	   err[0]=OE5_ERR_WRONGLINK;
	   if (log!=NULL) {
             fprintf(log,"HEY! This should not happen,"
                         "can't read message here\n"
                         "     in the middle of a message!\n"
                    );
             OUTNOW(log);
           }
	   s=EOFMSG;
	 }
       }
     }
     free(buf);
}


unsigned long *GetMessageStarts(FILE *in,FILE *log,int *err)
{
unsigned long adress;
unsigned long messages;
unsigned long *msgstarts;

  // First we seek to the beginning of the file.

  fseek(in,0,SEEK_SET);

  // The 13th long word catches the adress of the 
  // message header index. We read it and seek to it

  forward(in,12,err);
  adress=ReadLong(in,err);
  fseek(in,adress,SEEK_SET);
  if (log!=NULL) {
    fprintf(log,"Index starts at %08x\n",adress);OUTNOW(log);
  }

  // Now we're in the index.
  // Start reading: The 5th long word, from second byte gives
  // us the right number of messages. Thus, we need only 3 bytes
  // of the 4 byte long.
  // Next allocate msgstarts to this number of messages

  forward(in,4,err);
  messages=ReadTripple(in,err);
  if (log!=NULL) {
    fprintf(log,"%ld (0x%lx) messages in the index\n",messages,messages);
    OUTNOW(log);
  }
  msgstarts=(unsigned long *) malloc(sizeof(unsigned long)*(messages+1));

  // Now we're going to read these messages index starts.
  // Forward to the first one right now.

  forward(in,1,err);
  {int i;
    for(i=0;i<messages;i++) {
      msgstarts[i]=ReadLong(in,err);
      if (log!=NULL) {
        fprintf(log,"message index %d starts at %08x\n",i,msgstarts[i]);
        OUTNOW(log);
      }

      // Forward to the next (2 longs on the way)

      forward(in,2,err);
    }

    // End of messages is indicated by a zero member in msgstarts[]
    
    msgstarts[i]=0;
  }

  // Now convert these message index starts to the real message
  // starts
  // Seek to the index position of msgstarts[]; and read the 7e
  // long word. Here also the first byte should be skipped.
  //
  // Its a tripple again (strange behaviour of OE5 --> Max folder len
  // is thereby 16Mb????). Of course only the first block of a
  // message actually needs to be in this first 16Mb though I would
  // have opted for a 32bits at least here.
  //

  {int i;
    for(i=0;i<messages;i++) {unsigned long offset;
      fseek(in,msgstarts[i],SEEK_SET);
      forward(in,6,err);
      offset=ReadTripple(in,err);
      if (log!=NULL) {
        fprintf(log,"message %d (index %08x) starts at %08x\n",
                    i,msgstarts[i],offset
               );
        OUTNOW(log);
      }
      msgstarts[i]=offset;
    }
  }

  // Now we kan go seeking out the messages (they can be
  // fragmented, so we're going to read blocks here).

  return msgstarts;
}

void FreeMessageStarts(unsigned long *msgstarts,FILE *log,int *err)
{
   free(msgstarts);
}

///////////////////////////////////////////////////////////////////////////
// The supporting functions
///////////////////////////////////////////////////////////////////////////

unsigned long ReadLong(FILE *in,int *err)
//
// Note: The OE5 file comes from the Wintel platform,
// so the endian is known ==> 0x12345678 maps to 78 56 34 12
//
{
unsigned long from,to;
unsigned char *mem=(unsigned char *) &from;

  // We read the long from file directly into a memory location

  fread(&from,sizeof(from),1,in);

  // Now we just do higher order calculation on the
  // individual bytes in variable mem.

  to=mem[3];
  to<<=8;to|=mem[2];
  to<<=8;to|=mem[1];
  to<<=8;to|=mem[0];

  // return the right value (adress, size, number off)

  return to;
}

unsigned long ReadTripple(FILE *in,int *err)
//
// Note: The OE5 file comes from the Wintel platform,
// so the endian is known ==> 0x12345678 maps to 78 56 34 12
// Some of the numbers used in the .DBX format seem three bytes
// long instead of the 32bit value 24bit value is used.
// 
// This function skips the first byte of the read long.
//
{
unsigned long from,to;
unsigned char *mem=(unsigned char *) &from;

  // We read the long from file directly into a memory location

  fread(&from,sizeof(from),1,in);

  // Now we just do higher order calculation on the
  // individual bytes in variable mem.

  to=mem[3];
  to<<=8;to|=mem[2];
  to<<=8;to|=mem[1];

  // return the right value (adress, size, number off)

  return to;
}

void WriteSkip0x0D(unsigned char *buf,int size,int n,FILE *out,int *err)
//
// This function omits the 0x0D's of the MSDOS format
// Simply DOS to UNIX.
//
{
int i,k;
  for(i=0,k=0;i<size;i++) {
    buf[k]=buf[i];
    if (buf[i]!=13) { k+=1; }
  }
  fwrite(buf,k,n,out);
}

void forward(FILE *in,int steps,int *err)
//
// This function forwards steps longs in the file in
//
{
unsigned long word;
  fseek(in,steps*sizeof(word),SEEK_CUR);
}
