/******************************************************************************
* Pinfo is a lynx-style info and manual viewer. It is written by              *
* Przemek Borys <pborys@dione.ids.pl>. Copying policy is GPL                  *
******************************************************************************/

#include "common_includes.h"

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif

char *version = VERSION;
int DontHandleWithoutTagTable = 0;

char *curfile = 0;		/* currently viewed filename */

char *pinfo_start_node = 0;	/* node specified by --node option */

void strip_file_from_info_suffix (char *file);	/* 
						 * strip `.info' suffix from
						 * "file" 
						 */
char *addinfosuffix (char *file);	/* add `.info' suffix to "file" */

int
main (int argc, char *argv[])
{
  int filenotfound = 0;
  char filename[256];
  WorkRVal work_return_value =
  {0, 0};
  int i;
  int command_line_option;
  FILE *id = NULL;
  long lines = 0;		/* line count in message */
  char **message = 0;		/* this will hold node's text */
  char *type = 0;		/* this will hold the node's header */
  char *tmp;
  int tag_table_pos = 1;
#ifdef HAVE_GETOPT_LONG
  static struct option long_options[] =
  {
    {"help", 0, 0, 'h'},
    {"version", 0, 0, 'v'},
    {"manual", 0, 0, 'm'},
    {"raw-filename", 0, 0, 'r'},
    {"apropos", 0, 0, 'a'},
    {"cut-man-headers", 0, 0, 'c'},
    {"empty-lines", 0, 0, 'e'},
    {"dont-handle-without-tag-table", 0, 0, 'd'},
    {"force-manual-tag-table", 0, 0, 't'},
    {"node", 1, 0, 'n'},
    {"long-manual-links", 0, 0, 'l'},
    {0, 0, 0, 0}};
#endif
  signal_handler ();		/* take care of SIGSEGV, SIGTERM, SIGINT */
  searchagain.type=0;
  searchagain.search=0;
  initlocale ();
  inithistory ();
  printf ("Przemek's Info Viewer v%s\n", version);
  initpaths ();
  parse_config ();		/* read config information */
  if (argc == 1)		/* if no arguments were given */
    {
      id = openinfo ("dir", 0);
      curfile = xmalloc (150);
      strcpy (curfile, "dir");
    }
  if (strlen (argv[0]) >= 3)
    if (strstr (argv[0], "man") != NULL)	/* handle any 'man' alias to 'pinfo' */
      {
	printf (_ ("Looking for man page...\n"));
	strcpy (filename, "");
	for (i = 1; i < argc; i++)	/* pass all arguments to the `man' 
					   command (manhandler calls `man') */
	  {
	    strcat (filename, argv[i]);
	    strcat (filename, " ");
	  }
	exit (handlemanual (filename));
      }
#ifdef HAVE_GETOPT_LONG

/******************************************************************************
* Parse command line options (getopt)                                         *
******************************************************************************/

  do
    {
      command_line_option = getopt_long (argc, argv, "hvmracdetnl", long_options, NULL);
      switch (command_line_option)
	{
	case 'l':
	  LongManualLinks=1;
	  break;
	case 'n':
	  if(!optarg)
	    {
	      printf(_("--node option used without argument\n"));
	      exit(1);
	    }
	  pinfo_start_node=malloc(strlen(optarg)+1);
	  strcpy(pinfo_start_node,optarg);
	  break;
	case 't':
	  ForceManualTagTable = 1;
	  break;
	case 'h':
	  printf (_ ("Usage:\n" \
		     "%s [options] [info|manual]\n" \
		     "Options:\n" \
		     "--help, -h: help\n" \
		     "--version, -v: version\n" \
		     "--manual, -m: use man page\n" \
		     "--raw-filename, -r: use raw filename\n" \
		     "--apropos, -a: call apropos if nothing found\n"),
		  argv[0]);
	  exit (0);
	case 'v':
	  exit (0);
	case 'm':
	  printf (_ ("Looking for man page...\n"));
	  strcpy (filename, "");
	  for (i = optind; i < argc; i++)
	    {
	      strcat (filename, argv[i]);
	      strcat (filename, " ");
	    }
	  exit (handlemanual (filename));
	case 'r':
	  strncpy (filename, argv[argc - 1], 200);
	  checkfilename (filename);	/* security check */
	  id = openinfo (filename, 0);	/* try without '.info' suffix */
	  break;
	case 'a':
	  use_apropos = 1;
	  break;
	case 'c':
	  CutManHeaders = 1;
	  break;
	case 'd':
	  DontHandleWithoutTagTable = 1;
	  break;
	case 'e':
	  CutEmptyManLines = 1;
	  break;
	}
    }
  while (command_line_option != EOF);
/*****************************************************************************/
#endif

  if (argc > 1)
    {
      strncpy (filename, argv[argc - 1], 200);	/* the paths will be searched
						   by openinfo() */
      checkfilename (filename);	/* security check */

      curfile = xmalloc (strlen (filename) + 100);	/* leave some space for
							   `.info' suffix */
      strcpy (curfile, filename);
    }

  if (id == NULL)		/* try to open given filename with appended `.info' */
    {
      strcat (filename, ".info");
      id = openinfo (filename, 0);
    }
  if (id == NULL)		/* if still nothing, try to use man page instead */
    {
      printf (_ ("Error: could not open info file, trying manual\n"));
      strncpy (filename, argv[argc - 1], 200);	/* security check */
      exit (handlemanual (filename));
    }
  if (seek_indirect (id))	/* search for indirect entries, if any */
    {
      read_item (id, &type, &message, &lines);
      load_indirect (message, lines);
    }


  if (seek_tag_table (id) != 2)	/* load tag table if such exists... */
    {
      if (ForceManualTagTable == 0)
	{
	  read_item (id, &type, &message, &lines);
	  load_tag_table (message, lines);
	}
      else
	{
	  if (indirect)
	    create_indirect_tag_table ();
	  else
	    {
	      fseek (id, SEEK_SET, 0);
	      create_tag_table (id);
	    }
	}
    }
  else
    /* ...otherwise try to create one */
    {
      printf (_ ("Warning: tag table not found...\n"));
      if (!DontHandleWithoutTagTable)
	{
	  printf (_ ("Trying to create alternate tag table...\n"));
	  create_tag_table (id);
	  if (TagTableEntries < 1)	/* if there weren't found any info
					   entries */
	    {
	      printf (_ ("This doesn't look like info file...\n"));
	      return 1;
	    }
	}
      else
	return 1;
    }
    
  if(pinfo_start_node)
    {
      tag_table_pos=gettagtablepos(pinfo_start_node);
      if(tag_table_pos==-1)
        {
          printf(_("Specified node does not exist...\n"));
          return 1;
        }
    }
    
  init_curses ();		/* initialize curses screen interface */

  do
    {
      seeknode (tag_table_pos, &id);	/* set seek offset for given node */
      read_item (id, &type, &message, &lines);	/* read the node */

      if (!filenotfound)	/* handle goto/link where no file was found 
				   -- see bellow */
	addinfohistory (curfile, tag_table[tag_table_pos].nodename, -1, -1);
      else
	filenotfound = 0;
      work_return_value = work (&message, &type, &lines, id, tag_table_pos);
      if (work_return_value.node)
	{
	  if (work_return_value.file[0] == 0)	/* no cross-file link selected */
	    {
	      int tmppos = gettagtablepos (work_return_value.node);
	      if (tmppos != -1)
		tag_table_pos = tmppos;
	    }
	  else
	    /* file was specified */
	    {
	      strip_file_from_info_suffix (work_return_value.file);
	      if (strcmp (curfile, work_return_value.file) == 0)
		/* file name was the same with the file currently viewed */
		{
		  int tmppos = gettagtablepos (work_return_value.node);
		  if (tmppos != -1)
		    tag_table_pos = tmppos;
		}
	      else
		/* open new info file */
		{
		  char *tmp;
		  fclose (id);
		  tmp = addinfosuffix (work_return_value.file);
		  id = openinfo (tmp, 0);
		  xfree (tmp);
		  tmp = 0;
		  if (id == NULL)	/* if the file doesn't exist */
		    {
		      attrset (bottomline);
		      mvhline (maxy - 1, 0, ' ', maxx);
		      mvaddstr (maxy - 1, 0, _ ("File not found. Press any key..."));
		      move (0, 0);
		      attrset (normal);
		      getch ();
		      filenotfound = 1;
		      if (infohistory.length)
			{
			  npos = infohistory.pos[infohistory.length];
			  ncursor = infohistory.cursor[infohistory.length];
			}
		      strip_file_from_info_suffix (curfile);
		      tmp = addinfosuffix (curfile);
		      id = openinfo (tmp, 0);
		      xfree (tmp);
		      tmp = 0;
		      if (id == NULL)
			{
			  closeprogram ();
			  printf (_ ("Unexpected error.\n"));
			  return 1;
			}
		    }
		  else
		    /* if we succeeded in opening new file */
		    {
		      if (curfile)
			{
			  xfree (curfile);
			  curfile = 0;
			}
		      curfile = xmalloc (strlen (work_return_value.file) + 150);
		      strcpy (curfile, work_return_value.file);
		      freeindirect ();
		      if (seek_indirect (id))
			{
			  read_item (id, &type, &message, &lines);
			  load_indirect (message, lines);
			}
		      freetagtable ();
		      if (seek_tag_table (id) != 2)
			{
			  if (ForceManualTagTable == 0)
			    {
			      read_item (id, &type, &message, &lines);
			      load_tag_table (message, lines);
			    }
			  else
			    {
			      if (indirect)
				create_indirect_tag_table ();
			      else
				{
				  fseek (id, SEEK_SET, 0);
				  create_tag_table (id);
				}
			    }
			}
		      else
			{
			  if (!DontHandleWithoutTagTable)
			    {
			      TagTableEntries = 0;
			      mvhline (maxy - 1, 0, ' ', maxx);
			      mvaddstr (maxy - 1, 0, _ ("Tag table not found. Trying to create alternate..."));
			      create_tag_table (id);
			      if (TagTableEntries < 1)
				{
				  closeprogram ();
				  printf (_ ("This doesn't look like info file...\n"));
				  return 1;
				}
			    }
			  else
			    return 1;
			}
		      if (work_return_value.node[0] != 0)
			{
			  int tmptagtablepos = gettagtablepos (work_return_value.node);
			  if (tmptagtablepos != -1)
			    tag_table_pos = tmptagtablepos;
			  else
			    tag_table_pos = 1;
			}
		      else
			tag_table_pos = 1;

		    }		/* end: open new info file -- file exists */
		}		/* end: open new info file */
	    }			/* end: file name was specified */
	}			/* end: node was specified in work return value */
    }
  while (work_return_value.node);
  fclose (id);
  closeprogram ();
  freelinks ();			/* free's at the end are optional, but look nice :) */
  freeitem (&type, &message, &lines);
  freetagtable ();
  freeindirect ();
  return 0;
}
void
strip_file_from_info_suffix (char *file)
{
  if (strlen (file) > 5)
    {
      if (strcmp (file + strlen (file) - 5, ".info") == 0)
	{
	  file = file + strlen (file) - 5;
	  *file = 0;
	}
    }
}
char *
addinfosuffix (char *info)
{
  char *withsuffix = xmalloc (strlen (info) + 150);
  strcpy (withsuffix, info);
  if (strlen (info) == 3)
    {
      if (strcmp ("dir", info) != 0)
	strcat (withsuffix, ".info");
    }
  else
    strcat (withsuffix, ".info");

  return withsuffix;
}
