/*

------------------------------------------------------------------------------

A license is hereby granted to reproduce this software source code and
to create executable versions from this source code for personal,
non-commercial use.  The copyright notice included with the software
must be maintained in all copies produced.

THIS PROGRAM IS PROVIDED "AS IS". THE AUTHOR PROVIDES NO WARRANTIES
WHATSOEVER, EXPRESSED OR IMPLIED, INCLUDING WARRANTIES OF
MERCHANTABILITY, TITLE, OR FITNESS FOR ANY PARTICULAR PURPOSE.  THE
AUTHOR DOES NOT WARRANT THAT USE OF THIS PROGRAM DOES NOT INFRINGE THE
INTELLECTUAL PROPERTY RIGHTS OF ANY THIRD PARTY IN ANY COUNTRY.

Copyright (c) 1995, John Conover, All Rights Reserved.

Comments and/or bug reports should be addressed to:

    john@johncon.com (John Conover)

------------------------------------------------------------------------------

tsfBm.c, fractional brownian noise generator-generates a time
series. The idea is to produce a programmable power spectrum
distribution. See "Chaos and Order in the Capital Markets", Edgar
E. Peters, John Wiley \& Sons, New York, New York, 1991, ISBN
0-471-53372-6, pp 211, or "Fractals", Jens Feder, Plenum Press, New
York, New York, 1988, ISBN 0-306-42851-2, pp 173, referencing
Mandelbrot and Wallis, 1969.

Note: this program is computationally intensive-typically, about 30
minutes per 100 samples on a 20 Mhz. 386/387 machine.

Note: these programs use the following functions from other
references:

    ran1, which returns a uniform random deviate between 0.0 and
    1.0. See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 210, referencing Knuth.

    gasdev, which returns a normally distributed deviate with zero
    mean and unit variance, using ran1 () as the source of uniform
    deviates. See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 217.

    gammln, which returns the log of the results of the gamma
    function.  See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 168.

$Revision: 0.0 $
$Date: 1995/11/18 20:28:55 $
$Id: tsfBm.c,v 0.0 1995/11/18 20:28:55 john Exp $
$Log: tsfBm.c,v $
Revision 0.0  1995/11/18 20:28:55  john
Initial version


*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <getopt.h>

static char rcsid[] = "$Id: tsfBm.c,v 0.0 1995/11/18 20:28:55 john Exp $"; /* program version */
static char copyright[] = "Copyright (c) 1995, John Conover, All Rights Reserved"; /* the copyright banner */

#ifdef __STDC__

static const char *error_message[] = /* error message index array */

#else

static char *error_message[] = /* error message index array */

#endif

{
    "",
    "Generate a fractional brownian noise time series\nUsage: tsfBm [-d] [-h n] [-m n] [-n n] [-t] [-v] number\n    number is the number of samples in the time series\n    -d output fBm(t) - fBm(t - 1) instead of sum\n    -h n, Hurst coefficient\n    -m n, length of memory effects\n    -n n, number of increments in each time step\n    -t sample's time will be included in the output time series\n    -v print the program's version information\n",
    "Error allocating memory\n"
};

#define NOERROR 0 /* error values, one for each index in the error message array */
#define EARGS 1
#define EALLOC 2

#ifndef PI /* make sure PI is defined */

#define PI 3.141592653589793 /* pi to 15 decimal places as per CRC handbook */

#endif

#ifdef __STDC__

static double ran1 (int *idum);
static double gasdev (int *idum);
static double gamm (double xx);
static double gammln (double xx);

#else

static double ran1 ();
static double gasdev ();
static double gamm ();
static double gammln ();

#endif

#ifdef __STDC__

int main (int argc,char *argv[])

#else

int main (argc,argv)
int argc;
char *argv[];

#endif

{
    int number, /* number of records in time series */
        retval = EARGS, /* return value, assume not enough arguments */
        j, /* index counter */
        idem = -1, /* random number initialize flag */
        size = 0, /* size of the array of random numbers of gaussian distribution */
        newsize, /* size of the array of random numbers of gaussian distribution */
        t = 0, /* print time of samples flag, 0 = no, 1 = yes */
        d = 0, /* print differences instead of sum flag, 1 = yes, 0 = no */
        u = 0, /* math variable */
        k = 1, /* math variable */
        l = 1, /* math variable */
        m = 700, /* length of memory effect */
        n = 8, /* number of increments in each time step */
        c; /* command line switch */

    double g, /* gamma of h + 0.5 */
           h = (double) 0.5, /* hurst coefficient */
           e, /* math variable */
           e1 = (double) 0.0, /* math variable */
           e2 = (double) 0.0, /* math variable */
           e3 = (double) 0.0, /* math variable */
           a, /* constant term in equation */
           delta, /* delta between t and t - 1 of the fBm */
           sum = (double) 0.0, /* running sum of fBm */
           *x = (double *) 0; /* array of random numbers of gaussian distribution */

    while ((c = getopt (argc, argv, "dh:m:n:tv")) != EOF) /* for each command line switch */
    {

        switch (c) /* which switch? */
        {

            case 'd': /* request printing differences instead of sum */

                d = 1; /* yes, set the print the differences instead of sum flag */
                break;

            case 'h': /* request for hurst coefficient? */

                h = atof (optarg);  /* yes, save the hurst coefficient */
                break;

            case 'm': /* request for length of memory effect? */

                m = atoi (optarg); /* yes, save the length of memory effect */
                break;

            case 'n': /* request for number of increments in each time step? */

                n = atoi (optarg); /* yes, save the number of increments in each time step */
                break;

            case 't': /* request printing time of samples? */

                t = 1; /* yes, set the print time of samples flag */
                break;

            case 'v':

                (void) printf ("%s\n", rcsid); /* print the version */
                (void) printf ("%s\n", copyright); /* print the copyright */
                optind = argc; /* force argument error */

            default: /* illegal switch? */

                optind = argc; /* force argument error */
                break;
        }

    }

    if (argc - optind > 0) /* enough arguments? */
    {
        number = atoi (argv[optind]); /* number of records in time series */
        g = gamm (h + (double) 0.5); /* compute the gamma of H + 0.5 */
        retval = NOERROR; /* assume no error */
        a = pow ((double) n, -h) / g; /* compute the constant term in equation */

        for (u = 0; u < number; u++) /* for each sample in the time series */
        {
            e1 = (double) 0.0; /* initialize first and second terms */
            e3 = (double) 0.0;

            for (k = 1; k <= n; k ++) /* first summation */
            {

                if ((newsize = (1 + n * (m + u) - k)) > size) /* array of random numbers of gaussian distribution large enough? */
                {

                    if (size == 0) /* no, array of random numbers of gaussian distribution been allocated yet? */
                    {

                        if ((x = (double *) malloc ((newsize + 1) * sizeof (double))) == (double *) 0) /* no, allocate space for the array of random numbers of gaussian distribution */
                        {
                            retval = EALLOC;  /* assume error allocating memory */
                            break;
                        }

                        for (j = 0; j <= newsize; j++) /* for each new element in the array of random numbers of gaussian distribution */
                        {
                            x[j] = gasdev (&idem); /* get the random number */
                        }

                    }

                    else
                    {

                        if ((x = (double *) realloc (x, (newsize + 1) * sizeof (double))) == (double *) 0) /* yes, reallocate space for the array of random numbers of gaussian distribution */
                        {
                            retval = EALLOC;  /* assume error allocating memory */
                            break;
                        }

                        for (j = size + 1; j <= newsize; j++) /* for each new element in the array of random numbers of gaussian distribution */
                        {
                            x[j] = gasdev (&idem); /* get the random number */
                        }

                    }

                    size = newsize; /* new size of array of random numbers with gaussian distribution */
                }

                e = (pow ((double) k, h - (double) 0.5) * x[1 + n * (m + u) - k]);
                e1 = e1 + e; /* summation of first term */
            }

            if (retval != NOERROR) /* any errors? */
            {
                break; /* yes, stop */
            }

            for (l = 1; l <= n * (m - 1); l ++) /* second summation */
            {
                e2 = (pow ((double) (n + l), (h - (double) 0.5)) - pow ((double) l, h - (double) 0.5)) * x[1 + n * (m - 1 + u) - l];
                e3 = e3 + e2; /* summation of second term */
            }

            delta = a * (e1 + e3); /* calculation of increment */

            if (t == 1) /* print time of samples? */
            {
                (void) printf ("%d\t", u); /* yes, print the sample's time */
            }

            if (d == 1) /* print the differences instead of sum flag set? */
            {
                (void) printf ("%f\n", delta); /* yes, print the record */
            }

            else
            {
                sum = sum + delta; /* save the running sum of the fBm */
                (void) printf ("%f\n", sum); /* no, print the record */
            }

        }

        if (x != (double *) 0) /* array of random numbers of gaussian distribution still allocated? */
        {
            free (x); /* yes, free the array of random numbers of gaussian distribution */
        }

    }

    (void) fprintf (stderr, "%s", error_message[retval]); /* print any errors */
    exit (retval); /* exit with the error value */

#ifdef LINT

    return (0); /* for lint formalities */

#endif

}

#define M1 259200
#define IA1 7141
#define IC1 54773
#define RM1 (1.0/M1)
#define M2 134456
#define IA2 8121
#define IC2 28411
#define RM2 (1.0/M2)
#define M3 243000
#define IA3 4561
#define IC3 51349

/*

Returns a uniform random deviate between 0.0 and 1.0. Set idum to any
negative value to initialize or reinitialize the sequence. See
"Numerical Recipes in C: The Art of Scientific Computing," William
H. Press, Brian P. Flannery, Saul A. Teukolsky, William T. Vetterling,
Cambridge University Press, New York, 1988, ISBN 0-521-35465-X, page
210, referencing Knuth.

*/

#ifdef __STDC__

static double ran1 (int *idum)

#else

static double ran1 (idum)
int *idum;

#endif

{
    static int iff = 0;

    static long ix1,
                ix2,
                ix3;

    static double r[98];

    int j;

    double temp;

    if (*idum < 0 || iff == 0) /* initialize on first call even if idum is not negative */
    {
        iff = 1;
        ix1 = (IC1 - (*idum)) % M1; /* seed first routine */
        ix1 = (IA1 * ix1 + IC1) % M1;
        ix2 = ix1 % M2; /* use first to seed second routine */
        ix1 = (IA1 * ix1 +IC1) % M1;
        ix3 = ix1 % M3; /* use first to seed third routine */

        for (j = 1; j <= 97; j++) /* fill table with sequential uniform deviates generated by first two routines */
        {
            ix1 = (IA1 * ix1 + IC1) % M1;
            ix2 = (IA2 * ix2 + IC2) % M2;
            r[j] = (ix1 + ix2 * RM2) * RM1; /* low and high order pieces combined here */
        }

        *idum = 1;
    }

    ix1 = (IA1 * ix1 + IC1) % M1; /* except when initializing, this is the start-generate the next number for each sequence */
    ix2 = (IA2 * ix2 + IC2) % M2;
    ix3 = (IA3 * ix3 + IC3) % M3;
    j = 1 + ((97 * ix3)/M3); /* use the third sequence to get an integer between 1 and 97 */

    if (j > 97 || j < 1)
    {
        (void) fprintf (stderr, "RAN1: This can not happen.\n");
        exit (1);
    }

    temp = r[j]; /* return that table entry */
    r[j] = (ix1 + ix2 * RM2) * RM1; /* refill the table's entry */
    return (temp);
}

#ifdef TEST_RAN1

/*

Calculates PI statistically using volume of unit n-sphere.  Test
driver for ran1 (). See "Numerical Recipes: Example Book (C),"
William T. Vetterling, Saul A. Teukolsky, William H. Press, Brian
P. Flannery, Cambridge University Press, New York, 1988, ISBN
0-521-35746-2, page 82.

*/

#include <stdio.h>
#include <math.h>

#ifndef PI

#define PI 3.141592653589793 /* pi to 15 decimal places as per CRC handbook */

#endif

#ifdef __STDC__

static int twotoj (int j);
static double fnc (double x1, double x2, double x3, double x4);
static double ran1 (int *idum);

#else

static int twotoj ();
static double fnc ();
static double ran1 ();

#endif

#ifdef __STDC__

void main (void)

#else

void main ()

#endif

{
    int i,
        idum = -1,
        j,
        k,
        jpower;

    double x1,
           x2,
           x3,
           x4,
           iy[4],
           yprob[4];

    /* Calculates PI statistically using volume of unit n-sphere */

    for (i = 1; i <= 3; i ++)
    {
        iy[i] = (double) 0.0;
    }

    (void) printf ("\nvolume of unit n-sphere, n = 2, 3, 4\n");
    (void) printf ("points\t\tPI\t\t(4/3)*PI\t(1/2)*PI^2\n\n");

    for (j = 1; j <= 14; j ++)
    {

        for (k = twotoj (j - 1); k <= twotoj (j); k ++)
        {
            x1 = ran1 (&idum);
            x2 = ran1 (&idum);
            x3 = ran1 (&idum);
            x4 = ran1 (&idum);

            if (fnc (x1, x2, (double) 0.0, (double) 0.0) < (double) 1.0)
            {
                ++ iy[1];
            }

            if (fnc (x1, x2, x3, (double) 0.0) < (double) 1.0)
            {
                ++ iy[2];
            }

            if (fnc (x1, x2, x3, x4) < (double) 1.0)
            {
                ++ iy[3];
            }

        }

        jpower=twotoj (j);
        yprob[1] = (double) 4.0 * iy[1] / jpower;
        yprob[2] = (double) 8.0 * iy[2] / jpower;
        yprob[3] = (double) 16.0 * iy[3] / jpower;
        (void) printf ("%6d\t%12.6f\t%12.6f\t%12.6f\n", jpower, yprob[1], yprob[2], yprob[3]);
    }

    (void) printf ("\nactual\t%12.6f\t%12.6f\t%12.6f\n", (double) PI, 4.0 * (double) PI / (double) 3.0, (double) 0.5 * (double) PI * (double) PI);
}

#endif

/*

Returns a normally distributed deviate with zero mean and unit
variance, using ran1 () as the source of uniform deviates. Set idum to
any negative value to initialize or reinitialize the sequence. See
"Numerical Recipes in C: The Art of Scientific Computing," William
H. Press, Brian P. Flannery, Saul A. Teukolsky, William T. Vetterling,
Cambridge University Press, New York, 1988, ISBN 0-521-35465-X, page
217.

*/

#ifdef __STDC__

static double gasdev (int *idum)

#else

static double gasdev (idum)
int *idum;

#endif

{
    static int iset = 0;

    static double gset;

    double fac,
           r,
           v1,
           v2;

    if (iset == 0)
    {

        do /* no deviate */
        {
            v1 = 2.0 * ran1 (idum) - 1.0; /* get two uniform numbers in the square extending from -1 to +1 in each direction */
            v2 = 2.0 * ran1 (idum) - 1.0;
            r = v1 * v1 + v2 * v2; /* see if they are in the unit circle */
        }
        while (r >= 1.0); /* if not, try again */

        fac = sqrt (-2.0 * log (r) / r); /* make the Box-Muller transformation to get two normal deviates, return one, save the other for next call */
        gset = v1 * fac;
        iset = 1; /* set flag */
        return (v2 * fac);
    }

    else
    {
        iset = 0; /* extra deviat from last time, unset the flag an return it */
        return (gset);
    }

}

/*

Returns the results of the gamma function with argument xx.

Uses the reflection formula,

    gamma (1 - z) = pi / (gamma (z) * sin (pi * z))
                  = (pi * z) / (gamma (1 + z) * sin (pi * z))

to compute gamma (xx), 0 < xx < 1

*/

#ifdef __STDC__

static double gamm (double xx)

#else

static double gamm (xx)
double xx;

#endif

{
    double retval, /* return value */
           z; /* 1 - xx for reflection when xx < 1 */

    if (xx < (double) 1.0) /* xx < 1? */
    {
        z = (double) 1.0 - xx; /* yes, compute z */
        retval = ((double) PI * z) / (exp (gammln (1 + z)) * sin ((double) PI * z)); /* compute gamm (1 - z) = gamm (xx) */
    }

    else
    {
        retval = exp (gammln (xx)); /* no, compute gamm (xx) */
    }

    return (retval);
}

/*

Returns the log of the results of the gamma function with argument xx.
See "Numerical Recipes in C: The Art of Scientific Computing," William
H. Press, Brian P. Flannery, Saul A. Teukolsky, William T. Vetterling,
Cambridge University Press, New York, 1988, ISBN 0-521-35465-X, page
168.

*/

#ifdef __STDC__

static double gammln (double xx)

#else

static double gammln (xx)
double xx;

#endif

{
    static double cof[6] =
    {
        (double) 76.18009173,
        (double) -86.50532033,
        (double) 24.01409822,
        (double) -1.231739516,
        (double) 0.120858003e-2,
        (double) -0.536382e-5
    };

    double x,
           tmp,
           ser;

    int j;

    x = xx - (double) 1.0;
    tmp = x + (double) 5.5;
    tmp -= (x + (double) 0.5) * log (tmp);
    ser = (double) 1.0;

    for (j = 0; j <= 5; j ++)
    {
        x += (double) 1.0;
        ser += cof[j] / x;
    }

    return (-tmp + log (2.50662827456 * ser));
}
