/*

------------------------------------------------------------------------------

A license is hereby granted to reproduce this software source code and
to create executable versions from this source code for personal,
non-commercial use.  The copyright notice included with the software
must be maintained in all copies produced.

THIS PROGRAM IS PROVIDED "AS IS". THE AUTHOR PROVIDES NO WARRANTIES
WHATSOEVER, EXPRESSED OR IMPLIED, INCLUDING WARRANTIES OF
MERCHANTABILITY, TITLE, OR FITNESS FOR ANY PARTICULAR PURPOSE.  THE
AUTHOR DOES NOT WARRANT THAT USE OF THIS PROGRAM DOES NOT INFRINGE THE
INTELLECTUAL PROPERTY RIGHTS OF ANY THIRD PARTY IN ANY COUNTRY.

Copyright (c) 1995, John Conover, All Rights Reserved.

Comments and/or bug reports should be addressed to:

    john@johncon.com (John Conover)

------------------------------------------------------------------------------

tslogistic.c, logistic function generator-generates a time series.
The idea is to produce a function of the form y(t) = c / (1 + e^(-(at
+ b))).  See "The Art of Modeling Dynamic Systems", Foster Morrison,
John Wiley & Sons, New York, New York, 1991, pp 100, or "Predictions",
Theodore Modis, Simon & Schuster, New York, New York, 1992, ISBN
0-471-52004-7, pp 230.

Note: these programs use the following functions from other
references:

    ran1, which returns a uniform random deviate between 0.0 and
    1.0. See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 210, referencing Knuth.

    gasdev, which returns a normally distributed deviate with zero
    mean and unit variance, using ran1 () as the source of uniform
    deviates. See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 217.

    gammln, which returns the log of the results of the gamma
    function.  See "Numerical Recipes in C: The Art of Scientific
    Computing," William H. Press, Brian P. Flannery, Saul
    A. Teukolsky, William T. Vetterling, Cambridge University Press,
    New York, 1988, ISBN 0-521-35465-X, page 168.

$Revision: 0.0 $
$Date: 1995/11/18 20:28:55 $
$Id: tslogistic.c,v 0.0 1995/11/18 20:28:55 john Exp $
$Log: tslogistic.c,v $
Revision 0.0  1995/11/18 20:28:55  john
Initial version


*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <getopt.h>

static char rcsid[] = "$Id: tslogistic.c,v 0.0 1995/11/18 20:28:55 john Exp $"; /* program version */
static char copyright[] = "Copyright (c) 1995, John Conover, All Rights Reserved"; /* the copyright banner */

#ifdef __STDC__

static const char *error_message[] = /* error message index array */

#else

static char *error_message[] = /* error message index array */

#endif

{
    "",
    "Generate a logistic function time series\nUsage: tslogistic [-t] [-v] a b c number\nwhere the logistic function is y(t) = c / (1 + e^(-(at + b)))\n    number is the number of samples in the time series\n    -t sample's time will be included in the output time series\n    -v print the program's version information\n"
};

#define NOERROR 0 /* error values, one for each index in the error message array */
#define EARGS 1

#ifdef __STDC__

int main (int argc,char *argv[])

#else

int main (argc,argv)
int argc;
char *argv[];

#endif

{
    int number, /* number of records in time series */
        retval = EARGS, /* return value, assume not enough arguments */
        i, /* counter of number of records in time series */
        t = 0, /* print time of samples flag, 0 = no, 1 = yes */
        c; /* command line switch */

    double a, /* first parameter of logistic equation */
           b, /* second parameter of logistic equation */
           s; /* scaling factor of logistic equation */

    while ((c = getopt (argc, argv, "s:tv")) != EOF) /* for each command line switch */
    {

        switch (c) /* which switch? */
        {

            case 't': /* request printing time of samples? */

                t = 1; /* yes, set the print time of samples flag */
                break;

            case 'v':

                (void) printf ("%s\n", rcsid); /* print the version */
                (void) printf ("%s\n", copyright); /* print the copyright */
                optind = argc; /* force argument error */

            default: /* illegal switch? */

                optind = argc; /* force argument error */
                break;
        }

    }

    if (argc - optind > 3) /* enough arguments? */
    {
        retval = NOERROR; /* assume no error */

        a = atof (argv[optind]); /* first parameter of logistic equation */
        b = atof (argv[optind + 1]); /* second parameter of logistic equation */
        s = atof (argv[optind + 2]); /* scaling factor of logistic equation */
        number = atoi (argv[optind + 3]); /* number of records in time series */

        for (i = 0; i < number; i ++) /* for each record in the time series */
        {

            if (t == 1) /* print time of samples? */
            {
                (void) printf ("%d\t", i); /* yes, print the sample's time */
            }

            (void) printf ("%f\n", s / ((double) 1.0 + exp (-(b + (a * (double) i))))); /* print the record */
        }

    }

    (void) fprintf (stderr, "%s", error_message[retval]); /* print any errors */
    exit (retval); /* exit with the error value */

#ifdef LINT

    return (0); /* for lint formalities */

#endif

}
