/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <limits.h>
#include <fcntl.h>
#include <curses.h>
#include <sys/types.h>
#include <sys/time.h>
#include "version.h"
#include "main.h"
#include "error.h"
#include <sys/ioctl.h>

/************ Prototypes ************/

void main_trace_dumb(int n);
void main_trace_quiet(int);
void main_trace_curses(int n);
void main_finish_dumb();
void main_finish_quiet();
void main_finish_curses();
void main_init_curses();
void main_init_dumb();
void main_init_quiet();
void main_update_curses();
void main_update_dumb();
void main_update_quiet();
void main_do_options(int argc, char *argv[], char *envp[]);
void info();

/************************************/

char error_progname[PATH_MAX];

void main_trace_dumb(int n)
{
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX], tmp[PATH_MAX];

	if (n == 0) {
		return;
	}
	if (getprogress(src,dest,conv) == -1) exit(0);
	if (*conv) {
		if (strcmp(conv,tmp)) {
			printf("%s\n",conv);
			strcpy(tmp,conv);
		}
	} else if (*dest) {
		if (strcmp(dest,tmp)) {
			printf("%s\n",dest);
			strcpy(tmp,dest);
		}
	}
}

void main_trace_quiet(int n)
{
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX];
	if (getprogress(src,dest,conv) == -1) exit(0);
}

void main_trace_curses(int n)
{
	static int start;
	static struct timeval after;
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX];

	if (n == 0) {
		gettimeofday(&after, 0);
		start = after.tv_sec;
		return;
	}
	if (getprogress(src,dest,conv) == -1) exit(0);
	if (*conv) {
		attron(A_BOLD);
		mvprintw(17,17,conv);
		attroff(A_BOLD);
	} else {
		attron(A_BOLD);
		mvprintw(17,17,dest);
		attroff(A_BOLD);
		gettimeofday(&after, 0);
		mvprintw(5, 47, "%i:%i.%i",(after.tv_sec-start)/3600,
			 ((after.tv_sec-start)/60)%60,(after.tv_sec-start)%60);
	}

	refresh();
}

void main_finish_dumb()
{
	printf("done.\n");
}

void main_finish_quiet()
{
}

void main_finish_curses()
{
	flushinp();
	refresh();
	endwin();
}

void (*main_trace_func)(int) = main_trace_curses;

void (*main_finish_func)() = main_finish_curses;

void main_finish()
{
	(*main_finish_func)();
	fprintf(stderr,"%s\n",error_text);
}

void main_init_curses()
{
	char tmp1[PATH_MAX], tmp2[PATH_MAX];

	initscr(); cbreak(); noecho();		// init ncurses
	leaveok(stdscr, TRUE);			// disable cursor
	wresize(stdscr,24,80);			// resize win to 80x24
	refresh();				// flush the screen 
	nodelay(stdscr,1);			// don't wait for input in getch
	start_color();				// set colors
	init_pair(1,COLOR_WHITE,COLOR_BLUE);
	init_pair(2,COLOR_RED,COLOR_BLUE);
	init_pair(3,COLOR_CYAN,COLOR_BLUE);
	wbkgd(stdscr, COLOR_PAIR(1));
	wattrset(stdscr, COLOR_PAIR(1));
	attron(A_BOLD);
	newwin(0,0,0,0);
	box(stdscr, 0, 0 );							// 80x24 box
	
	mvprintw(14, 2, "Keys : ");

	inquire("width",tmp1);
	inquire("height",tmp2);
	mvprintw(7,16," %sx%s",tmp1,tmp2);
	
	attroff(A_BOLD);
	mvprintw(14,10,"Q	- Quit");
	mvprintw(15,10,"P	- Pause");
	mvprintw(5, 32,"Time (secs.) :");
	mvprintw(7, 2, "Image size   :");
	mvprintw(6, 2, "FPS          :");
	mvprintw(9, 2, "Status       :");
	mvprintw(17, 2,"Grabbing into:");
	
	attron(A_BOLD);
	mvprintw(9, 17, "Working...");		// status

	wattrset(stdscr, COLOR_PAIR(2));
	mvprintw(3, 1, "------------------------------------------------------------------------------");
	mvprintw(11, 1, "------------------------------------------------------------------------------");
	wattrset(stdscr, COLOR_PAIR(3));
	attron(A_BOLD);
	mvprintw(0,2,"< bttvgrab " VERSION " (v4l) by J. Walter/A. Kopacz >");
	mvprintw(23,30,"< http://moes.pmnet.uni-oldenburg.de/bttvgrab/ >");
	wattrset(stdscr, COLOR_PAIR(1));
	
	refresh();
	
	wgetch(stdscr);
}

void main_init_dumb()
{
	setbuf(stdout,NULL);
	setbuf(stdin,NULL);
}

void main_init_quiet()
{
	setbuf(stdout,NULL);
	setbuf(stdin,NULL);
}

void (*main_init_func)() = main_init_curses;

void main_update_curses()
{
	switch(getch()) { 
	case 'p': case 'P':
		nodelay(stdscr,0);
		attron(A_BOLD);
		mvprintw(9,17,"PAUSED - Push any key (except s/q) to continue");
		attroff(A_BOLD);
		mvprintw(15, 10, "S	- Sync HDDs      ");
		flushinp();
		control(C_STOP);
		switch(getch()) {
		case 's': case 'S':
			attron(A_BOLD);
			mvprintw(9,17,"Syncing HDDs...                                ");
			refresh();	      
			sync();
			attron(A_BOLD);
			mvprintw(9,17,"PAUSED - Press any key (except q) to continue");
			attroff(A_BOLD);
			mvprintw(15, 10,"                                    ");
			refresh();
			flushinp();
			switch(getch()) {
			case 'q': case 'Q':
				exit(0);
				break;
				
			default:
				break;
			}
			break;
		case 'q': case 'Q':
			exit(0);
			
		default: 
			break;
			
		}
		control(C_START);
		attron(A_BOLD);
		mvprintw(9, 17, "Working...                                    ");
		attroff(A_BOLD);
		mvprintw(15, 10, "P	- Pause    ");
		attron(A_BOLD);
		mvprintw(16, 10,"                                    ");
		nodelay(stdscr,1);
		break;
		
	case 'q': case 'Q':
		control(C_STOP);
		exit(0);
		
	default: 
		break;
	}
}

void main_update_dumb()
{
	switch(getc(stdin)) { 
	case '\n':
		control(C_STOP);
		printf("paused\n");
		if (getc(stdin) == 'q') exit(0);
		control(C_START);
		break;
	case 'q':
		exit(0);
	default:
		break;
	}
}

void main_update_quiet()
{
	main_update_dumb();
}

void (*main_update)() = main_update_curses;


/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////                                             ///////////////
/////////////                 Main loop                   ///////////////
/////////////                                             ///////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[], char *envp[])
{
	fd_set fds;

	strcpy(error_progname,argv[0]);
	main_init();

	FD_ZERO(&fds);

	configure("input_file_fmt","pmm");
	main_do_options(argc, argv, envp);

	(*main_init_func)();
	atexit(main_finish);
	control(C_START);
loop:
	FD_SET(0,&fds);
	FD_SET(main_inpipe,&fds);
	select(main_inpipe+1,&fds,NULL,NULL,NULL);

	if (FD_ISSET(main_inpipe,&fds)) (*main_trace_func)(1);
	if (FD_ISSET(0,&fds)) (*main_update)();
	goto loop;
}
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////

#define OPTSET(c,opt,val) case c: configure(#opt,val); break;
#define OPT(c,opt) OPTSET(c,opt,optarg);

void main_do_options(int argc, char *argv[], char *envp[])
{
	int optchar, height = 0;
	char tmp[PATH_MAX], *p;

	while ((optchar = getopt(argc, argv, "N:m:W:w:x:b:B:rc:s:i:q:l:f:g:F:D:o:tyQd:h")) != -1) {
		switch (optchar) {	
		OPT('w',width);
		OPT('m',msleep);
		case 'W':
			height = 1;
		OPT(1,height);
		OPT('x',first_conv);
		OPT('b',video_bitrate);
		OPT('B',audio_bitrate);
		OPTSET('r',lowdisk,"1");
		OPT('c',conv_log);
		case 's':
			p = strchr(optarg,':');
			if (p) {
				*p = 0;
				configure("input_transport",optarg);
				*p = ':';
				configure("inputname",p+1);
			} else configure("inputname",optarg);
			break;
		OPT('i',input_file_fmt);
		OPT('q',quality);
		OPT('l',limit);
		case 'f':
			p = strchr(optarg,':');
			if (p) {
				*p = 0;
				configure("conv_transport",optarg);
				*p = ':';
				configure("convname",p+1);
			} else configure("convname",optarg);
			break;
		OPT('g',grablog);
		OPT('F',soundfile);
		OPT('D',sounddevice);
		OPT('o',conv_file_fmt);
		OPT('N',norm);
		OPTSET('t',do_trace,"1");
		OPTSET('y',do_sync,"1");
		OPTSET('Q',do_sound,"0");
		case 'd':
			ASSERT(strlen(optarg) == 1,(ARGPMSG("display type must be one of c,d,q")));
			switch (*optarg) {
			case 'c':
				main_init_func = main_init_curses;
				main_trace_func = main_trace_curses;
				main_update = main_update_curses;
				main_finish_func = main_finish_curses;
				break;
			case 'd':
				main_init_func = main_init_dumb;
				main_trace_func = main_trace_dumb;
				main_update = main_update_dumb;
				main_finish_func = main_finish_dumb;
				break;
			case 'q':
				main_init_func = main_init_quiet;
				main_trace_func = main_trace_quiet;
				main_update = main_update_quiet;
				main_finish_func = main_finish_quiet;
				break;
			default:
				ASSERT(0,(ARGPMSG("display type must be one of c,d,q")));
				break;
			}
			break;
		case 'h': 
		default:
			printf("bttvconvert " VERSION "\n");
			printf("(c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>\n");
			printf("Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/\n\n");

			printf("Options:\n");
			printf("-g [logname]     : file name of grab log, any options present in\n");
			printf("                   it will override preceding options\n");
			printf("                   if missing, you MUST specify -i and (for raw input) -w\n");
			printf("-t               : trace, allow gaps in file numbers as created by\n");
			printf("                   bttvgrab -t, you MUST specify -l as well\n");
			printf("-w [32-768]      : width, default = 640 (height=480)\n");
			printf("                   height is fixed at (3/4 * width)\n");
			printf("-o [format]      : output file format, default = first, available:\n");
			inquire("conv_file_fmt list",tmp);
			printf("%s\n",tmp);
			printf("-i [format]      : input file format, default = pmm, available:\n");
			inquire("input_file_fmt list",tmp);
			printf("%s\n",tmp);
			printf("-q [1-200]       : image quality, default = 75, values > 100 select better\n");
			printf("                   algorithm, currently used by jpg and pmm\n");
			printf("-l [1-?]         : limit, exit after <limit> pictures.\n");
			printf("-s [URL]         : filename for input picture(s)\n");
			printf("-f [URL]         : filename for output picture(s)\n");
			printf("                   if it ends with '/', a default filename is appended\n");
			printf("                   otherwise use %%04d (or similar) for numbering\n");
			printf("                   currently supported methods: file: (default) and\n");
			printf("                   net: (use server address as name), and\n");
			printf("                   webcam: (like file:, plus some help files)\n");
			printf("-m [sleeptime]   : sleep time in msec between frames, useful for X output\n");
			printf("-d [c | d | q]   : display type (curses/dumb/quiet), default = curses\n");
			printf("-h               : this help screen.\n\n");
			printf("\nMPEG options:\n");
			printf("-x [0-?]         : exclude first n frames, sound is synchronized accordingly\n");
			printf("-Q               : quiet, don't convert sound\n");
			printf("-b [bitrate]     : video bitrate * 1000, default = 1152\n");
			printf("-B [bitrate]     : audio bitrate * 1000, default = 128\n");
			printf("-F [filename]    : audio WAV filename\n");
			printf("-c [filename]    : conversion log, set this if you experience problems\n");
			printf("-r               : remove any used input files, DANGEROUS\n");
			printf("-N [normname]    : video norm, available: PAL, NTSC, SECAM\n");
			exit(0);
		}
	}
	if (!height) {
		inquire("width",tmp);
		sprintf(tmp,"%i",atoi(tmp)*3/4);
		configure("height",tmp);
	}
}
