/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#include <fcntl.h>
#include <curses.h>
#include <sys/types.h>
#include <sys/time.h>
#include "version.h"
#include "main.h"
#include "error.h"
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/videodev.h>

/************ Prototypes ************/

void main_trace_dumb(int n);
void main_trace_quiet(int);
void main_trace_curses(int n);
void main_finish_dumb();
void main_finish_quiet();
void main_finish_curses();
void main_init_curses();
void main_init_dumb();
void main_init_quiet();
void main_update_curses();
void main_update_dumb();
void main_update_quiet();
void main_do_options(int argc, char *argv[], char *envp[]);
void info();

/************************************/

char error_progname[PATH_MAX];

void main_trace_dumb(int n)
{
	static int oldsec, oldusec;
	static double old2fps, oldfps, fps;
	static struct timeval after;
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX];

	if (n == 0) {
		gettimeofday(&after, 0);
		fps = oldfps = 0;
		return;
	}
	if (getprogress(src,dest,conv) == -1) exit(0);
	if (*conv) printf("%s\r",conv);
	else {
		oldsec = after.tv_sec;
		oldusec = after.tv_usec;
		old2fps = oldfps;
		oldfps = fps;
		gettimeofday(&after, 0);
		fps = (1/((after.tv_sec + after.tv_usec/1.0e6) - (oldsec + oldusec/1.0e6)))+0.05;
		printf("%s (%2.01f fps)\n",dest,(fps+oldfps+old2fps)/3);
	}
}

void main_trace_quiet(int n)
{
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX];
	if (getprogress(src,dest,conv) == -1) exit(0);
}

void main_trace_curses(int n)
{
	static int oldsec, oldusec, start, nr, rc;
	static double old2fps, oldfps, fps;
	static struct timeval after;
	static char src[PATH_MAX], dest[PATH_MAX], conv[PATH_MAX];

	if (n == 0) {
		gettimeofday(&after, 0);
		start = after.tv_sec;
		fps = oldfps = 0;
		nr = -1;
		return;
	}
	nr += (rc = getprogress(src,dest,conv));
	if (rc == -1) exit(0);
	if (*conv) {
		attron(A_BOLD);
		mvprintw(17,17,conv);
		attroff(A_BOLD);
	} else {
		int i = -1;
		sscanf(dest,"%*[^0-9]%d",&i);
		attron(A_BOLD);
		mvprintw(17,17,dest);
		attroff(A_BOLD);
		oldsec = after.tv_sec;
		oldusec = after.tv_usec;
		old2fps = oldfps;
		oldfps = fps;
		gettimeofday(&after, 0);
		fps = (1/((after.tv_sec + after.tv_usec/1.0e6) - (oldsec + oldusec/1.0e6)))+0.05;
		mvprintw(6, 14, "%2.01f (%2.01f)   ", (fps+oldfps+old2fps)/3,after.tv_sec==start?0:(double)nr/(double)(after.tv_sec-start));
		mvprintw(5, 45, "%i:%i.%i ",(after.tv_sec-start)/3600,
			 ((after.tv_sec-start)/60)%60,(after.tv_sec-start)%60);
		mvprintw(5, 14, "%i  ",nr);
		mvprintw(6, 45, "%i  ", i-nr+1);
	}

	refresh();
}

void main_finish_dumb()
{
	printf("done.\n");
}

void main_finish_quiet()
{
}

void main_finish_curses()
{
	flushinp();
	refresh();
	endwin();
}

void (*main_finish_func)() = main_finish_curses;

void main_finish()
{
	(*main_finish_func)();
	fprintf(stderr,"%s\n",error_text);
}

void main_init_curses()
{
	char tmp1[PATH_MAX], tmp2[PATH_MAX];

	initscr(); cbreak(); noecho();		// init ncurses
	leaveok(stdscr, TRUE);			// disable cursor
	wresize(stdscr,24,80);			// resize win to 80x24
	refresh();				// flush the screen 
	start_color();				// set collors
	init_pair(1,COLOR_WHITE,COLOR_BLUE);
	init_pair(2,COLOR_RED,COLOR_BLUE);
	init_pair(3,COLOR_CYAN,COLOR_BLUE);
	wbkgd(stdscr, COLOR_PAIR(1));
	wattrset(stdscr, COLOR_PAIR(1));
	attron(A_BOLD);
	newwin(0,0,0,0);
	box(stdscr, 0, 0 );							// 80x24 box
	
	mvprintw(14, 2, " Keys: ");

	inquire("width",tmp1);
	inquire("height",tmp2);
	mvprintw(7,14," %sx%s",tmp1,tmp2);
	
	attroff(A_BOLD);
	mvprintw(14,10,"Q	- Quit");
	mvprintw(15,10,"P	- Pause");
	mvprintw(5, 2, " Image Nr.:");
	mvprintw(7, 2, "Image size:");
	mvprintw(6, 2, "       FPS:");
	mvprintw(9, 2, "    Status:");

	mvprintw(5, 32,"       Time:");
	mvprintw(6, 32,"Frames lost:");
	

	mvprintw(17, 2,"Grabbing into:");

	attron(A_BOLD);
	mvprintw(9, 14, "Working...");		// status

	wattrset(stdscr, COLOR_PAIR(2));
	mvprintw(3, 1, "------------------------------------------------------------------------------");
	mvprintw(11, 1, "------------------------------------------------------------------------------");
	wattrset(stdscr, COLOR_PAIR(3));
	attron(A_BOLD);
	mvprintw(0,2,"< bttvgrab " VERSION " (v4l) by J. Walter/A. Kopacz >");
	mvprintw(23,30,"< http://moes.pmnet.uni-oldenburg.de/bttvgrab/ >");
	wattrset(stdscr, COLOR_PAIR(1));
	
	refresh();
}

void main_init_dumb()
{
	setbuf(stdout,NULL);
	setbuf(stdin,NULL);
}

void main_init_quiet()
{
	setbuf(stdout,NULL);
	setbuf(stdin,NULL);
}

void (*main_init_func)() = main_init_curses;

void main_update_curses()
{
	switch(getch()) { 
	case 'p': case 'P':
		nodelay(stdscr,0);
		attron(A_BOLD);
		mvprintw(9,14,"PAUSED - Push any key (except s/q) to continue");
		attroff(A_BOLD);
		mvprintw(15, 10, "S	- Sync HDDs      ");
		flushinp();
		control(C_PAUSE);
		switch(getch()) {
		case 's': case 'S':
			attron(A_BOLD);
			mvprintw(9,17,"Syncing HDDs...                                ");
			refresh();	      
			sync();
			attron(A_BOLD);
			mvprintw(9,14,"PAUSED - Press any key (except q) to continue");
			attroff(A_BOLD);
			mvprintw(15, 10,"                                    ");
			refresh();
			flushinp();
			switch(getch()) {
			case 'q': case 'Q':
				exit(0);
				break;
				
			default:
				break;
			}
			break;
		case 'q': case 'Q':
			exit(0);
			
		default: 
			break;
			
		}
		main_trace_curses(0);
		control(C_CONT);
		attron(A_BOLD);
		mvprintw(9, 14, "Working...                                    ");
		attroff(A_BOLD);
		mvprintw(15, 10, "P	- Pause    ");
		attron(A_BOLD);
		mvprintw(16, 10,"                                    ");
		nodelay(stdscr,1);
		break;
		
	case 'q': case 'Q':
		control(C_STOP);
		exit(0);
		
	default: 
		break;
	}
}

void main_update_dumb()
{
	switch(getc(stdin)) { 
	case '\n':
		control(C_STOP);
		printf("paused\n");
		if (getc(stdin) == 'q') exit(0);
		control(C_START);
		break;
	case 'q':
		exit(0);
	default:
		break;
	}
}

void main_update_quiet()
{
	main_update_dumb();
}

void (*main_update)() = main_update_curses;
void (*main_trace_func)(int) = main_trace_curses;


/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////                                             ///////////////
/////////////                 Main loop                   ///////////////
/////////////                                             ///////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[], char *envp[])
{
	fd_set fds;

	strcpy(error_progname,argv[0]);

	main_init();

	main_do_options(argc, argv, envp);

	FD_ZERO(&fds);

	(*main_init_func)();
	atexit(main_finish);
	(*main_trace_func)(0);
	control(C_START);
loop:
	FD_SET(0,&fds);
	FD_SET(main_inpipe,&fds);
	select(main_inpipe+1,&fds,NULL,NULL,NULL);

	if (FD_ISSET(main_inpipe,&fds)) (*main_trace_func)(1);
	if (FD_ISSET(0,&fds)) (*main_update)();
	goto loop;
}
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////

#define OPTSET(c,opt,val) case c: configure(#opt,val); break;
#define OPT(c,opt) OPTSET(c,opt,optarg);

void main_do_options(int argc, char *argv[], char *envp[])
{
	int optchar, height = 0;
	char tmp[PATH_MAX], *p;
	configure("width","640");

	while ((optchar = getopt(argc, argv, "N:G:S:k:W:w:q:s:l:f:R:C:B:F:D:o:g:d:tihQy")) != -1) {
		switch (optchar) {
		OPT('w',width);
		case 'W':
			height = 1;
		OPT(1,height);
		OPT('G',grabdevice);
		OPT('k',skip);
		OPT('N',norm);
		OPT('S',channel);
		OPT('q',quality);
		OPT('R',soundrate);
		OPT('C',soundchannels);
		OPT('B',soundbits);
		OPT('s',sleeptime);
		OPT('l',limit);
		OPT('o',grab_file_fmt);
		OPT('g',grablog);
		OPT('F',soundfile);
		OPT('D',sounddevice);
		case 'f':
			p = strchr(optarg,':');
			if (p) {
				*p = 0;
				configure("grab_transport",optarg);
				*p = ':';
				configure("grabname",p+1);
			} else configure("grabname",optarg);
			break;
		OPTSET('t',do_trace,"1");
		OPTSET('y',do_sync,"1");
		OPTSET('Q',do_sound,"0");
		case 'i': info();
		case 'd':
			ASSERT(strlen(optarg) == 1,(ARGPMSG("display type must be one of c,d,q,t")));
			switch (*optarg) {
			case 't': {
				char *command;
				int i, len = 0;
				for (i = 1; i < argc; i++) len += strlen(argv[i]) + 1;
				len += strlen(WISH " " DATADIR "/bttvgrab.tcl -- ");
				command = malloc(len);
				strcpy(command,WISH " " DATADIR "/bttvgrab.tcl --");
				for (i = 1; i < argc; i++) {
					strcat(command," ");
					strcat(command,argv[i]);
				}
				NOTNEG1(execle("/bin/sh","/bin/sh","-c",command,NULL,envp),(ERRMSG("executing "WISH)));
				exit(0);
			}
			case 'c':
				main_init_func = main_init_curses;
				main_trace_func = main_trace_curses;
				main_update = main_update_curses;
				main_finish_func = main_finish_curses;
				break;
			case 'd':
				main_init_func = main_init_dumb;
				main_trace_func = main_trace_dumb;
				main_update = main_update_dumb;
				main_finish_func = main_finish_dumb;
				break;
			case 'q':
				main_init_func = main_init_quiet;
				main_trace_func = main_trace_quiet;
				main_update = main_update_quiet;
				main_finish_func = main_finish_quiet;
				break;
			default:
				ASSERT(0,(ARGPMSG("display type must be one of c,d,q,t")));
				break;
			}
			break;
		case 'h': 
		default:
			printf("bttvgrab " VERSION "\n");
			printf("(c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>\n");
			printf("Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/\n\n");

			printf("Options:\n");
			printf("-w [32-768]      : width, default = 640\n");
			printf("-W [32-576]      : height, default = width*3/4\n");
			printf("-o [format]      : output file format, default = last, available:\n");
			inquire("grab_file_fmt list",tmp);
			printf("%s\n",tmp);
			printf("-t               : trace lost frames by leaving out filenumbers\n");
			printf("-g [filename]    : create a grab logfile, default = disabled\n");
			printf("-G [devicename]  : grab device, default = /dev/bttv\n");
			printf("-S [channelnr]   : source channel number, default=0\n");
			printf("-N [normname]    : video norm, available: PAL, NTSC, SECAM, AUTO (default)\n");
			printf("-q [1-200]       : image quality, default = 75, values > 100 select better\n");
			printf("                   algorithm, currently used by jpg and pmm\n");
			printf("-s [1-?]         : sleeptime, default = 1\n");
			printf("                   sleeping time in seconds\n");
			printf("-k [n]           : frame skip interval, grab a frame after <n> frames\n");
			printf("                   skipped, or skip a frame after -<n> frames grabbed\n");
			printf("-l [1-?]         : limit, exit after <limit> pictures.\n");
			printf("-f [URL]         : filename/URL for output-picture(s)\n");
			printf("                   if it ends with '/', a default filename is appended\n");
			printf("                   otherwise use %%04d (or similar) for numbering\n");
			printf("                   currently supported methods: file: (default),\n");
			printf("                   net: (use server address as name), and\n");
			printf("                   webcam: (like file:, plus some help files)\n");
			printf("-F [filename]    : filename for output wav, default is ./grab.wav\n");
			printf("                   if it ends with '/', a default filename is appended\n");
			printf("-D [devicename]  : device for recording, default is /dev/dsp\n");
			printf("-Q               : quiet, do not record sound.\n");
			printf("-R [8000-48000]  : sampling rate, default = 44100\n");
			printf("-C [1-4]         : sound channels, default = 2(stereo)\n");
			printf("-B [8 | 16]      : sound bits, default = 16\n");
			printf("-d [c|d|q|t]     : display type (curses/dumb/quiet/TclTk), default = curses\n");
			printf("-y               : synchronous output, use this if you have large\n");
			printf("                 : video sequences to grab, needs *fast* hard disk\n");
			printf("-i               : shows some v4l informations.\n");
			printf("-h               : this help screen.\n\n");
			exit(0);
		}
	}
	if (!height) {
		inquire("width",tmp);
		sprintf(tmp,"%i",atoi(tmp)*3/4);
		configure("height",tmp);
	}
}

void info()
{
	struct video_capability vcapability;
	struct video_tuner vtuner;
	struct video_buffer vbuffer;
	struct video_channel vchan;
	int i, fbttv;
	const char *device = "/dev/bttv";
	char *device_cap[] = {
		"capture", "tuner", "teletext", "overlay", "chromakey", "clipping",
		"frameram", "scales", "monochrome", NULL };

	printf("bttvgrab "    VERSION    " - Video4Linux Informations\n");
	printf("*****************************************************\n\n");

	SAFE_OPEN(fbttv,device,O_RDWR);

	NOTNEG1(ioctl(fbttv,VIDIOCGCAP,&vcapability),(ERRMSG("VIDIOCGCAP")));
	
	NOTNEG1(ioctl(fbttv,VIDIOCGFBUF,&vbuffer),(ERRMSG("VIDIOCGFBUF")));
	
	printf("card_name     : %s\n",vcapability.name);
	printf("flags         :");
	for (i = 0; device_cap[i] != NULL; i++) {
		if (vcapability.type & (1 << i)) printf(" %s",device_cap[i]);
	}
	
	printf("\nchannels      : %d (",vcapability.channels);
	for (i = 0; i < vcapability.channels; i++) {
		vchan.channel = i;
		if (ioctl(fbttv,VIDIOCGCHAN,&vchan) != -1) printf(" %s",vchan.name);
	}
	printf(" )\n");
	printf("audio devices : %d\n",vcapability.audios);
	printf("size (max/min): %d*%d/%d*%d\n",vcapability.maxwidth,vcapability.maxheight,vcapability.minwidth,vcapability.minheight);

	vtuner.tuner = 0;
	if (ioctl(fbttv,VIDIOCGTUNER,&vtuner) != -1) {
		printf("tuner         : %s (Freq.: %lu-%lu)\n",vtuner.name,vtuner.rangelow,vtuner.rangehigh);
	}

	printf("grab_buffer   : base=%p size=%dx%d depth=%d bpl=%d\n",vbuffer.base,vbuffer.width,vbuffer.height,vbuffer.depth,vbuffer.bytesperline);
	printf("\n");
	exit(0);
}
