/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <values.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include "convert.h"
#include "error.h"
#include "configuration.h"
#define SYMBOL(x) convert_ ## x

int convert_inwidth = 0;
int convert_inheight = 0;
int convert_outwidth = 0;
int convert_outheight = 0;
int convert_infmt = 0;
int convert_outfmt = 0;
int convert_bufsize = 0;
int convert_inbytes = 0;
int convert_inpal = 0;
int convert_outpal = 0;
int convert_in_altcol = 0;
int convert_out_altcol = 0;
int convert_bytes = 0;

unsigned char **convert_outbuf = NULL, **convert_inbuf = NULL;

int convert_fbs = 0;
int convert_5to8tab[32];
int convert_6to8tab[64];
int convert_8to5tab[256];
int convert_8to6tab[256];
int convert_256to5tab[256];
int convert_256to9tab[256];
int convert_32to5tab[32];
int convert_64to9tab[64];
int convert_palette[256][3];
int convert_palette16[256][3];
int convert_paletteYCC[256][3];
int convert_paletteG[256];
int convert_backpal[5][9][5];

// YCC<->RGB code taken from IJG's jpeglib-v6b
// Copyright (C) 1991-1996, Thomas G. Lane.

#define NO_C            -128
#define CBCR_OFFSET     ((int) 128 << 16)
#define ONE_HALF        ((int) 1 << 15)
#define FIX(x)          ((int) ((x) * (double)(1L<<16) + 0.5))
#define R_Y_OFF         0                       /* offset to R => Y section */
#define G_Y_OFF         (1*256)      /* offset to G => Y section */
#define B_Y_OFF         (2*256)      /* etc. */
#define R_CB_OFF        (3*256)
#define G_CB_OFF        (4*256)
#define B_CB_OFF        (5*256)
#define R_CR_OFF        B_CB_OFF                /* B=>Cb, R=>Cr are the same */
#define G_CR_OFF        (6*256)
#define B_CR_OFF        (7*256)
#define TABLE_SIZE      (8*256)
int convert_rgb_ycc_tab[TABLE_SIZE];
int convert_Crrtab[256];
int convert_Cbbtab[256];
int convert_Crgtab[256];
int convert_Cbgtab[256];
unsigned char convert_range[5*256+128];
unsigned char *convert_rangetable = NULL;

typedef void (*convert_func)(int fbnr);

convert_func convert_doit = NULL;

/************ Prototypes ************/

void convert_init24(int in);
void convert_nil24(int fbnr);
void convert_24to8(int fbnr);
void convert_24to16(int fbnr);
void convert_24to422(int fbnr);
void convert_24to420(int fbnr);
void convert_24toG(int fbnr);
void convert_init8(int in);
void convert_nil8(int fbnr);
void convert_8to24(int fbnr);
void convert_8to16(int fbnr);
void convert_8to422(int fbnr);
void convert_8to420(int fbnr);
void convert_8toG(int fbnr);
void convert_init16(int in);
void convert_nil16(int fbnr);
void convert_16to24(int fbnr);
void convert_16to8(int fbnr);
void convert_16to422(int fbnr);
void convert_16to420(int fbnr);
void convert_16toG(int fbnr);
void convert_init422(int in);
void convert_nil422(int fbnr);
void convert_422to24(int fbnr);
void convert_422to8(int fbnr);
void convert_422to16(int fbnr);
void convert_422to420(int fbnr);
void convert_422toG(int fbnr);
void convert_init420(int in);
void convert_nil420(int fbnr);
void convert_420to24(int fbnr);
void convert_420to8(int fbnr);
void convert_420to16(int fbnr);
void convert_420to422(int fbnr);
void convert_420toG(int fbnr);
void convert_initG(int in);
void convert_nilG(int fbnr);
void convert_Gto24(int fbnr);
void convert_Gto16(int fbnr);
void convert_Gto422(int fbnr);
void convert_Gto420(int fbnr);
void convert_Gto8(int fbnr);
void convert_nil(int fbnr);
void convert_init();
void convert_start();
void convert_stop();
void convert_end();
int convert_convert(int fbnr);

/************************************/

#define CONVERT_COLORS 6
const int convert_color_bpps[CONVERT_COLORS] = {24,8,16,16,12,8};
const char *convert_color_names[CONVERT_COLORS] = {"RGB24","PAL8","RGB16","YUV422","YUV420","GREY8"};


START_OPTIONS
INTOPTION(inwidth,0,MAXINT);
INTOPTION(inheight,0,MAXINT);
INTOPTION(outwidth,0,MAXINT);
INTOPTION(outheight,0,MAXINT);
OPTION(infmt) {
	OPTIONSET(int i;
		  if (value[strlen(value)-1] == 'P') {
			  convert_inpal = 1;
			  value[strlen(value)-1] = 0;
		  } else convert_inpal = 0;

		  if (value[strlen(value)-1] == 'S') {
			  convert_in_altcol = 1;
			  value[strlen(value)-1] = 0;
		  } else convert_in_altcol = 0;

		  for (i = 0; i < CONVERT_COLORS; i++) {
			  if (!strcmp(convert_color_names[i],value)) {
				  convert_infmt = i;
				  break;
			  }
		  }
		  ASSERT(i < CONVERT_COLORS,(MSG("unknown color format '%s'"),value)););
	OPTIONGET(strcpy(value,convert_color_names[convert_infmt]););
}
OPTION(outfmt) {
	OPTIONSET(int i;
		  if (value[strlen(value)-1] == 'P') {
			  convert_outpal = 1;
			  value[strlen(value)-1] = 0;
		  } else convert_outpal = 0;

		  if (value[strlen(value)-1] == 'S') {
			  convert_out_altcol = 1;
			  value[strlen(value)-1] = 0;
		  } else convert_out_altcol = 0;

		  for (i = 0; i < CONVERT_COLORS; i++) {
			  if (!strcmp(convert_color_names[i],value)) {
				  convert_outfmt = i;
				  break;
			  }
		  }
		  ASSERT(i < CONVERT_COLORS,(MSG("unknown color format '%s'"),value)););
	OPTIONGET(strcpy(value,convert_color_names[convert_outfmt]););
}
INTOPTION(fbs,0,MAXINT) {
	OPTIONSET(SAFE_FREE(convert_inbuf);
		  SAFE_CALLOC(convert_inbuf,convert_fbs,sizeof(void *));
		  SAFE_FREE(convert_outbuf);
		  SAFE_CALLOC(convert_outbuf,convert_fbs,sizeof(void *)););
}
PTROPTION(inbuf,convert_fbs);
PTRINFO(outbuf,convert_fbs);
END_OPTIONS

/*********** storage macros ***********/
#define GET_BGR24(r,g,b,ptr) b = *(ptr++); g = *(ptr++); r = *(ptr++);
#define PUT_BGR24(r,g,b,ptr) *(ptr++) = b; *(ptr++) = g; *(ptr++) = r;
#define GET_RGB24(r,g,b,ptr) r = *(ptr++); g = *(ptr++); b = *(ptr++);
#define PUT_RGB24(r,g,b,ptr) *(ptr++) = r; *(ptr++) = g; *(ptr++) = b;
#define GET_8BIT(x,ptr) x = *(ptr++);
#define PUT_8BIT(x,ptr) *(ptr++) = x;
#define GET_RGB16(r,g,b,ptr) r = (*((unsigned short *)ptr) >> 11) & 0x1f; g = (*((unsigned short *)ptr) >> 5) & 0x3f; b = *(((unsigned short *)ptr)++) & 0x1f;
#define PUT_RGB16(r,g,b,ptr) *(((unsigned short *)ptr)++) = ((r & 0x1f) << 11) | ((g & 0x3f) << 5) | (b & 0x1f);
#define GET_YUV420(y0,y1,y2,y3,Cb,Cr,ptr1,ptr2) y0=*(ptr1++); Cr=*(ptr1++); y1=*(ptr1++); Cb=*(ptr1++); y2=*(ptr2++); y3=*(ptr2++);
#define PUT_YUV420(y0,y1,y2,y3,Cb,Cr,ptr1,ptr2) *(ptr1++)=y0; *(ptr1++)=Cr; *(ptr1++)=y1; *(ptr1++)=Cb; *(ptr2++)=y2; *(ptr2++)=y3;
#define GET_YUV420PAL(y0,y1,y2,y3,Cb,Cr,pY0,pY1,pCb,pCr) y0=*(pY0++); Cr=*(pCr++); y1=*(pY0++); Cb=*(pCb++); y2=*(pY1++); y3=*(pY1++);
#define PUT_YUV420PAL(y0,y1,y2,y3,Cb,Cr,pY0,pY1,pCb,pCr) *(pY0++)=y0; *(pCr++)=Cr; *(pY0++)=y1; *(pCb++)=Cb; *(pY1++)=y2; *(pY1++)=y3;
#define GET_YUV422S(y0,y1,Cb,Cr,ptr) y0 = *(ptr++); Cb = *(ptr++); y1 = *(ptr++); Cr = *(ptr++);
#define PUT_YUV422S(y0,y1,Cb,Cr,ptr) *(ptr++) = y0; *(ptr++) = Cb; *(ptr++) = y1; *(ptr++) = Cr;
#define GET_YUV422SP(y0,y1,Cb,Cr,ptrY,ptrCb,ptrCr) y0 = *(ptrY++); Cb = *(ptrCr++); y1 = *(ptrY++); Cr = *(ptrCb++);
#define PUT_YUV422SP(y0,y1,Cb,Cr,ptrY,ptrCb,ptrCr) *(ptrY++) = y0; *(ptrCr++) = Cb; *(ptrY++) = y1; *(ptrCb++) = Cr;
/**************************************/

/************ conversion macros ***************/
#define RGB242PAL8(r,g,b,c) c = convert_backpal[convert_256to5tab[r]][convert_256to9tab[g]][convert_256to5tab[b]];
#define RGB242RGB16(ri,gi,bi,ro,go,bo) ro = convert_8to5tab[ri]; go = convert_8to6tab[gi]; bo = convert_8to5tab[bi];
#define RGB242YUV(r,g,b,y,cb,cr) \
	y = (char)((convert_rgb_ycc_tab[r+R_Y_OFF] + convert_rgb_ycc_tab[g+G_Y_OFF] + convert_rgb_ycc_tab[b+B_Y_OFF]) >> 16); \
	cb= (char)((convert_rgb_ycc_tab[r+R_CB_OFF] + convert_rgb_ycc_tab[g+G_CB_OFF] + convert_rgb_ycc_tab[b+B_CB_OFF]) >> 16); \
	cr= (char)((convert_rgb_ycc_tab[r+R_CR_OFF] + convert_rgb_ycc_tab[g+G_CR_OFF] + convert_rgb_ycc_tab[b+B_CR_OFF]) >> 16);
#define RGB242GREY8(r,g,b,l) \
	l = (char)((convert_rgb_ycc_tab[r+R_Y_OFF] + convert_rgb_ycc_tab[g+G_Y_OFF] + convert_rgb_ycc_tab[b+B_Y_OFF]) >> 16);

#define PAL82RGB24(c,r,g,b) r = convert_palette[(int)c][0]; g = convert_palette[(int)c][1]; b = convert_palette[(int)c][2];
#define PAL82RGB16(c,r,g,b) r = convert_palette16[(int)c][0]; g = convert_palette16[(int)c][1]; b = convert_palette16[(int)c][2];
#define PAL82YUV(c,y,cb,cr) y = convert_paletteYCC[(int)c][0]; cb = convert_paletteYCC[(int)c][1]; cr = convert_paletteYCC[(int)c][2];
#define PAL82GREY8(c,l) l = convert_paletteG[(int)c];

#define RGB162RGB24(ri,gi,bi,ro,go,bo) ro = convert_5to8tab[ri]; go = convert_6to8tab[gi]; bo = convert_5to8tab[bi];
#define RGB162PAL8(r,g,b,c) c = convert_backpal[convert_32to5tab[r]][convert_64to9tab[g]][convert_32to5tab[b]];
#define RGB162YUV(r,g,b,y,cb,cr) RGB162RGB24(r,g,b,r,g,b) RGB242YUV(r,g,b,y,cb,cr)
#define RGB162GREY8(r,g,b,l) RGB162RGB24(r,g,b,r,g,b) RGB242GREY8(r,g,b,l)

#define YUV2RGB24(y,cb,cr,r,g,b) \
	r = convert_rangetable[y + convert_Crrtab[cr]]; \
	g = convert_rangetable[y + ((int) (convert_Cbgtab[cb] + convert_Crgtab[cr]) >> 16)]; \
	b = convert_rangetable[y + convert_Cbbtab[cb]];
#define YUV2RGB16(y,cb,cr,r,g,b) YUV2RGB24(y,cb,cr,r,g,b) RGB242RGB16(r,g,b,r,g,b)
#define YUV2PAL8(y,cb,cr,c) c = convert_backpal[convert_256to5tab[convert_rangetable[y + convert_Crrtab[cr]]]] \
						[convert_256to9tab[convert_rangetable[y + ((int) (convert_Cbgtab[cb] + convert_Crgtab[cr]) >> 16)]]] \
						[convert_256to5tab[convert_rangetable[y + convert_Cbbtab[cb]]]]
#define YUV2GREY8(y,cb,cr,l) l = y;

#define GREY82RGB24(l,r,g,b) r = g = b = l;
#define GREY82PAL8(l,c) RGB242PAL8(l,l,l,c)
#define GREY82RGB16(l,r,g,b) RGB242RGB16(l,l,l,r,g,b)
#define GREY82YUV(l,y,cb,cr) RGB242YUV(l,l,l,y,cb,cr)
/**********************************************/

/******************* scaling macros *******************/
#define INTERPOL2(x1,x2) x1
#define INTERPOL4(x1,x2,x3,x4) x1
#define addINTERPOLn(x,x2,n) do {x += x2; n++} while (0)
#define INTERPOLn(x,n) (x/n)
/******************************************************/

#define ADD(symbol) convert_ ## symbol

convert_func convert_inittable[CONVERT_COLORS] = {
	ADD(init24),
	ADD(init8),
	ADD(init16),
	ADD(init422),
	ADD(init420),
	ADD(initG)
};

convert_func convert_table[CONVERT_COLORS][CONVERT_COLORS] = {
/*BGR24 -> x */	{ ADD(nil24),   ADD(24to8),  ADD(24to16),  ADD(24to422),  ADD(24to420),  ADD(24toG)  },
/*PAL8  -> x */ { ADD(8to24),   ADD(nil8),   ADD(8to16),   ADD(8to422),   ADD(8to420),   ADD(8toG)   },
/*RGB16 -> x */ { ADD(16to24),  ADD(16to8),  ADD(nil16),   ADD(16to422),  ADD(16to420),  ADD(16toG)  },
/*422   -> x */ { ADD(422to24), ADD(422to8), ADD(422to16), ADD(nil422),   ADD(422to420), ADD(422toG) },
/*420   -> x */ { ADD(420to24), ADD(420to8), ADD(420to16), ADD(420to422), ADD(nil420),   ADD(420toG) },
/*GREY  -> x */ { ADD(Gto24),   ADD(Gto8),   ADD(Gto16),   ADD(Gto422),   ADD(Gto420),   ADD(nilG)   }
};

////////////////////////// 24bit
void convert_init24(int in)
{
	ASSERT((in && !convert_inpal) || (!in && !convert_outpal),(INTMSG("palettized in-/output not supported for RGB24")));
	if (convert_infmt == convert_outfmt && convert_in_altcol == convert_out_altcol) convert_doit = convert_nil;
}
void convert_nil24(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		if (convert_in_altcol) {
			GET_BGR24(r,g,b,ptr);
		} else {
			GET_RGB24(r,g,b,ptr);
		}
		if (convert_out_altcol) {
			PUT_BGR24(r,g,b,ptro);
		} else {
			PUT_RGB24(r,g,b,ptro);
		}
	}
}

void convert_24to8(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		if (convert_in_altcol) {
			GET_BGR24(r,g,b,ptr);
		} else {
			GET_RGB24(r,g,b,ptr);
		}
		RGB242PAL8(r,g,b,r);
		PUT_8BIT(r,ptro);
	}
}

void convert_24to16(int fbnr)
{
	unsigned int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		if (convert_in_altcol) {
			GET_BGR24(r,g,b,ptr);
		} else {
			GET_RGB24(r,g,b,ptr);
		}
		RGB242RGB16(r,g,b,r,g,b);
		PUT_RGB16(r,g,b,ptro);
	}
}

void convert_24to422(int fbnr)
{
	int r,g,b, y1,cb1,cr1, y2,cb2,cr2;
	unsigned char *ptr = convert_inbuf[fbnr];
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_in_altcol) {
			GET_BGR24(r,g,b,ptr);
			RGB242YUV(r,g,b,y1,cb1,cr1);
			GET_BGR24(r,g,b,ptr);
			RGB242YUV(r,g,b,y2,cb2,cr2);
		} else {
			GET_RGB24(r,g,b,ptr);
			RGB242YUV(r,g,b,y1,cb1,cr1);
			GET_RGB24(r,g,b,ptr);
			RGB242YUV(r,g,b,y2,cb2,cr2);
		}
		if (convert_outpal) {
			PUT_YUV422SP(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),
				      ptro0,ptro1,ptro2);
		} else {
			PUT_YUV422S(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),ptro0);
		}
	}
}

void convert_24to420(int fbnr)
{
	int r,g,b;
	int y1,cb1,cr1, y2,cb2,cr2, y3,cb3,cr3, y4,cb4,cr4;
	unsigned char *ptr0, *ptr1 = convert_inbuf[fbnr];
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptro0 = ptro01;
		if (!convert_outpal) ptro01 += convert_outwidth*2;
		else ptro01 += convert_outwidth;

		ptr0 = ptr1;
		ptr1 += (convert_inwidth*3);
		w = convert_inwidth/2;

		while (w--) {
			if (convert_in_altcol) {
				GET_BGR24(r,g,b,ptr0);
				RGB242YUV(r,g,b,y1,cb1,cr1);
				GET_BGR24(r,g,b,ptr0);
				RGB242YUV(r,g,b,y2,cb2,cr2);
				GET_BGR24(r,g,b,ptr1);
				RGB242YUV(r,g,b,y3,cb3,cr3);
				GET_BGR24(r,g,b,ptr1);
				RGB242YUV(r,g,b,y4,cb4,cr4);
			} else {
				GET_RGB24(r,g,b,ptr0);
				RGB242YUV(r,g,b,y1,cb1,cr1);
				GET_RGB24(r,g,b,ptr0);
				RGB242YUV(r,g,b,y2,cb2,cr2);
				GET_RGB24(r,g,b,ptr1);
				RGB242YUV(r,g,b,y3,cb3,cr3);
				GET_RGB24(r,g,b,ptr1);
				RGB242YUV(r,g,b,y4,cb4,cr4);
			}

			if (convert_outpal) {
				PUT_YUV420PAL(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					      ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					   ptro0,ptro01);
			}
		}
	}
}

void convert_24toG(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		if (convert_in_altcol) {
			GET_BGR24(r,g,b,ptr);
		} else {
			GET_RGB24(r,g,b,ptr);
		}
		RGB242GREY8(r,g,b,r);
		PUT_8BIT(r,ptro);
	}
}

/////////////////////////// 8 bit
void convert_init8(int in)
{
	ASSERT((in && !convert_inpal) || (!in && !convert_outpal),(INTMSG("palettized in-/output not supported for RGB8")));
	ASSERT((in && !convert_in_altcol) || (!in && !convert_out_altcol),(INTMSG("color swapped in-/output not supported for RGB8")));
	if (convert_infmt == convert_outfmt) convert_doit = convert_nil;
}

void convert_nil8(int fbnr)
{
	ERROR(INTMSG("convert_nil8 called"));
}

void convert_8to24(int fbnr)
{
	int c,r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(c,ptr);
		PAL82RGB24(c,r,g,b);
		if (convert_out_altcol) {
			PUT_BGR24(r,g,b,ptro);
		} else {
			PUT_RGB24(r,g,b,ptro);
		}
	}
}

void convert_8to16(int fbnr)
{
	int c,r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(c,ptr);
		PAL82RGB16(c,r,g,b);
		PUT_RGB16(r,g,b,ptro);
	}
}

void convert_8to422(int fbnr)
{
	unsigned int c, y1,cb1,cr1, y2,cb2,cr2;
	unsigned char *ptr = convert_inbuf[fbnr];
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		GET_8BIT(c,ptr);
		PAL82YUV(c,y1,cb1,cr1);
		GET_8BIT(c,ptr);
		PAL82YUV(c,y2,cb2,cr2);
		if (convert_outpal) {
			PUT_YUV422SP(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),
				      ptro0,ptro1,ptro2);
		} else {
			PUT_YUV422S(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),ptro0);
		}
	}
}

void convert_8to420(int fbnr)
{
	int c;
	int y1,cb1,cr1, y2,cb2,cr2, y3,cb3,cr3, y4,cb4,cr4;
	unsigned char *ptr0, *ptr1 = convert_inbuf[fbnr];
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptro0 = ptro01;
		if (!convert_outpal) ptro01 += convert_outwidth*2;
		else ptro01 += convert_outwidth;

		ptr0 = ptr1;
		ptr1 += convert_inwidth;
		w = convert_inwidth/2;

		while (w--) {
			GET_8BIT(c,ptr0);
			PAL82YUV(c,y1,cb1,cr1);
			GET_8BIT(c,ptr0);
			PAL82YUV(c,y2,cb2,cr2);
			GET_8BIT(c,ptr1);
			PAL82YUV(c,y3,cb3,cr3);
			GET_8BIT(c,ptr1);
			PAL82YUV(c,y4,cb4,cr4);
			if (convert_outpal) {
				PUT_YUV420PAL(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					      ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					   ptro0,ptro01);
			}
		}
	}
}

void convert_8toG(int fbnr)
{
	int c;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(c,ptr);
		PAL82GREY8(c,c);
		PUT_8BIT(c,ptro);
	}
}

/////////////////////////// 16 bit
void convert_init16(int in)
{
	ASSERT((in && !convert_inpal) || (!in && !convert_outpal),(INTMSG("palettized in-/output not supported for RGB16")));
	ASSERT((in && !convert_in_altcol) || (!in && !convert_out_altcol),(INTMSG("color swapped in-/output not supported for RGB16")));
	if (convert_infmt == convert_outfmt) convert_doit = convert_nil;
}

void convert_nil16(int fbnr)
{
	ERROR(INTMSG("convert_nil16 called"));
}

void convert_16to24(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_RGB16(r,g,b,ptr);
		RGB162RGB24(r,g,b,r,g,b);
		if (convert_out_altcol) {
			PUT_BGR24(r,g,b,ptro);
		} else {
			PUT_RGB24(r,g,b,ptro);
		}
	}
}

void convert_16to8(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_RGB16(r,g,b,ptr);
		RGB162PAL8(r,g,b,b);
		PUT_8BIT(b,ptro);
	}
}

void convert_16to422(int fbnr)
{
	int r,g,b, y1,cb1,cr1, y2,cb2,cr2;
	unsigned char *ptr = convert_inbuf[fbnr];
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		GET_RGB16(r,g,b,ptr);
		RGB162YUV(r,g,b,y1,cb1,cr1);
		GET_RGB16(r,g,b,ptr);
		RGB162YUV(r,g,b,y2,cb2,cr2);
		if (convert_outpal) {
			PUT_YUV422SP(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),
				      ptro0,ptro1,ptro2);
		} else {
			PUT_YUV422S(y1,y2,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),ptro0);
		}
	}
}

void convert_16to420(int fbnr)
{
	int r,g,b;
	int y1,cb1,cr1, y2,cb2,cr2, y3,cb3,cr3, y4,cb4,cr4;
	unsigned char *ptr0, *ptr1 = convert_inbuf[fbnr];
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptro0 = ptro01;
		if (!convert_outpal) ptro01 += convert_outwidth*2;
		else ptro01 += convert_outwidth;

		ptr0 = ptr1;
		ptr1 += (convert_inwidth*2);
		w = convert_inwidth/2;

		while (w--) {
			GET_RGB16(r,g,b,ptr0);
			RGB162YUV(r,g,b,y1,cb1,cr1);
			GET_RGB16(r,g,b,ptr0);
			RGB162YUV(r,g,b,y2,cb2,cr2);
			GET_RGB16(r,g,b,ptr1);
			RGB162YUV(r,g,b,y3,cb3,cr3);
			GET_RGB16(r,g,b,ptr1);
			RGB162YUV(r,g,b,y4,cb4,cr4);

			if (convert_outpal) {
				PUT_YUV420PAL(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					      ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y1,y2,y3,y4,INTERPOL4(cb1,cb2,cb3,cb4),INTERPOL4(cr1,cr2,cr3,cr4),
					   ptro0,ptro01);
			}
		}
	}
}

void convert_16toG(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_RGB16(r,g,b,ptr);
		RGB162GREY8(r,g,b,b);
		PUT_8BIT(b,ptro);
	}
}

/////////////////////////// YUV422
void convert_init422(int in)
{
	ASSERT((in && convert_in_altcol) || (!in  && convert_out_altcol),(INTMSG("only color swapped in-/output supported for YUV422")));
	if (convert_infmt == convert_outfmt && convert_inpal == convert_outpal) convert_doit = convert_nil;
}

void convert_nil422(int fbnr)
{
	int y0, y1, cb, cr;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_inpal) {
			GET_YUV422SP(y0,y1,cb,cr,ptr0,ptr1,ptr2);
		} else {
			GET_YUV422S(y0,y1,cb,cr,ptr0);
		}
		if (convert_outpal) {
			PUT_YUV422SP(y0,y1,cb,cr,ptro0,ptro1,ptro2);
		} else {
			PUT_YUV422S(y0,y1,cb,cr,ptro0);
		}
	}
}

void convert_422to24(int fbnr)
{
	int r,g,b, y0, y1, cb, cr;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_inpal) {
			GET_YUV422SP(y0,y1,cb,cr,ptr0,ptr1,ptr2);
		} else {
			GET_YUV422S(y0,y1,cb,cr,ptr0);
		}
		if (convert_out_altcol) {
			YUV2RGB24(y0,cb,cr,r,g,b);
			PUT_BGR24(r,g,b,ptro);
			YUV2RGB24(y1,cb,cr,r,g,b);
			PUT_BGR24(r,g,b,ptro);
		} else {
			YUV2RGB24(y0,cb,cr,r,g,b);
			PUT_BGR24(r,g,b,ptro);
			YUV2RGB24(y1,cb,cr,r,g,b);
			PUT_RGB24(r,g,b,ptro);
		}
	}
}

void convert_422to8(int fbnr)
{
	int c, y0, y1, cb, cr;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_inpal) {
			GET_YUV422SP(y0,y1,cb,cr,ptr0,ptr1,ptr2);
		} else {
			GET_YUV422S(y0,y1,cb,cr,ptr0);
		}
		YUV2PAL8(y0,cb,cr,c);
		PUT_8BIT(c,ptro);
		YUV2PAL8(y1,cb,cr,c);
		PUT_8BIT(c,ptro);
	}
}

void convert_422to16(int fbnr)
{
	int r,g,b, y0, y1, cb, cr;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_inpal) {
			GET_YUV422SP(y0,y1,cb,cr,ptr0,ptr1,ptr2);
		} else {
			GET_YUV422S(y0,y1,cb,cr,ptr0);
		}
		YUV2RGB16(y0,cb,cr,r,g,b);
		PUT_RGB16(r,g,b,ptro);
		YUV2RGB16(y1,cb,cr,r,g,b);
		PUT_RGB16(r,g,b,ptro);
	}
}

void convert_422to420(int fbnr)
{
	int y0,y1,cb1,cr1;
	int y2,y3,cb2,cr2;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptro0 = ptro01;
		if (!convert_outpal) ptro1 += convert_outwidth*2;
		else ptro01 += convert_outwidth;

		if (!convert_inpal) {
			ptr1 = ptr0;
			ptr0 += convert_inwidth*2;
		}
		w = convert_inwidth/2;

		while (w--) {
			if (convert_inpal) {
				GET_YUV422SP(y0,y1,cb1,cr1,ptr0,ptr1,ptr2);
				GET_YUV422SP(y2,y3,cb2,cr2,ptr0,ptr1,ptr2);
			} else {
				GET_YUV422S(y0,y1,cb1,cr1,ptr1);
				GET_YUV422S(y2,y3,cb2,cr2,ptr0);
			}
			if (convert_outpal) {
				PUT_YUV420PAL(y0,y1,y2,y3,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),
					      ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y0,y1,y2,y3,INTERPOL2(cb1,cb2),INTERPOL2(cr1,cr2),
					   ptro0,ptro01);
			}
		}
	}
}

void convert_422toG(int fbnr)
{
	int y0,y1,cb,cr;
	unsigned char *ptr0 = convert_inbuf[fbnr], *ptr1 = ptr0+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*convert_inheight;
	unsigned char *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		if (convert_inpal) {
			GET_YUV422SP(y0,y1,cb,cr,ptr0,ptr1,ptr2);
		} else {
			GET_YUV422S(y0,y1,cb,cr,ptr0);
		}
		PUT_8BIT(y0,ptro);
		PUT_8BIT(y1,ptro);
	}
}

/////////////////////////// YUV420
void convert_init420(int in)
{
	ASSERT((in && !convert_in_altcol) || (!in  && !convert_out_altcol),(INTMSG("color swapped in-/output not supported for YUV420")));
	if (convert_infmt == convert_outfmt && convert_inpal == convert_outpal) convert_doit = convert_nil;
}

void convert_nil420(int fbnr)
{
	int y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		ptro0 = ptro1;
		ptro1 += convert_inwidth*3;

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}
			if (convert_outpal) {
				PUT_YUV420PAL(y0,y1,y2,y3,cb,cr,ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y0,y1,y2,y3,cb,cr,ptro0,ptro01);
			}
		}
	}
}

void convert_420to24(int fbnr)
{
	int r,g,b, y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro1 = convert_outbuf[0], *ptro0;
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		ptro0 = ptro1;
		ptro1 += convert_inwidth*3;

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}

			if (convert_out_altcol) {
				YUV2RGB24(y0,cb,cr,r,g,b);
				PUT_BGR24(r,g,b,ptro0);
				YUV2RGB24(y1,cb,cr,r,g,b);
				PUT_BGR24(r,g,b,ptro0);
				YUV2RGB24(y2,cb,cr,r,g,b);
				PUT_BGR24(r,g,b,ptro1);
				YUV2RGB24(y3,cb,cr,r,g,b);
				PUT_BGR24(r,g,b,ptro1);
			} else {
				YUV2RGB24(y0,cb,cr,r,g,b);
				PUT_RGB24(r,g,b,ptro0);
				YUV2RGB24(y1,cb,cr,r,g,b);
				PUT_RGB24(r,g,b,ptro0);
				YUV2RGB24(y2,cb,cr,r,g,b);
				PUT_RGB24(r,g,b,ptro1);
				YUV2RGB24(y3,cb,cr,r,g,b);
				PUT_RGB24(r,g,b,ptro1);
			}
		}
	}
}

void convert_420to8(int fbnr)
{
	int c, y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro1 = convert_outbuf[0], *ptro0;
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		ptro0 = ptro1;
		ptro1 += convert_inwidth;

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}
			YUV2PAL8(y0,cb,cr,c);
			PUT_8BIT(c,ptro0);
			YUV2PAL8(y1,cb,cr,c);
			PUT_8BIT(c,ptro0);
			YUV2PAL8(y2,cb,cr,c);
			PUT_8BIT(c,ptro1);
			YUV2PAL8(y3,cb,cr,c);
			PUT_8BIT(c,ptro1);
		}
	}
}

void convert_420to16(int fbnr)
{
	int r,g,b, y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro1 = convert_outbuf[0], *ptro0;
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		ptro0 = ptro1;
		ptro1 += convert_inwidth*2;

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}
			YUV2RGB16(y0,cb,cr,r,g,b);
			PUT_RGB16(r,g,b,ptro0);
			YUV2RGB16(y1,cb,cr,r,g,b);
			PUT_RGB16(r,g,b,ptro0);
			YUV2RGB16(y2,cb,cr,r,g,b);
			PUT_RGB16(r,g,b,ptro1);
			YUV2RGB16(y3,cb,cr,r,g,b);
			PUT_RGB16(r,g,b,ptro1);
		}
	}
}

void convert_420to422(int fbnr)
{
	int y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		if (!convert_outpal) {
			ptro1 = ptro0;
			ptro0 += convert_inwidth*2;
		}

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}
			if (convert_outpal) {
				PUT_YUV422SP(y0,y1,cb,cr,ptro0,ptro1,ptro2);
				PUT_YUV422SP(y2,y3,cb,cr,ptro0,ptro1,ptro2);
			} else {
				PUT_YUV422S(y0,y1,cb,cr,ptro1);
				PUT_YUV422S(y2,y3,cb,cr,ptro0);
			}
		}
	}
}

void convert_420toG(int fbnr)
{
	int y0, y1, y2, y3, cb, cr;
	unsigned char *ptr0, *ptr01 = convert_inbuf[fbnr], *ptr1 = ptr01+convert_inwidth*convert_inheight, *ptr2 = ptr1+convert_inwidth/2*((convert_inheight+1)/2);
	unsigned char *ptro1 = convert_outbuf[0], *ptro0;
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptr0 = ptr01;
		if (!convert_inpal) ptr01 += convert_inwidth*2;
		else ptr01 += convert_inwidth;

		ptro0 = ptro1;
		ptro1 += convert_inwidth;

		w = convert_inwidth/2;
		while (w--) {
			if (convert_inpal) {
				GET_YUV420PAL(y0,y1,y2,y3,cb,cr,ptr0,ptr01,ptr1,ptr2);
			} else {
				GET_YUV420(y0,y1,y2,y3,cb,cr,ptr0,ptr01);
			}
			PUT_8BIT(y0,ptro0);
			PUT_8BIT(y1,ptro0);
			PUT_8BIT(y2,ptro1);
			PUT_8BIT(y3,ptro1);
		}
	}
}

/////////////////////////// grey
void convert_initG(int in)
{
	ASSERT((in && !convert_inpal) || (!in && !convert_outpal),(INTMSG("palettized in-/output not supported for GREY8")));
	ASSERT((in && !convert_in_altcol) || (!in && !convert_out_altcol),(INTMSG("color swapped in-/output not supported for GREY8")));
	if (convert_infmt == convert_outfmt) convert_doit = convert_nil;
}

void convert_nilG(int fbnr)
{
	ERROR(INTMSG("convert_nilG called"));
}

void convert_Gto24(int fbnr)
{
	int b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(b,ptr);
		PUT_RGB24(b,b,b,ptro);
	}
}

void convert_Gto16(int fbnr)
{
	int r,g,b;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(b,ptr);
		GREY82RGB16(b,r,g,b);
		PUT_RGB16(r,g,b,ptro);
	}
}

void convert_Gto422(int fbnr)
{
	int y1,y2;
	unsigned char *ptr = convert_inbuf[fbnr];
	unsigned char *ptro0 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*convert_inheight;
	int cnt = convert_inwidth*convert_inheight/2;

	while (cnt--) {
		GET_8BIT(y1,ptr);
		GET_8BIT(y2,ptr);
		if (convert_outpal) {
			PUT_YUV422SP(y1,y2,NO_C,NO_C,ptro0,ptro1,ptro2);
		} else {
			PUT_YUV422S(y1,y2,NO_C,NO_C,ptro0);
		}
	}
}

void convert_Gto420(int fbnr)
{
	int y1,y2,y3,y4;
	unsigned char *ptr0, *ptr1 = convert_inbuf[fbnr];
	unsigned char *ptro0 = NULL, *ptro01 = convert_outbuf[0], *ptro1 = ptro0+convert_inwidth*convert_inheight, *ptro2 = ptro1+convert_inwidth/2*((convert_inheight+1)/2);
	int w, h = (convert_inheight+1)/2;

	while (h--) {
		ptro0 = ptro01;
		if (!convert_outpal) ptro01 += convert_outwidth*2;
		else ptro01 += convert_outwidth;

		ptr0 = ptr1;
		ptr1 += convert_inwidth;

		w = convert_inwidth/2;
		while (w--) {
			GET_8BIT(y1,ptr0);
			GET_8BIT(y2,ptr0);
			GET_8BIT(y3,ptr1);
			GET_8BIT(y4,ptr1);
			if (convert_outpal) {
				PUT_YUV420PAL(y1,y2,y3,y4,NO_C,NO_C,
					      ptro0,ptro01,ptro1,ptro2);
			} else {
				PUT_YUV420(y1,y2,y3,y4,NO_C,NO_C,ptro0,ptro01);
			}
		}
	}
}

void convert_Gto8(int fbnr)
{
	int c;
	unsigned char *ptr = convert_inbuf[fbnr], *ptro = convert_outbuf[0];
	int cnt = convert_inwidth*convert_inheight;

	while (cnt--) {
		GET_8BIT(c,ptr);
		GREY82PAL8(c,c);
		PUT_8BIT(c,ptro);
	}
}


///////////////////

void convert_nil(int fbnr)
{
	memcpy(convert_outbuf[0], convert_inbuf[fbnr], convert_bytes);
}

void convert_init()
{
	int tg[9] = {0,32,64,96,128,160,192,224,255};
	int trb[5] = {0,64,128,192,255};
	int i, last;
	int r,g,b, ro, go, bo, l, y, cb, cr;
	double x;
	
	if (convert_rangetable) return;

	// 16 <-> 24 bit conversion
	x = 255.0/31.0;
	for (i = 0; i < 32; i++) convert_5to8tab[i] = (int)((x*((double)i))+0.5);
	x = 255.0/63.0;
	for (i = 0; i < 64; i++) convert_6to8tab[i] = (int)((x*((double)i))+0.5);
	x = 31.0/255.0;
	for (i=0; i < 256; i++) convert_8to5tab[i] = (int)((x*((double)i))+0.5);
	x = 63.0/255.0;
	for (i=0; i < 256; i++) convert_8to6tab[i] = (int)((x*((double)i))+0.5);

	// YCbCr <-> 24bit conversion
	for (i = 0; i < 256; i++) {
		convert_rgb_ycc_tab[i+R_Y_OFF] = FIX(0.29900) * i;
		convert_rgb_ycc_tab[i+G_Y_OFF] = FIX(0.58700) * i;
		convert_rgb_ycc_tab[i+B_Y_OFF] = FIX(0.11400) * i + ONE_HALF;
		convert_rgb_ycc_tab[i+R_CB_OFF] = (-FIX(0.16874)) * i;
		convert_rgb_ycc_tab[i+G_CB_OFF] = (-FIX(0.33126)) * i;
		convert_rgb_ycc_tab[i+B_CB_OFF] = FIX(0.50000) * i + CBCR_OFFSET + ONE_HALF-1;
		convert_rgb_ycc_tab[i+G_CR_OFF] = (-FIX(0.41869)) * i;
		convert_rgb_ycc_tab[i+B_CR_OFF] = (-FIX(0.08131)) * i;
	}
	for (i = 0, x = -128; i < 256; i++, x++) {
		convert_Crrtab[i] = ((int)(FIX(1.40200) * x + ONE_HALF) >> 16);
		convert_Cbbtab[i] = ((int)(FIX(1.77200) * x + ONE_HALF) >> 16);
		convert_Crgtab[i] = (- FIX(0.71414)) * x;
		convert_Cbgtab[i] = (- FIX(0.34414)) * x + ONE_HALF;
	}
	convert_rangetable = convert_range+256;
	memset(convert_range,0,256);
	for (i = 0; i < 256; i++) convert_rangetable[i] = i;
	memset(convert_rangetable+256,255,384);
	memset(convert_rangetable+256+384,0,384);
	memcpy(convert_rangetable+256+384+384,convert_rangetable,128);

	// 8bit pal <-> 24bit conversion
	last = 0;
	for (i = 0; i < 256; i++) {
		if (last < 4 && (i-trb[last]) > (trb[last+1]-i)) last++;
		convert_256to5tab[i] = last;
	}
	for (i = 0; i < 32; i++) convert_32to5tab[i] = convert_256to5tab[convert_5to8tab[i]];
	last = 0;
	for (i = 0; i < 256; i++) {
		if (last < 8 && (i-tg[last]) > (tg[last+1]-i)) last++;
		convert_256to9tab[i] = last;
	}
	for (i = 0; i < 64; i++) convert_64to9tab[i] = convert_256to9tab[convert_6to8tab[i]];

	for (i = 0; i < 16; i++) convert_palette[i][0] = convert_palette[i][1] = convert_palette[i][2] = 0;
	for (g=0; g<9; g++) for (r=0; r<5; r++) for (b=0;b<5;b++) {
		convert_palette[i][0] = trb[r];
		convert_palette[i][1] = tg[g];
		convert_palette[i][2] = trb[b];
		convert_backpal[r][g][b] = i;

		RGB242RGB16(trb[r],tg[g],trb[b],ro,go,bo);
		convert_palette16[i][0] = ro;
		convert_palette16[i][1] = go;
		convert_palette16[i][2] = bo;

		RGB242GREY8(trb[r],tg[g],trb[b],l);
		convert_paletteG[i] = l;

		RGB242YUV(trb[r],tg[g],trb[b],y,cb,cr);
		convert_paletteYCC[i][0] = y;
		convert_paletteYCC[i][1] = cb;
		convert_paletteYCC[i][2] = cr;

		i++;
	}
	for (; i < 256; i++) convert_palette[i][0] = convert_palette[i][1] = convert_palette[i][2] = 0;
	
	atexit(convert_end);
}


void convert_start()
{
	ASSERT(convert_inwidth,(INTMSG("unknown image width")));
	ASSERT(convert_inheight,(INTMSG("unknown image height")));
	ASSERT(convert_inwidth == convert_outwidth && convert_inheight == convert_outheight,(INTMSG("scaling not supported yet")));

	convert_doit = convert_table[convert_infmt][convert_outfmt];
	(*(convert_inittable[convert_infmt]))(1);
	(*(convert_inittable[convert_outfmt]))(0);

	convert_bytes = ((convert_outheight+1)*convert_outwidth*convert_color_bpps[convert_outfmt]+7)/8;

	if (convert_doit == convert_nil) {
		memcpy(convert_outbuf,convert_inbuf,convert_fbs*sizeof(void *));
		return;
	}
	SAFE_FREE(convert_outbuf);
	SAFE_MALLOC(convert_outbuf,sizeof(void *));
	convert_fbs = 1;
	SAFE_MALLOC(convert_outbuf[0],convert_bytes);
}

void convert_stop()
{
	if (convert_outbuf && convert_doit != convert_nil) {
		SAFE_FREE(convert_outbuf[0]);
	}
}

void convert_end()
{
	convert_stop();
	SAFE_FREE(convert_inbuf);
	SAFE_FREE(convert_outbuf);
}

int convert_convert(int fbnr)
{
	if (convert_doit == convert_nil) return fbnr;
	(*convert_doit)(fbnr);
	return 0;
}
