/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <errno.h>
#include <values.h>
#include <linux/types.h>
#include <linux/videodev.h>
// #include <bttv.h>
#include "grab.h"
#include "mutex.h"
#include "error.h"
#include "configuration.h"
// #include <bt848.h>
#define SYMBOL(x) grab_ ## x
#undef ENDFUNC
#define ENDFUNC(n) grab_error(n)

char grab_device[PATH_MAX]="/dev/bttv";
int grab_width = 640;
int grab_height = 480;
int grab_format = VIDEO_PALETTE_RGB24;
char grab_log[PATH_MAX] = "";
int grab_sleep = 0;

int grab_fbttv = 0, grab_fbs = 4, grab_channels = 1;
unsigned char **grab_buf = NULL;
int *grab_buf_picnr, *grab_bufst, *grab_bufqueue, *grab_buffers = NULL;
int grab_picnr = 0, grab_shmid, grab_mutexid = 0;
pid_t grab_child = 0, grab_parent = 0, grab_norm = 3;
FILE *grab_log_fp = NULL;
int grab_skip = 0, grab_skip_count = 0;
struct video_mbuf grab_mbuf;
struct video_channel grab_channel;

#define ST_EMPTY 0
#define ST_ARMED 1
#define ST_READY 2
#define ST_INUSE 3

/************ Prototypes ************/

void grab_init();
void grab_start();
void grab_parent_start();
void grab_child_start();
int grab_get();
int grab_get_internal();
int grab_get_picnr();
void grab_release();
void grab_stop();
void grab_error(int n);
void grab_end();

/************************************/

#define GRAB_NORMS 4
const char *grab_norms[GRAB_NORMS] = {"PAL","NTSC","SECAM","AUTO"};
const int grab_normvals[GRAB_NORMS] = {VIDEO_MODE_PAL,VIDEO_MODE_NTSC,VIDEO_MODE_SECAM,VIDEO_MODE_AUTO};

#define GRAB_COLORS 6
const char *grab_color_names[GRAB_COLORS] = {"RGB24","RGB16","PAL8","YUV422","YUV420", "GREY8"};
const char *grab_color_subst[GRAB_COLORS] = {"RGB24S","RGB16","PAL8","YUV422S","YUV422S", "GREY8"};
const int grab_color_values[GRAB_COLORS] = {VIDEO_PALETTE_RGB24,
					    VIDEO_PALETTE_RGB565,
					    VIDEO_PALETTE_HI240,
					    VIDEO_PALETTE_YUV422,
					    VIDEO_PALETTE_YUV422,
					    VIDEO_PALETTE_GREY};

START_OPTIONS
INTOPTION(sleep,0,1);
STROPTION(device) {
	OPTIONSET(struct video_capability vc;
		  int fh;
		  if ((fh = open(grab_device,O_RDONLY)) == -1) {
			  grab_channels = 1;
			  grab_channel.channel = 0;
			  grab_channel.norm = grab_normvals[3];
		  } else {
			  NOTNEG1(ioctl(fh,VIDIOCGCAP,&vc),(ERRMSG("VIDIOCGCAP")));
			  grab_channels = vc.channels;
			  grab_channel.channel = 0;
			  grab_channel.norm = grab_normvals[3];
			  SAFE_CLOSE(fh,grab_device);
		  }
		  );
}
INTINFO(channels);
OPTION(channel list) {
	OPTIONGET(struct video_channel vch;
		  int fh;
		  if ((fh = open(grab_device,O_RDONLY)) == -1) {
			  strcpy(value,"unknown");
		  } else {
			  *value = 0;
			  for (vch.channel = 0; vch.channel < grab_channels; vch.channel++) {
				  NOTNEG1(ioctl(fh,VIDIOCGCHAN,&vch),(ERRMSG("VIDIOCGCHAN")));
				  strcat(value,vch.name);
				  strcat(value," ");
			  }
			  if (!(*value)) value[strlen(value)-1] = 0;
			  SAFE_CLOSE(fh,grab_device);
		  }
		  );
}
OPTION(channel) {
	OPTIONSET(int fh;
		  int ch = atoi(value);
		  ASSERT(ch >= 0 && ch <= grab_channels,(INTMSG("channel out of range")));
		  grab_channel.channel = ch;
		  if ((fh = open(grab_device,O_RDONLY)) != -1) {
			  ioctl(fh,VIDIOCSCHAN,&grab_channel);
			  SAFE_CLOSE(fh,grab_device);
		  }
		);
	OPTIONGET(sprintf(value,"%i",grab_channel.channel););
}
OPTION(norm) {
	OPTIONSET(int fh; int i;
		  for (i = 0; i < GRAB_NORMS; i++) {
			  if (!strcmp(value,grab_norms[i])) {
				  break;
			  }
		  }
		  ASSERT(i < GRAB_NORMS,(MSG("Unknown video norm '%s'"),value));
		  grab_channel.norm = grab_normvals[i];
		  grab_norm = i;
		  if ((fh = open(grab_device,O_RDONLY)) != -1) {
			  ioctl(fh,VIDIOCSCHAN,&grab_channel);
			  SAFE_CLOSE(fh,grab_device);
		  }
		);
	OPTIONGET(strcpy(value,grab_norms[grab_norm]););
}
INTOPTION(width,32,768);
INTOPTION(height,32,576);
INTOPTION(format,0,MAXINT);
OPTION(color_fmt) {
	OPTIONSET(int i;
		  for (i = 0; i < GRAB_COLORS; i++) {
			  if (!strcmp(value,grab_color_names[i])) {
				  grab_format = grab_color_values[i];
				  break;
			  }
		  }
		  ASSERT(i < GRAB_COLORS,(INTMSG("unknown color format '%s'"),value)););
	OPTIONGET(int i;
		  for (i = 0; i < GRAB_COLORS; i++) {
			  if (grab_format == grab_color_values[i]) {
				  strcpy(value,grab_color_subst[i]);
				  break;
			  }
		  }
		  ASSERT(i < GRAB_COLORS,(INTMSG("unknown color format '%i'"),grab_format)););
}
INTINFO(fbs);
PTRINFO(buf,grab_fbs);
STROPTION(log);
INTOPTION(skip,-100,100);
END_OPTIONS

void grab_init()
{
	if (grab_parent) return;
	grab_parent = getpid();
	atexit(grab_end);
	memset(&grab_channel,0,sizeof(grab_channel));
	grab_channel.channel=0;
	grab_channel.norm=grab_normvals[grab_norm];

	NOTNEG1(mutex_init(),                                          (ERRMSG("mutex_init")));
	NOTNULL((void *)(grab_mutexid = mutex_open(2)),                (ERRMSG("open %i mutexes"),2));

	grab_configure("device","/dev/bttv");
}

void grab_start()
{
	struct video_capability vcapability;
	int i;

	if (grab_fbttv || grab_child) return;

	SAFE_OPEN(grab_fbttv,grab_device,O_RDWR);

	NOTNEG1(ioctl(grab_fbttv,VIDIOCGMBUF,&grab_mbuf),(ERRMSG("VIDIOCGMBUF")));
	grab_fbs = grab_mbuf.frames;

	NOTNEG1(grab_shmid = shmget(IPC_PRIVATE, (1+(grab_fbs*3))*sizeof(int), 0600),(ERRMSG("shmget")));
	NOTNEG1((int)(grab_buffers = (int *)shmat(grab_shmid, 0, 0)),  (ERRMSG("shmat")));

	*grab_buffers=grab_fbs;
	grab_bufqueue = grab_buffers+1;
	grab_bufst = grab_bufqueue+grab_fbs;
	grab_buf_picnr = grab_bufst+grab_fbs;
	for (i = 0; i < grab_fbs; i++) {
		grab_bufqueue[i] = -1;
		grab_bufst[i] = ST_EMPTY;
		grab_buf_picnr[i] = -1;
	}

	NOTNEG1(ioctl(grab_fbttv,VIDIOCGCAP,&vcapability),(ERRMSG("VIDIOCGCAP")));

	RANGE(grab_width,vcapability.minwidth,vcapability.maxwidth,"grab width",MSG);
	RANGE(grab_height,vcapability.minheight,vcapability.maxheight,"grab height",MSG);

	// start grab thread
	NOTNEG1(grab_child = fork(),(ERRMSG("fork")));

	if (grab_child) grab_parent_start();
	else grab_child_start();
}

void grab_parent_start()
{
	int i;

	SAFE_MALLOC(grab_buf,grab_fbs*sizeof(char *));
	grab_buf[0] = (void *)-1;
	NOTNEG1((int)(grab_buf[0]=mmap(0,grab_mbuf.size,PROT_READ|PROT_WRITE,MAP_SHARED,grab_fbttv,0)),
		(ERRMSG("mmap %i bytes"),grab_mbuf.size));

	for (i = 1; i < *grab_buffers; i++) {
		grab_buf[i] = grab_buf[i-1]+(grab_mbuf.size/grab_mbuf.frames);
	}

	mutex_release(grab_mutexid,0);
	mutex_try(grab_mutexid,2);
}

#if 0
#define mutex_grab(a,b) NOTNEG1(mutex_grab(a,b),(ERRMSG("mutex_grab")))
#define mutex_release(a,b) NOTNEG1(mutex_release(a,b),(ERRMSG("mutex_release")))
#define mutex_wait(a,b) NOTNEG1(mutex_wait(a,b),(ERRMSG("mutex_wait")))
#define mutex_waitforgrab(a,b) NOTNEG1(mutex_waitforgrab(a,b),(ERRMSG("mutex_waitforgrab")))
#endif

#define ioctl(a,b,c) NOTNEG1(ioctl(a,b,c),(ERRMSG(#b)))

void grab_child_start()
{
	struct video_mmap *gb;
	struct timeval grabtime;
	int nextnum, newnum, tmp, i, grab_bufnum;

	if (*grab_log) {
		SAFE_FOPEN(grab_log_fp,grab_log,"w");
	}

	mutex_grab(grab_mutexid, 0);
	SAFE_MALLOC(gb,(*grab_buffers)*sizeof(struct video_mmap));

	for (i = 0; i < *grab_buffers; i++) {
		gb[i].width = grab_width;
		gb[i].height = grab_height;
		gb[i].format = grab_format;
		gb[i].frame=i;
	}

	grab_bufnum = *grab_buffers;
	/////////////////// grab loop ///////////////////////
//	printf("-0 = 0\n");
	ioctl(grab_fbttv, VIDIOCMCAPTURE, &(gb[0]));
//	printf("-1 = 1\n");
	ioctl(grab_fbttv, VIDIOCMCAPTURE, &(gb[1]));

	grab_bufst[0]++;
	grab_bufst[1]++;
	grab_buf_picnr[0] = grab_picnr++;
	grab_buf_picnr[1] = grab_picnr++;

	mutex_release(grab_mutexid,0);

	nextnum = 0;
	newnum = 1;
	if (grab_bufnum == 2) goto loop2;
loop:
	ioctl(grab_fbttv, VIDIOCSYNC, &nextnum);

	mutex_grab(grab_mutexid,0);

	grab_bufst[nextnum]++;
	for (i = 0; i < grab_bufnum; i++) if (grab_bufst[i] == ST_EMPTY) break;
	if (i == grab_bufnum) {
		tmp = nextnum;
		nextnum = newnum;
		newnum = tmp;
		if (*grab_log) fprintf(grab_log_fp,"%i lost\n",grab_buf_picnr[newnum]);
	} else {
		if (*grab_log) {
			gettimeofday(&grabtime,0);
			fprintf(grab_log_fp,"%i %li %li\n",grab_buf_picnr[nextnum],grabtime.tv_sec,grabtime.tv_usec);
		}
		tmp = i;
		for (i = 0; i < grab_bufnum; i++) if (grab_bufqueue[i] == -1) {
			grab_bufqueue[i] = nextnum;
			break;
		}
		nextnum = newnum;
		newnum = tmp;
	}
	grab_bufst[newnum] = ST_ARMED;
	grab_buf_picnr[newnum] = grab_picnr++;

	mutex_release(grab_mutexid,0);
	mutex_release(grab_mutexid,1);

//	printf("-%i = %i\n",newnum,grab_picnr);
	ioctl(grab_fbttv, VIDIOCMCAPTURE, gb+newnum);

	goto loop;

// if we have only 2 grab_buffers, we have to wait for the main thread each frame
loop2:
	ioctl(grab_fbttv, VIDIOCSYNC, &nextnum);

	mutex_grab(grab_mutexid,0);
		
	grab_bufst[nextnum]++;
	grab_bufqueue[0] = nextnum;
	tmp = nextnum;
	nextnum = newnum;
	newnum = tmp;
	grab_bufst[newnum] = ST_ARMED;
	mutex_release(grab_mutexid,0);
	mutex_release(grab_mutexid,1);
	mutex_waitforgrab(grab_mutexid,1);

	ioctl(grab_fbttv, VIDIOCMCAPTURE, gb+newnum);

	goto loop2;

}

int grab_get()
{
	if (grab_skip < 0) {
		grab_skip_count--;
		if (grab_skip_count == grab_skip) {
			grab_skip_count = 0;
			grab_get_internal();
			grab_release();
		}
	} else if (grab_skip > 0) {
		int i;
		for (i = 0; i < grab_skip; i++) {
			grab_get_internal();
			grab_release();
		}
	}
	return grab_get_internal();
}

int grab_get_internal()
{
	int x;
	if (grab_sleep) {
		mutex_release(grab_mutexid,0);
		mutex_grab(grab_mutexid,1);
		mutex_try(grab_mutexid,1);
		mutex_grab(grab_mutexid,0);
		x = 1;
		while (grab_bufqueue[x] != -1) x++;
		x = grab_bufqueue[x-1];
	} else {
		mutex_grab(grab_mutexid,0);
		if (grab_bufqueue[0] == -1) {
			mutex_release(grab_mutexid,0);
			mutex_wait(grab_mutexid,1);
			mutex_grab(grab_mutexid,0);
		}
		x = grab_bufqueue[0];
		mutex_release(grab_mutexid,0);
	}

	return x;
}

int grab_get_picnr()
{
	return grab_buf_picnr[grab_bufqueue[0]];
}


void grab_release()
{
	int i;
	if (grab_sleep) {
		for (i = 0; i < *grab_buffers; i++) {
			grab_bufst[grab_bufqueue[0]] = ST_EMPTY;
			grab_bufqueue[i]  = -1;
		}
	} else {
		mutex_grab(grab_mutexid,0);
		
		grab_bufst[grab_bufqueue[0]] = ST_EMPTY;
		for (i = 1; i < *grab_buffers; i++) grab_bufqueue[i-1] = grab_bufqueue[i];
		grab_bufqueue[(*grab_buffers)-1]  = -1;
		
		mutex_release(grab_mutexid,0);
		mutex_grab(grab_mutexid,1);
	}
}

void grab_stop()
{
	struct shmid_ds x;
	struct sigaction old;

	if (grab_buffers) {
		if (grab_buf) {
			if (grab_buf[0] != (void *)-1) munmap(grab_buf[0],grab_mbuf.size);
			SAFE_FREE(grab_buf);
		}
		shmdt((void *)grab_buffers);
		grab_buffers = NULL;
	}

	if (getpid() == grab_parent) {
		if (grab_child && grab_child != -1) {
			sigaction(SIGCLD,NULL,&old);
			signal(SIGCLD,SIG_IGN);
			kill(grab_child,SIGINT);
			waitpid(grab_child,NULL,0);
			grab_child = 0;
			sigaction(SIGCLD,&old,NULL);
		}

		if (grab_shmid && grab_shmid != -1) {
			shmctl(grab_shmid,IPC_RMID,&x);
			grab_shmid = 0;
		}

	} else if (!grab_child) {
		if (grab_log_fp) {
			fclose(grab_log_fp);
			grab_log_fp = NULL;
		}
	}

	if (grab_fbttv) {
		if (grab_fbttv != -1) close(grab_fbttv);
		grab_fbttv = 0;
	}
}

void grab_error(int n)
{
	if (getpid() == grab_parent) grab_stop();
	else exit(n);
}

void grab_end()
{
	grab_stop();
//	printf("%s\n",error_text);
	if (getpid() == grab_parent && grab_mutexid) {
		mutex_close(grab_mutexid);
		grab_mutexid = 0;
	}
}
