/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <values.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <limits.h>
#include <fcntl.h>
#include <curses.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#ifndef MSGMAX
#define MSGMAX 4056
#endif
#include <sys/shm.h>
#include <setjmp.h>
#include "version.h"
#include "output.h"
#include "grab.h"
#include "sound.h"
#include "error.h"
#include "mpeg.h"
#include "read.h"
#include "main.h"
#include "convert.h"
#include "transport.h"
#include "configuration.h"
#define SYMBOL(x) main_ ## x

/************ Prototypes ************/

void control(int action);
void configure(const char *option, const char *value);
void inquire(const char *option, char value[MSGMAX-96]);
int getprogress(char *src, char *dest, char *conv);
void setprogress(const char *src, char type);
void setprogressresult(int end);
void setprogressdest(const char *dest);
void setprogresssrc(const char *src);
void setprogressconv(const char *conv);
int main_convert_one(int n);
void main_mpeg_remove(int n);
void main_start();
void main_init();
void main_end();
void error_abort(int n);

/************************************/

void main_inquire(const char *, char *);

char *error_text = NULL;
int error_id = 0;
char pre_error_text[4096];

sigjmp_buf main_abort;
pid_t main_child = 0, main_parent = 0;
int main_cmdq = -1, main_statq = -1, main_inpipe = -1, main_outpipe = -1, main_errorpipe = -1, main_erroutpipe = -1;
int main_running = -1;

#define CONFIGURE 1
#define CONTROL   2
#define INQUIRE   3

#define P_SRC  's'
#define P_DEST 'd'
#define P_CONV 'c'
#define P_RESULT 'r'

#define PROGRESS 1
#define REPLY    2

struct confmsgbuf {
	struct msgbuf msg;
	char option[64];
	char value[MSGMAX-96];
};

char srcprogress[MSGMAX-96] = "";
char destprogress[MSGMAX-96] = "";
char convprogress[MSGMAX-96] = "";

struct progmsgbuf {
	struct msgbuf msg;
	char progress[MSGMAX-96];
} progress;

void control(int action)
{
	struct msgbuf msg;
	ASSERT(action == C_START || action == C_STOP || action == C_PAUSE || action == C_CONT,(INTMSG("unknown control code")));
	msg.mtype = CONTROL;
	msg.mtext[0] = action;

	NOTNEG1(msgsnd(main_cmdq,&msg,sizeof(struct msgbuf)-sizeof(long),0),(ERRMSG("message send")));
}

void configure(const char *option, const char *value)
{
	struct confmsgbuf msg;
	ASSERT(main_child != 0,(INTMSG("configure called from child thread")));

	ASSERT(strlen(value) <= MSGMAX-96 && strlen(value) >= 0,(INTMSG("configure value too long '%s' (option %s)"),value,option));
	ASSERT(strlen(option) <= 64,(INTMSG("configure option name too long")));
	msg.msg.mtype = CONFIGURE;
	strcpy(msg.option,option);
	strcpy(msg.value,value);

	NOTNEG1(msgsnd(main_cmdq,(struct msgbuf *)&msg,sizeof(struct confmsgbuf)-sizeof(long),0),(ERRMSG("message send")));
}

void inquire(const char *option, char value[MSGMAX-96])
{
	struct confmsgbuf msg;
	ASSERT(main_child != 0,(INTMSG("inquire called from child thread")));

	ASSERT(strlen(option) <= 64,(INTMSG("configure option name too long")));
	msg.msg.mtype = INQUIRE;
	strcpy(msg.option,option);

	NOTNEG1(msgsnd(main_cmdq,(struct msgbuf *)&msg,sizeof(struct confmsgbuf)-sizeof(long),0),(ERRMSG("message send")));
	NOTNEG1(msgrcv(main_statq,(struct msgbuf *)&msg,sizeof(struct confmsgbuf)-sizeof(long),REPLY,0),(ERRMSG("message receive")));

	strcpy(value,msg.value);
}

int getprogress(char *src, char *dest, char *conv)
{
	int rc, num=0;
	ASSERT(main_child != 0,(INTMSG("getprogress called from child thread")));

	rc = read(main_inpipe,&progress,sizeof(struct progmsgbuf));
	while (rc>=0) {
		ASSERT(rc == sizeof(struct progmsgbuf),(INTMSG("invalid data on progress pipe")));
		switch (progress.msg.mtext[0]) {
		case P_RESULT:
			*src = *srcprogress = 0;
			*dest = *destprogress = 0;
			*conv = *convprogress = 0;
			return -(*(progress.progress));
			break;
		case P_SRC:
			strcpy(srcprogress,progress.progress);
			break;
		case P_DEST:
			strcpy(destprogress,progress.progress);
			num++;
			break;
		case P_CONV:
			strcpy(convprogress,progress.progress);
			break;
		}
		rc = read(main_inpipe,&progress,sizeof(struct progmsgbuf));
	} 
	ASSERT(errno == EAGAIN,(ERRMSG("message receive")));

	strcpy(src,srcprogress);
	strcpy(dest,destprogress);
	strcpy(conv,convprogress);
	return num;
}

void setprogress(const char *src, char type)
{
	ASSERT(strlen(src) < MSGMAX-96,(INTMSG("progress message too long")));
	strcpy(progress.progress,src);
	progress.msg.mtext[0] = type;
	SAFE_WRITE(main_outpipe,&progress,sizeof(struct progmsgbuf),"pipe");
}

void setprogressresult(int end)
{
	*(progress.progress) = end;
	progress.msg.mtext[0] = P_RESULT;
	SAFE_WRITE(main_outpipe,&progress,sizeof(struct progmsgbuf),"pipe");
}

void setprogressdest(const char *dest)
{
	setprogress(dest,P_DEST);
}

void setprogresssrc(const char *src)
{
	setprogress(src,P_SRC);
}

void setprogressconv(const char *conv)
{
	setprogress(conv,P_CONV);
}


/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
int main_do_sound = 1;
int main_sleeptime = 0;
int main_limit = 0;
int main_width = 640;
int main_height = 480;
int main_do_trace = 0;
int main_do_grab = 1;
int main_do_convert = 0;
int main_mpeg = 0;
int main_lowdisk = 0;
int main_always_abort = 1;
char main_grablog[PATH_MAX] = "";
char main_convfmt[PATH_MAX] = "";
char main_convname[PATH_MAX] = "";
char main_conv_transport[PATH_MAX] = "file";
int main_first_conv = 0;
char main_tmpname[PATH_MAX] = "";
int main_msleep;

START_OPTIONS
OPTREDIRECT(grabdevice,grab,device);
OPTREDIRECT(channel,grab,channel);
OPTREDIRECT(norm,grab,norm) {
	if (!strcmp(value,"NTSC")) mpeg_configure("ntsc","1");
	else mpeg_configure("ntsc","0");
}
OPTREDIRECT(channel list,grab,channel list);
INTOPTION(do_sound,0,1) FORWARD(mpeg,sound);
INTOPTION(msleep,0,MAXINT);
INTOPTION(sleeptime,0,MAXINT) {
	OPTIONSET(if (main_sleeptime) grab_configure("sleep","1"); else grab_configure("sleep","0"););
}
OPTREDIRECT(skip,grab,skip);
INTOPTION(limit,0,MAXINT) {
	char w[50];
	OPTIONSET(sprintf(w,"%i",atoi(value)-1);
		  if (atoi(value) > 0) mpeg_configure("last",w););
}
INTOPTION(width,32,768) {
	FORWARD(grab,width);
	FORWARD(read,width);
	FORWARD(output,width);
	FORWARD(convert,inwidth);
	FORWARD(convert,outwidth);
}
INTOPTION(height,32,768) {
	FORWARD(grab,height);
	FORWARD(read,height);
	FORWARD(output,height);
	FORWARD(convert,inheight);
	FORWARD(convert,outheight);
}
INTOPTION(do_trace,0,1);
INTOPTION(first_conv,0,MAXINT) FORWARD(mpeg,first);
OPTREDIRECT(video_bitrate,mpeg,bitrate);
OPTREDIRECT(audio_bitrate,mpeg,audio_rate);
INTOPTION(lowdisk,0,1) FORWARD(mpeg,lowdisk);
OPTREDIRECT(conv_log,mpeg,logname);
OPTREDIRECT(inputname,transport,rname);
OPTREDIRECT(quality,output,quality);
OPTREDIRECT(grabname,transport,wname);
STROPTION(convname);
INTOPTION(always_abort,0,1);
STROPTION(grablog) FORWARD(grab,log);
OPTREDIRECT(soundfile,sound,file) FORWARD(mpeg,wavname);
OPTREDIRECT(soundrate,sound,rate);
OPTREDIRECT(soundchannels,sound,channels);
OPTREDIRECT(soundbits,sound,bits);
OPTREDIRECT(sounddevice,sound,device);
OPTREDIRECT(do_sync,transport,sync);
OPTREDIRECT(input_transport,transport,rformat str);
OPTREDIRECT(grab_transport,transport,wformat str);
STROPTION(conv_transport);
OPTION(transport list) {
	OPTIONGET(
		char tmp[PATH_MAX];
		int i;
		transport_inquire("format nr",tmp);
		i = atoi(tmp);
		value[0] = 0;
		while(i-->0) {
			sprintf(tmp,"%i",i);
			transport_configure("wformat",tmp);
			transport_inquire("wformat str",tmp);
			strcat(value," ");
			strcat(value,tmp);
		}
		);
}
OPTION(input_file_fmt) {
	OPTIONSET(if (strchr(value,'-')) *(strchr(value,'-')) = 0;);
	FORWARD(read,file_fmt str);
	OPTIONSET(main_do_grab = 0;);
	OPTIONGET(if (main_do_grab) value[0] = 0;);
}
OPTREDIRECT(grab_file_fmt,output,file_fmt str) {
	OPTIONSET(main_do_grab = 1;);
	OPTIONGET(if (!main_do_grab) value[0] = 0;);
}
OPTION(conv_file_fmt) {
	OPTIONSET(
		if (value[0] == 0) {
			main_do_convert = 0;
		} else {
			main_do_convert = 1;
			if (!strcmp(value,"mpeg-1")) {
				strcpy(main_convfmt,"yuv");
				mpeg_configure("type","1");
				main_mpeg = 1;
			} else if (!strcmp(value,"mpeg-2")) {
				strcpy(main_convfmt,"yuv");
				mpeg_configure("type","2");
				main_mpeg = 1;
			} else {
				strcpy(main_convfmt,value);
				main_mpeg = 0;
			}
		});
	OPTIONGET(
		if (!main_do_convert) {
			value[0] = 0;
		} else {
			if (main_mpeg) {
				char w[2];
				mpeg_inquire("type",w);
				strcpy(value,"mpeg-");
				strcat(value,w);
			} else {
				strcpy(value,main_convfmt);
			}
		});
}
OPTION(grab_file_fmt list) {
	OPTIONGET(
		char tmp[PATH_MAX];
		int i;
		output_inquire("file_fmt nr",tmp);
		i = atoi(tmp);
		value[0] = 0;
		while(i-->0) {
			sprintf(tmp,"%i",i);
			output_configure("file_fmt",tmp);
			output_inquire("file_fmt str",tmp);
			strcat(value," ");
			strcat(value,tmp);
		}
		);
}
OPTION(conv_file_fmt list) {
	main_inquire("grab_file_fmt list",value);
	strcat(value," mpeg-1 mpeg-2");
}
OPTION(input_file_fmt list) {
	OPTIONGET(
		char tmp[PATH_MAX];
		int i;
		read_inquire("file_fmt nr",tmp);
		i = atoi(tmp);
		value[0] = 0;
		while(i-->0) {
			sprintf(tmp,"%i",i);
			read_configure("file_fmt",tmp);
			read_inquire("file_fmt str",tmp);
			strcat(value," ");
			strcat(value,tmp);
		}
		);
}
END_OPTIONS

int main_convert_one(int n)
{
	if (!read_read(0,n) && !main_do_trace) {
		return 0;
	}
	convert_convert(0);
	output_write(0,n);
	if (main_lowdisk) transport_rremove(n);
	return 1;
}

void main_mpeg_remove(int n)
{
	transport_wremove(n);
}

void main_start()
{
	struct msgbuf msg;
	struct msqid_ds msq_stat;
	int xgrab;
	static int picnr;
	char tmp[PATH_MAX];
	picnr = 0;

	if (sigsetjmp(main_abort,1)) {
		return;
	}
	setprogressresult(0);

	main_running = 1;

	if (main_do_grab) {
		if (sigsetjmp(main_abort,1)) {
			grab_stop();
			sound_stop();
			output_stop();
			convert_stop();
			return;
		}

		output_inquire("color_fmt",tmp);
		convert_configure("outfmt",tmp);
		convert_inquire("outfmt",tmp);
		grab_configure("color_fmt",tmp);
		grab_inquire("color_fmt",tmp);
		convert_configure("infmt",tmp);
		grab_configure("fbs","32");
		grab_start();
		grab_inquire("fbs",tmp);
		convert_configure("fbs",tmp);
		grab_inquire("buf",tmp);
		convert_configure("inbuf",tmp);
		convert_start();

		convert_inquire("fbs",tmp);
		output_configure("fbs",tmp);
		convert_inquire("outbuf",tmp);
		output_configure("buf",tmp);
		output_start();

		if (main_do_sound) sound_start();

		while (1) {
			if (main_limit && main_limit == picnr) break;
			
			NOTNEG1(msgctl(main_cmdq,IPC_STAT,&msq_stat),(ERRMSG("IPC_STAT msq")));
			if (msq_stat.msg_qnum) {
				NOTNEG1(msgrcv(main_cmdq,&msg,sizeof(struct msgbuf)-sizeof(long),0,0),(ERRMSG("message receive")));
				ASSERT(msg.mtype == CONTROL,(INTMSG("no configure/inquire messages while running")));
				switch (msg.mtext[0]) {
				case C_STOP:
					sound_stop();
					grab_stop();
					output_stop();
					return;
				case C_START:
				case C_CONT:
					break;
				case C_PAUSE:
					do {
						NOTNEG1(msgrcv(main_cmdq,&msg,sizeof(struct msgbuf)-sizeof(long),0,MSG_NOERROR),(ERRMSG("message receive")));
						ASSERT(msg.mtype == CONTROL,(INTMSG("no configure/inquire messages while running")));
						switch (msg.mtext[0]) {
						case C_STOP:
							sound_stop();
							grab_stop();
							output_stop();
							return;
						case C_START:
						case C_PAUSE:
						case C_CONT:
							break;
						default:
							ERROR(INTMSG("unknown control message '%c'"),msg.mtext[0]);
							exit(1);
						}
					} while (msg.mtext[0] != C_CONT);
					break;
				default:
					ERROR(INTMSG("unknown control message '%c'"),msg.mtext[0]);
					exit(1);
				}
			}
			
			xgrab = grab_get();
			xgrab = convert_convert(xgrab);
			if (!main_do_trace) output_write(xgrab, picnr);
			else output_write(xgrab, grab_get_picnr());
			grab_release();
			picnr++;
			if (main_sleeptime) sleep(main_sleeptime);
		}
		sound_stop();
		grab_stop();
		output_stop();
		if (main_do_convert) {
			output_inquire("file_fmt str",tmp);
			if (strchr(tmp,'-')) *(strchr(tmp,'-')) = 0;
			read_configure("file_fmt str",tmp);
			transport_inquire("wname",main_tmpname);
			transport_configure("rname",main_tmpname);
			transport_inquire("wformat str",tmp);
			transport_configure("rformat str",tmp);
		}
	}
	if (main_do_convert) {
		char name[PATH_MAX];

		setprogressresult(2);

		if (sigsetjmp(main_abort,1)) {
			read_stop();
			mpeg_stop();
			output_stop();
			convert_stop();
			return;
		}

		read_start();

		read_inquire("fbs",name);
		convert_configure("fbs",name);
		read_inquire("buf",name);
		convert_configure("inbuf",name);

		read_inquire("width",name);
		convert_configure("inwidth",name);
		convert_configure("outwidth",name);
		output_configure("width",name);
		mpeg_configure("width",name);

		read_inquire("height",name);
		convert_configure("inheight",name);
		convert_configure("outheight",name);
		output_configure("height",name);
		mpeg_configure("height",name);

		if (main_mpeg) output_configure("file_fmt str","yuv");
		else output_configure("file_fmt str",main_convfmt);
		/*if (*main_conv_transport)*/ transport_configure("wformat str",main_conv_transport);

		read_inquire("color_fmt",name);
		convert_configure("infmt",name);
		output_inquire("color_fmt",name);
		convert_configure("outfmt",name);

		convert_start();

		convert_inquire("fbs",name);
		output_configure("fbs",name);
		convert_inquire("outbuf",name);
		output_configure("buf",name);

		if (main_mpeg) {
			tmpnam(main_tmpname);
			strcat(main_tmpname,"%04d.yuv");
			mpeg_configure("tmpname",main_tmpname);
			mpeg_configure("trace",main_do_trace?"1":"0");

			transport_configure("wname",main_tmpname);

			output_start();

			mpeg_start();

			output_stop();
			read_stop();
			convert_stop();
		} else {
			/*if (*main_convname)*/ transport_configure("wname",main_convname);

			output_start();

			picnr = main_first_conv;
			while (1) {
				if (main_limit && main_limit == picnr) break;
				
				
				NOTNEG1(msgctl(main_cmdq,IPC_STAT,&msq_stat),(ERRMSG("IPC_STAT msq")));
				if (msq_stat.msg_qnum) {
					NOTNEG1(msgrcv(main_cmdq,&msg,sizeof(struct msgbuf)-sizeof(long),0,0),(ERRMSG("message receive")));
					ASSERT(msg.mtype == CONTROL,(INTMSG("no configure/inquire messages while running")));
					switch (msg.mtext[0]) {
					case C_STOP:
						read_stop();
						output_stop();
						return;
					case C_PAUSE:
					case C_CONT:
					case C_START:
						break;
					default:
						ERROR(INTMSG("unknown control message '%c'"),msg.mtext[0]);
						exit(1);
					}
				}
				if (!main_convert_one(picnr)) break;
				picnr++;
				if (main_msleep) usleep(main_msleep*1000);
			}
			read_stop();
			convert_stop();
			output_stop();
		}
	}
}


void main_init()
{
	struct confmsgbuf msg;
	struct sigaction sa;
	int rc, pipes[2];
	char *tmp;

	if (main_child) return;
	error_text = pre_error_text;
	error_text[0] = 0;
	main_parent = getpid();
	atexit(main_end);

	NOTNEG1(error_id = shmget(IPC_PRIVATE,4096,0600),(ERRMSG("shmget")));
	NOTNEG1((int)(tmp = (char *)shmat(error_id,NULL,0)),(ERRMSG("shmat")));
	NOTNEG1(shmctl(error_id,IPC_RMID,0),(ERRMSG("shmctl")));
	error_text = tmp;
	error_text[0] = 0;

	sa.sa_handler = error_abort;
	sa.sa_flags = SA_NOCLDSTOP;
	sa.sa_restorer = NULL;
	NOTNEG1(sigfillset(&(sa.sa_mask)),(ERRMSG("sigfillset")));
	NOTNEG1(sigdelset(&(sa.sa_mask),SIGQUIT),(ERRMSG("sigdelset")));
	NOTNEG1(sigaction(SIGCHLD,&sa,NULL),(ERRMSG("sigaction")));
	sa.sa_handler = exit;
	NOTNEG1(sigaction(SIGINT,&sa,NULL),(ERRMSG("sigaction")));
	NOTNEG1(sigaction(SIGTERM,&sa,NULL),(ERRMSG("sigaction")));

	NOTNEG1(main_cmdq = msgget(IPC_PRIVATE,0600),(ERRMSG("message queue open")));
	NOTNEG1(main_statq = msgget(IPC_PRIVATE,0600),(ERRMSG("message queue open")));

	NOTNEG1(pipe(pipes),(ERRMSG("pipe")));
	main_inpipe = pipes[0];
	main_outpipe = pipes[1];
	fcntl(main_inpipe,F_SETFL,O_NONBLOCK);
	NOTNEG1(pipe(pipes),(ERRMSG("pipe")));
	main_erroutpipe = pipes[1];
	main_errorpipe = pipes[0];
//	fcntl(main_erroutpipe,F_SETFL,O_NONBLOCK);

	NOTNEG1(main_child = fork(),(ERRMSG("fork")));
	
	if (main_child) {
		main_running = -1;
		return;
	}
	main_running = 0;

	transport_otrace = setprogressdest;
	transport_itrace = setprogresssrc;
	mpeg_strace_func = setprogresssrc;
	mpeg_trace_func = setprogressconv;
	mpeg_remove_func = main_mpeg_remove;
	mpeg_conv_func = main_convert_one;

	grab_init();
	convert_init();
	output_init();
	read_init();
	mpeg_init();
	sound_init();
	progress.msg.mtype = PROGRESS;
	
main_loop:
	rc = msgrcv(main_cmdq,(struct msgbuf *)&msg,sizeof(struct confmsgbuf)-sizeof(long),0,0);
	ASSERT(rc > 0 || errno == EINTR,(ERRMSG("message receive")));
	if (rc > 0) switch (msg.msg.mtype) {
	case CONFIGURE:
		main_configure(msg.option,msg.value);
		break;

	case INQUIRE:
		main_inquire(msg.option,msg.value);
		msg.msg.mtype = REPLY;
		NOTNEG1(msgsnd(main_statq,(struct msgbuf *)&msg,sizeof(struct confmsgbuf)-sizeof(long),0),(ERRMSG("message send")));
		break;

	case CONTROL:
		switch (msg.msg.mtext[0]) {
		case C_STOP:
			break;
		case C_START:
			main_start();
			main_running = -1;
			setprogressresult(1);
			break;
		default:
			ERROR(INTMSG("unknown control message '%c'"),msg.msg.mtext[0]);
			exit(1);
		}
		break;
	default:
			ERROR(INTMSG("unknown message type '%li'"),msg.msg.mtype);
			exit(1);
	}

	goto main_loop;
}


void main_end()
{
	struct sigaction old;

	if (main_parent == getpid()) {
		if (main_child && main_child != -1) {
			sigaction(SIGCLD,NULL,&old);
			signal(SIGCLD,SIG_IGN);
			kill(main_child,SIGTERM);
			waitpid(main_child,NULL,0);
			sigaction(SIGCLD,&old,NULL);
			main_child = -1;
		}

		if (main_cmdq != -1) {
			msgctl(main_cmdq,IPC_RMID,NULL);
			main_cmdq = 0;
		}

		if (main_statq != -1) {
			msgctl(main_statq,IPC_RMID,NULL);
			main_statq = 0;
		}

		if (main_inpipe != -1) {
			SAFE_CLOSE(main_inpipe,"pipe");
			SAFE_CLOSE(main_outpipe,"pipe");
			main_inpipe = -1;
			main_outpipe = -1;
		}
	}
}

void error_abort(int n)
{
	char c = 0;
	if (getpid() == main_parent) exit(1);
	write(main_erroutpipe,&c,sizeof(c));
#ifdef DEBUG
	fprintf(stderr,"%s [%i/%i]\n",error_text,n,main_running);
#endif
	if (main_running >= 0 && !main_always_abort) {
		if (main_running) {
			main_running = 0;
			siglongjmp(main_abort,1);
		}
	} else {
		exit(1);
	}
}
