/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <unistd.h>
#include <stdlib.h>
#include <sys/ipc.h>
#include <sys/sem.h>

/************ Prototypes ************/

int mutex_init();
void mutex_end();
int mutex_open(int n);
int mutex_close(int id);
int sync_on();
void sync_off();
int mutex_wait(int id, int n);
int mutex_waitforgrab(int id, int n);
int mutex_grab(int id, int n);
int mutex_try(int id, int n);
int mutex_release(int id, int n);
int mutex_isSet(int id, int n);
int mutex_isFree(int id, int n);

/************************************/

#define SAFETY 	if (sync_on()) return -1;\
		if (id >= mutex_nr || id <= 0 || mutex_semid[id] == -1) {\
			sync_off();\
			return -1;\
		}\
		id = mutex_semid[id];\
		sync_off()


struct sembuf mutex_semgrab, mutex_semrelease, mutex_semtest, mutex_semtry;
int *mutex_semid = NULL, mutex_nr = 0, mutex_syncid = -1;
pid_t mutex_creator, *mutex_semcreator;

int mutex_init()
{
	if (mutex_semid) return 0;
	mutex_creator = getpid();
	atexit(mutex_end);

	if (!(mutex_semid = malloc(sizeof(int)))) return -1;;
	if (!(mutex_semcreator = malloc(sizeof(pid_t)))) return -1;;

	mutex_nr = 1;
	mutex_semgrab.sem_op = -1;
	mutex_semgrab.sem_flg = 0;

	mutex_semtry.sem_op = -1;
	mutex_semtry.sem_flg = IPC_NOWAIT;

	mutex_semtest.sem_op = 0;
	mutex_semtest.sem_flg = IPC_NOWAIT;

	mutex_semrelease.sem_op = 1;
	mutex_semrelease.sem_flg = 0;

	if ((mutex_syncid = semget(IPC_PRIVATE, 1, 0600)) == -1) return -1;
	sync_off();

	return 0;
}

void mutex_end()
{
	if (getpid() == mutex_creator) semctl(mutex_syncid,0,IPC_RMID,(union semun)0);
	if (mutex_semid) {
		int i = 0;
		while (++i < mutex_nr) {
			if (mutex_semid[i] == -1) continue;
			if (getpid() == mutex_semcreator[i]) semctl(mutex_semid[i],0,IPC_RMID,(union semun)0);
		}
		free(mutex_semid);
	}
}

int mutex_open(int n)
{
	int semid, *buffer;
	if (mutex_semid == NULL) return -1;

	if (sync_on()) return -1;
	buffer = realloc(mutex_semid,(mutex_nr+1)*sizeof(int));
	if (!buffer) {
		sync_off();
		return -1;
	}
	mutex_semid = buffer;
	buffer = realloc(mutex_semcreator,(mutex_nr+1)*sizeof(pid_t));
	if (!buffer) {
		sync_off();
		return -1;
	}
	mutex_semcreator = buffer;
	if ((semid = semget(IPC_PRIVATE, n, 0600)) == -1) {
		sync_off();
		return -1;
	}
	mutex_semid[mutex_nr] = semid;
	mutex_semcreator[mutex_nr] = getpid();
	mutex_nr++;
	sync_off();
	return mutex_nr-1;
}

int mutex_close(int id)
{
	if (sync_on()) return -1;
	if (id >= mutex_nr || id <= 0 || mutex_semid[id] == -1) {
		sync_off();
		return -1;
	}

	if (semctl(mutex_semid[id],0,IPC_RMID,(union semun)0) == -1) {
		sync_off();
		return -1;
	}
	mutex_semid[id] = -1;
	sync_off();
	return 0;
}

int sync_on()
{
	mutex_semgrab.sem_num = 0;
	if (semop(mutex_syncid,&mutex_semgrab,1)) return -1;
	return 0;
}

void sync_off()
{
	mutex_semrelease.sem_num = 0;
	semop(mutex_syncid,&mutex_semrelease,1);
}

int mutex_wait(int id, int n)
{
	struct sembuf action[2] = {mutex_semgrab, mutex_semrelease};
	SAFETY;

	action[0].sem_num = action[1].sem_num = n;
	return semop(id,action,2);
}

int mutex_waitforgrab(int id, int n)
{
	struct sembuf action = mutex_semtest;
	SAFETY;

	action.sem_num = n;
	action.sem_flg = 0;
	return semop(id,&action,1);
}

int mutex_grab(int id, int n)
{
	struct sembuf action = mutex_semgrab;
	SAFETY;

	action.sem_num = n;
	return semop(id,&action,1);
}

int mutex_try(int id, int n)
{
	struct sembuf action = mutex_semtry;
	SAFETY;

	action.sem_num = n;
	return semop(id,&action,1);
}

int mutex_release(int id, int n)
{
	struct sembuf action0 = mutex_semtest, action1 = mutex_semrelease;
	SAFETY;

	action0.sem_num = n;
	if (semop(id,&action0,1)) return 1;

	action1.sem_num = n;
	return semop(id,&action1,1);
}

int mutex_isSet(int id, int n)
{
	struct sembuf action = mutex_semtest;
	SAFETY;

	action.sem_num = n;
	return !(semop(id,&action,1));
}

int mutex_isFree(int id, int n)
{
	return !(mutex_isSet(id,n));
}
