/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

// see below for adding new output modules

#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <values.h>
#define OUTPUT_INTERNAL
#include "output.h"
#include "error.h"
#include "configuration.h"
#define SYMBOL(x) output_ ## x

int output_width = 0;
int output_height = 0;
int output_file_fmt = 0;
int output_quality = 75;

unsigned char ***output_ptrs = NULL, **output_buf = NULL;
int output_fbs = 0;
int output_bytes;
/************ Prototypes ************/

void output_init();
void output_start();
void output_stop();
void output_end();
void output_write(int fbnr, int picnr);
void output_remove(int fbnr);

/************************************/
#define OUTPUT_COLORS 7
const int output_color_bpps[OUTPUT_COLORS] = {24,8,16,16,-12,8,24};
const char *output_color_names[OUTPUT_COLORS] = {"RGB24","PAL8","RGB16","YUV422S","YUV420","GREY8","RGB24S"};

#define ADD_FORMAT(name,symbol,color) { \
	output_ ## symbol ## _start, \
	output_ ## symbol ## _write, \
	output_ ## symbol ## _stop, \
	#color, \
	#name \
}

struct OUTPUT_FORMAT_STRUCT {
	void (*start)();
	void (*write)(int fbnr, int picnr);
	void (*stop)();
	const char *color;
	char *name;
};

////////////////////////////////////////////////////////////////////////////////
///////////////////////// output module registration ///////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Step 1: include header file
#include "output_gif.h"
#include "output_jpg.h"
#include "output_raw.h"
#include "output_pmm.h"
#include "output_yuv.h"
#include "output_pnm.h"
#ifdef X11
#include "output_X.h"
#endif

// Step 2: increase number of formats
#ifdef X11
#define OUTPUT_FORMATS 21
#else
#define OUTPUT_FORMATS 18
#endif

// Step 3: fill in structure with ADD_FORMAT
//         supply literal name, symbol name, color format (w/o OUTPUT_ prefix)
struct OUTPUT_FORMAT_STRUCT output_formats[OUTPUT_FORMATS] = {
	ADD_FORMAT(jpg,    jpg, YUV422S),
	ADD_FORMAT(gif,    gif, PAL8),
	ADD_FORMAT(raw8,   raw, PAL8),
	ADD_FORMAT(raw16,  raw, RGB16),
	ADD_FORMAT(raw24,  raw, RGB24S),
	ADD_FORMAT(rawyuv, raw, YUV420),
	ADD_FORMAT(rawyuv422, raw, YUV422S),
	ADD_FORMAT(rawgrey, raw, GREY8),
	ADD_FORMAT(yuv,    yuv, YUV422S),
	ADD_FORMAT(pmm-fast,     pmm, PAL8),
	ADD_FORMAT(pmm-grey,     pmm, GREY8),
	ADD_FORMAT(pmm-bestgrey, pmm, GREY8),
	ADD_FORMAT(pmm-fastest,  pmm, PAL8),
	ADD_FORMAT(pmm-good, pmm, YUV422S),
	ADD_FORMAT(pmm-best, pmm, YUV422S),
	ADD_FORMAT(pgm, pnm, GREY8),
	ADD_FORMAT(ppm, pnm, RGB24),
#ifdef X11
	ADD_FORMAT(X8, X, PAL8),
	ADD_FORMAT(X16, X, RGB16),
	ADD_FORMAT(X24, X, RGB24),
#endif
	ADD_FORMAT(jpg8, jpg8, GREY8)
};

// Step 4: add the new object file to Makefile.in
// Step 5: create the source and header file, see output_jpg.c/h for an example

// finished
////////////////////////////////////////////////////////////////////////////////

START_OPTIONS
// STROPTION(webcam_name);
INTOPTION(width,0,MAXINT);
INTOPTION(height,0,MAXINT);
INTOPTION(file_fmt,0,OUTPUT_FORMATS-1);
OPTION(file_fmt nr) {
	OPTIONGET(sprintf(value,"%i",OUTPUT_FORMATS););
}
OPTION(file_fmt str) {
	OPTIONSET(int i;
		  for (i = 0; i < OUTPUT_FORMATS; i++) {
			  if (!strcmp(output_formats[i].name,value)) {
				  output_file_fmt = i;
				  break;
			  }
		  }
		  ASSERT(i < OUTPUT_FORMATS,(MSG("unknown output format '%s'"),value)););
	OPTIONGET(strcpy(value,output_formats[output_file_fmt].name););
}
OPTION(color_fmt) {
	OPTIONGET(strcpy(value,output_formats[output_file_fmt].color););
	OPTIONSET(ASSERT(!strcmp(value,output_formats[output_file_fmt].color),(ERRMSG("color format '%s' not supported for file format '%s'"),value,output_formats[output_file_fmt].name)););
}
INTOPTION(quality,0,200);
INTOPTION(fbs,0,MAXINT) {
	OPTIONSET(SAFE_FREE(output_buf);
		  SAFE_CALLOC(output_buf,output_fbs,sizeof(void *)););
}
PTROPTION(buf,output_fbs);
INTINFO(bytes);
END_OPTIONS

void (*output_write_func)(int fbnr, int picnr);
void (*output_stop_func)() = NULL;

void output_init()
{
	atexit(output_end);
	transport_init();
}

void output_start()
{
	int i,j,c;

	ASSERT(output_width,(INTMSG("unknown image width")));
	ASSERT(output_height,(INTMSG("unknown image height")));
	ASSERT(output_fbs,(INTMSG("image buffers not set")));

	c = 0;
	for (i = 0; i < OUTPUT_COLORS; i++) {
		if (!strcmp(output_formats[output_file_fmt].color,output_color_names[i])) {
			c = output_color_bpps[i];
			break;
		}
	}
	ASSERT(c,(INTMSG("unknown color format")));
	if (c > 0) output_bytes = (((output_width * c) + 7) / 8) * output_height;
	else output_bytes = (((output_width * -c) + 7) / 8) * output_height + (output_height%2) * output_width/2;

	SAFE_CALLOC(output_ptrs,output_fbs,sizeof(void *));
	for (j = 0; j < output_fbs; j++) {
		ASSERT(output_buf[j],(INTMSG("image buffer %i missing"),j));
		SAFE_MALLOC(((void *)output_ptrs[j]),output_height*sizeof(char *));
		for (i = 0; i < output_height; i++) {
			output_ptrs[j][i] = output_buf[j]+(i*output_bytes/output_height);
		}
	}

	
	output_write_func = output_formats[output_file_fmt].write;
	output_stop_func = output_formats[output_file_fmt].stop;

	(*output_formats[output_file_fmt].start)();
}

void output_stop()
{
	int i;
	if (output_stop_func) (*output_stop_func)();
	output_stop_func = NULL;

	transport_wstop();

	if (output_ptrs) {
		for (i = 0; i < output_fbs; i++) SAFE_FREE(output_ptrs[i]);
		SAFE_FREE(output_ptrs);
	}
}

void output_end()
{
	output_stop();
	SAFE_FREE(output_buf);
}

void output_write(int fbnr, int picnr)
{
	(*output_write_func)(fbnr,picnr);
}
