/* 
   bttvgrab 0.15.0 [1999-01-18]
   (c) 1998, 1999 by Joerg Walter <trouble@moes.pmnet.uni-oldenburg.de>
   Maintained by: Joerg Walter
   Current version at http://moes.pmnet.uni-oldenburg.de/bttvgrab/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

// see below for adding new transport modules

#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#define TRANSPORT_INTERNAL
#include "transport.h"
#include "error.h"
#include "configuration.h"
#define SYMBOL(x) transport_ ## x

int transport_unread_cnt = 0;
int transport_unread_data;
int transport_rformat = 0;
int transport_wformat = 0;
int transport_sync = 0;
char transport_rname[PATH_MAX] = "./";
char transport_wname[PATH_MAX] = "./";
char transport_rcname[PATH_MAX] = "./";
char transport_wcname[PATH_MAX] = "./";
char *transport_rdef = NULL, *transport_wdef = NULL;
int transport_rfd = -1;
int transport_wfd = -1;
FILE *transport_rfilep = NULL;
FILE *transport_wfilep = NULL;
pid_t transport_parent = 0;
int transport_rmulti = 1;
int transport_wmulti = 1;
struct transport_signature transport_rsig, transport_wsig;

 void (*transport_otrace)(const char *s);
 void (*transport_itrace)(const char *s);

/************ Prototypes ************/

void transport_init();
void transport_rdetect(char *format, int *width, int *height);
void transport_rmkdef();
void transport_wmkdef();
int transport_rstart(int multi);
int transport_wstart(const char *format, int width, int height, int multi);
void transport_rstop();
void transport_wstop();
void transport_rclose();
void transport_wclose();
void transport_end();
FILE *transport_fwframe(int nr);
FILE *transport_fwstart(const char *format, int width, int height, int multi);
int transport_wframe(int nr);
void transport_write(void *ptr, unsigned long cnt);
FILE *transport_frframe(int *nr);
FILE *transport_frstart(int multi);
int transport_rframe(int *nr);
void transport_read(void *ptr, unsigned long cnt);
void transport_rremove(int nr);
void transport_wremove(int nr);
void transport_write_int(int x);
int transport_read_int();
void transport_unread_int(int n);
void transport_unframe();

/************************************/

#define ADD_FORMAT(name,symbol) { \
	transport_ ## symbol ## _rstart, \
	transport_ ## symbol ## _wstart, \
	transport_ ## symbol ## _rframe, \
	transport_ ## symbol ## _wframe, \
	transport_ ## symbol ## _rremove, \
	transport_ ## symbol ## _wremove, \
	transport_ ## symbol ## _rstop, \
	transport_ ## symbol ## _wstop, \
	transport_ ## symbol ## _unframe, \
	transport_ ## symbol ## _rdetect, \
	#name \
}

struct TRANSPORT_FORMAT_STRUCT {
	void (*rstart)();
	void (*wstart)();
	void (*rframe)(int *nr);
	void (*wframe)(int nr);
	void (*rremove)(int nr);
	void (*wremove)(int nr);
	void (*rstop)();
	void (*wstop)();
	void (*unframe)();
	void (*rdetect)();
	char *name;
};

////////////////////////////////////////////////////////////////////////////////
///////////////////////// transport module registration ////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Step 1: include header file
#include "transport_file.h"
#include "transport_net.h"
#include "transport_webcam.h"

// Step 2: increase number of formats
#define TRANSPORT_FORMATS 3

// Step 3: fill in structure with ADD_FORMAT
//         supply literal name, symbol name
struct TRANSPORT_FORMAT_STRUCT transport_formats[TRANSPORT_FORMATS] = {
	ADD_FORMAT(file, file),
	ADD_FORMAT(webcam, webcam),
	ADD_FORMAT(net, net)
};

// Step 4: add the new object file to Makefile.in
// Step 5: create the source and header file, see transport_file.c/h for an example

// finished
////////////////////////////////////////////////////////////////////////////////

START_OPTIONS
STROPTION(rname);
STROPTION(wname);
INTOPTION(rformat,0,TRANSPORT_FORMATS-1);
INTOPTION(wformat,0,TRANSPORT_FORMATS-1);
OPTION(format nr) {
	OPTIONGET(sprintf(value,"%i",TRANSPORT_FORMATS););
}
OPTION(rformat str) {
	OPTIONSET(int i;
		  for (i = 0; i < TRANSPORT_FORMATS; i++) {
			  if (!strcmp(transport_formats[i].name,value)) {
				  transport_rformat = i;
				  break;
			  }
		  }
		  ASSERT(i < TRANSPORT_FORMATS,(MSG("unknown transport format '%s'"),value)););
	OPTIONGET(strcpy(value,transport_formats[transport_rformat].name););
}
OPTION(wformat str) {
	OPTIONSET(int i;
		  for (i = 0; i < TRANSPORT_FORMATS; i++) {
			  if (!strcmp(transport_formats[i].name,value)) {
				  transport_wformat = i;
				  break;
			  }
		  }
		  ASSERT(i < TRANSPORT_FORMATS,(MSG("unknown transport format '%s'"),value)););
	OPTIONGET(strcpy(value,transport_formats[transport_wformat].name););
}
INTOPTION(sync,0,1);
END_OPTIONS

void transport_init()
{
	if (transport_parent) return;
	transport_parent = getpid();
	atexit(transport_end);
}

void transport_rdetect(char *format, int *width, int *height)
{
	(*transport_itrace)(transport_rname);
	transport_rsig.version = TRANSPORT_CURRENT_VERSION;
	transport_rsig.width = *width;
	transport_rsig.height = *height;
	strcpy(transport_rsig.format,format);

	(*transport_formats[transport_rformat].rdetect)();

	ASSERT(transport_rsig.version == TRANSPORT_CURRENT_VERSION,(MSG("input stream is rev. %i, need rev. %i"),transport_rsig.version,TRANSPORT_CURRENT_VERSION));
	*width = transport_rsig.width;
	*height = transport_rsig.height;
	strcpy(format,transport_rsig.format);
}

void transport_rmkdef()
{
	transport_rdef = transport_rname+strlen(transport_rname);
	if (transport_rmulti) {
		strcat(transport_rname,"p_%04d.");
	} else {
		strcat(transport_rname,"output.");
	}
	strcat(transport_rname,transport_rsig.format);
}

void transport_wmkdef()
{
	transport_wdef = transport_wname+strlen(transport_wname);
	if (transport_wmulti) {
		strcat(transport_wname,"p_%04d.");
	} else {
		strcat(transport_wname,"output.");
	}
	strcat(transport_wname,transport_wsig.format);
}

int transport_rstart(int multi)
{
	transport_rmulti = multi;
	transport_unread_cnt = 0;
	(*transport_formats[transport_rformat].rstart)();
	return transport_rfd;
}

int transport_wstart(const char *format, int width, int height, int multi)
{
	(*transport_otrace)(transport_wname);
	transport_wmulti = multi;
	transport_wsig.version = TRANSPORT_CURRENT_VERSION;
	transport_wsig.width = width;
	transport_wsig.height = height;
	strcpy(transport_wsig.format,format);
	(*transport_formats[transport_wformat].wstart)();

	return transport_wfd;
}

void transport_rstop()
{
	(*transport_formats[transport_rformat].rstop)();
	transport_rclose();
	if (transport_rdef) *transport_rdef = 0;
}

void transport_wstop()
{
	(*transport_formats[transport_wformat].wstop)();
	transport_wclose();
	if (transport_wdef) *transport_wdef = 0;
}

void transport_rclose()
{
	if (transport_rfd >= 0) {
		if (transport_rfilep) {
			SAFE_FCLOSE(transport_rfilep,transport_rcname);
			transport_rfilep = NULL;
		} else {
			SAFE_CLOSE(transport_rfd,transport_rcname);
		}
		transport_rfd = -1;
	}
}

void transport_wclose()
{
	if (transport_wfd >= 0) {
		if (transport_wfilep) {
			SAFE_FCLOSE(transport_wfilep,transport_wcname);
			transport_wfilep = NULL;
		} else {
			SAFE_CLOSE(transport_wfd,transport_wcname);
		}
		transport_wfd = -1;
	}
}

void transport_end()
{
	transport_rstop();
	transport_wstop();
}

FILE *transport_fwframe(int nr)
{
	transport_wframe(nr);
	if (!transport_wfilep) NOTNULL((transport_wfilep = fdopen(transport_wfd,"w")),OPENMSG(transport_wcname));
	return transport_wfilep;
}

FILE *transport_fwstart(const char *format, int width, int height, int multi)
{
	transport_wstart(format, width, height, multi);
	if (!transport_wfilep && transport_wfd != -1) NOTNULL((transport_wfilep = fdopen(transport_wfd,"w")),OPENMSG(transport_wname));
	return transport_wfilep;
}

int transport_wframe(int nr)
{
	char tmp[PATH_MAX];

	if (transport_sync) fdatasync(transport_wfd);
	if (transport_wmulti) {
		transport_wclose();
		sprintf(transport_wcname, transport_wname, nr);
		sprintf(tmp,"%s [%i]",transport_wcname,nr);
	} else {
		sprintf(tmp,"%s [%i]",transport_wname,nr);
	}
	(*transport_otrace)(tmp);

	(*transport_formats[transport_wformat].wframe)(nr);

	return transport_wfd;
}

void transport_write(void *ptr, unsigned long cnt)
{
	int x;

	if (transport_wfilep) {
		while (cnt > 0) {
			ASSERT((x = fwrite(ptr,01,cnt,transport_wfilep)) > 0,WRITEMSG(transport_wcname));
			cnt -= x;
			ptr += x;
		}
	} else {
		while (cnt > 0) {
			ASSERT((x = write(transport_wfd,ptr,cnt)) >= 0,WRITEMSG(transport_wcname));
			cnt -= x;
			ptr += x;
		}
	}
}

FILE *transport_frframe(int *nr)
{
	transport_rframe(nr);
	if (!transport_rfilep) NOTNULL((transport_rfilep = fdopen(transport_rfd,"r")),OPENMSG(transport_rcname));
	return transport_rfilep;
}

FILE *transport_frstart(int multi)
{
	transport_rstart(multi);
	if (!transport_rfilep && transport_rfd != -1) NOTNULL((transport_rfilep = fdopen(transport_rfd,"r")),OPENMSG(transport_rname));
	return transport_rfilep;
}

int transport_rframe(int *nr)
{
	char tmp[PATH_MAX];

	if (transport_rmulti) {
		transport_rclose();
		sprintf(transport_rcname, transport_rname,*nr);
		sprintf(tmp,"%s [%i]",transport_rcname,*nr);
	} else {
		sprintf(tmp,"%s [%i]",transport_rname,*nr);
	}
	(*transport_itrace)(tmp);

	(*transport_formats[transport_rformat].rframe)(nr);

	return transport_rfd;
}

void transport_read(void *ptr, unsigned long cnt)
{
	int x;

	if (transport_rfilep) {
		while (cnt > 0) {
			x = fread(ptr,01,cnt,transport_rfilep);
			if (x <= 0){
				if (ferror(transport_rfilep)) ERROR READMSG(transport_rcname);
				else ERROR(MSG("end of file (read %s)"),transport_rcname);
				END(1);
			}
			cnt -= x;
			ptr += x;
		}
	} else {
		while (cnt > 0) {
			ASSERT((x = read(transport_rfd,ptr,cnt)) > 0,READMSG(transport_rcname));
			cnt -= x;
			ptr += x;
		}
	}
}

void transport_rremove(int nr)
{
	if (transport_rmulti) (*transport_formats[transport_rformat].rremove)(nr);
}

void transport_wremove(int nr)
{
	if (transport_wmulti) (*transport_formats[transport_wformat].wremove)(nr);
}

void transport_write_int(int x)
{
	unsigned char c;

	c = x & 0xff;
	transport_write(&c,1);
	x >>= 8;
	c = x & 0xff;
	transport_write(&c,1);
	x >>= 8;
	c = x & 0xff;
	transport_write(&c,1);
	x >>= 8;
	c = x & 0xff;
	transport_write(&c,1);
}

int transport_read_int()
{
	unsigned char c;
	int x;

	if (transport_unread_cnt) {
		transport_unread_cnt = 0;
		return transport_unread_data;
	}

	transport_read(&c,1);
	x = c;
	transport_read(&c,1);
	x |= c << 8;
	transport_read(&c,1);
	x |= c << 16;
	transport_read(&c,1);
	x |= c << 24;

	return x;
}

void transport_unread_int(int n)
{
	transport_unread_cnt = 1;
	transport_unread_data = n;
}

void transport_unframe()
{
	(*transport_formats[transport_rformat].unframe)();
}
