/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     confirm.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function calls
*
*  Description:   This file holds the function check_streams
**************************************************************************/

#include <fcntl.h>
#include "list.h"

/* function prototypes */
LINK next_node(LINK);
INT hex_compare(CHAR *, CHAR *, INT);

/*************************************************************************
*  Function:      check_streams()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         stream linked list, number of video streams,
*                 number of audio streams
*
*  Effects:       returns a stream linked list with more info
*
*  Description:   This function confirms if a file is actually an MPEG
*  A/V stream, then, if it is, it finds the picture or audio rate,
*  otherwise, it stops the program.
**************************************************************************/

LINK check_streams(LINK streams_list,INT *num_video,INT *num_audio)
{
  INT fd, n;

/* video stream = 1110 xxxx */
  CHAR video_stream_id = {0xe0};

/* audio stream = 110x xxxx */
  CHAR audio_stream_id = {0xc0};
  CHAR test_header[VIDEOHEADERSIZE];

/* MPEG Video: sequence_header_code */
  CHAR video_header[VIDEOHEADERSIZE]={0x00,0x00,0x01,0xb3};

/* MPEG Audio: syncword */
  CHAR audio_header[AUDIOHEADERSIZE]={0xff,0xf0};
  LINK current_stream;

  current_stream = streams_list;           /* point to first node in list    */
  while (current_stream != NULL)           /* while we aren't at end of list */
  {
    if ((fd=open(current_stream->filename,O_RDONLY))<0) /* open stream       */
    {
      printf("Error, can't find: %s\n",current_stream->filename); /* on err  */
      exit(1);                             /* print err msg and exit program */
    }

    else
    {
      if ((n=read(fd,test_header,VIDEOHEADERSIZE)<0))  /* read stream header */
      {
	printf("Error reading %s\n",current_stream->filename); /* on err     */
	exit(1);                           /* print err msg and exit program */
      }

      if (!hex_compare(test_header,video_header,VIDEOHEADERSIZE)) 
	                                   /* test if header is video        */
      {
	current_stream->isvideo=1;         /* set isvideo bit                */
	                                   /* tell user that stream video    */
	++(*num_video);                    /* incrmnt # of video streams     */

	if (*num_video > MAXVIDEO)         /* if we have too many            */
	{
	  printf("Too many video streams.\n"); /* stop program               */
	  exit(1);
	}    

	current_stream->stream_id=video_stream_id; /* create stream id       */
	++video_stream_id;                 /* incrmnt for next video stream  */

/* find picture rate */
	if ((n=read(fd,test_header,VIDEOHEADERSIZE)<0)) /* read next 32 bits */
	{
	  printf("Error reading %s\n", current_stream->filename); /* on err  */
	  exit(1); /* print error message and exit program */
	}

	else
	{
	  test_header[3] = test_header[3] & 0x0f; /* flush all but the last
						     4 bits which are the
						     picture_rate            */

	  switch(test_header[3])                  /* find out value          */
	  {
	  case 0x00:
	    printf("Forbidden picture_rate value: %s\n", /* Binary 0000    */
		   current_stream->filename);            /* forbidden      */
	    break;
	  case 0x01:
	    current_stream->rate=23.976;                 /* Binary 0001    */
	    break;                                       /* 23.976 fps     */
	  case 0x02:
	    current_stream->rate=24;                     /* Binary 0010    */
	    break;                                       /* 24 fps         */
	  case 0x03:
	    current_stream->rate=25;                     /* Binary 0011    */
	    break;                                       /* 25 fps         */
	  case 0x04:
	    current_stream->rate=29.97;                  /* Binary 0100    */
	    break;                                       /* 29.97 fps      */
	  case 0x05:
	    current_stream->rate=30;                     /* Binary 0101    */
	    break;                                       /* 30 fps         */
	  case 0x06:
	    current_stream->rate=50;           /* Binary 0110              */
	    break;                             /* 50 fps                   */
	  case 0x07:
	    current_stream->rate=59.94;        /* Binary 0111              */
	    break;                             /* 59.94 fps                */
	  case 0x08:
	    current_stream->rate=60;           /* Binary 1000              */
	    break;                             /* 60 fps                   */
	  default:
	    printf("Reserved value for picture_rate: %s\n", /* all other   */
		   current_stream->filename);  /* values are               */
	    exit(1);                           /* reserved                 */
	    break;
	  }
	}
      }

      else
      {
	test_header[0] = test_header[0] & 0xff; /* look at first 12 bits   */
	test_header[1] = test_header[1] & 0xf0;
	
	if (!hex_compare(test_header,audio_header,AUDIOHEADERSIZE))
	                                      /* test if header is audio   */
	{
	  current_stream->isvideo=0;          /* reset isvideo bit         */
	  /* tell the user that current stream is MPEG audio */
	  ++(*num_audio);                     /* incrmnt # of audio        */
   
	  if (*num_audio > MAXAUDIO)          /* if we have too many       */
	  {
	    printf("Too many audio streams.\n"); /* stop program           */
	    exit(1);
	  }

	  current_stream->stream_id=audio_stream_id; /* create stream id   */
	  ++audio_stream_id;                  /* incrmnt for next audio    */

/* find sampling frequency */
	  test_header[2]=test_header[2]&0x0c; /* strip all but 2 bits      */
	                                      /* which are the             */
	                                      /* sampling_frequency        */
	  switch(test_header[2])              /* determine value of bits   */
	  {
	  case 0x00:
	    current_stream->rate = 44100;     /* Binary 00                 */
	    break;                            /* 44.1 Khz                  */
	  case 0x04:
	    current_stream->rate = 48000;     /* Binary 01                 */
	    break;                            /* 48 Khz                    */
	  case 0x08:
	    current_stream->rate = 32000;     /* Binary 10                 */
	    break;                            /* 32 Khz                    */
	  default:
	    printf("Reserved value for sampling_frequency: %s\n",
		   current_stream->filename); /* Binary 1100, reserved     */
	    exit(1);
	    break;
	  }
      	}

      	else /* otherwise, file is non-MPEG */
	{
	  printf("Error, unrecognized file format: %s\n",
		 current_stream->filename);
	  exit(1); /* print error message and exit */
	}    

      }

      close(fd); /* close stream */
    }

    current_stream = next_node(current_stream);
  }

  return streams_list;
}
