/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     hex.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function file
*
*  Description:   This file contains the functions that manipulate strings
*  as if they were long hex values
**************************************************************************/

#include "defines.h"

/*************************************************************************
*  Function:      hex_compare()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         string 1, string 2, # chars to compare
*
*  Effects:       returns 1 if equal, 0 if not equal
*
*  Description:   This function compares two string for equality.
**************************************************************************/

INT hex_compare(CHAR *str1, CHAR *str2, INT n)
{
  INT EQUALITY = 0;
  INT i = 0;

  for (i=0; i<n; i++)                           /* while both strings left */
  {
    if (str1[i] != str2[i])                     /* if current bytes don't  */
    {                                           /* match, set EQUALITY and */
      EQUALITY = 1;                             /* break out of loop       */
      break;
    }
  }

  return EQUALITY;
}

/*************************************************************************
*  Function:      scr_reset()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         system clock reference
*
*  Effects:       none
*
*  Description:   This function sets the system clock reference to it's
*  maximum possible value.
**************************************************************************/

void scr_reset(CHAR *scr)
{
  scr[0] = 0x2f;                                /* set SCR to highest val  */
  scr[1] = 0xff;
  scr[2] = 0xff;
  scr[3] = 0xff;
  scr[4] = 0xff;
}

/*************************************************************************
*  Function:      stamp_compare()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         time stamp 1, time stamp 2
*
*  Effects:       returns 1 if 1>2, -1 if 1<2, 0 1=2
*
*  Description:   This function compares two 5 CHAR strings as if they
*  were 33-bit MPEG time stamps (SCR format assumed).
**************************************************************************/

INT stamp_compare(CHAR *str1, CHAR *str2)
{
  INT i, equality = 0;

  for (i = 0; i < 5; i++)                       /* loop through string     */
  {
    if (str1[i] > str2[i])                      /* if str1 > str2          */
    {
      equality = 1;                             /* set equality = 1        */
      break;                                    /* break out of loop       */
    }

    else if (str1[i] < str2[i])                 /* if str1 < str2          */
    {
      equality = -1;                            /* set equality = -1       */
      break;                                    /* break out of loop       */
    }
  }  

  return equality;                              /* return equality         */
}

/*************************************************************************
*  Function:      stamp_copy()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         destination stamp, source stamp
*
*  Effects:       none
*
*  Description:   This function copies one stamp to another.
**************************************************************************/

void stamp_copy(CHAR *dest, CHAR *source)
{
  INT i;

  for (i = 0; i < 5; i++)                       /* loop through stamp      */
    dest[i] = source[i];                        /* copy CHAR i             */
}

/*************************************************************************
*  Function:      hex_left_shift()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         string, length of string, # chars to shift
*
*  Effects:       none
*
*  Description:   This function does a binary shift of a string by n bytes.
**************************************************************************/

void hex_left_shift(CHAR *str, INT length, INT n)
{
  INT i,j;

  for (i=0; i<n; i++)                           /* shift string n times    */
  {
    for (j=1; j<length; j++)                    /* cycle though string     */
      str[j-1] = str[j];                        /* shifting each CHAR      */
    str[j-1] = 0x00;                            /* and clearing last CHAR  */
  }
}

/*************************************************************************
*  Function:      stamp_add()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         time stamp, unsigned integer
*
*  Effects:       none
*
*  Description:   This function adds n to an MPEG stamp (5-byte string,
*  SCR format).
**************************************************************************/

void stamp_add(CHAR *time_stamp, unsigned INT n)
{
  unsigned CHAR stamp[5];
  unsigned INT i;

  stamp[0] = (time_stamp[0] & 0x08) >> 3;       /* extract actual time     */
  stamp[1] = ((time_stamp[0] & 0x06) << 5) |    /* from time stamp form    */
             ((time_stamp[1] & 0xfc) >> 2);
  stamp[2] = ((time_stamp[1] & 0x03) << 6) |
             ((time_stamp[2] & 0xfc) >> 2);
  stamp[3] = ((time_stamp[2] & 0x02) << 6) |
             ((time_stamp[3] & 0xfe) >> 1);
  stamp[4] = ((time_stamp[3] & 0x01) << 7) |
             ((time_stamp[4] & 0xfe) >> 1);

  for(i=0; i < n; i++)                          /* add 1 n times to stamp  */
  {
    if (stamp[4] == 0xff)                       /* if least byte full      */ 
      if (stamp[3] == 0xff)                     /* if next byte full       */
	if (stamp[2] == 0xff)                   /* if next byte full       */
	  if (stamp[1] == 0xff)                 /* if next byte full       */
	    if (stamp[0] == 0x01)               /* if last byte full       */
	    {
	      printf("Time stamp overflow error.\n"); /* time stamp full   */
	      exit(1);                          /* print error and exit    */
	    }

            else
	    {
	      stamp[4] = 0x00;
	      stamp[3] = 0x00;
	      stamp[2] = 0x00;
	      stamp[1] = 0x00;
	      ++stamp[0];
	    }

          else
	  {
	    stamp[4] = 0x00;
	    stamp[3] = 0x00;
	    stamp[2] = 0x00;
	    ++stamp[1];
	  }

        else
	{
	  stamp[4] = 0x00;
	  stamp[3] = 0x00;
	  ++stamp[2];
	}

      else
      {
	stamp[4] = 0x00;
	++stamp[3];
      }

    else ++stamp[4];
  }
 
  time_stamp[0] = time_stamp[0] & 0xf1;         /* clear time bits from time */
  time_stamp[1] = time_stamp[1] & 0x00;         /* stamp in prep. for adding */
  time_stamp[2] = time_stamp[2] & 0x01;         /* the new bits              */
  time_stamp[3] = time_stamp[3] & 0x00;
  time_stamp[4] = time_stamp[4] & 0x01;

/* add time data to time_stamp */
  time_stamp[0] = time_stamp[0] | (((stamp[0]&0x01)<<3)|(stamp[1]&0xc0)>>5);
  time_stamp[1] = time_stamp[1] | (((stamp[1]&0x3f)<<2)|(stamp[2]&0xc0)>>6);
  time_stamp[2] = time_stamp[2] | (((stamp[2]&0x3f)<<2)|(stamp[3]&0x80)>>6);
  time_stamp[3] = time_stamp[3] | (((stamp[3]&0x7f)<<1)|(stamp[4]&0x80)>>7);
  time_stamp[4] = time_stamp[4] | ((stamp[4]&0x7f)<<1);
}

/*************************************************************************
*  Function:      itostr()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         string, integer
*
*  Effects:       none
*
*  Description:   This function takes an integer value and converts it
*  into a string.
**************************************************************************/
void itostr(CHAR *output, INT *input, INT length)
{
  INT i;

  for(i=length-1; i>-1; i--)
    output[i] = (CHAR)(*input >> (8*((length-1)-i)));
}

/*************************************************************************
*  Function:      make_mux()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         string mux rate, integer mux rate
*
*  Effects:       none
*
*  Description:   This function takes an integer value for a mux rate and
*  encodes it as a string which can be written to a file as an MPEG file
*  format mux rate.
**************************************************************************/

void make_mux(CHAR *mux_rate, INT mux)
{
  CHAR copy[4];;

  mux = (INT)(mux/50);                              /* get a value in 50bps  */
  
  mux_rate[2] = (CHAR)(mux);                        /* convert mux to string */
  mux_rate[1] = (CHAR)(mux >> 8);
  mux_rate[0] = (CHAR)(mux >> 16);

  copy[2] = mux_rate[2];                            /* make a backup copy    */
  copy[1] = mux_rate[1];
  copy[0] = mux_rate[0];

  mux_rate[2] = ((copy[2] & 0x7f)<<1) | 0x01;       /* put mux in mux_rate   */
  mux_rate[1] = ((copy[2] & 0x80)>>7) | ((copy[1] & 0x7f)<<1); /* form       */
  mux_rate[0] = ((copy[1] & 0x80)>>7) | ((copy[0] & 0x3f)<<1) | 0x80;
}



