/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     list.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function calls
*
*  Description:   A series of functions for creating, handling, and deleting
*  linked lists
**************************************************************************/

#include <stdlib.h>
#include "list.h"

/*************************************************************************
*  Function:      create_waitlist()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         none
*
*  Effects:       initializes a linked list
*
*  Description:   This function creates a linked list that will hold all
*  the wait times for each I or P frame (PTS vs DTS) found in the current
*  stream
**************************************************************************/

WAIT_LINK create_waitlist()
{
  WAIT_LINK head;

  head = malloc(sizeof(WAIT_ELEMENT));       /* Create head                 */
  head->wait_time = 0;                       /* set first frame wait to zero*/
  head->next = NULL;                         /* head points to NULL for now */
  return head;
}

/*************************************************************************
*  Function:      create_list()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         none
*
*  Effects:       initializes a linked list
*
*  Description:   This function creates a linked list that will hold all
*  all the information on one A/V stream, including pointers to the file-
*  offsets, wait times, and frame types
**************************************************************************/

LINK create_list()
{
  LINK head;

  head = malloc(sizeof(ELEMENT));            /* Create head                 */
  head = NULL;                               /* head points to NULL for now */
  return head;
}

/*************************************************************************
*  Function:      create_sublist()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         none
*
*  Effects:       initializes a linked list
*
*  Description:   This function creates a linked list that will hold all
*  all the frame type and frame file offset information for a single
*  A/V stream
**************************************************************************/

SUB_LINK create_sublist()
{
  SUB_LINK head;

  head = malloc(sizeof(SUB_ELEMENT));        /* Create head                 */
  head -> next = NULL;                       /* head points to NULL for now */
  return head;
}

/*************************************************************************
*  Function:      add_node()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       add a node to the stream linked list
*
*  Description:   This function creates a node with stream information and
*  adds it to the linked list which holds all the streams' information
**************************************************************************/

LINK add_node(LINK head)
{
  LINK new_node;
/*  CHAR pts[5] = {0x21,0x00,0x01,0x00,0x01};  */ /* set's a zero values PTS     */

  new_node = malloc(sizeof(ELEMENT));        /* Create a new node           */

  new_node->framewritten = 0;                /* set amount written = 0      */
  new_node->newframe = 1;                    /* start encoding w/new frame  */
  new_node->current_offset = 0;              /* reset current_offset        */
  new_node->isfinished = 0;                  /* reset isfinished            */
  new_node->last_pts[0] = 0x21;              /* set first PTS               */
  new_node->last_pts[1] = 0x00;
  new_node->last_pts[2] = 0x01;
  new_node->last_pts[3] = 0x00;
  new_node->last_pts[4] = 0x01;

  new_node->buffer_change = 1;               /* unset bit that tells wether */
                                             /* an STD values have been     */
                                             /* for the first time only     */
  new_node -> next = head;                   /* insert new node between     */
  head = new_node;                           /* head and the next node      */

  return head;
}

/*************************************************************************
*  Function:      del_node()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       delete a node to the stream linked list
*
*  Description:   This function deletes a node with stream information from
*  the linked list which holds all the streams' information
**************************************************************************/

LINK del_node(LINK head)
{
  LINK temp_ptr;

  temp_ptr = head;

  if (head->next != NULL)
    head = head->next;

  else
    head = NULL;

  free(temp_ptr);

  return head;
}

/*************************************************************************
*  Function:      add_offset()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list, offset value, frame type
*
*  Effects:       add a node to the offset linked list
*
*  Description:   This function adds a node with frame type and file offset
*  information to the offset linked list which holds all this information
*  for that particular stream.
**************************************************************************/

SUB_LINK add_offset(SUB_LINK head, INT data, CHAR frame_type)
{
  SUB_LINK new_node, current_node;

  current_node = head;
  while (current_node -> next != NULL)       /* Get to end of sub-list      */
    current_node = current_node -> next;
 
  new_node = malloc(sizeof(SUB_ELEMENT));    /* Create a new node           */
  new_node -> offset = data;                 /* store offset                */
  new_node -> frame_type = frame_type;       /* store frame_type            */
  new_node -> next = NULL;                   /* insert new nod between      */
  current_node -> next = new_node;           /* head and the next node      */
  return head;
}

/*************************************************************************
*  Function:      del_offset()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       delete a node to the offset linked list
*
*  Description:   This function deletes a node with frame type and file offset
*  information from the offset linked list which holds all this information
*  for a particular stream.
**************************************************************************/

SUB_LINK del_offset(SUB_LINK head)
{
  SUB_LINK temp_ptr;

  temp_ptr = head;
  if (head->next != NULL)
    head = head->next;

  else
    head = NULL;

  free(temp_ptr);

  return head;
}

/*************************************************************************
*  Function:      add_wait()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list, wait time
*
*  Effects:       add a node to the wait time linked list
*
*  Description:   This function adds a node with I and P wait time information 
*  (PTS vs DTS) to the linked list which holds all this information for a
*  particular stream.
**************************************************************************/

WAIT_LINK add_wait(WAIT_LINK head, INT wait_time)
{
  WAIT_LINK new_node, current_node;

  current_node = head;
  while (current_node->next != NULL)         /* Get to end of sub-list      */
    current_node = current_node->next;

  new_node = malloc(sizeof(WAIT_ELEMENT));   /* Create a new node           */
  new_node ->wait_time = wait_time;          /* store wait time             */
  new_node -> next = NULL;                   /* insert new node between     */
  current_node -> next = new_node;           /* head and the next node      */
  return head;
}

/*************************************************************************
*  Function:      del_wait()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       delete a node from the wait time linked list
*
*  Description:   This function deletes a node with I and P wait time 
*  information (PTS vs DTS) from the linked list which holds all this
*  information for a particular stream.
**************************************************************************/

WAIT_LINK del_wait(WAIT_LINK head)
{
  WAIT_LINK temp_ptr;

  temp_ptr = head;
  if (head->next != NULL)
    head = head->next;

  else
    head = NULL;

  free(temp_ptr);

  return head;
}

/*************************************************************************
*  Function:      next_node()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       returns a pointer to the next node in the list
*
*  Description:   This function takes a pointer to the streams list and
*  returns a pointer to the next node in the streams list 
**************************************************************************/

LINK next_node(LINK head)
{
  LINK *next_node;

  next_node = &(head -> next);               /* point to the next node      */
  return *next_node;
}

/*************************************************************************
*  Function:      print_list()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list
*
*  Effects:       prints stream node information
*
*  Description:   This function takes a pointer to a stream node and prints
*  MPEG header information that is contained in the node.
**************************************************************************/

void print_list (LINK head)
{
  if (head == NULL);                         /* if end of list, quit        */
  else                                       /* otherwise, print info       */
  {
    print_list(head->next);
    if (head->isvideo) printf("\nMPEG Video Stream\n");
    else printf("\nMPEG Audio Steam\n");
    printf("Filename        : %s\n",head->filename);
    printf("Start Time      : %0.2f sec.\n",head->startime);
    if (head->isvideo) 
      printf("Rate            : %0.2f frames/sec.\n",head->rate);
    else printf("Rate            : %0.0f samples/sec.\n",head->rate);
    printf("# of Frames     : %d\n",head->no_frames);
    printf("Max. Frame Size : %d bytes\n", head->max_frame_size);
  }
}
