/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     offset.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function calls
*
*  Description:   This file hold the function find_offsets
**************************************************************************/

#include <stdio.h>
#include <fcntl.h>
#include "list.h"

/* function prototypes */
SUB_LINK create_sublist(void);
WAIT_LINK create_waitlist(void);
SUB_LINK add_offset(SUB_LINK, INT, CHAR);
WAIT_LINK add_wait(WAIT_LINK, INT);
LINK next_node(LINK);
void stamp_add(CHAR *, unsigned INT);
void read_bytes(size_t *, size_t *, INT, CHAR *, CHAR *, INT);
INT hex_compare(CHAR *, CHAR *, INT);
void hex_left_shift(CHAR *, INT, INT);
void update_buffer(size_t *, size_t *, INT, CHAR *);
CHAR type_frame(CHAR);

/*************************************************************************
*  Function:      find_offsets()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         linked list, maximum mux rate, maximum buffer size
*
*  Effects:       goes through every A/V stream finding all frames
*
*  Description:   This function takes a linked list of all the A/V streams
*  in this system stream and for each stream, creates a list of that
*  stream's frames' location.  For I and P video frames, it also finds the
*  amount of time they must wait between decoding and presentation, by
*  finding how many B frames are located between.  It also finds the
*  largest frame in each stream and computes the individual stream mux
*  rate, as well as the maximum overall mux rate.
**************************************************************************/

LINK find_offsets(LINK streams_list, INT *max_mux_rate, INT *max_buff_size)
{
  INT fd, /* n, */ offset = 0, I_wait = 0, special = 0;
  INT current_mux, samples_per_frame = 0;
  INT frame = 0 /* , max_frame_size = 0*/ ;
  CHAR buffer[VIDEOHEADERSIZE], buffer2[AUDIOHEADERSIZE];
  CHAR readbuffer[READSIZE];
  size_t readsize = READSIZE, i = READSIZE;
  CHAR picture_start_code[4] = {0x00,0x00,0x01,0x00};
  CHAR audio_frame[AUDIOHEADERSIZE];
  CHAR image_type;
  CHAR sequence_end_code[4] = {0x00,0x00,0x01,0xb7};
  LINK current_stream;

  current_stream = streams_list;             /* point to first node         */

  while (current_stream != NULL)             /* while not at end of list    */
  {
    printf("\nProcessing %s\n",current_stream->filename);
    current_stream->scsr =                   /* calculate SCPR or SCASR     */
      CLOCKFREQUENCY/current_stream->rate;
    current_stream->next_frame_time =        /* calculate startime in terms */
      (unsigned INT)(CLOCKFREQUENCY*current_stream->startime); 
                                             /* of system clock             */
    stamp_add(current_stream->last_pts, current_stream->next_frame_time);
                                             /* set last_pts to starttime   */


    if ((fd=open(current_stream->filename,   /* open current stream file    */
	 O_RDONLY))<0)
    {
      printf("Error, can't find %s\n",       /* on error print error msg    */
	     current_stream->filename);
      exit(1);                               /* and exit program            */
    }

    current_stream->offsets=create_sublist();/* create sub-list for offsets */

/* Video Stream */
    if (current_stream->isvideo)             /* if current stream is video  */
    {

      current_stream->waits=create_waitlist(); /* create wait-list for I P  */

/* find first frame */
      read_bytes(&i,&readsize,fd,readbuffer,buffer,VIDEOHEADERSIZE); /*read */
                                             /* first 4 bytes               */

      do
       {
	hex_left_shift(buffer,VIDEOHEADERSIZE,1); /* shift buffer to left   */

	update_buffer(&i, &readsize, fd, readbuffer); /* update readbuffer  */
        buffer[3] = readbuffer[i++];         /* read in the next byte       */
	++offset;                            /* increment offset, byte read */
      } while(hex_compare(buffer,picture_start_code,VIDEOHEADERSIZE));
                                             /* till picture start is found */

/* find frame type for first frame */
      read_bytes(&i,&readsize,fd,readbuffer,buffer,VIDEOHEADERSIZE); /* read*/
                                             /* next 4 bytes                */

      image_type = buffer[1] & 0x38;         /* flush all but 3 bits        */
      image_type=type_frame(image_type);     /* determine image type        */
      offset = offset + 4;                   /* increment offset for bytes  */

/* find the rest of the frames */
      do
      {
	
	if (readsize == 0)                   /* if unexpected EOF occures   */
	{
	  printf("Unexpected EOF in %s\n",current_stream->filename);/* print*/
	  exit(1);                           /* error message and exit      */
	}

	hex_left_shift(buffer,VIDEOHEADERSIZE,1); /* shift buffer to left   */

	buffer[3] = readbuffer[i++];         /* read in next byte           */
	update_buffer(&i, &readsize, fd, readbuffer);

	++offset;                            /* increment offset, byte read */
	if (!hex_compare(buffer,picture_start_code,VIDEOHEADERSIZE))/* if   */
	{                                    /* current buffer=start frame  */
	  current_stream->offsets=add_offset(current_stream->offsets,
					     offset,image_type);
	  ++frame;                           /* increment frame counter     */
	  if (offset > current_stream->max_frame_size) /* store greatest    */
	    current_stream->max_frame_size=offset; /* file offset as max.   */

	  offset = 0;                        /* reset offset                */
          if (!(frame % UPDATEPERIOD))       /* every ten frames print some */
	  {                                  /* update message and fflush   */
	    printf("\r%d frames processed",frame); /* to force a printout   */
	    fflush(stdout);
	  }

/* find frame type */
	  read_bytes(&i,&readsize,fd,readbuffer,buffer,VIDEOHEADERSIZE);
	                                     /* read next 4 bytes           */

	  image_type = buffer[1] & 0x38;     /* strip all but 3 bits        */
	  image_type=type_frame(image_type); /* determine image type        */
          if (image_type=='I'||image_type=='P') /* if special frame         */
	  {
	    if (special==1)                  /* if second I or P frame      */
	    {
              current_stream->waits=add_wait(current_stream->waits,I_wait);
                                             /* add the current I_wait      */
	      I_wait = 0;                    /* reset I_wait                */
	    }

            else special = 1;                /* set special                 */
	  }

          else ++I_wait;                     /* if B frame, incr. I_wait    */

	  offset = offset + 4;
	}

      } while(hex_compare(buffer,sequence_end_code,VIDEOHEADERSIZE));
                                             /* repeat till end of stream   */
      current_stream->offsets = add_offset(current_stream->offsets,
					   offset,image_type);
                                             /* add last offset to sublist  */
      ++frame;                               /* icrement for last frame     */
      current_stream->no_frames = frame;     /* store number of frames      */
      printf("\r%d frames processed",frame); /* print last update info      */
      fflush(stdout);
      current_stream->waits=add_wait(current_stream->waits,I_wait);
      I_wait = 0;                            /* add last wait and reset     */
      printf("\n");                          /* print a CRLF                */
      if (offset > current_stream->max_frame_size) /* store greatest file   */
	current_stream->max_frame_size=offset; /* offset as max.            */
      offset = 0;                            /* reset offset                */
      frame = 0;                             /* reset frame                 */
    }

/* Audio Stream */
    else                                     /* otherwise, it's an audio    */
    {

      read_bytes(&i,&readsize,fd,readbuffer,audio_frame,AUDIOHEADERSIZE);
                                             /* read frame header           */
      
      read_bytes(&i,&readsize,fd,readbuffer,buffer2,AUDIOHEADERSIZE);
                                             /* read first 2 bytes          */

      offset = offset + 2;                   /* increment offset for 2 bytes*/

      do
      {
	hex_left_shift(buffer2,AUDIOHEADERSIZE,1); /* shift buffer2 1 byte  */

        buffer2[1] = readbuffer[i++];        /* read in next byte           */
        update_buffer(&i, &readsize, fd, readbuffer); /* update buffer      */

	++offset;                            /* increment offset for byte   */

	if (!hex_compare(buffer2,audio_frame,AUDIOHEADERSIZE)) /* if buffer */
	{                                    /* is an audio frame then      */
	  ++frame;                           /* increment number of frames  */
	  current_stream->offsets=add_offset(current_stream->offsets,
					     offset,'A');
	                                     /* store offset in sublist     */

	  if (offset>current_stream->max_frame_size) /* store greatest      */
	    current_stream->max_frame_size=offset; /* offset as max         */

	  offset = 0;                        /* reset offset                */

	  if (!(frame % UPDATEPERIOD))       /* update the display every 10 */
	  {                                  /* frames                      */
	    printf("\r%d frames processed",frame);
	    fflush(stdout);

	  }

	}

      } while(readsize);                     /* read until EOF              */

      current_stream->offsets = add_offset(current_stream->offsets,offset,'A');
                                             /* store last offset           */
      ++frame;                               /* increment no. of frames     */
      current_stream->no_frames = frame;     /* store no. of frames         */
      printf("\r%d frames processed",frame); /* print last update info      */
      fflush(stdout);
      printf("\n");                          /* print a CRLF                */
      if (offset > current_stream->max_frame_size) /* store greatest offset */
	current_stream->max_frame_size=offset; /* as max                    */

      offset = 0;                            /* clear offset                */
      frame = 0;                             /* clear no. of frames         */
    }
    
    close(fd);                               /* close current stream        */
                                            
    if(current_stream->isvideo)              /* for video stream            */
      current_mux =                          /* get current stream mux rate */
	(INT)((current_stream->max_frame_size*CLOCKFREQUENCY)/
	      current_stream->scsr);

    else                                     /* for audio stream            */
    {
      audio_frame[1] = audio_frame[1] & 0x06; /* stip out all but layer bits*/

      switch(audio_frame[1])                 /* find out which layer        */
      {
      case 0x06:                             /* 11 Layer I                  */
	samples_per_frame = 384;             /* 384 samples per frame       */
	break;
      case 0x04:                             /* 10 Layer II                 */
	samples_per_frame = 1152;            /* 1152 samples per frame      */
	break;
      case 0x02:                             
	samples_per_frame = 1152;            /* 01 Layer III                */
	break;                               /* 1152 samples per frame      */
      default:
	printf("Unknown audio layer in %s\n",current_stream->filename);
	exit(1);
	break;
      }

                                             /* calculate audio mux         */
      current_mux =
	(INT)((current_stream->max_frame_size*CLOCKFREQUENCY)/
	      (current_stream->scsr)/(samples_per_frame));

      current_stream->scsr =                /* calculate new SCSR           */
	current_stream->scsr * samples_per_frame;
    }

    current_stream->stream_mux_rate = current_mux; /* store current mux rate*/
                                             /* increment max_mux rate      */
    *max_mux_rate = *max_mux_rate + current_mux;

                                             /* get max. buffer size        */
    *max_buff_size = *max_buff_size + current_stream->max_frame_size;

    current_stream = next_node(current_stream); /* goto next stream         */
  }

  return streams_list;
} 
