/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     pack.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function calls
*
*  Description:   This file contains the function write_pack
**************************************************************************/

#include "pack.h"
#include "list.h"
#include "mta.h"

/* function prototypes */
void make_mux(CHAR *, INT);
void itostr(CHAR *, void *, INT);

/*************************************************************************
*  Function:      write_pack()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         multiplex timing array, stream linked list, output file,
*                 system clock reference, max mux rate, number of audio
*                 streams, number of video streams, flag: does this pack
*                 need a system header, number of bytes left in the pack,
*                 fixed pack size
*
*  Effects:       returns an updated multiplex timing array
*
*  Description:   This function writes the necessary pack header info.
*  and updates the needed variables.
**************************************************************************/

MTA_LINK write_pack(MTA_LINK mta, LINK system_streams, INT output, CHAR *scr, 
		    INT max_mux_rate, INT num_audio, INT num_video, 
		    INT *isheader, INT *pack_bytes_left, INT pack_size)
{

  CHAR pack_start_code[4]={0x00,0x00,0x01,0xba};
  CHAR system_header_start_code[4]={0x00,0x00,0x01,0xbb};
  SHORT header_length = PACKHEADERLENGTH;
  CHAR header_length_data[2];
  CHAR mux_rate[3];
  CHAR reserved_byte = RESERVEDBYTE;
  LINK current_stream;
  struct data1 audio_bound_data;
  struct data2 video_bound_data;
  struct buffer buff_info;
#if LITTLE_ENDIAN
  CHAR buff_info_data[2];
#endif
  CHAR pad_id = 0xbe;

  if (pack_size) *pack_bytes_left = *pack_bytes_left-12; /* subtract         */
                                                 /* for pack header          */

/* write pack_start_code */
  if(write(output,pack_start_code,4) != 4)       /* write pack start code    */
  {
    printf("Error writing to output\n");         /* on error                 */
    exit(1);                                     /* print err msg and exit   */
  }

/* write system_clock_reference */
  if(write(output,scr,5) != 5)                   /* write SCR time stamp     */
  {
    printf("Error writing to output\n");         /* on error                 */
    exit(1);                                     /* print err msg and exit   */
  }

/* write mux_rate */
  make_mux(mux_rate,max_mux_rate);               /* create mux_rate from INT */
  if(write(output,mux_rate,3) != 3)              /* write mux_rate           */
  {
    printf("Error writing to output\n");         /* on error                 */
    exit(1);                                     /* print err msg and exit   */
  }

/* write SYSTEM HEADER */
  if (*isheader)                                 /* if system header needed  */
  {

    if(pack_size) *pack_bytes_left = *pack_bytes_left - 12;  /* subtract     */
                                                 /* for system header        */

/* write system_header_start_code */
    if(write(output,system_header_start_code,4) != 4) /* write s_h_s_c       */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }

/* write header_length */
    header_length = header_length+(num_audio*3)+(num_video*3); /* calc h_l   */

    if(pack_size) header_length = header_length+3; /* add 3 bytes, pad stuff */

    itostr(header_length_data,&header_length,2);  /* convert SHORT to string  */

    if(write(output,header_length_data,2) != 2)  /* write header_length      */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }

/* write rate_bound (for now: rate_bound=mux_rate) */
    if(write(output,mux_rate,3) != 3)            /* write rate_bound         */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }
    
/* write audio_bound, fixed_flag, CSPS_flag */
    audio_bound_data.audio_bound = num_audio;    /* set audio bound          */
    audio_bound_data.fixed_flag = FIXEDFLAG;     /* set fixed_flag           */
    audio_bound_data.csps_flag = CSPSFLAG;       /* set CSPS_flag            */
    if(write(output,&audio_bound_data,1) != 1)   /* write the above three    */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }

/* write system_audio_lock_flag, system_video_lock_flag,video_bound */
    video_bound_data.system_audio_lock_flag=SYSTEMAUDIOLOCKFLAG; /* set data */
    video_bound_data.system_video_lock_flag=SYSTEMVIDEOLOCKFLAG;
    video_bound_data.marker_bit = 1;
    video_bound_data.video_bound = num_video;
    if(write(output,&video_bound_data,1) != 1)   /* write data               */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }

/* write reserved_byte */
    if(write(output,&reserved_byte,1) != 1)      /* write reserved byte      */
    {
      printf("Error writing to output\n");       /* on error                 */
      exit(1);                                   /* print err msg and exit   */
    }

/* write all stream id's, STD_buffer_bound_scale, STD_buffer_size_bound */
    current_stream = system_streams;
    do
    {
      if(write(output,&(current_stream->stream_id),1) != 1)/* write stream_id*/
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);                                 /* print err msg and exit   */
      }

      buff_info.flag = 3;                        /* set first 2 bits         */
      if(current_stream->isvideo)                /* if stream is video       */
      {
	buff_info.std_buffer_bound_scale = 1;    /* set s_b_b_s = 1          */
	buff_info.std_buffer_size_bound =        /* calc s_b_s_b             */
	  (current_stream->stream_mux_rate / 1024) + 1;
      }
      else                                       /* else if stream is audio  */
      {
	buff_info.std_buffer_bound_scale = 0;    /* set s_b_b_s = 0          */
	buff_info.std_buffer_size_bound =        /* calc s_b_s_b             */
	  (current_stream->stream_mux_rate / 128) + 1;
      }

#if LITTLE_ENDIAN
      itostr(buff_info_data,&buff_info,2);        /* convert SHORT to string  */

      if(write(output,buff_info_data,2) != 2)    /* write buff_info          */
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);                                 /* print err msg and exit   */
      }

#else
      if(write(output,&buff_info,2) != 2)        /* write buff_info          */
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);
      }
#endif

      if(pack_size) *pack_bytes_left = *pack_bytes_left-3;       /* subtract */
                                                 /* for each stream_id, etc. */
      
      current_stream=current_stream->next;       /* go to next node          */
    } while (current_stream != NULL);            /* till end of stream       */

/* write padding stream id, STD_buffer_bound_scale, STD_buffer_size_bound */
    if(pack_size)                                /* if in fixed pack mode    */
    {
      if(write(output,&pad_id,1) != 1)           /* write padding stream id  */
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);                                 /* print err msg and exit   */
      }

      buff_info.flag = 3;                        /* set first 2 bits         */
      buff_info.std_buffer_bound_scale = 1;      /* set s_b_b_s = 1          */
      buff_info.std_buffer_size_bound = (pack_size/1024)+1; /* set s_b_s_b   */

#if LITTLE_ENDIAN
      itostr(buff_info_data,&buff_info,2);        /* convert SHORT to string  */

      if(write(output,buff_info_data,2) != 2)    /* write pad buff_info      */
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);                                 /* print err msg and exit   */
      }

#else
      if(write(output,&buff_info,2) != 2)        /* write pad buff_info      */
      {
	printf("Error writing to output\n");     /* on error                 */
	exit(1);                                 /* print err msg and exit   */
      }
#endif

      *pack_bytes_left = *pack_bytes_left-3;     /* subtract 3 bytes for info*/
    }
    
    *isheader = *isheader - 1;
  }

  return mta;                                    /* return mta               */
}
  
