/**************************************************************************
* Copyright (c) 1994 The Multimedia Communications Lab, Boston University.
* All rights reserved.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose, without fee, and without written agreement is
* hereby granted, provided that the above copyright notice and the following
* two paragraphs appear in all copies of this software.
*
* IN NO EVENT SHALL BOSTON UNIVERSITY BE LIABLE TO ANY PARTY FOR
* DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
* OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BOSTON
* UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* BOSTON UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
* ON AN "AS IS" BASIS, AND BOSTON UNIVERSITY HAS NO OBLIGATION TO
* PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*
**************************************************************************/

/*************************************************************************
*  File Name:     packet.c
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Usage:         function file
*
*  Description:   This file contains the function write_packet
**************************************************************************/

#include <fcntl.h>
#include "list.h"
#include "mta.h"
#include "packet.h"

/* function prototypes */
SUB_LINK del_offset(SUB_LINK);
void stamp_add(CHAR *, unsigned INT);
INT stamp_compare(CHAR *, CHAR *);
void stamp_copy(CHAR *, CHAR *);
void itostr(CHAR *, void *, INT);

/*************************************************************************
*  Function:      write_packet()
*  Creation Date: 12 Aug 1994
*  Author:        Ziv Yaar
*                 email  zyaar@bu.edu
*                 Multimedia Communications Lab, Boston University
*                  Professor T.D.C. Little tdcl@flash.bu.edu
*  Input:         multiplex timing array, output file, packet size,
*                 system clock reference, bitrate
*
*  Effects:       returns an updated multiplex timing array
*
*  Description:   This function writes out one MPEG system packet worth
*  of data based on the highest priority (first) stream pointed to in the
*  multiplex timing array.
**************************************************************************/

MTA_LINK write_packet(MTA_LINK mta, INT output, INT packet_size,
		      CHAR *scr, unsigned INT bitrate)
{
  MTA_LINK current = mta;
  INT num_stuff = NUMSTUFFBYTES + 12;
  CHAR packet_start_code_prefix[3]={0x00,0x00,0x01};
  CHAR no_ptsdts = 0x0f;
  CHAR buffer[MAXBUFFER], stuffing=0xff, empty=0x00;
  CHAR dts[5], pts[5];
  unsigned SHORT packet_length;
  CHAR packet_length_data[2];
  INT input,n,i;
  struct buffer buff;
#if LITTLE_ENDIAN
  CHAR buff_data[2];
#endif
  INT header = mta->stream->newframe;

/* delete any accidentally blank frame */
  if(current->stream->offsets->frame_type != 'I' &&   /* if frame unrecognize*/
     current->stream->offsets->frame_type != 'P' &&
     current->stream->offsets->frame_type != 'B' &&
     current->stream->offsets->frame_type != 'A')
    mta->stream->offsets = del_offset(mta->stream->offsets); /* delete it    */

  if((input=open(current->stream->filename,O_RDONLY))<0) /* open input strm  */
  {
    printf("Error reading %s\n",current->stream->filename); /* on error      */
    exit(1);                                          /* print err msg,quit  */
  }

/* write packet_start_code */
  if(write(output,packet_start_code_prefix,3) != 3)   /* write packet start  */
  {
    printf("Error writing to output\n");              /* on error            */
    exit(1);                                          /* print err msg, quit */
  }

/* write stream id */
  if(write(output,&(current->stream->stream_id),1) != 1) /* write stream id  */
  {
    printf("Error writing to output\n");                 /* on error         */
    exit(1);
  }

  packet_length=num_stuff+(unsigned SHORT)packet_size;/* set packet_length   */
  if (current->stream->buffer_change)                 /* if change in buffers*/
    num_stuff = num_stuff - 2;                        /* add 2 bytes for STDs*/

  if (header)                                         /* if packet has header*/
  {
    if((current->stream->offsets->frame_type == 'I' ||/* if frame type I or P*/
       current->stream->offsets->frame_type == 'P') &&/* and wait time isn't */
       (current->stream->waits->wait_time != 0))      /* zero                */
      num_stuff = num_stuff - 10;                     /* incr for PTS and DTS*/

    else num_stuff = num_stuff - 5;                   /* else, incr for PTS  */
  }

  else --num_stuff;                                   /* 1 byte for 0000 1111*/

/* write packet length */
  itostr(packet_length_data,&packet_length,2);         /* SHORT to string     */

  if(write(output,packet_length_data,2) != 2)         /* write packet_length */
  {
    printf("Error writing to output.\n");             /* on error            */
    exit(1);                                          /* print err and exit  */
  }

/* write stuffing bytes */
  for(i=0; i<num_stuff; i++)                          /* write stuffing bytes*/
  {
    if(write(output,&stuffing,1) != 1)                /* write stuffing byte */
    {
      printf("Error writing to output\n");            /* on error            */
      exit(1);                                        /* print error and exit*/
    }
  }

/* write STD_buffer_scale and STD_buffer_size */
  if (current->stream->buffer_change)                 /* if STD_buffer, add  */
  {
    buff.flag = 1;                                    /* Write '01' first2bit*/

    if(current->stream->isvideo)                      /* if stream video     */
    {
      buff.std_buffer_scale = 1;                      /* s_b_s = 1           */
      buff.std_buffer_size =                          /* calc s_b_s          */
	(current->stream->stream_mux_rate / 1024) + 1;
    }

    else                                              /* else, stream audio  */
    { 
      buff.std_buffer_scale = 0;                      /* s_b_s = 0           */
      buff.std_buffer_size =                          /* calc s_b_s          */
	(current->stream->stream_mux_rate / 128) + 1;
    }

#if LITTLE_ENDIAN
    itostr(buff_data,&buff,2);                         /* SHORT to string     */

    if(write(output,buff_data,2) != 2)                /* write buff info     */
    {
      printf("Error writing to output\n");            /* on error            */
      exit(1);                                        /* print err msg, exit */
    }

#else
    if(write(output,&buff,2) != 2)                    /* write buff info     */
    {
      printf("Error writing to output\n");            /* on error            */
      exit(1);                                        /* print err msg, exit */
    }
#endif

    current->stream->buffer_change = 0;               /* clear buffer_change */
  }    

/* write PTS or PTS/DTS */
  if (header)                                         /* if PTS/DTS add      */
  {
    if ((current->stream->offsets->frame_type=='I' || /* if frame type I or P*/
	 current->stream->offsets->frame_type=='P') &&/* and wait time isn't */
	(current->stream->waits->wait_time != 0))     /* zero                */
    {
      dts[0] = current->stream->last_pts[0];          /* copy PTS to DTS     */
      dts[1] = current->stream->last_pts[1];
      dts[2] = current->stream->last_pts[2];
      dts[3] = current->stream->last_pts[3];
      dts[4] = current->stream->last_pts[4];

      pts[0] = dts[0];                                /* make backup of PTS  */
      pts[1] = dts[1];
      pts[2] = dts[2];
      pts[3] = dts[3];
      pts[4] = dts[4];

      pts[0] = (pts[0] & 0x0f) | 0x30;                /* '0010' -> '0011'    */
      dts[0] = (dts[0] & 0x0f) | 0x10;                /* '0010' -> '0001'    */

      stamp_add(pts,(INT)(current->stream->scsr *     /* generate offset PTS */
		     current->stream->waits->wait_time));

      if(write(output,pts,5) != 5)                    /* write PTS           */
      {
	printf("Error writing to output\n");          /* on error            */
	exit(1);                                      /* print err msg, exit */
      }

      if(write(output,dts,5) != 5)                    /* write DTS           */
      {
	printf("Error writing to output\n");          /* on error            */
	exit(1);                                      /* print err msg, exit */
      }           
    }

    else if(write(output,current->stream->last_pts,5)!=5) /* write PTS       */
         {
	   printf("Error writing to output\n");       /* on error            */
	   exit(1);                                   /* print err msg, exit */
	 }
  }


  else                                                /* else write 0000 1111*/
  {
    if(write(output,&no_ptsdts,1) != 1)               /* write 0000 1111     */
    {
      printf("Error writing to output\n");            /* on error            */
      exit(1);                                        /* print err and exit  */
    }
  }

  if(lseek(input,current->stream->current_offset,0) == -1) /* goto current   */
  {                                                   /* pos. in input file  */
    printf("Error seeking in %s\n",current->stream->filename); /* on error   */
    exit(1);                                          /* print err msg, exit */
  }

  if((n=read(input,buffer,packet_size))<0)            /* read 1 data packet  */
  {
    printf("Error reading %s\n",current->stream->filename); /* on error      */
    exit(1);                                          /* print err msg, exit */
  }

  current->stream->current_offset =                   /* incr. curr. offset  */
    current->stream->current_offset + packet_size;    /* by packet size      */

  if(write(output,buffer,n) != n)                     /* write 1 data packet */
  {
    printf("Error writing to output\n");              /* on error            */
    exit(1);                                          /* print err and exit  */
  }

  if (n<packet_size)                                  /* if datapack<packet  */
  {
    for(i=n; i<packet_size; i++)
      if(write(output,&empty,1) != 1)
      {
	printf("Error writing to output\n");          /* on error            */
	exit(1);                                      /* print err and exit  */
      }
                                                      /* place EOF stuff here*/
  }

  if(close(input)<0)                                  /* close input file    */
  {
    printf("Error closing %s\n",current->stream->filename); /* on error      */
    exit(1);                                          /* print err msg, exit */
  }

  if(!bitrate)                                        /* if variable bitrate */
    if(stamp_compare(current->stream->last_pts,scr)<0)/* if pts < scr        */
      stamp_copy(scr,current->stream->last_pts);      /* then scr = pts      */

  return mta;                                         /* return mta          */
}




