

  #define SCSI_BUFSIZE 32000

  #define INQUIRY_CMD     0x12
  #define INQUIRY_CMDLEN  6
  #define INQUIRY_REPLY_LEN 96
  #define INQUIRY_VENDOR  8       /* Offset in reply data to vendor name */

  #define READ_CMD	0x08
  #define READ_CMDLEN  6
  #define WRITE_CMD	0x0A
  #define WRITE_CMDLEN  6


  /* Example program to demonstrate the generic SCSI interface */
  #include <stdio.h>
  #include <unistd.h>
  #include <string.h>
  #include <fcntl.h>
  #include <errno.h>
  #include <signal.h>
  #include "sg.h"


  #define SCSI_OFF sizeof(struct sg_header)

  static unsigned char cmd[SCSI_OFF + 1000];      /* SCSI command buffer */
  int fd;                               /* SCSI device/file descriptor */


#define READ_MAXBUFSIZE 60000          	/* maximal 10k mit einem Mal        */
#define WRITE_MAXBUFSIZE 4096         	/* maximal 4k mit einem Mal        */

extern int errno;
int	scan_break = 0;

/*------------------------------------------------------------*/
/* Exit Handler                                               */
/*------------------------------------------------------------*/
void scan_ende(void)
{

scan_break = 1;
}

/*-------------------------------------------------------------------------*/
/* protected read                                                          */
/*-------------------------------------------------------------------------*/
int p_read(int fd,char *buffer, long size)
{
  long  rest = size;
  long  pos  = 0;
  long  erg;
  
  for(;;)
  {
    if(rest <= READ_MAXBUFSIZE)
      erg = read(fd,&buffer[pos],rest);
    else  
      erg = read(fd,&buffer[pos],READ_MAXBUFSIZE);

    if(erg >= 0)
    {
      rest -= erg;
      pos += erg;
      if(rest <=0)
        return pos;
      if(erg == 0)
        return pos;
    }
    else
    {
      if(errno != 4)
        return -1;
      if(pos == size)
        return pos;
    }
  }
}

/*-------------------------------------------------------------------------*/
/* protected Write                                                         */
/*-------------------------------------------------------------------------*/
int 	p_write (int fd, char *buffer, long size)
{
  long  rest = size;
  long  pos  = 0;
  long  erg;
  
  for (;;)
  {
    if (rest <= WRITE_MAXBUFSIZE)
      erg = write (fd, &buffer[pos], rest);
    else  
      erg = write (fd, &buffer[pos], WRITE_MAXBUFSIZE);

    if (erg >= 0)
    {
      rest -= erg;
      pos += erg;
      if (rest <= 0)
        return pos;
      if (erg == 0)
        return pos;
    }
    else
    {
      if (errno != 4)
        return -1;
      if (pos == size)
        return pos;
    }
  }
}

/*------------------------------------------------------------*/
/* request vendor brand and model                             */
/*------------------------------------------------------------*/

  /* process a complete SCSI cmd. Use the generic SCSI interface. */
  static int handle_SCSI_cmd(unsigned cmd_len,         /* command length */
                             unsigned in_size,         /* input data size */
                             unsigned char *i_buff,    /* input buffer */
                             unsigned out_size,        /* output data size */
                             unsigned char *o_buff     /* output buffer */
                             )
  {
      int status = 0;
      struct sg_header *sg_hd;

      /* safety checks */
      if (!cmd_len) return -1;            /* need a cmd_len != 0 */
      if (!i_buff) return -1;             /* need an input buffer != NULL */
  #ifdef SG_BIG_BUFF
      if (SCSI_OFF + cmd_len + in_size > SG_BIG_BUFF) return -1;
      if (SCSI_OFF + out_size > SG_BIG_BUFF) return -1;
  #else
      if (SCSI_OFF + cmd_len + in_size > 4096) return -1;
      if (SCSI_OFF + out_size > 4096) return -1;
  #endif

      if (!o_buff) out_size = 0;      /* no output buffer, no output size */

      /* generic SCSI device header construction */
      sg_hd = (struct sg_header *) i_buff;
      sg_hd->reply_len   = SCSI_OFF + out_size;
      sg_hd->twelve_byte = cmd_len == 12;
  #if     0
      sg_hd->pack_len    = SCSI_OFF + cmd_len + in_size; /* not necessary */
      sg_hd->pack_id;     /* not used */
      sg_hd->other_flags; /* not used */
  #endif


      /* send command */
      status = p_write( fd, i_buff, SCSI_OFF + cmd_len + in_size );
      if ( status < 0 || status != SCSI_OFF + cmd_len + in_size ||
                         sg_hd->result ) {
          /* some error happened */
          fprintf( stderr, "write(generic) result = 0x%x cmd = 0x%x\n",
                      sg_hd->result, i_buff[SCSI_OFF] );
          perror("");
          return status;
      }

      if (!o_buff) o_buff = i_buff;       /* buffer pointer check */

      /* retrieve result */
      status = p_read( fd, o_buff, SCSI_OFF + out_size);
      if ( status < 0 || status != SCSI_OFF + out_size || sg_hd->result ) {
          /* some error happened */
          fprintf( stderr, "read(generic) status = 0x%x, result = 0x%x, "
                           "cmd = 0x%x\n",
                           status, sg_hd->result, o_buff[SCSI_OFF] );
          fprintf( stderr, "read(generic) sense "
                  "%x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x\n",
                  sg_hd->sense_buffer[0],         sg_hd->sense_buffer[1],
                  sg_hd->sense_buffer[2],         sg_hd->sense_buffer[3],
                  sg_hd->sense_buffer[4],         sg_hd->sense_buffer[5],
                  sg_hd->sense_buffer[6],         sg_hd->sense_buffer[7],
                  sg_hd->sense_buffer[8],         sg_hd->sense_buffer[9],
                  sg_hd->sense_buffer[10],        sg_hd->sense_buffer[11],
                  sg_hd->sense_buffer[12],        sg_hd->sense_buffer[13],
                  sg_hd->sense_buffer[14],        sg_hd->sense_buffer[15]);
      }
      /* Look if we got what we expected to get */
      if (status == SCSI_OFF + out_size) status = 0; /* got them all */

      return status;  /* 0 means no error */
  }

/*------------------------------------------------------------*/
/* request vendor brand and model                             */
/*------------------------------------------------------------*/
  void inquiry ( char *str )
  {
    unsigned char cmdblk [ INQUIRY_CMDLEN ] =
        { INQUIRY_CMD,  /* command */
                    0,  /* lun/reserved */
                    0,  /* page code */
                    0,  /* reserved */
    INQUIRY_REPLY_LEN,  /* allocation length */
                    0 };/* reserved/flag/link */

    strcpy(str,"");		/* preload empty string		*/

    memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

    /*
     * +------------------+
     * | struct sg_header | <- cmd
     * +------------------+
     * | copy of cmdblk   | <- cmd + SCSI_OFF
     * +------------------+
     */

    if (handle_SCSI_cmd(sizeof(cmdblk), 0, cmd,
                        INQUIRY_REPLY_LEN, str )) {
        fprintf( stderr, "Inquiry failed\n" );
        exit(2);
    }
  }

/*------------------------------------------------------------*/
/* send data to Scanner                                       */
/*------------------------------------------------------------*/
  void scanner_send( unsigned char *buffer, unsigned short anz)
  {
    unsigned char retbuffer[ SCSI_OFF +100 ];

    unsigned char cmdblk [ WRITE_CMDLEN ] =
        { WRITE_CMD,  /* command */
                    0,  /* lun/reserved */
                    0,  /* page code */
                    0,  /* reserved */
                  anz,  /* allocation length */
                    0 };/* reserved/flag/link */

    memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );
    memcpy( cmd + SCSI_OFF+sizeof(cmdblk), buffer, anz );

    if (handle_SCSI_cmd(sizeof(cmdblk), 
			anz, cmd,                 /* Input */
                           0, retbuffer) )       /* Output*/
    {
        fprintf( stderr, "Write failed\n" );
        exit(2);
    }
  }

/*------------------------------------------------------------*/
/* recive data from scanner                                   */
/*------------------------------------------------------------*/
  int scanner_rec( unsigned char *buffer, unsigned short *anz,int *status,int *lcount)
  {
    unsigned char retbuffer[ SCSI_OFF + SCSI_BUFSIZE + 1000 ];
    int recanz;
    
    unsigned char cmdblk [ READ_CMDLEN ] =
        { READ_CMD,  /* command */
                    0,  /* lun/reserved */
                    0,  /* page code */
                    0,  /* reserved */
                   100,  /* allocation length */
                    0 };/* reserved/flag/link */

    unsigned char cmdblk1 [ READ_CMDLEN ] =
        { READ_CMD,  /* command */
                    0,  /* lun/reserved */
                    0,  /* page code */
                    0,  /* reserved */
                   10,  /* allocation length */
                    0 };/* reserved/flag/link */

    memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

    retbuffer[SCSI_OFF+4] = 0;           /* nur vorloeschen */
    retbuffer[SCSI_OFF+5] = 0;

    if (handle_SCSI_cmd(sizeof(cmdblk), 
			0, cmd,                 /* Input */
                        100, retbuffer) )           /* Output*/
    {
        fprintf( stderr, "Read failed\n" );
        exit(2);
    }

    if(retbuffer[SCSI_OFF] != 0x02)
      {
      *anz = 0;
      *status=0;
      *lcount = 0;

      memcpy(buffer,retbuffer+SCSI_OFF,10);

      return -1;
      }

    *status = retbuffer[SCSI_OFF+1];
    *anz = retbuffer[SCSI_OFF+2] + 256*retbuffer[SCSI_OFF+3];
    *lcount = retbuffer[SCSI_OFF+4] + 256*retbuffer[SCSI_OFF+5];
 
    recanz = (*anz) * (*lcount);

    if(recanz < 256)
      {
        cmdblk1[4] = (unsigned char)recanz;
      }
    else
      {
        cmdblk1[2] = 1;
        cmdblk1[4] = (unsigned char)((recanz) / 256);
      }

    memcpy( cmd + SCSI_OFF, cmdblk1, sizeof(cmdblk1) );

    if (handle_SCSI_cmd(sizeof(cmdblk1), 
			0, cmd,                 /* Input */
                        SCSI_BUFSIZE, retbuffer) )       /* Output*/
    {
        fprintf( stderr, "Read failed\n" );
        exit(2);
    }

    memcpy(buffer,retbuffer+SCSI_OFF,recanz);
    return 0;
  }

/*------------------------------------------------------------*/
/* Teste, ob Befehl korrekt ausgefuehrt wurde                 */
/*------------------------------------------------------------*/
void testack(char *str,int val)
{
  unsigned char buf[SCSI_BUFSIZE + 1000];
  int status,lcount,erg;
  unsigned short anz;

  erg = scanner_rec(buf,&anz,&status,&lcount);

  if(buf[0] != 0x06)
  {
    fprintf(stderr,str,val);
    exit(1);
  }
}

/*------------------------------------------------------------*/
/* Help screen                                                */
/*------------------------------------------------------------*/
void scan_help(void)
{
  printf("usage: scan [-x -y -w -h -d -?] filename\n");
  printf("  Options\n");
  printf("        -x  <val> x starting point in pixel\n");
  printf("        -y  <val> y starting point in pixel\n");
  printf("        -w  <val> width in pixel\n");
  printf("        -h  <val> height in pixel\n");
  printf("        -r  <val> scanner resolution in dpi\n");
  printf("        -s  <val> scanner sharpness value\n");
  printf("        -g  <val> scanner gamma value\n");
  printf("        -b  <val> scanner brightness (-3 to 3)\n");
  printf("        -o <file> Name of outpu File\n");
  printf("        -m monochrome mode\n");
  printf("        -i monochrome bitmap Mode (1bit/Pixel)\n");
  printf("        -a halftoning Mode 1=disable\n");
  printf("        -q <file> query Capabilities, write result to file\n");

}

/*------------------------------------------------------------*/
/* Main Proceccor                                             */
/*------------------------------------------------------------*/
 int main( int argc , char *argv[] )
 {
  struct   sigaction aktion;
   unsigned char buf[SCSI_BUFSIZE + 1000];
   unsigned char rgb_line[SCSI_BUFSIZE + 1000];
   unsigned char tmp[SCSI_BUFSIZE + 1000];
   char          inq_str[256];
   char		 str[100];
   char		 output_fil[255]="output.ppm";
   int status,lcount,erg;
   unsigned short anz;
   int i,j;
   int dpi  = 200;
   int  x=0;
   int  y=0;
   int  w=1400;
   int  h=1000;
   int  color;
   FILE *fil;
   int	tiefe=8;
   int  farben=3;
   int  sharpness=1;
   int  gamma = 2;
   int  brightness=0;
   int  c;              /* Rueckgabewert fuer GETOPT                    */
   int  fast = 0;
   int  halftone = 1;
   int  zoom = 100;
   int  debug = 0;
   int  query_mode = 0;
   char query_file[255]="query.conf";

         /* Uebergebene Parameter auswerten          */
  while ((c = getopt(argc, argv, "d?x:y:w:h:r:o:s:g:b:mifa:l:z:q:")) != -1)
  {
    switch (c) 
    {
      case 'd':		/* Switch on Debug messages		*/
        debug = 1;
	printf ("scan: DEBUG-Mode\n");
        break; 

      case 'x':		/* x Position of Picture                 	*/
        x=atoi(optarg);
        break;

      case 'y':		/* y Position of Picture                 	*/
        y=atoi(optarg);
        break;

      case 'w':		/* width of Picture                 	*/
        w=atoi(optarg);
        break;

      case 'h':		/* heigth of Picture                 	*/
        h=atoi(optarg);
        break;

      case 'r':		/* Scanner resolution                 	*/
        dpi=atoi(optarg);
        break;

      case 's':		/* Scanner sharpness                 	*/
        sharpness=atoi(optarg);
        break;

      case 'g':		/* Scanner gamma value                 	*/
        gamma=atoi(optarg);
        break;

      case 'b':		/* Scanner brightness value                	*/
        brightness=atoi(optarg);
        break;

      case 'o':		/* Output File                   	*/
        strcpy(output_fil,optarg);
        break;

      case 'm':		/* monochrome mode                   	*/
        farben = 1;
        break;

      case 'i':		/* monochrome mode image scanning         	*/
        tiefe = 1;
        farben = 1;
        break;

      case 'f':		/* fast mode        	*/
        fast = 1;
        break;

      case 'a':		/* halftoning mode        	*/
        halftone = atoi(optarg);
        break;
        
      case 'z':		/* zoom        	*/
        zoom = atoi(optarg);
        break;

      case 'q':		/* query resolutions and Scannerlevel  	*/
        query_mode = 1;
        strcpy(query_file,optarg);
        break;
        
      case '?':
	scan_help ();
        exit (0);
        break;

      default:
        scan_help();
    }
  }

  if(debug)
  {
    printf("Parameters: x=%d y=%d w=%d h=%d\n",x,y,w,h);
  }


  aktion.sa_handler 	= (void *) scan_ende;
  sigemptyset(&aktion.sa_mask);
  aktion.sa_flags 	= SA_RESTART;

  if (sigaction (SIGTERM, &aktion, NULL) < 0)
  {
    printf("cant install Signal Handler (TERM)\n");
    exit(1);
  }
  
  if (sigaction (SIGINT, &aktion, NULL) < 0)
  {
    printf("cant install Signal Handler (CTRL-C)\n");
    exit(1);
  }

      
/*-- Check for Epson Scanner in Devices ------------*/
   j=0;
   for(i=0;i<8;i++)
   {
     sprintf(str,"/dev/sg%c",i+'0');
     fd = open(str, O_RDWR);
     if (fd < 0) 
     {
       if(debug)
         fprintf( stderr, "scan: can't open %s read/write.\n",str );
     }
     else
         /* print some fields of the Inquiry result */
     {
       inquiry(inq_str);
       if(debug)
         fprintf(stderr, "Inq String >%s<\n", &inq_str[SCSI_OFF+8]);
       if(inq_str[SCSI_OFF] == 3)
       {
         if(strncmp(&inq_str[SCSI_OFF+8],"EPSON ",6) == 0)
         {
           j =1;
           break;
         }
       }
       close(fd);
     }
   }

   if(!j)
   {
     fprintf(stderr,"scan: no EPSON Scanner found\n");
     exit(1);
   }

   scanner_send("\033I",2);
   erg = scanner_rec(buf,&anz,&status,&lcount);

   if(debug)
     fprintf(stderr,"Scanner read erg=%d anz=%d status=%d lcount=%d\n",erg,anz,status,lcount);

   if(debug)
     for(i=0;i<anz+100;i++)
       fprintf(stderr,"%02X ",buf[i]);
   if(debug)
     fprintf(stderr,"\n");

/*--------query mode, read id Buf and write to File----*/
   if(query_mode)
   {
     scanner_send("\033I",2);

     erg = scanner_rec(buf,&anz,&status,&lcount);

     fil = fopen(query_file,"w");
     if(fil)
     {
       fwrite(buf,1,anz,fil);
       fclose(fil);
     }
     else
       fprintf(stderr,"ep_scan: unable to open %s\n",query_file);
     exit(0);
   }
/*--------Set Sharpness----*/
   scanner_send("\033Q",2);
   testack("scan: Error transfering query identity command\n",0);

   buf[0] = sharpness;
   scanner_send(buf,1);

   testack("scan: Error setting sharpness to %d\n",sharpness);

/*--------Set Halftonig Mode ----*/
   scanner_send("\033B",2);
   testack("scan: Error transfering set halftoning command\n",0);

   buf[0] = halftone;
   scanner_send(buf,1);

   testack("scan: Error setting halftoning to %d\n",sharpness);

/*--------Set GAMMA----*/
   scanner_send("\033Z",2);
   testack("scan: Error transfering gamma correction command\n",0);

   buf[0] = gamma;                /* 1 = CRT B */
   scanner_send(buf,1);

   testack("scan: Error setting gamma correction to 0x%x\n",gamma);

/*--------Set Bightness----*/
   scanner_send("\033L",2);
   testack("scan: Error transfering brightness command\n",0);

   buf[0] = brightness;
   scanner_send(buf,1);
   testack("scan: Error setting brightness to %d\n",brightness);

/*--------Set Zoom----*/
   scanner_send("\033H",2);
   testack("scan: Error transfering zoom command\n",0);

   buf[0] = zoom;
   buf[1] = zoom;
   scanner_send(buf,2);
   testack("scan: Error setting zoom to %d\n",zoom);

/*--------Resolution in dpi----*/
   scanner_send("\033R",2);
   testack("scan: Error transfering resolution command\n",0);

   buf[0] = dpi & 0xFF;
   buf[1] = (dpi >> 8) & 0xFF;
   buf[2] = dpi & 0xFF;
   buf[3] = (dpi >> 8) & 0xFF;
   scanner_send(buf,4);

   testack("scan: Error setting resolution to %d dpi\n",dpi);
   
/*--------Scanning AREA----*/
   scanner_send("\033A",2);
   testack("scan: Error transfering area command\n",0);

   buf[0] = x & 0xFF;
   buf[1] = (x >> 8) & 0xFF;
   buf[2] = y & 0xFF;
   buf[3] = (y >> 8) & 0xFF;
   buf[4] = w & 0xFF;
   buf[5] = (w >> 8) & 0xFF;
   buf[6] = h & 0xFF;
   buf[7] = (h >> 8) & 0xFF;
   scanner_send(buf,8);

   testack("scan: Error setting scanning area w=%d\n",w);


/*-------Color Scan Line Sequence Mode-----*/
   scanner_send("\033C",2);
   testack("scan: Error transfering scan mode command\n",0);

   if(farben == 3)
     buf[0] = 0x02;       /* color in line sequence mode  */
   else
     buf[0] = 0;          /* monochrome, no dropout color */
   scanner_send(buf,1);

   testack("scan: Error setting number of colors to %d\n",farben);

/*--------Line Counter----*/
   scanner_send("\033d",2);
   testack("scan: Error transfering line counter command\n",0);

   i = SCSI_BUFSIZE / w / tiefe * 8 ;
   if(i>255)i=255;
   i/=farben;
   i*=farben;
   if(i<1)i=1;

   buf[0] = i;
   scanner_send(buf,1);

   testack("scan: Error setting line counter to %d\n",i);

/*--------Dataformat 8Bit/Pixel----*/
   scanner_send("\033D",2);
   testack("scan: Error transfering deepness command\n",0);

   buf[0] = tiefe;
   scanner_send(buf,1);

   testack("scan: Error setting bit/pixel to %d\n",tiefe);


/*-----Scan Speed 0=normal 1=fast-------*/
   scanner_send("\033g",2);
   testack("scan: Error transfering scan speed command\n",0);

   buf[0] = fast;
   scanner_send(buf,1);

   testack("scan: Error setting scan speed to %d\n",fast);

/*------------*/
   color = 0;

   fil = fopen(output_fil,"w");
   if(fil == NULL)
   {
     printf("cant open %s for writing\n",output_fil);

   }
   
   if(farben == 3)
     fprintf(fil,"P6\n# Erzeugt mit Uli's GT8000 Treiber\n%d %d\n255\n",w,h);   
   else if(farben == 1)
     if(tiefe == 8)
       fprintf(fil,"P5\n# Erzeugt mit Uli's GT8000 Treiber\n%d %d\n255\n",w,h);   
     else if(tiefe == 1)
       fprintf(fil,"P4\n# Erzeugt mit Uli's GT8000 Treiber\n%d %d\n",w,h);   
      
   scanner_send("\033G",2);

   do
   {
     erg = scanner_rec(buf,&anz,&status,&lcount);

     if(debug)
       fprintf(stderr,"Scanner read erg=%d anz=%d status=%d lcount=%d\n",erg,anz,status,lcount);


     for(i=0;i<lcount;i++)
     {
       if(farben == 3)
       {
         memcpy(&rgb_line[color*w],&buf[i*anz],anz);

         color++;
         if(color == farben)
         {
           for(j=0;j<w;j++)
           {
             tmp[j*3+0] = rgb_line[j+w];
             tmp[j*3+1] = rgb_line[j];
             tmp[j*3+2] = rgb_line[j+2*w];

             color = 0;
           }
           fwrite(tmp,w,3,fil);
         }
       }
       else if(farben == 1)
       {
         int k;

         if(tiefe == 1)		/* monochrome Bitmaps are inverted! */
           for(k=0;k<anz;k++)
             buf[i*anz+k] = ~buf[i*anz+k];
         fwrite(&buf[i*anz],anz,1,fil);
       }
     }
     if( ! ( status & 0x20 ) )
     {
       if(!scan_break)
         scanner_send("\x06",1);  
       else
       {
	 printf("Scan cancelled\n");
         scanner_send("\x18",1);  
         testack("scan: Error aborting scan !! %d\n",0);
         fclose(fil);
	 exit(0); 
       }
     }
   } while( ! ( status & 0x20 ) );

   fclose(fil);

   return 0;      
 }

