# This file is a part of TkFax.
# fax-file.tcl
# This is for converting files in various format to files in fax(g3) format
# and add them to the fax page list.

proc fax_file_window {} {
#
# --- File Attachment Window ---
#
    global fax

    # The name of the window

    set wn $fax(mwn).attach
    if [winfo exists $wn] {return}
    toplevel $wn
    wm title $wn "TkFax: File Attachment"

    label $wn.1 -text "TkFax: File Attachment" -width 24 -pady 4 -relief ridge
    label $wn.2 -text " Directory" -width 10 -anchor w
    button $wn.u -text Update -width 6 -command fax_update_filelist
    entry $wn.3 -width 30 -textvariable fax(curdir) -font $fax(f1) -fg $fax(c1)
    label $wn.4 -text " Files:" -width 4 -anchor w
    label $wn.5 -text "" -width 36 -anchor w -fg red
    listbox $wn.6 -width 28 -height 12 -font $fax(f1) \
            -yscrollcommand "$wn.7 set" -xscrollcommand "$wn.8 set"
    scrollbar $wn.7 -orient vertical   -command "$wn.6 yview"
    scrollbar $wn.8 -orient horizontal -command "$wn.6 xview"
    frame $wn.9
    button $wn.a -text "ATTACH" -command fax_attach_file -width 8 -pady 5
    button $wn.b -text "Close"  -command "destroy $wn"   -width 6 -pady 5
    label $wn.c -text "" -padx 7 -pady 0 -relief sunken

    grid $wn.1 -row 1 -column 1 -rowspan 1 -columnspan 4 -sticky news -pady 5
    grid $wn.2 -row 2 -column 1 -rowspan 1 -columnspan 3 -sticky news
    grid $wn.u -row 2 -column 3 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.3 -row 3 -column 1 -rowspan 1 -columnspan 4 -sticky news
    grid $wn.4 -row 4 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.5 -row 4 -column 2 -rowspan 1 -columnspan 3 -sticky news
    grid $wn.6 -row 5 -column 1 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.7 -row 5 -column 3 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.8 -row 6 -column 1 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.9 -row 5 -column 4 -rowspan 2 -columnspan 1 -sticky news
    grid $wn.a -row 7 -column 1 -rowspan 1 -columnspan 2 -sticky news -pady 5
    grid $wn.b -row 7 -column 3 -rowspan 1 -columnspan 2 -sticky news -pady 5
    grid $wn.c -row 6 -column 3 -rowspan 1 -columnspan 1

    # Radiobuttons for selecting file types

    label $wn.9.0 -text "File Type" -width 8 -pady 0 -anchor w
    radiobutton $wn.9.1 -variable fax(attachfiletype) -value auto -text auto \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.2 -variable fax(attachfiletype) -value fax -text fax \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.3 -variable fax(attachfiletype) -value text -text text \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.4 -variable fax(attachfiletype) -value tex -text tex \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.5 -variable fax(attachfiletype) -value dvi -text dvi \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.6 -variable fax(attachfiletype) -value ps -text ps \
                        -width 6 -pady 0 -anchor w 
    radiobutton $wn.9.7 -variable fax(attachfiletype) -value image -text image \
                        -width 6 -pady 0 -anchor w 
    pack $wn.9.0 $wn.9.1 $wn.9.2 $wn.9.3 \
         $wn.9.4 $wn.9.5 $wn.9.6 $wn.9.7 -side top

    # Globally used widgets

    set fax(attachmsg) $wn.5
    set fax(filelist)  $wn.6

    # Keyboard and mouse bindings

    bind $wn.3 <KeyPress-Return> { cd $fax(curdir) ; fax_update_filelist }
    bind $wn.6 <Double-Button-1> fax_attach_file

    # Initialization

    set fax(attachfiletype) auto
    fax_update_filelist
}

proc fax_update_filelist {} {
#
# Fills the filelist from the current directory.
#
    global fax
    $fax(filelist) delete 0 end
    set filelist [exec ls -a $fax(curdir)]
    foreach i $filelist {
        if {[file isdirectory $fax(curdir)/$i] && $i!="."} {
            $fax(filelist) insert end [format " %9s  %-17s " "Directory" $i]
        }
    }
    foreach i $filelist {
        if { [file isfile $fax(curdir)/$i] } {
            set size [expr [file size $fax(curdir)/$i] / 1024.0]
            $fax(filelist) insert end [format " %6.0f  %-20s " $size $i]
        }
    }
}

proc fax_attach_file {} {
#
# If currently selected file is a directory, change to that directory.
# If a ordinary file, try to convert it into g3 format and add to
# the fax page list.
#
    global fax

    # When no file is selected, do nothing.
    if {[$fax(filelist) curselection]=={}} { return }

    # Clear the message
    fax_attach_msg ""

    # Get filename and test if it is a directory or a file.
    # If directory, list the files in that directory and return.
    set f [lindex [selection get] 1]
    if [file isdirectory $fax(curdir)/$f] {
        set xx [expr [string last / $fax(curdir)]-1]
        if {$f==".."} {
            if {$xx < 0} { set fax(curdir) "/"
            } else { set fax(curdir) [string range $fax(curdir) 0 $xx] }
        } else {
            if {$fax(curdir)=="/"} { set fax(curdir) "/$f"
            } else { set fax(curdir) "$fax(curdir)/$f" }
        }
        fax_update_filelist
        return
    }

    # Divide the file name into prefix and surfix
    set xx [string last . $f]
    if {$xx < 0} {
        set prefix $f
        set surfix ""
    } else {
        set prefix [string range $f 0 [expr $xx-1]]
        set surfix [string range $f [expr $xx+1] end]
    }

    # Determine the type(format) of the file (from surfix).
    if {$fax(attachfiletype)=="auto"} {
        switch $surfix {
            g3      -
            fax     {set type fax}
            txt     -
            text    {set type text}
            tex     {set type tex}
            dvi     {set type dvi}
            eps     -
            ps      {set type ps}
            jpg     -
            gif     -
            pnm     -
            pbm     {set type image}
            default {set type unknown}
        }
    } else {
        set type $fax(attachfiletype)
    }

    # Convert the file into g3 format and add it to the page list.
    switch $type {
        fax {
            exec cp -f $fax(curdir)/$f $fax(tmpdir)
            fax_addto_pagelist $f
        }
        text {
            fax_text2fax $fax(curdir)/$f
        }
        tex {
            fax_tex2dvi $fax(curdir)/$prefix
            fax_dvi2ps $prefix
            fax_ps2fax $prefix
        }
        dvi {
            fax_dvi2ps $fax(curdir)/$prefix
            fax_ps2fax $prefix
        }
        ps {
            fax_ps2fax $fax(curdir)/$f
        }
        image {
            fax_image2pbm $fax(curdir)/$f
            fax_pbm2fax $prefix.pbm
        }
        unknown {
            fax_attach_msg "$f: unkown filetype"
            return
        }
    }
    fax_attach_msg "$f: attached"
}

proc fax_attach_msg { msg } {
#
# Show messages on the file attach window.
#
    global fax
    $fax(attachmsg) configure -text $msg
}

#
# File converting procedures
# All processes are done in fax(tmpdir) directory.
#

proc fax_text2fax { fn } {
#
# Convert a text file into g3 file using ghostscript and
# add it to the page list.
#
    global fax
    cd $fax(tmpdir)
    set prefix [get_prefix $fn]
    set command "$fax(ghostscript) \
        -q -dNOPAUSE -sDEVICE=faxg3 -r204x196 -sOutputFile=$prefix.%03d.g3 \
        -- gslp.ps -f$fax(gstextfont) -B $fn 2>/dev/null </dev/null"
    fax_exec $command
    set faxes [fax_exec "ls $prefix.\[0-9\]\[0-9\]\[0-9\].g3"]
    fax_addto_pagelist $faxes
}

proc fax_tex2dvi { fn } {
#
# Generate a dvi file from a tex file.
#
    global fax
    cd $fax(tmpdir)
    fax_exec_xterm "$fax(latex) $fn"
}

proc fax_dvi2ps { fn } {
#
# Generate a ps file from a dvi file.
#
    global fax
    cd $fax(tmpdir)
    fax_exec_xterm "$fax(dvips) $fn"
}

proc fax_ps2fax { fn } {
#
# Convert a ps file to a g3 file using ghostscript and
# add it to the page list.
#
    global fax
    cd $fax(tmpdir)
    set prefix [get_prefix $fn]
    if {$prefix==$fn} {
        set file $fn.ps
    } else {
        set file $fn
    }
    set command "$fax(ghostscript) \
        -q -dNOPAUSE -sDEVICE=faxg3 -r204x196 -sOutputFile=$prefix.%03d.g3 \
        $file </dev/null"
    fax_exec $command
    set faxes [fax_exec "ls $prefix.\[0-9\]\[0-9\]\[0-9\].g3"]
    fax_addto_pagelist $faxes
}

proc fax_pbm2fax { fn } {
#
# Convert a pbm file into a g3 file using pbm2g3
# and add it to the page list.
#
    global fax
    cd $fax(tmpdir)
    set prefix [get_prefix $fn]
    set command "$fax(pbm2g3) $fn > $prefix.g3"
    fax_exec $command
    fax_addto_pagelist $prefix.g3
}

proc fax_image2pbm { fn } {
#
# Convert any image file into a pbm file using convert from ImageMagic.
#
    global fax
    cd $fax(tmpdir)
    set prefix [get_prefix $fn]
    set command "$fax(convert) $fn $prefix.pbm"
    fax_exec $command
}

