# This file is a part of TkFax.
#

#
# Phone book database has the following structure:
# Each group has
#	- group name
#	- group description
#	- list of group members
# Each group member consists of
#	- fax number
#	- recipient
#	- reference
#	- description
#
# Phone book database is kept in two variables:
#
#   fax(grouplist)
#     = { {group0|description0} {group0|description0} ... }
#
#   fax(group-groupname)
#     = { {faxnumber0|recipient0|reference0|description0} ... }
#
# For delemeter(|), [TAB] will be used.
#

#
# Phone Book Window

proc fax_phonebook {} {
    global fax

    # Name of this window

    set wn $fax(mwn).pbook
    set fax(phonebookwin) $wn
    if [winfo exists $wn] {return}
    toplevel $wn
    wm title $wn "TkFax: Fax Number Book"

    # Top label

    frame $wn.0 -bd 10
    label $wn.0.l -text "TkFax: Fax Number Book" -padx 20 -pady 4 -relief ridge
    pack $wn.0.l -side left -fill x -expand yes

    # List of groups

    frame $wn.g
    frame $wn.g.l
    label $wn.g.l.1 -text "Group Name" -width 20 -anchor w
    label $wn.g.l.2 -text "Description" -width 40 -anchor w -padx 4
    pack $wn.g.l.1 $wn.g.l.2 -side left -fill x
    text $wn.g.t -width 60 -height 5 \
         -tabs {5c left 15c left} -wrap none \
         -xscroll "$wn.g.x set" -yscroll "$wn.g.y set" \
         -relief sunken -font $fax(f1)
    scrollbar $wn.g.x -command "$wn.g.t xview" -orient horizontal
    scrollbar $wn.g.y -command "$wn.g.t yview"
    grid $wn.g.l  -row 0 -column 0 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.g.t  -row 1 -column 0 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.g.y  -row 1 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.g.x  -row 2 -column 0 -rowspan 1 -columnspan 1 -sticky news

    # Group Entries

    frame $wn.j -bd 2 -relief ridge
    label $wn.j.1 -text "Group Name"  -anchor e -width 10
    label $wn.j.2 -text "Description" -anchor e -width 10
    entry $wn.j.3 -textvariable fax(tmpgroupname) -width 20 \
          -font $fax(f1) -fg $fax(c1)
    entry $wn.j.4 -textvariable fax(tmpgroupdescription) -width 40 \
          -font $fax(f1) -fg $fax(c1)
    button $wn.j.a -text Clear  -command fax_pbook_group_clear \
           -width 6 -pady 0
    button $wn.j.b -text Add    -command fax_pbook_group_add \
           -width 6 -pady 0
    button $wn.j.c -text Remove -command fax_pbook_group_remove \
           -width 6 -pady 0
    grid $wn.j.1  -row 1 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.j.2  -row 2 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.j.3  -row 1 -column 2 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.j.4  -row 2 -column 2 -rowspan 1 -columnspan 4 -sticky news
    grid $wn.j.a  -row 1 -column 5 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.j.b  -row 1 -column 6 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.j.c  -row 2 -column 6 -rowspan 1 -columnspan 1 -sticky news

    # spare space

    label $wn.x -text "" -pady 0

    # List of fax numbers

    frame $wn.l
    frame $wn.l.l
    label $wn.l.l.1 -text "Fax Number" -width 20 -anchor w
    label $wn.l.l.2 -text "Recipient"  -width 40 -anchor w -padx 4
    pack $wn.l.l.1 $wn.l.l.2 -side left -fill x
    text $wn.l.t -width 60 -height 10 -state disabled \
         -tabs {5c left 15c left 25c left} -wrap none \
         -xscroll "$wn.l.x set" -yscroll "$wn.l.y set" \
         -relief sunken -font $fax(f1)
    scrollbar $wn.l.x -command "$wn.l.t xview" -orient horizontal
    scrollbar $wn.l.y -command "$wn.l.t yview"
    grid $wn.l.l  -row 0 -column 0 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.l.t  -row 1 -column 0 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.l.y  -row 1 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.l.x  -row 2 -column 0 -rowspan 1 -columnspan 1 -sticky news

    # Fax Entries and Add and Edit buttons

    frame $wn.i -bd 2 -relief ridge
    label $wn.i.1 -text "Fax Number"  -anchor e -width 10
    label $wn.i.2 -text "Recipient"   -anchor e -width 10
    label $wn.i.3 -text "Reference"   -anchor e -width 10
    label $wn.i.4 -text "Description" -anchor e -width 10
    entry $wn.i.5 -textvariable fax(tmpnumber) \
          -width 20 -font $fax(f1) -fg $fax(c1)
    entry $wn.i.6 -textvariable fax(tmprecipient) \
          -width 40 -font $fax(f1) -fg $fax(c1)
    entry $wn.i.7 -textvariable fax(tmpreference) \
          -width 40 -font $fax(f1) -fg $fax(c1)
    entry $wn.i.8 -textvariable fax(tmpdescription) \
          -width 40 -font $fax(f1) -fg $fax(c1)
    button $wn.i.a -text Clear   -command fax_pbook_faxnum_clear \
           -width 6 -pady 0
    button $wn.i.b -text Add     -command fax_pbook_faxnum_add \
           -width 6 -pady 0
    button $wn.i.c -text Replace -command fax_pbook_faxnum_replace \
           -width 6 -pady 0
    button $wn.i.d -text Remove  -command fax_pbook_faxnum_remove \
           -width 6 -pady 0
    button $wn.i.e -text Sort    -command fax_pbook_faxnum_sort \
           -width 6 -pady 0
    grid $wn.i.1  -row 1 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.2  -row 2 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.3  -row 3 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.4  -row 4 -column 1 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.5  -row 1 -column 2 -rowspan 1 -columnspan 2 -sticky news
    grid $wn.i.6  -row 2 -column 2 -rowspan 1 -columnspan 4 -sticky news
    grid $wn.i.7  -row 3 -column 2 -rowspan 1 -columnspan 4 -sticky news
    grid $wn.i.8  -row 4 -column 2 -rowspan 1 -columnspan 4 -sticky news
    grid $wn.i.a  -row 1 -column 5 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.b  -row 1 -column 6 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.c  -row 2 -column 6 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.d  -row 3 -column 6 -rowspan 1 -columnspan 1 -sticky news
    grid $wn.i.e  -row 4 -column 6 -rowspan 1 -columnspan 1 -sticky news

    # Select and Close Buttons

    frame $wn.b -bd 10
    button $wn.b.1 -text "Select Fax Number" \
           -command fax_pbook_select_faxnumber \
           -width 30 -pady 2
    button $wn.b.3 -text Close -command fax_pbook_close -width 6 -pady 2
    pack $wn.b.1 $wn.b.3 -side left -fill x -expand yes

    # Packing of all frames

    pack $wn.0 $wn.g $wn.j $wn.x $wn.l $wn.i $wn.b -side top -fill x -expand yes

    # Set the globally accessed windows

    set fax(grouplistwin)  $wn.g.t
    set fax(numberlistwin) $wn.l.t

    # Key and Mouse bindings

    bind $wn.i.5 <KeyPress-Return> {focus $fax(phonebookwin).i.6}
    bind $wn.i.6 <KeyPress-Return> {focus $fax(phonebookwin).i.7}
    bind $wn.i.7 <KeyPress-Return> {focus $fax(phonebookwin).i.8}
    bind $wn.i.8 <KeyPress-Return> fax_pbook_faxnum_add
    bind $wn.i.5 <KeyPress-Down> {focus $fax(phonebookwin).i.6}
    bind $wn.i.6 <KeyPress-Down> {focus $fax(phonebookwin).i.7}
    bind $wn.i.7 <KeyPress-Down> {focus $fax(phonebookwin).i.8}
    bind $wn.i.6 <KeyPress-Up> {focus $fax(phonebookwin).i.5}
    bind $wn.i.7 <KeyPress-Up> {focus $fax(phonebookwin).i.6}
    bind $wn.i.8 <KeyPress-Up> {focus $fax(phonebookwin).i.7}
    bind $wn.j.3 <KeyPress-Return> {focus $fax(phonebookwin).j.4}
    bind $wn.j.4 <KeyPress-Return> fax_pbook_group_add
    bind $wn.j.3 <KeyPress-Down> {focus $fax(phonebookwin).j.4}
    bind $wn.j.4 <KeyPress-Up> {focus $fax(phonebookwin).j.3}
    bind $fax(numberlistwin) <Button-1>      fax_pbook_faxnum_setcur
    bind $fax(numberlistwin) <Button-3>      fax_pbook_faxnum_unsetcur
    bind $fax(numberlistwin) <Double-Button> fax_pbook_select_faxnumber
    bind $fax(grouplistwin)  <Button-1>      fax_pbook_group_setcur
    bind $fax(grouplistwin)  <Button-3>      fax_pbook_group_unsetcur
    bind $fax(grouplistwin)  <Double-Button> fax_pbook_select_group

    # Initialization

    set fax(curgroupname)		"All"
    set fax(tmpgroupname)		"All"
    set fax(tmpgroupdescription)	"Fax numbers from all groups"
    set fax(tmpnumber)      		""
    set fax(tmprecipient)   		""
    set fax(tmpreference)   		""
    set fax(tmpdescription) 		""
    fax_pbook_display_grouplist
    fax_pbook_display_faxnumberlist All
}

#
# Procedures for phone book window
#

proc fax_pbook_read_faxnumberlist {} {
#
# This procedure reads group list file into the variable fax(grouplist).
# Then it reads each group file into the variable fax(group-$groupname).
#
    global fax

    # Read group list into a variable fax(grouplist).

    set fax(grouplist) {}
    if [file exist $fax(pbookgroupfile)] {
        set fid [open $fax(pbookgroupfile) r]
        while {[set line [gets $fid]]!={}} {
            lappend fax(grouplist) "$line"
        }
    }

    # Read fax number list into variables fax(group-$groupname)

    foreach group $fax(grouplist) {
        set groupname [lindex [split $group \t] 0]
        set fax(group-$groupname) {}
        set file $fax(pbookdir)/$groupname.pbd
        if [file exist $file] {
            set fid [open $file r]
            while {[set line [gets $fid]]!={}} {
                lappend fax(group-$groupname) "$line"
            }
        } else {
            puts "TkFax: Fax list file $file does not exits."
        }
    }

}

proc fax_pbook_save_faxnumberlist {} {
#
# This saves fax number list stored in fax(group-groupname) into a file
# $fax(pbookdir)/groupname.pbd, if changes are made.
#
    global fax
    set gfile "$fax(pbookdir)/group.pbd"
    set gid [open $gfile w]
    foreach group $fax(grouplist) {
        puts $gid $group
        set groupname [lindex $group 0]
        set file "$fax(pbookdir)/$groupname.pbd"
        set fid [open $file w]
        foreach member $fax(group-$groupname) {
            puts $fid $member
        }
        close $fid
    }
    close $gid
}

proc fax_pbook_display_grouplist {} {
#
# This displays group list on $fax(grouplistwin),
#
    global fax
    $fax(grouplistwin) configure -state normal
    $fax(grouplistwin) delete 1.0 end
    $fax(grouplistwin) insert end "All\tFax numbers from all groups\n"
    foreach item $fax(grouplist) {
        $fax(grouplistwin) insert end "$item\n"
    }
    $fax(grouplistwin) delete {end -1 char}
    $fax(grouplistwin) configure -state disabled
}

proc fax_pbook_group_setcur {} {
#
# This shows the group name and description selected by mouse.
#
    global fax
    $fax(grouplistwin) tag delete cur
    $fax(grouplistwin) tag configure cur -foreground red -background skyblue
    $fax(grouplistwin) tag add cur {current linestart} {current lineend +1 char}
    set line [$fax(grouplistwin) get {current linestart} {current lineend}]
    set elements [split $line \t]
    set fax(curgroupname) [lindex $elements 0]
    set fax(tmpgroupname) [lindex $elements 0]
    set fax(tmpgroupdescription) [lindex $elements 1]
    fax_pbook_display_faxnumberlist $fax(tmpgroupname)
}

proc fax_pbook_group_unsetcur {} {
    global fax
    $fax(grouplistwin) tag delete cur
    set fax(tmpgroupname) ""
    set fax(tmpgroupdescription) ""
}

proc fax_pbook_group_clear {} {
    global fax
    set fax(tmpgroupname)		""
    set fax(tmpgroupdescription)	""
    focus $fax(phonebookwin).j.3
}

proc fax_pbook_group_add {} {
    global fax

    # check groupname to see it is non-empty or already used.

    set groupname $fax(tmpgroupname)
    set replace 0
    if {$groupname==""} { return }
    if {$groupname=="All"} {
        tk_messageBox -icon error -message "Group All is reserved." \
            -type ok
        return
    }
    foreach igroup $fax(grouplist) {
        set igroupname [lindex [split $igroup \t] 0]
        if {$groupname==$igroupname} {
            set q [tk_dialog .dialog "TkFax Dialog" \
                   "Group name $groupname is already used." {} 0 {OK}]
            return
        }
    }

    # Add a line to a list and create a new list variable.

    set line "$groupname\t$fax(tmpgroupdescription)"
    $fax(grouplistwin) configure -state normal
    $fax(grouplistwin) insert end \n$line
    $fax(grouplistwin) yview end
    $fax(grouplistwin) configure -state disabled
    lappend fax(grouplist) $line
    set fax(group-$groupname) {}

    # Check whether database file already exists. If so, try to read it in.

    set file $fax(pbookdir)/$groupname.pbd
    if [file exist $file] {
        set q [tk_dialog .dialog "TkFax Dialog" \
  "Phone book database file $file already exists. Do you want to read it in?" \
               {} 0 {Yes} {No}]
        if {$q==0} {
            set fid [open $file r]
            while {[set line [gets $fid]]!={}} {
                lappend fax(group-$groupname) "$line"
            }
        }
    }

    # Ready for another addition.

    focus $fax(phonebookwin).j.3
}

proc fax_pbook_group_remove {} {
    global fax
    set line [$fax(grouplistwin) get cur.first {cur.last -1 char}]
    set groupname [lindex [split $line \t] 0]
    if {$groupname=="All"} {
        tk_messageBox -icon error -message "Group All cannot be removed." \
            -type ok
        return
    }
    $fax(grouplistwin) configure -state normal
    $fax(grouplistwin) delete cur.first cur.last
    $fax(grouplistwin) configure -state disabled
    fax_pbook_reset_grouplist
    unset fax(group-$groupname)
    set file $fax(pbookdir)/$groupname.pbd
    set q [tk_dialog .dialog "TkFax Dialog" \
           "Do you want to delete a phone book database file $file?" \
           {} 0 {Yes} {No}]
    if {$q==0} {
        exec rm -f $file
    }
}

proc fax_pbook_reset_grouplist {} {
    global fax
    set fax(grouplist) {}
    for {set i 2} {$i<[$fax(grouplistwin) index end]} {incr i} {
        lappend fax(grouplist) [$fax(grouplistwin) get $i.0 $i.end]
    }
}

proc fax_pbook_select_group {} {
#
# This select a group for faxing.
#
    global fax
    set fax(number)      group:$fax(tmpgroupname)
    set fax(recipient)   $fax(tmpgroupdescription)
    destroy $fax(phonebookwin)
}

proc fax_pbook_display_faxnumberlist { groupname } {
#
# This displays fax number list of a selected group on $fax(numberlistwin).
#
    global fax
    $fax(numberlistwin) configure -state normal
    $fax(numberlistwin) delete 1.0 end

    if { $groupname == "All" } {
        set fax(group-All) {}
        foreach group $fax(grouplist) {
            set groupname [lindex $group 0]
            foreach member $fax(group-$groupname) {
                lappend fax(group-All) "$member"
            }
        }
        set fax(group-All) \
            [lsort -command fax_recipient_compare $fax(group-All)]
        foreach member $fax(group-All) {
            $fax(numberlistwin) insert end "$member\n"
        }
    } else {
        foreach member $fax(group-$groupname) {
            $fax(numberlistwin) insert end "$member\n"
        }
    }

    $fax(numberlistwin) delete {end -1 char}
    $fax(numberlistwin) configure -state disabled
}

#
# This displays faxnumber element on entry windows when left mouse button is
# pressed on $fax(numberlistwin) and deselect it when right mouse button is
# pressed.

proc fax_pbook_faxnum_setcur {} {
    global fax
    $fax(numberlistwin) tag delete cur
    $fax(numberlistwin) tag configure cur -foreground red -background skyblue
    $fax(numberlistwin) tag add cur \
                                {current linestart} {current lineend +1 char}
    set line [$fax(numberlistwin) get {current linestart} {current lineend}]
    set elements [split $line \t]
    set fax(tmpnumber)      [lindex $elements 0]
    set fax(tmprecipient)   [lindex $elements 1]
    set fax(tmpreference)   [lindex $elements 2]
    set fax(tmpdescription) [lindex $elements 3]
}

proc fax_pbook_faxnum_unsetcur {} {
    global fax
    $fax(numberlistwin) tag delete cur
    set fax(tmpnumber)      ""
    set fax(tmprecipient)   ""
    set fax(tmpreference)   ""
    set fax(tmpdescription) ""
}

#
# These two procedures sort the faxnumberlist.

proc fax_pbook_faxnum_sort {} {
    global fax
    set fax(group-$fax(curgroupname)) \
        [lsort -command fax_recipient_compare $fax(group-$fax(curgroupname))]
    fax_pbook_display_faxnumberlist $fax(curgroupname)
}

proc fax_recipient_compare { a b } {
    set x [lindex [split $a \t] 1]
    set y [lindex [split $b \t] 1]
    return [string compare $x $y]
}

proc fax_pbook_faxnum_clear {} {
#
# This clears entry windows.
#
    global fax
    set fax(tmpnumber)      ""
    set fax(tmprecipient)   ""
    set fax(tmpreference)   ""
    set fax(tmpdescription) ""
    focus $fax(phonebookwin).i.5
}

proc fax_pbook_faxnum_add {} {
#
# This adds a new fax number to the current group.
#
    global fax
    if {$fax(tmpnumber) != {}} {
        $fax(numberlistwin) configure -state normal
        set line \
"$fax(tmpnumber)\t$fax(tmprecipient)\t$fax(tmpreference)\t$fax(tmpdescription)"
        set insertpoint [$fax(numberlistwin) index end]
        $fax(numberlistwin) insert end "\n$line"
        if {[$fax(numberlistwin) get 1.0]=="\n"} {
            $fax(numberlistwin) delete 1.0
        }
        $fax(numberlistwin) yview end
        $fax(numberlistwin) configure -state disabled
        lappend fax(group-$fax(curgroupname)) $line
        fax_pbook_faxnum_clear
        focus $fax(phonebookwin).i.5
    }
}

proc fax_pbook_faxnum_remove {} {
#
# This removes a fax number from the current group.
#
    global fax
    $fax(numberlistwin) configure -state normal
    $fax(numberlistwin) delete cur.first cur.last
    $fax(numberlistwin) configure -state disabled
    fax_pbook_reset_numberlist
}

proc fax_pbook_reset_numberlist {} {
#
# This resets fax(group-groupname) from the fax numbers listed on
# $fax(numberlistwin)
#
    global fax
    set fax(group-$fax(curgroupname)) {}
    for {set i 1} {$i<[$fax(numberlistwin) index end]} {incr i} {
        lappend fax(group-$fax(curgroupname)) \
            [$fax(numberlistwin) get $i.0 $i.end]
    }
}

proc fax_pbook_faxnum_replace {} {
#
# This replaces the selected faxnumber with corrected info.
#
    global fax
    $fax(numberlistwin) configure -state normal
    set insertpoint [$fax(numberlistwin) index cur.first]
    set endpoint [$fax(numberlistwin) index end]
    $fax(numberlistwin) delete cur.first cur.last
    set line \
"$fax(tmpnumber)\t$fax(tmprecipient)\t$fax(tmpreference)\t$fax(tmpdescription)"
    if {$insertpoint<[expr $endpoint-1]} {
        $fax(numberlistwin) insert $insertpoint "$line\n"
    } else {
        $fax(numberlistwin) insert $insertpoint "\n$line"
    }
    $fax(numberlistwin) configure -state disabled
    fax_pbook_reset_numberlist
}

proc fax_pbook_select_faxnumber {} {
#
# This selects current faxnumber for faxing.
#
    global fax
    set fax(number)      $fax(tmpnumber)
    set fax(recipient)   $fax(tmprecipient)
    set fax(reference)   $fax(tmpreference)
    set fax(description) $fax(tmpdescription)
    destroy $fax(phonebookwin)
}

proc fax_pbook_close {} {
#
# This closes the phone book window.
#
    global fax
    destroy $fax(phonebookwin)
}

