/*
 * arc.C
 * 
 * Drawing circular arcs
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static pl_vec pt[3];
static int so_far;

inline pl_angle calc_angle(const pl_vec& p, const pl_vec& center) {
  return angle_of(p - center);
}

static Widget arcVariant;
advertise int arc_variant;

advertise void create_arc_variants(Widget parent)
{
  arcVariant = 
    create_choice(parent, CHOICE_MODE, "inputVariant", NIL, &arc_variant);
  add_to_choice("three points");
  add_to_choice("center, right, left");
  add_to_choice("center, left, right");
}

void Arc::setchoice(void)
{
  manage_variant(arcVariant);
}

static pl_vec three_center(const pl_vec& p1,
			      const pl_vec& p2,
			      const pl_vec& p3)
{
  pl_vec p;
  
  pl_vec q1 = 0.5 * (p2 - p1);
  pl_line l1(p1+q1, normalised(q1).normal());
  pl_vec q2 = 0.5 * (p3 - p1);
  pl_line l2(p1+q2, normalised(q2).normal());
  (void)intersect(p, l1, l2);
  return p;
}

static void draw_arc(const pl_vec& p)
// draw arc depending on input mode
{
  switch (arc_variant) {
  case 0:
    // arc through three points
    if (so_far == 1) {
      draw_line(pt[0], p);
    } else {
      pl_vec q = three_center(pt[0], pt[1], p);
      pl_angle alpha0(calc_angle(pt[0], q));
      pl_angle alpha1(calc_angle(pt[1], q));
      pl_angle alpha2(calc_angle(p, q));
      if (!alpha1.lies_between(alpha0, alpha2)) {
	alpha1 = alpha2;
	alpha2 = alpha0;
	alpha0 = alpha1;
      }
      draw_arc(q, clearance(q, p), alpha0, alpha2);
    }
    return;
  case 1:
  case 2:
    // arc defined by center and two points
    if (so_far == 1) {
      draw_circle(pt[0], clearance(pt[0], p));
    } else {
      pl_angle alpha1(calc_angle((arc_variant == 1 ? pt[1] : p), pt[0]));
      pl_angle alpha2(calc_angle((arc_variant == 2 ? pt[1] : p), pt[0]));
      draw_arc(pt[0], clearance(pt[1], pt[0]), alpha1, alpha2);
    }
    return;
  }
}
      
void Arc::redraw(void)
{
  canvas(COL_CREATION);
  draw_arc(curmp);
}

void Arc::outline(void)
{
  draw_arc(d.center(), d.radius(), d.small_angle(), d.big_angle());
}

void Arc::outline_rotated(const pl_vec& fp, const pl_angle alpha)
{
  pl_arc dt(d);
  pl_rotra tfm = pl_rotra(alpha, fp) * pl_rotra(-fp);
  dt.transform(tfm);
  draw_arc(dt.center(), dt.radius(), dt.small_angle(), dt.big_angle());
}

void Arc::outline_stretched(const s_coord xs, const s_coord ys)
{
  s_coord factor = xs < ys ? xs : ys;
  draw_arc(pl_vec(xs * d.center().x(), ys * d.center().y()),
	   factor * d.radius(), d.small_angle(), d.big_angle());
}

static void arc_draw_arrow(const pl_vec& p, pl_angle alpha,
			   s_coord orientation, s_coord arsize)
{
  pl_vec dir(sin_pp(alpha), -cos_pp(alpha));
  draw_arrow(p + (orientation * dir), p, arsize);
}

void Arc::draw(void)
{
  if (!fill.is_empty() && !transparent_bt.on()) {
    fill.setcol();
#ifdef MIXED
    pl_vecreray appr;
    d.inner_approximation_d(appr, 0.5/(screen_factor.x()*zoom_factor));
    bgn_polygon();
    for (int i = 0; i < appr.size(); i++)
      put_vertex(appr[i]);
    end_polygon();
#else
    draw_filled_arc(d.center(), d.radius(), d.small_angle(), d.big_angle());
#endif
  }
  if (!stroke.is_empty()) {
    stroke.setcol();
    if (arrow & 1)
      arc_draw_arrow(d.begin_vt(), d.begin_angle(), -1, arsize);
    if (arrow & 2)
      arc_draw_arrow(d.end_vt(), d.end_angle(), +1, arsize);
    set_linestyle(dash);
    set_linewidth((lwidth*zoom_factor < 1.0) ? 1 : int(lwidth*zoom_factor));
    draw_arc(d.center(), d.radius(), d.small_angle(), d.big_angle());
  }
}

void Arc::stretch(const s_coord xs, const s_coord ys)
{
  if (xs == ys) {
    d.scale(xs);
    return;
  }
  put_msg("cannot stretch a circular arc");
  d.set_center(pl_vec(xs * d.center().x(), ys * d.center().y()));
  s_coord factor = xs < ys ? xs : ys;
  d.set_radius(factor * d.radius());
}

void Arc::push(pl_vec& mousep, int button, Boolean )
{
  if (drawing_mode != DRAWING) {
    // left mouse button pressed, starting line mode
    clear_selection();
    pt[0] = mousep;
    curmp = mousep;
    so_far = 1;
    nonidle(DRAWING);
    mouse_msg("second point", "", "");
    redraw_pup();
  } else {
    if (button != ((so_far == 1) ? LEFT_MOUSE : MIDDLE_MOUSE))
      return;
    // next point of circle
    pt[so_far++] = mousep;
    curmp = mousep;
    if (so_far < 3) {
      mouse_msg("", "final point", "");
      redraw_pup();
      return;
    }
    // arc is finished, compute and create it
    Arc *ar;
    int arw = arrow_type;
    
    if (arc_variant == 0) {
      pl_vec q = three_center(pt[0], pt[1], pt[2]);
      pl_angle alpha0(calc_angle(pt[0], q));
      pl_angle alpha1(calc_angle(pt[1], q));
      pl_angle alpha2(calc_angle(pt[2], q));
      s_coord diff;
      if (alpha1.lies_between(alpha0, alpha2))
	diff = pl_angle(alpha2-alpha0).normalised_value(0.0);
      else {
	// make sure arc is counterclockwise
	diff = pl_angle(alpha0-alpha2).normalised_value(0.0);
	alpha0 = alpha2;
	if (arw == 1 || arw == 2)
	  arw = 3 - arw;
      }
      ar = new Arc(pl_arc(q, clearance(pt[1], q), alpha0, diff));
    } else {
      // arc_variant == 1 or 2
      pl_angle alpha1(calc_angle(pt[arc_variant] , pt[0]));
      pl_angle alpha2(calc_angle(pt[3 - arc_variant], pt[0]));
      s_coord diff = pl_angle(alpha2-alpha1).normalised_value(0.0);
      ar = new Arc(pl_arc(pt[0], clearance(pt[1], pt[0]), alpha1, diff));
    }

    ar->setcurrent();
    ar->fill.mkempty();
    ar->arrow = arw;
    ar->arsize = arrow_size[mipe_mode];
	     
    pic->add_object(ar);
    object_added();
    idlefy(2);
  }
}

s_coord Arc::dist(const pl_vec& p)
{
  if (fill.is_empty() || !enable_interior_bt.on()) {
    return(clearance(p, d, select_dist()));
  }
  pl_edge e(d.begin_vt(), d.end_vt());
  pl_line l(e.supporting_line());
  if (clearance(p, d.center()) < d.radius()
      && ((d.enclosed_angle().value() > 0) ? is_to_the_right(p,l) :
	  is_to_the_left(p,l)))
    return 0;
  return min(clearance(p,d,select_dist()),
	     clearance(p,e,select_dist()));
}

void Arc::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  snap_vertex(mp, p, d.center());
  snap_vertex(mp, p, d.begin_vt());
  snap_vertex(mp, p, d.end_vt());
}

void Arc::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  snap_vertex(mp, p, d.begin_vt());
  snap_vertex(mp, p, d.end_vt());
  pl_vec v = d.center() + d.radius() * normalised(mp - d.center());
  if (calc_angle(v, d.center()).lies_between(d.small_angle(), d.big_angle())
      && (clearance(mp, v) < clearance(mp, p)))
    p = v;
}

void Arc::close_edge(const pl_vec& mp, Iedge_rr& s)
// compute if arc is close enough and return it
{
  if (clearance(mp, d) < snap_dist())
    s.append(Iedge(d));
}

void Arc::save_postscript(ostream& fh)
// save arc in postscript format
{
  fh << "% Arc\n";
  ps_dash(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
  }
  if (!stroke.is_empty() && arrow) {
    stroke.save(fh, NIL, " ");
    if (arrow & 1) {
      pl_vec dir(sin_pp(d.begin_angle()), -cos_pp(d.begin_angle()));
      fh << d.begin_vt() << " " << arsize << " " << dir << " arw\n";
    }
    if (arrow & 2) {
      pl_vec dir(-sin_pp(d.end_angle()), cos_pp(d.end_angle()));
      fh << d.end_vt() << " " << arsize << " " << dir << " arw\n";
    }
  }
  fh << "% xy\n";
  fh << d.center() << " % r\n";
  fh << d.radius() << " % ang\n";
  fh << (d.begin_angle() / M_PI * 180.0) << " "
     << (d.end_angle() / M_PI  * 180.0) << " np arc\n";
  fill.save(fh, "fi", (stroke.is_empty() ? " fi\n" : " sfi\n"));
  stroke.save(fh, "sk", " sk\n");
  fh << "% End\n\n";
}

void Arc::save_properties(ostream& fh)
// save arc in properties format
{
  fh << "% Arc\n";
  dash_properties(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
  }
  fh << "% xy " << d.center() << "\n";
  fh << "% r " << d.radius() << "\n";
  fh << "% ang ";
  fh << (d.small_angle() / M_PI * 180.0)
     << " " << (d.big_angle() / M_PI  * 180.0) << "\n";
  stroke.save(fh, "sk", NIL);
  fill.save(fh, "fi", NIL);
  fh << "% End\n\n";
}

static char *fieldlist[] = {
  "?2ss", "..", "!2xy", "..", "!1r", "!2ang", "..", "?2ar", ".." };

Arc::Arc(istream& fh)
// read arc item  from file
{
  // no necessary initialization:

  ps_in_value[0] = 65535.0;
  ps_in_value[1] = 1.0;
  ps_in_value[7] = 0.0;
  ps_in_value[8] = arrow_size[mipe_mode];
  ps_okay &= ps_read_entry(fh, 9, fieldlist);
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.read(1);

  dash = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];
  pl_angle alpha0(ps_in_value[5] * M_PI / 180.0);
  pl_angle alpha1(ps_in_value[6] * M_PI / 180.0);
  arrow = int(ps_in_value[7]);
  arsize = ps_in_value[8];
  d = pl_arc(pl_vec(ps_in_value[2], ps_in_value[3]),
	     ps_in_value[4],
	     alpha0, pl_angle(alpha1-alpha0).normalised_value(0.0));
}
