/*
 * bitmap.C
 * 
 * Bitmaps on Ipe's canvas
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

Bitmap::Bitmap(Bitmap& b)
{
  pos    = b.pos;
  ur     = b.ur;
  in_color = b.in_color;
  bits   = b.bits->copy();
  screen = NIL;
}

void Bitmap::outline(void)
{
  draw_rectangle(pos, ur);
}

void Bitmap::outline_rotated(const pl_vec& p, const pl_angle alpha)
{
  push_matrix();
  translate((pl_rotra(alpha, p) * (pos - p)) - pos);
  outline();
  pop_matrix();
}

void Bitmap::draw(void)
{
#ifdef MIXED
  pl_vec size = (ur - pos);
  pl_vec bpos = pos;
  
  if (size.x() < 0) {
    pixmode(PM_RTOL, 1);
    bpos += pl_vec(size.x(), 0);
  }    
  if (size.y() < 0) {
    pixmode(PM_TTOB, 0);
    bpos += pl_vec(0, size.y());
  } else
    pixmode(PM_TTOB, 1);
  
  size = zoom_factor * pl_vec(screen_factor.x() * size.x(),
			      screen_factor.y() * size.y());
  s_coord scrwid = fabs_pp(size.x());
  s_coord scrhei = fabs_pp(size.y());
  s_coord xfactor = int(scrwid / bits->width);
  s_coord yfactor = int(scrhei / bits->height);
  
  if (xfactor < 1.0) xfactor = 1.0;
  int swidth = int(scrwid / xfactor);
  if (yfactor < 1.0) yfactor = 1.0;
  int sheight = int(scrhei / yfactor);

  bpos = bpos + canvas_pan;
  bpos = zoom_factor * pl_vec(screen_factor.x() * bpos.x(),
			      screen_factor.y() * bpos.y()) + canvas_origin;
  
  int mx = int(bpos.x() + 1.0);
  int my = int(bpos.y() + 1.0);
  
  if (!screen || screen->width != swidth || screen->height != sheight) {
    // make new bitmap for screen
    if (screen && screen->unlink()) delete screen;
    
    unsigned long *words = new unsigned long[swidth * sheight];
    int xb, yb;
    s_coord pixwidth = s_coord(swidth - 1.0) / s_coord(bits->width - 1.0);
    s_coord pixheight = s_coord(sheight - 1.0) / s_coord(bits->height - 1.0);
    
    for (int y = 0; y < sheight; y++) {
      for (int x = 0; x < swidth; x++) {
	// compute color at that point
	xb = int(x / pixwidth);
	yb = int(y / pixheight);
	words[swidth * y + x] = bits->bits()[bits->width * yb + xb];
      }
    }
    screen = new BitmapBits(words);
    screen->width = swidth;
    screen->height = sheight;
  }
  rectzoom(xfactor, yfactor);
  lrectwrite(mx, my, mx+swidth-1, my+sheight-1, screen->bits());
  rectzoom(1.0, 1.0);
  pixmode(PM_TTOB, 0);
  pixmode(PM_RTOL, 0);

#else
  // draw bitmap using X ?
  stroke.setcol();
  outline();
#endif
}

void Bitmap::transform(const pl_rotra& tfm)
{
  pl_vec oldpos = pos;
  pos = tfm * pos;
  ur = ur + (pos - oldpos);
}

void Bitmap::stretch(const s_coord xs, const s_coord ys)
{
  pos = pl_vec(xs * pos.x(), ys * pos.y());
  ur = pl_vec(xs * ur.x(), ys * ur.y());
}

s_coord Bitmap::dist(const pl_vec& mp)
{
  if (enable_interior_bt.on()) {
    pl_box bb;
    bb.add_first_point(pos);
    bb.add_point(ur);
    return clearance(mp, bb);
  }
  s_coord d = select_dist(), d1;
  if ((d1 = clearance(mp, pl_edge(pos, pl_vec(ur.x(), pos.y())))) < d)
    return d1;
  if ((d1 = clearance(mp, pl_edge(ur, pl_vec(ur.x(), pos.y())))) < d)
    return d1;
  if ((d1 = clearance(mp, pl_edge(pos, pl_vec(pos.x(), ur.y())))) < d)
    return d1;
  if ((d1 = clearance(mp, pl_edge(ur, pl_vec(pos.x(), ur.y())))) < d)
    return d1;
  return d;
}

static void save_bits(ostream& fh, Boolean in_color, BitmapBits* bits)
{
  long charcnt = 0;
  char buf[3];
  int k, kmax = in_color ? 3 : 1;
  unsigned long word;
  for (int i = 0; i < bits->width*bits->height; i++) {
    word = bits->bits()[i];
    for (k = 0; k < kmax; k++) {
      sprintf(buf, "%.2x", word & 0xff);
      word >>= 8;
      fh << buf;
      charcnt++;
    }
    if (charcnt > 32) {
      charcnt = 0;
      fh << "\n";
    }
  }
}

void Bitmap::save_postscript(ostream& fh)
// save bitmap in postscript format
{
  fh << "% Bitmap\n";
  fh << (in_color ? 3 : 1) * bits->width << " pix\n";
  fh << "% bb\n" << (ur - pos) << "\n% xy\n" << pos
     << " gsts\n"
     << "% px\n" << bits->width << " " << bits->height << "\n";
  fh << "% bits " << bits->width * bits->height
     << (in_color ? " 1\nkimg\n\n" : " 0\nimg\n\n");
  save_bits(fh, in_color, bits);
  fh << "\ngr\n% End\n\n";
}

//
// This is a bit dirty here: If "write_raw_bitmaps" is set,
// Bitmap::save_properties saves in RAW format
// Hopefully I will come up with a cleaner scheme at some point.
//
advertise int write_raw_bitmaps InitializeWith(0);

void Bitmap::save_properties(ostream& fh)
// save bitmap in properties
{
  fh << "% Bitmap\n";
  fh << "% bb " << (ur - pos) << "\n% xy " << pos << "\n"
     << "% px " << bits->width << " " << bits->height << "\n";
  if (!write_raw_bitmaps) {
    // write Postscript style bitmap
    fh << "% bits " << bits->width * bits->height << " "
       << (in_color ? 1 : 0) << "\n";
    save_bits(fh, in_color, bits);
    fh << "\n% End\n\n";
  } else {
    // write RAW bitmap
    long nwords = bits->width * bits->height;
    fh << "% bits " << nwords << " " << (in_color ? 9 : 8) << "\n";
    if (in_color) {
      // write 32 bits per pixel
      fh.write((char *) bits->bits(), sizeof(unsigned long), nwords);
    } else {
      // write 8 bits per pixel
      char *pix = new char[nwords];
      register char *out;
      register unsigned long *inp, *end;
      inp = bits->bits();
      end = inp + nwords;
      out = pix;
      while (inp < end)
	*out++ = char(*inp++);
      fh.write(pix, 1, nwords);
      delete [] pix;
    }
    fh << "\n\n% End\n\n";
  }    
}

static char *fieldlist[] = {
  "!2xy", "..", "!2px", "..", "?2bb", "..", "!$bits" };

static unsigned long *ps_in_bitmap = NIL;
static Boolean is_in_color;

Bitmap::Bitmap(istream& fh)
// read bitmap from file
{
  bits = screen = NIL;
  
  ps_okay &= ps_read_entry(fh, 7, fieldlist);
  if (!ps_okay)
    return;

  stroke.mkblack();
  fill.mkempty();

  pos = pl_vec(ps_in_value[0], ps_in_value[1]);
  if (ps_in_defd[4])
    ur = pos + pl_vec(ps_in_value[4], ps_in_value[5]);
  else
    ur = pos + pl_vec(ps_in_value[2], ps_in_value[3]);
  in_color = is_in_color;
  bits   = new BitmapBits(ps_in_bitmap);
  bits->width  = short(ps_in_value[2]);
  bits->height = short(ps_in_value[3]);

  if (in_color) {
    // is it REALLY a color bitmap ?
    register unsigned long *pix, *end;
    register char r, g, b;
    pix = bits->bits();
    end = pix + bits->width * bits->height;
    while (pix < end) {
      r =  char(*pix);
      g =  char(*pix >> 8);
      b =  char(*pix++ >> 16);
      if (r != g || g != b)
	return;
    }
    in_color = FALSE;
  }
}

advertise void ps_read_bitmap(istream& fh)
{
  ps_in_bitmap = NIL;
  long nwords, mode;
  
  fh >> nwords >> mode;
  is_in_color = Boolean(mode & 1);

  if (mode & 0x08) {
    // this is a RAW bitmap
    ps_in_bitmap = new unsigned long[nwords];
    char ch;
    do {
      fh.get(ch);
    } while (ch != '\n');
    // now comes the bitmap:
    if (mode & 0x01) {
      // nwords x 32 bits
      fh.read((char *) ps_in_bitmap, sizeof(unsigned long), nwords);
    } else {
      // nwords x 8 bits
      char *pix = new char[nwords];
      fh.read(pix, 1, nwords);
      // convert to unsigned longs
      register char *inp, *end;
      register unsigned long *out;
      inp = pix;
      end = pix + nwords;
      out = ps_in_bitmap;
      while (inp < end) {
	*out++ = (*inp << 16) | (*inp << 8) | (*inp);
	inp++;
      }
      delete [] pix;
    }
  } else {
    // this is a bitmap in Postscript style format
  
    long nchars = (is_in_color ? 6 : 2) * nwords;
    char *strbits = new char[nchars];
    char ch, *p = strbits;
    for (int i = 0; i < nchars; i++) {
      do {
	fh.get(ch);
	if (!fh.good()) {
	  ps_okay = FALSE;
	  return;
	}
      } while (!(('0' <= ch && ch <= '9') || ('a' <= ch && ch <= 'f')));
      *p++ = ch;
    }
    p = strbits;
    ps_in_bitmap = new unsigned long[nwords];
    char buf[3];
    buf[2] = '\0';
    short red, green, blue;
    for (i = 0; i < nwords; ) {
      // grab words
      buf[0] = *p++;
      buf[1] = *p++;
      red = short(strtol(buf, NULL, 16));
      if (is_in_color) {
	buf[0] = *p++;
	buf[1] = *p++;
	green = short(strtol(buf, NULL, 16));
	buf[0] = *p++;
	buf[1] = *p++;
	blue = short(strtol(buf, NULL, 16));
	ps_in_bitmap[i++] = (blue * 0x10000) | (green * 0x100) | red;
      } else {
	ps_in_bitmap[i++] = (red * 0x10000) | (red * 0x100) | red;
      }
    }
    delete [] strbits;
  }
}
