/*
 * canvas.C
 * 
 * Creation and event handling for the canvas
 *
 * $Modified: Sunday, September 18, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"
#include <Xm/DrawingA.h>
#include <X11/keysym.h>
#ifdef MIXED
#include <X11/Xirisw/GlxMDraw.h>
#endif

advertise KeyState key_state;

advertise Widget canvasWidget;
advertise Window canvas_window;
advertise XPoint canvas_size;

#ifdef MIXED
////////////////////////// GL mode //////////////////////////////

static Window popup_window;

static GLXconfig glxConfig [] = {
  { GLX_OVERLAY, GLX_BUFSIZE, 2},
  { GLX_NORMAL,  GLX_RGB,    TRUE },
  { GLX_NORMAL,  GLX_DOUBLE, FALSE },
  { 0, 0, 0}
};

static Boolean use_pups=FALSE; // if TRUE, use popup instead of overlay

#endif

///////////////////// POP UP PLANE //////////////////////////////

static int pup_bit = 1;

static void setup_canvas(void)
{
  push_matrix();
  translate(canvas_origin);
  scale(screen_factor);
  scale(zoom_factor);
  translate(canvas_pan);
}

advertise void canvas(short cidx)
{
  sys_color[cidx].syscol(pup_bit);
  sys_color[cidx].syscol(3);
  set_color(pup_bit);
}

////////////////////////////// FIFI //////////////////////////////

static pl_vec last_mousep;
static pl_vec fifipos;

static void draw_fifi()
// show a little cross at snapping location, if snap is on
{
  if ((drawing_mode == IDLE || drawing_mode == MOVING ||
       (drawing_mode == DRAWING && curobj->type() != TEXT)) &&
      enable_fifi_bt.on() &&
      (snap_vertex_bt.on() ||
       snap_boundary_bt.on() ||
       snap_crossing_bt.on() ||
       snap_grid_bt.on() ||
       snap_directional_bt.on() ||
       snap_autodir_bt.on())) {
    // draw fifi in overlay plane
    s_coord hs = 4.0/zoom_factor;
    draw_line(fifipos + pl_vec(-hs, 0),
	      fifipos + pl_vec(hs, 0));
    draw_line(fifipos + pl_vec(0, -hs),
	      fifipos + pl_vec(0, hs));
  }
}

////////////////////// POPUP REDRAW HANDLER //////////////////

static void canvas_popup_callback()
{
  setup_canvas();
  // manage pop up planes double buffering
  write_mask(pup_bit);
  push_matrix();
#ifdef MIXED
  set_linestyle(0, FALSE);
  set_linewidth(1);
#else
  current_gc = popup_gc;
#endif

  switch (drawing_mode) {
  case MOVING:
    moving_redraw();
    break;
  case DRAWING:
    curobj->redraw();
    break;
  case DRAGGING:
    dragging_redraw();
    break;
  default:
    canvas(COL_SELECTION);
    redraw_selection();
    break;
  }
  pop_matrix();
  draw_fifi();
  pup_bit = 3 - pup_bit;
  write_mask(pup_bit);
#ifdef MIXED
  color(0);
  clear();
#else
  if (!monochrome) {
    set_color(0);
    XFillRectangle(display, canvas_window, popup_gc,
		   0, 0, canvas_size.x + 1, canvas_size.y + 1);
  }
#endif
  pop_matrix();
}

////////////////////////// NON IDLE /////////////////////////

advertise void nonidle(Drawing_mode m)
// set drawing_mode
{
  drawing_mode = m;
}

advertise void idlefy(int how)
// return to IDLE mode, reset things, redraw if wanted
{
  drawing_mode = IDLE;
  if (edit_object) {
    edit_object = FALSE;
    curobj = saved_curobj;
  }
  standard_mouse_msg();

  switch (how) {
  case 0:
    break;
  case 1:
    redraw_pup();
    break;
  case 2:
    redraw_canvas();
    break;
  }
}

//////////////////////////// REDRAW HANDLER //////////////////////////////

static void canvas_redraw_callback(void)
{
  DEBUG(DBG_CB, "REDRAW canvas callback", "");
  
  setup_canvas();
#ifdef MIXED
  // redraw background
  sys_color[COL_BACKGROUND].setcol();
  clear();
#else
  // in X, the background has already been cleared ???
  XClearWindow(display, canvas_window);
  current_gc = canvas_gc;
#endif
  // draw grid if necessary
  if (grid_visible)
    draw_grid();
  // redraw template, if necessary
  if (page_number > 0) pages[0]->nontext_draw();
  // redraw objects of current page
  pic->nontext_draw();
  if (page_number > 0) pages[0]->text_draw();
  pic->text_draw();
  draw_fixpoint();
  pop_matrix();
}

//////////////////////////// MOUSE POSITION /////////////////////////////

static pl_vec current_mouse_position(0,0);

advertise pl_vec get_mouse(s_coord mx, s_coord my)
{
  pl_vec mousep = pl_vec(mx, my) - canvas_origin;
  mousep = pl_vec(mousep.x() / (zoom_factor * screen_factor.x()),
		  mousep.y() / (zoom_factor * screen_factor.y()));
  mousep = mousep - canvas_pan;
  return mousep;
}

static pl_vec get_mouse(XEvent *event)
{
  int x, y;
  switch (event->type) {
  case KeyPress:
  case KeyRelease:
    x = event->xkey.x;
    y = event->xkey.y;
    break;
  case ButtonPress:
  case ButtonRelease:
    x = event->xbutton.x;
    y = event->xbutton.y;
    break;
  case MotionNotify:
    x = event->xmotion.x;
    y = event->xmotion.y;
    break;
  }
  current_mouse_position = get_mouse(x, canvas_size.y - y);
  return current_mouse_position;
}

advertise pl_vec get_mouse(void)
{
  return current_mouse_position;
}

//////////////////////////// INPUT HANDLER //////////////////////////////

static void update_key_state(XEvent *event, KeySym *pkeysym)
// should be called for Enter, KeyPress, and KeyRelease
{
  Boolean old_shift   = key_state.shift;
  Boolean old_control = key_state.control;
  if (event->type == EnterNotify || event->type == LeaveNotify) {
    // Enter/Leave Event
    key_state.shift   = ((event->xcrossing.state & ShiftMask) != 0);
    key_state.control = ((event->xcrossing.state & ControlMask) != 0);
  } else {
    // KeyPress or KeyRelease
    key_state.shift   = ((event->xkey.state & ShiftMask) != 0);
    key_state.control = ((event->xkey.state & ControlMask) != 0);

    KeySym keysym;
    Modifiers mod_ret;
    XtTranslateKeycode(display, event->xkey.keycode,
		       event->xkey.state, &mod_ret, &keysym);
    Boolean pressed = (event->type == KeyPress);
    switch (keysym) {
    case XK_Shift_L:
    case XK_Shift_R:
      key_state.shift = pressed;
      break;
    case XK_Control_L:
    case XK_Control_R:
      key_state.control = pressed;
      break;
    }
    if (pkeysym)
      *pkeysym = keysym;
  }
  // if idle and key_state has changed: update mouse message
  if (drawing_mode == IDLE
      && (key_state.shift != old_shift
	  || key_state.control != old_control)) {
    standard_mouse_msg();
  }
}

//////////////////////////// INPUT HANDLER //////////////////////////////

static void input_callback(Widget, XtPointer, XtPointer call_data)
{
#ifdef MIXED
  GlxDrawCallbackStruct *w = (GlxDrawCallbackStruct *) call_data;
#else
  XmDrawingAreaCallbackStruct *w = (XmDrawingAreaCallbackStruct *) call_data;
#endif

  // when Ipe is busy saving or loading a figure,
  // or just refreshing the display in the middle of a long operation,
  // we ignore all X input events on the canvas
  if (drawing_mode == BUSY)
    return;
  
  switch (w->event->type) {

    ////////////////////// KEYBOARD EVENT ///////////////////////////
  case KeyRelease:
    KeySym keysym;
    update_key_state(w->event, &keysym);
    DEBUG(DBG_CB, "KeyRelease with keysym ", keysym);
    // I don't know why the following should be necessary
    keysym &= 0xffff;
    // if M-ESC: cancel everything that is going on right now
    if ((keysym == XK_Escape || keysym == XK_Cancel)
	&& (w->event->xkey.state & Mod1Mask)) {
      if (drawing_mode == USERMACRO)
	kill_ium();
      idlefy(2);
      show_operation("abort!", "M-ESC");
      return;
    }
    if (keysym == XK_Return && drawing_mode == DRAWING) {
      curobj->key(0x0d);
      return;
    }
    return;
  case KeyPress:
    {
      KeySym keysym;
      update_key_state(w->event, &keysym);
      pl_vec mousep = get_mouse(w->event);
      // I don't know why the following should be necessary
      keysym &= 0xffff;
      DEBUG(DBG_CB, "KeyPress with keysym ", keysym);
      
      char key;
      if (keysym >= XK_space && keysym <= XK_asciitilde) {
	key  = char(keysym);
	if (w->event->xkey.state & ControlMask)
	  key &= 0x1f;
	if (w->event->xkey.state & Mod1Mask)
	  key |= 0x80;
      } else {
	switch (keysym) {
	case XK_BackSpace:
	  key = 'h' & 0x1f;
	  break;
	case XK_Left:
	  key = 'b' & 0x1f;
	  break;
	case XK_Right:
	  key = 'f' & 0x1f;
	  break;
	case XK_Up:
	  key = 'p' & 0x1f;
	  break;
	case XK_Down:
	  key = 'n' & 0x1f;
	  break;
	case XK_Home:
	  key = '<' | 0x80;
	  break;
	case XK_End:
	  key = '>' | 0x80;
	  break;
	default:
	  return;
	}
      }
      put_msg("");
      if (drawing_mode == DRAWING) {
	curobj->key(key);
	return;
      }
      if (drawing_mode == SELECTING && key == ' ')
	selecting_key();
      else
	shortcut(key);
      return;
    }
    ////////////////////// MOUSE PUSHED ///////////////////////////
  case ButtonPress:
    {
      // flush message area
      put_msg("");

      update_key_state(w->event, NIL);
      
      pl_vec mousep = get_mouse(w->event);
      int button = w->event->xbutton.button;
      DEBUG(DBG_CB, "ButtonPress at mouse ", mousep);
      
      switch (drawing_mode) {
      case IDLE:
	if (button == LEFT_MOUSE &&
	    !key_state.shift && !key_state.control) {
	  // starting to draw something
	  snap(mousep);
	  curobj->push(mousep, button, key_state.shift);
	  return;
	}
	if (button == RIGHT_MOUSE) {
	  // start SELECTING mode
	  selecting_push(mousep, key_state.shift, key_state.control);
	  return;
	}
	// we are starting MOVING mode
	moving_push(mousep, key_state.shift, key_state.control, button);
	return;
      case DRAWING:
	if (curobj->type() != TEXT) {
	  snap(mousep);
	}
	if (edit_object)
	  curobj->edit_push(mousep, button, key_state.shift);
	else
	  curobj->push(mousep, button, key_state.shift);
	return;
      case MOVING:
	moving_done(mousep);
	return;
      case SELECTING:
	selecting_done();
	return;
      case DRAGGING:
	dragging_done(mousep);
      case USERMACRO:
      case BUSY:
	return;
      }
      return;
    }
    ////////////////////// MOUSE RELEASED ///////////////////////////
  case ButtonRelease:
    { pl_vec mousep = get_mouse(w->event);
      DEBUG(DBG_CB, "ButtonRelease at mouse ", mousep);
      switch (drawing_mode) {
      case IDLE:
	return;
      case DRAWING:
	snap(mousep);
	curobj->release(mousep);
	return;
      case MOVING:
	moving_done(mousep);
	return;
      case SELECTING:
	selecting_done();
	return;
      case DRAGGING:
	dragging_done(mousep);
      case USERMACRO:
      case BUSY:
	return;
      }
      return;
    }
    ////////////////////// MOUSE MOVED ///////////////////////////
  case MotionNotify:
    { pl_vec mousep = get_mouse(w->event);
      int any_button = w->event->xmotion.state &
	(Button1Mask | Button2Mask | Button3Mask);

      // mouse has been moved ?
      if (mousep == last_mousep)
	return;
      last_mousep = mousep;

#ifndef MIXED
      // monochrome: erase old pop up plane
      if (monochrome) {
//	XSetForeground(display, popup_gc, WhitePixel(display, screen_num));
	if (drawing_mode != IDLE)
	  canvas_popup_callback();
      }
#endif
      
      switch (drawing_mode) {
      case IDLE:
	snap(mousep);
	break;
      case DRAWING:
	snap(mousep);
	curobj->mouse(mousep, any_button);
	break;
      case MOVING:
	snap(mousep);
	moving_mouse(mousep);
	break;
      case SELECTING:
	selecting_mouse(mousep);
	break;
      case DRAGGING:
	dragging_mouse(mousep);
	break;
      case USERMACRO:
      case BUSY:
	break;
      }
      fifipos = mousep;
      // in IDLE mode, this redraw is only necessary for fifi
#ifdef MIXED
      redraw_pup();
#else
      if (monochrome) {
//	XSetForeground(display, popup_gc, BlackPixel(display, screen_num));
	if (drawing_mode != IDLE)
	  canvas_popup_callback();
      } else
	redraw_pup();
#endif
      return;
    }
  case EnterNotify:
    update_key_state(w->event, NIL);
//    show_mouse_msg();
    return;
  case LeaveNotify:
    update_key_state(w->event, NIL);
//    blank_mouse_msg();
    return;
  }
}

#ifdef MIXED
///////////////// canvas Widget callbacks GL Version /////////////////

static void ginit_resize_callback(Widget, XtPointer client_data,
				  XtPointer call_data)
{
  GlxDrawCallbackStruct *w = (GlxDrawCallbackStruct *) call_data;
  int is_resize = int(client_data);
  
  DEBUG(DBG_GL, "CALLBACK: GL ginit/resize on window ", w->window);

  // get canvas window
  canvas_window = XtWindow(canvasWidget);

  concave(TRUE);

  GLXwinset(display, w->window);
  viewport(0, (Screencoord) w->width-1, 0, (Screencoord) w->height-1);
  ortho2(-0.5, w->width-0.5, -0.5, w->height-0.5);
  canvas_size.x = w->width;
  canvas_size.y = w->height;
  canvas_origin = pl_vec(rint_pp(w->width / 2.0), rint_pp(w->height / 2.0));
  if (is_resize) canvas_redraw_callback();
  
  GLXwinset(display, popup_window);
  viewport(0, (Screencoord) w->width-1, 0, (Screencoord) w->height-1);
  ortho2(-0.5, w->width-0.5, -0.5, w->height-0.5);
  if (is_resize) {
    canvas_popup_callback();
    gflush();
  }
}

static void expose_callback(Widget, XtPointer, XtPointer call_data)
{
  GlxDrawCallbackStruct *w = (GlxDrawCallbackStruct *) call_data;
  GLXwinset(display, w->window);
  DEBUG(DBG_GL, "CALLBACK: GL Expose on window ", w->window);

  canvas_redraw_callback();
  gflush();
}

static void popup_callback(Widget, XtPointer, XtPointer call_data)
{
  GlxDrawCallbackStruct *w = (GlxDrawCallbackStruct *) call_data;
  DEBUG(DBG_GL, "CALLBACK: GL popup on window ", w->window);

  GLXwinset(display, w->window);
  canvas_popup_callback();
  gflush();
}

#else
///////////////// canvas Widget callbacks X Version /////////////////

static void resize_callback(Widget, XtPointer, XtPointer)
{
  // get canvas window
  canvas_window = XtWindow(canvasWidget);
  if (!canvas_window)
    return;
  
  DEBUG(DBG_GFX, "CALLBACK: X resize on canvas window ", canvas_window);

  resize_canvas();
}

static void expose_callback(Widget, XtPointer, XtPointer call_data)
{
  XmDrawingAreaCallbackStruct *w = (XmDrawingAreaCallbackStruct *) call_data;
  
  DEBUG(DBG_GFX, "CALLBACK: X Expose on window ", w->window);
  canvas_redraw_callback();
  canvas_popup_callback();
}
#endif

#ifdef MIXED
///////////////////// create canvas GL VERSION  /////////////////////////

advertise void create_canvas_widget(Widget frame)
{
  // check if we must use pups
  if (getgdesc(GD_BITS_OVER_SNG_CMODE) < 2) {
    use_pups = TRUE;
    glxConfig[0].buffer = GLX_POPUP;
  }

  canvasWidget =
    XtVaCreateManagedWidget("canvas",
			    glxMDrawWidgetClass, frame,
			    GlxNglxConfig, glxConfig,
			    (use_pups ? GlxNusePopup : GlxNuseOverlay), True,
			    XmNnavigationType, XmEXCLUSIVE_TAB_GROUP,
			    NULL);
  
  XtAddCallback(canvasWidget, GlxNginitCallback,  ginit_resize_callback, 0);
  XtAddCallback(canvasWidget, GlxNresizeCallback,
		ginit_resize_callback, XtPointer(1));
  XtAddCallback(canvasWidget, GlxNexposeCallback, expose_callback, 0);
  XtAddCallback(canvasWidget,
		use_pups ? GlxNpopupExposeCallback : GlxNoverlayExposeCallback,
		popup_callback, 0);
  XtAddCallback(canvasWidget, GlxNinputCallback, input_callback, 0);
}

advertise void install_gl_interface(void)
{
  // save gl window and popup window
  XtVaGetValues(canvasWidget,
		(use_pups ? GlxNpopupWindow : GlxNoverlayWindow), &popup_window, NIL);
  
  // Install Gl colormap
  Window windows[3];
  windows[0] = popup_window;
  windows[1] = canvas_window;
  windows[2] = XtWindow(topLevel);
  XSetWMColormapWindows(XtDisplay(topLevel), XtWindow(topLevel),
			windows, 3);
}

#else
///////////////////// create canvas X VERSION  /////////////////////////

advertise void create_canvas_widget(Widget frame)
{
  canvasWidget =
    XtVaCreateManagedWidget("canvasX",
			    xmDrawingAreaWidgetClass, frame,
			    XmNnavigationType, XmEXCLUSIVE_TAB_GROUP,
			    XmNresizePolicy, XmRESIZE_ANY,
			    XmNbackground,  sys_color[COL_BACKGROUND].pixel(),
			    NULL);
  
  XtAddCallback(canvasWidget, XmNexposeCallback, expose_callback, 0);
  XtAddCallback(canvasWidget, XmNinputCallback,  input_callback,  0);
  XtAddCallback(canvasWidget, XmNresizeCallback, resize_callback, 0);
}

#endif  

///////////////////////// FORCE A REDRAW /////////////////////////    

advertise void redraw_canvas(void)
{
#ifdef MIXED  
  GLXwinset(display, canvas_window);
  canvas_redraw_callback();
  GLXwinset(display, popup_window);
  canvas_popup_callback();
#else
  canvas_redraw_callback();
  canvas_popup_callback();
#endif
}

advertise void redraw_pup(void)
{
#ifdef MIXED
  // Force redrawing popup plane, set popup window and call callback
  GLXwinset(display, popup_window);
#else
  if (monochrome)
    canvas_redraw_callback();
#endif
  canvas_popup_callback();
}

