/*
 * circle.C
 * 
 * Drawing circles
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".

 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static pl_vec pt[3];
static int so_far;

#ifdef MIXED
static void multm(Transform &tfm)
// apply matrix to window
{
  Matrix M;
  M[0][0] = tfm.A[0];
  M[0][1] = tfm.A[1];
  M[0][2] = 0;
  M[0][3] = 0;
  M[1][0] = tfm.A[2];
  M[1][1] = tfm.A[3];
  M[1][2] = 0;
  M[1][3] = 0;
  M[2][0] = 0;
  M[2][1] = 0;
  M[2][2] = 1;
  M[2][3] = 0;
  M[3][0] = tfm.A[4];
  M[3][1] = tfm.A[5];
  M[3][2] = 0;
  M[3][3] = 1;
  multmatrix(M);
}
#else
inline void multm(const Transform &tfm)
{
  tfstack[tfsp].premult(tfm);
}
#endif

static void tfmsave(ostream& fh, Transform &tfm)
{
  fh << tfm.A[0] << " " << tfm.A[1] << " " << tfm.A[2] << " " << tfm.A[3];
}

static Widget circleVariant;
advertise int circle_variant;

advertise void create_circle_variants(Widget parent)
{
  circleVariant = 
    create_choice(parent, CHOICE_MODE, "inputVariant", NIL, &circle_variant);
  add_to_choice("by radius");
  add_to_choice("diameter");
  add_to_choice("three points");
}

void Circle::setchoice(void)
{
  manage_variant(circleVariant);
}

static pl_vec three_center(const pl_vec& p1,
			      const pl_vec& p2,
			      const pl_vec& p3)
{
  pl_vec p;
  
  pl_vec q1 = 0.5 * (p2 - p1);
  pl_line l1(p1+q1, normalised(q1).normal());
  pl_vec q2 = 0.5 * (p3 - p1);
  pl_line l2(p1+q2, normalised(q2).normal());
  (void)intersect(p, l1, l2);
  return p;
}

static void draw_circle(const pl_vec& p)
// draw circle depending on input mode
{
  pl_vec q;
  
  switch (circle_variant) {
  case 0:
    // circle by radius
    draw_circle(pt[0], clearance(pt[0], p));
    break;
  case 1:
    // circle by diameter
    q = 0.5 * (p + pt[0]);
    draw_circle(q, clearance(q, p));
    break;
  case 2:
    // circle through three points
    if (so_far == 1) {
      draw_line(pt[0], p);
    } else {
      q = three_center(pt[0], pt[1], p);
      draw_circle(q, clearance(q, p));
    }
    break;
  }
}

void Circle::redraw(void)
{
  canvas(COL_CREATION);
  draw_circle(curmp);
}

void Circle::outline(void)
{
  push_matrix();
  multm(tfm);
  // set elliptic flag!
  draw_circle(pl_vec(0, 0), 1.0, TRUE);
  pop_matrix();
}

void Circle::draw(void)
{
  push_matrix();
  multm(tfm);
  if (!fill.is_empty() && !transparent_bt.on()) {
    fill.setcol();
    // set elliptic flag
    draw_filled_circle(pl_vec(0, 0), 1.0, TRUE);
  }
  if (!stroke.is_empty()) {
    stroke.setcol();
    set_linestyle(dash);
    set_linewidth((lwidth*zoom_factor < 1.0) ? 1 : int(lwidth*zoom_factor));
    // set elliptic flag
    draw_circle(pl_vec(0, 0), 1.0, TRUE);
  }
  pop_matrix();
}

void Circle::transform(const pl_rotra& tf)
{
  tfm.transform(tf);
}

void Circle::stretch(const s_coord xs, const s_coord ys)
{
  tfm.stretch(xs, ys);
}

void Circle::push(pl_vec& mousep, int button, Boolean )
{
  pl_vec q;
  
  if (drawing_mode != DRAWING) {
    // left mouse button pressed, starting line mode
    clear_selection();
    pt[0] = mousep;
    curmp = mousep;
    so_far = 1;
    nonidle(DRAWING);
    if (circle_variant == 2)
      mouse_msg("next point", "", "");
    else
      mouse_msg("", "other point", "");
    
  } else {
    if (button != ((circle_variant == 2 && so_far == 1)
		   ? LEFT_MOUSE : MIDDLE_MOUSE))
      return;
    // next point of circle
    curmp = mousep;
    pt[so_far++] = mousep;
    if (circle_variant == 2 && so_far < 3) {
      mouse_msg("", "final point", "");
      redraw_pup();
      return;
    }
    // circle is finished, compute and create it
    Circle *ci = new Circle;
    ci->fill.setcurrent(TRUE);
    ci->stroke.setcurrent(FALSE);
    if (!ci->fill.is_empty() && !enable_stroke_bt.on())
      ci->stroke.mkempty();
    ci->dash = linestyle_choice;
    ci->lwidth = line_width_list[line_width_choice];
    pl_disc d;
    
    switch (circle_variant) {
    case 0:
      // by radius
      d.set_center(pt[0]);
      d.set_radius(clearance(pt[1], pt[0]));
      break;
    case 1:
      // by diameter
      q = 0.5 * (pt[1] + pt[0]);
      d.set_center(q);
      d.set_radius(clearance(pt[1], q));
      break;
    case 2:
      // circle through three points
      q = three_center(pt[0], pt[1], pt[2]);
      d.set_center(q);
      d.set_radius(clearance(pt[1], q));
      break;
    }
    ci->tfm.stretch(d.radius(), d.radius());
    ci->tfm.transform(pl_rotra(d.center()));
    pic->add_object(ci);
    object_added();
    idlefy(2);
  }
}

s_coord Circle::radius(void)
// return radius of circle. For ellipses, returns rough estimate
{
  pl_vec e1(1,0), e2(0,1);
  tfm.lapply(e1);
  tfm.lapply(e2);
  return max(e1.length(), e2.length());
}

s_coord Circle::dist(const pl_vec& p)
{
  pl_vec pp(p);
  tfm.inverse().apply(pp);
  if (pp.length() <= 1.0 && !fill.is_empty() && enable_interior_bt.on())
    return 0;
  pl_vec q = normalised(pp);
  tfm.apply(q);
  return(clearance(p,q));
}

void Circle::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  pl_vec q(mp);
  tfm.inverse().apply(q);
  q = normalised(q);
  tfm.apply(q);
  if (clearance(mp, q) < clearance(mp, p))
    p = q;
}

void Circle::close_edge(const pl_vec& mp, Iedge_rr& s)
{
  if (tfm.is_similar() &&
      fabs_pp(clearance(mp, (tfm.transl())) - radius()) < snap_dist())
    s.append(pl_disc(tfm.transl(), radius()));
}

void Circle::save_postscript(ostream& fh)
// save circle in postscript format
{
  fh << "% Circle\n";
  ps_dash(fh);
  if (tfm.is_similar()) {
    // circle
    fh << "np % xy\n";
    fh << tfm.transl() << " % r\n";
    fh << radius() << " ci\n";
  } else {
    s_coord r = radius();
    Transform A(tfm);
    A.stretch(1.0/r, 1.0/r);
    fh << "% r\n" << r << " [ % tfm\n";
    tfmsave(fh, A);
    fh << " % xy\n" << tfm.transl() << " ] el\n";
  }
  fill.save(fh, "fi", (stroke.is_empty() ? " fi\n" : " sfi\n"));
  stroke.save(fh, "sk", " sk\n");
  if (!tfm.is_similar())
    fh << "gr\n";
  fh << "% End\n\n";
}

void Circle::save_properties(ostream& fh)
// save circle in properties format
{
  fh << "% Circle\n";
  dash_properties(fh);
  fh << "% xy ";
  fh << tfm.transl() << "\n";
  fh << "% r " << radius() << "\n";
  if (!tfm.is_similar()) {
    s_coord r = radius();
    Transform A(tfm);
    A.stretch(1.0/r, 1.0/r);
    fh << "% tfm\n";
    tfmsave(fh, A);
    fh << "\n";
  }
  stroke.save(fh, "sk", NIL);
  fill.save(fh, "fi", NIL);
  fh << "% End\n\n";
}

void Circle::bbox(pl_boundingbox& b)
{
  pl_angle beta;
  pl_vec dir;
  pl_unitvec exdir;
  pl_vec ex;
  for (beta = 0; beta < M_PI; beta = beta + M_PI_2) {
    dir = pl_vec(cos_pp(beta), sin_pp(beta));
    tfm.inverse().lapply(dir);
    exdir = normalised(dir).normal();
    ex = exdir;
    tfm.apply(ex);
    b.add_point(ex);
    ex = -exdir;
    tfm.apply(ex);
    b.add_point(ex);
  }
}

static char *fieldlist[] = {
  "?2ss", "..", "!2xy", "..", "!1r", "?[tfm" };

Circle::Circle(istream& fh)
// read circle item  from file
{
  // no necessary initialization:

  ps_in_value[0] = 65535.0;
  ps_in_value[1] = 1.0;
  ps_okay &= ps_read_entry(fh, 6, fieldlist);
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.read(1);

  dash = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];
  if (ps_in_defd[5]) {
    tfm = ps_in_transform;
  } 
  tfm.stretch(ps_in_value[4], ps_in_value[4]);
  tfm.transform(pl_vec(ps_in_value[2], ps_in_value[3]));
}


  
