/*
 * colors.C
 * 
 * definition of colors, and handling of color buttons
 *
 * $Modified: Sunday, September 18, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>

static Widget colorButton[17];

IpeColor *black_color, *white_color;
static IpeColor *first_color;

static Colormap xcolormap;
static Visual *visual;

static unsigned long ipe_pixel[MAX_COLORS];
static int num_colors = 0, next_pixel = 2;
#ifndef MIXED
static unsigned long overlay_plane[2];
#endif

static IColor color_choice[17];
advertise IColor sys_color[COL_LAST];

static char xLations[] = "#replace\
<Btn1Down>:	Arm()\n\
<Btn1Up>:	Activate() Disarm()\n\
<Btn2Down>:	Arm()\n\
<Btn2Up>:	Activate() Disarm()\n\
<EnterWindow>:	Enter()\n\
<LeaveWindow>:	Leave()";

static XtTranslations parsed_xlations;

static void set_color_ball(int j)
{
  XtVaSetValues(colorButton[j],
		XtVaTypedArg,
		XmNlabelString, XmRString,
		(j == current_color[1] ? "F" :
		 (j == current_color[0]) ? "S" : " "), 2, NIL);
}  

void color_cb(Widget, XtPointer client_data, XtPointer call_data)
{
  Iobject *ob;
  int isfill = 1;
  XEvent *event = ((XmPushButtonCallbackStruct *) call_data)->event;

  DEBUG(DBG_CB, "CALLBACK: color ", int(client_data));
  if (event->type == ButtonRelease &&
      ((event->xbutton.state & ShiftMask) ||
       (event->xbutton.button == MIDDLE_MOUSE)))
    isfill = 0;
  DEBUG(DBG_CB, "CALLBACK: isfill = ", isfill);

  int old_color = current_color[isfill];
  current_color[isfill] = int(client_data);
  set_color_ball(old_color);
  set_color_ball(current_color[isfill]);
  
  if (selection()) {
    IColor col;
    col.setcurrent(isfill);
    for (ob = pic->first; ob; ob = ob->next)
      if (ob->sel != UNSELECTED)
	if (isfill)
	  ob->setfill(col);
	else
	  ob->setstroke(col);
    undo_changed();
    redraw_canvas();
  }
}

//
//----------------------------------------------------------------------
//

unsigned long IColor::pixel(void)
{
  if (col == NIL) return 0;
  if (!col->pixeled) mkpixel();
  return col->pixel;
}

void IColor::setcurrent(Boolean flag)
// set to current color
{
  col = color_choice[current_color[flag]].col;
}

//
//----------------------------------------------------------------------
//
// COLOR I/O
//

void IColor::save(ostream& fh, char *save_ipe, char *save_ps)
// save color in file
{
  if (!col)
    return;
  Boolean gray = (col->red == col->green && col->red == col->blue);

  if (save_ipe) {
    fh << "% " << save_ipe << (gray ? "" : "c");
    if (save_ps)
      fh << "\n";
    else if (gray)
      fh << " " << col->red << "\n";
    else
      fh << " " << col->red << " " << col->green << " " << col->blue << "\n";
  }
  if (save_ps) {
    if (gray)
      fh << col->red << " sg" << save_ps;
    else
      fh << col->red << " " << col->green << " " << col->blue
	 << " sc" << save_ps;
  }
}

void IColor::save_latex(ostream& fh)
// save color in Latex format for ipe.sty
{
  if (col == black_color)
    fh << "\\IPEfs";
  else
    fh << "\\IPEcolfs{"
       << col->red << " " << col->green << " " << col->blue << "}";
}

void IColor::read(int a)
// read color from file
{
  col = find_color(ps_in_color[a][0], ps_in_color[a][1], ps_in_color[a][2]);
}

//
//----------------------------------------------------------------------
//
//  SETTING colors
//

void IColor::setcol(void)
{
#ifdef MIXED
  float cv[3];
  cv[0] = col->red; cv[1] = col->green; cv[2] = col->blue;
  c3f(cv);
#else
  XSetForeground(display, current_gc, pixel());
#endif
}

void IColor::syscol(short idx)
{
#ifdef MIXED
  mapcolor(idx, short(255*col->red),
	   short(255*col->green), short(255*col->blue));
#else
  if (!monochrome) {
    // map such that overlay combination i gives this color
    XColor xcolor;
    xcolor.flags = (DoRed | DoGreen | DoBlue);
    xcolor.red = (unsigned short)   (65535.0 * col->red);
    xcolor.green = (unsigned short) (65535.0 * col->green);
    xcolor.blue = (unsigned short)  (65535.0 * col->blue);
    unsigned long pixel = 0;
    if (idx & 1) pixel |= overlay_plane[0];
    if (idx & 2) pixel |= overlay_plane[1];
    
    for (int j = 0; j < next_pixel; j++) {
      xcolor.pixel = ipe_pixel[j] | pixel;
      XStoreColor(display, xcolormap, &xcolor);
    }
  }
#endif
}

#ifndef MIXED
advertise void set_color(short idx)
{
  if (monochrome) return;
  unsigned long pixel = 0;
  if (idx & 1)
    pixel |= overlay_plane[0];
  if (idx & 2)
    pixel |= overlay_plane[1];
  //  DEBUG(DBG_GFX, "Using foreground color for popup: ", pixel);
  XSetForeground(display, popup_gc, pixel);
}

advertise void write_mask(unsigned long idx)
{
  if (monochrome) return;
  unsigned long pixel = 0;
  if (idx & 1)
    pixel |= overlay_plane[0];
  if (idx & 2)
    pixel |= overlay_plane[1];
//  DEBUG(DBG_GFX, "Writemask for popup set: ", pixel);
  XSetPlaneMask(display, popup_gc, pixel);
}
#endif

//
//----------------------------------------------------------------------
//
// COLOR ALLOCATION
//

static Boolean message_shown = FALSE;

//
// mkpixel is only called when this color has never been allocated before
//
void IColor::mkpixel(void)
{
  XColor xcolor;
  xcolor.red = short (65535.0 * col->red);
  xcolor.green = short (65535.0 * col->green);
  xcolor.blue = short (65535.0 * col->blue);
#ifndef MIXED
  if (monochrome)
#endif
    {
      if (!(XAllocColor(display, xcolormap, &xcolor))) {
	cerr << "Warning: cannot allocate colormap entry.\n";
	col->pixel = (0.30 * col->red + 0.59 * col->green + 0.11 * col->blue)
	  < 0.5 ? ipe_pixel[0] : ipe_pixel[1];
      } else {
	col->pixel = xcolor.pixel;
      }
    }
#ifndef MIXED
  else {
    xcolor.flags = (DoRed | DoGreen | DoBlue);
    if (next_pixel < num_colors) {
      col->pixel = ipe_pixel[next_pixel++];
      xcolor.pixel = col->pixel;
      XStoreColor(display, xcolormap, &xcolor);
    } else {
      col->pixel = ipe_pixel[0];
      if (!message_shown)
	show_message("I have run out of my supply of color cells",
		     "From now on, colors will start looking wrong");
      message_shown = TRUE;
    }
  }
#endif
  col->pixeled = TRUE;
}

IpeColor *IColor::find_color(float r, float g, float b)
{
  if (r < 0)
    return NIL;
  for (IpeColor *p = first_color; p; p = p->next) {
    if (r == p->red && g == p->green && b == p->blue)
      return p;
  }
  // color not found, create it
  p = new IpeColor;
  p->red = r;
  p->green = g;
  p->blue = b;
  p->pixeled = FALSE;
  p->next = first_color;
  first_color = p;
  return p;
}

Boolean IColor::parse(char *color_name)
{
  XColor xcolor;
  if (!XParseColor(display, xcolormap, color_name, &xcolor)) {
    col = black_color;
    return FALSE;
  } else {
    col = find_color(float(xcolor.red)   / 65535.0,
		     float(xcolor.green) / 65535.0,
		     float(xcolor.blue)  / 65535.0);
    return TRUE;
  }
}

//
//----------------------------------------------------------------------
//
// CREATE color buttons
//

static unsigned long button_backgnd[16], button_foregnd[16];

advertise void fill_color_field(Widget field)
{
  Widget colorLabel =
    XtVaCreateManagedWidget("colorLabel",
			    xmLabelWidgetClass, field,
			    NIL);
  
  Widget fourbyfour =
    XtVaCreateManagedWidget("colorGrid",
			    xmRowColumnWidgetClass, field,
			    XmNnumColumns, 4,
			    XmNpacking, XmPACK_COLUMN,
			    NIL);

  parsed_xlations = XtParseTranslationTable(xLations);

  //
  // make color buttons with correct background color
  //
  for (int j = 0; j < 16; j++) {
    colorButton[j] =
      XtVaCreateManagedWidget("colorButton",
			      xmPushButtonWidgetClass, fourbyfour,
			      XmNtranslations, parsed_xlations,
			      XmNbackground, button_backgnd[j],
			      XmNforeground, button_foregnd[j],
			      NIL);
    XtAddCallback(colorButton[j], XmNactivateCallback, color_cb, XtPointer(j));
  }
  colorButton[16] =
    XtVaCreateManagedWidget("emptyColor",
			    xmPushButtonWidgetClass, field,
			    XmNtranslations, parsed_xlations,
			    NIL);
  XtAddCallback(colorButton[16], XmNactivateCallback, color_cb, XtPointer(16));
  
  XtVaCreateManagedWidget("emptyLabel",
			  xmLabelWidgetClass, field,
			  NIL);
  for (j = 0; j < 17; j++)
    set_color_ball(j);
}

//
//----------------------------------------------------------------------
//
// find best PseudoColor visual
//

#ifndef MIXED
static void best_visual(void)
{
  XVisualInfo *xv;
  XVisualInfo *bestv[6];
  XVisualInfo xvi;
  int xvn, i, j;
  
  // request a list of all supported visuals on screen
  xvi.screen = screen_num;
  if (!(xv = XGetVisualInfo(display, VisualScreenMask, &xvi, &xvn))) {
    cerr << "Fatal Error: Cannot get Visual Info!\n";
    exit(1);
  }

  // choose the visual with maximum depth in each class
  for (j = 0; j < 6; )
    bestv[j++] = NIL;
  
  for (i = 0; i < xvn; i++) {
    j = xv[i].c_class;
    
    // simply use visuals returned by GetVisualInfo and don't free xv
    if (!bestv[j] || xv[i].depth > bestv[j]->depth) {
      bestv[j] = xv + i;
    }
  }
  
  if (app.debug & DBG_GFX) {
    cerr << "No. of Visuals: " << xvn << "\n";
    for (j = 0; j < 6; j++)
      if (bestv[j])
	cerr << "Best " << j << ": Depth= " << bestv[j]->depth << "\n";
  }
  
  // determine visual: use best PseudoColor visual
  
  // if less than 6 bits, give up and use b&w
  
  if (bestv[PseudoColor]->depth < 6) {
    monochrome = TRUE;
    return;
  }
  
  visual = bestv[PseudoColor]->visual;
}    
#endif

//
//----------------------------------------------------------------------
//
// SETUP and ALLOCATE colors for two overlay planes (in X mode)
//

advertise void get_ipe_colormap(Arg &arg, int &ac)
{
#ifndef MIXED  
  // determine best PseudoColor Visual
  best_visual();
  xcolormap = DefaultColormap(display, screen_num);

  if (monochrome || !app.private_colors)
    return;

  xcolormap = XCreateColormap(display, RootWindow(display, screen_num),
			      visual, AllocNone);
  XtSetArg(arg, XmNcolormap, xcolormap); ac++;
#else
  xcolormap = DefaultColormap(display, screen_num);
#endif
}

advertise void initialize_ipe_colors(void)
{
  //
  // try to allocate colors for a colormap that permits
  //  two overlay planes
  //
#ifdef MIXED
  ipe_pixel[0] = BlackPixel(display, screen_num);
  ipe_pixel[1] = WhitePixel(display, screen_num);
#else
  int status = 0;
  num_colors = MAX_COLORS;

  if (!monochrome) {
    do {
      DEBUG(DBG_GFX, "Trying to allocate ", num_colors << " colors... ");
      status = XAllocColorCells(display, xcolormap, FALSE,
				overlay_plane, 2, ipe_pixel, num_colors);
      if (!status)
	num_colors = int(num_colors * COLOR_REDUCE_RATE);
    } while (!status && num_colors >= 2);

    if (!status) {
      // we have failed to allocate two overlay planes
      // for the time begin, let's give up --- b/w is implemented later
      cerr << "I could not allocate enough colors---"
	   << "running in black and white.\n";
      monochrome = TRUE;
    }
  }

  if (monochrome) {
    // only allocate Black and White
    ipe_pixel[0] = BlackPixel(display, screen_num);
    ipe_pixel[1] = WhitePixel(display, screen_num);
    num_colors = 2;
  } else {
    //
    // Wow, we did it!  We have allocated num_colors colors!
    //
    // First allocate black and white
    //
    XColor xcolor;
    xcolor.flags = (DoRed | DoGreen | DoBlue);
    xcolor.pixel = ipe_pixel[0];		// black
    xcolor.red = xcolor.green = xcolor.blue = 0;
    XStoreColor(display, xcolormap, &xcolor);
    xcolor.pixel = ipe_pixel[1];		// white
    xcolor.red = xcolor.green = xcolor.blue = 0xffff;
    XStoreColor(display, xcolormap, &xcolor);
  }
#endif
  
  //
  // now setup internal Ipe color system
  //
  black_color = new IpeColor;
  black_color->red = black_color->green = black_color->blue = 0.0;
  black_color->pixeled = TRUE;
  black_color->pixel   = ipe_pixel[0];
  black_color->next    = NIL;

  white_color = new IpeColor;
  white_color->red = white_color->green = white_color->blue = 1.0;
  white_color->pixeled = TRUE;
  white_color->pixel   = ipe_pixel[1];
  white_color->next    = black_color;

  first_color = white_color;
  next_pixel = 2;

  //
  // initialize color panel colors
  //
  
  for (int j = 0; j < 15; j++) {
    color_choice[j].col = black_color;
  }
  color_choice[15].col = white_color;
  color_choice[16].col = NIL;

  for (j = 0; j < COL_LAST; j++) {
    sys_color[j].col = black_color;
  }

  //
  // Now allocate the colors for the buttons
  //
  button_foregnd[15] = button_backgnd[0]  = ipe_pixel[0]; // Black
  button_foregnd[0]  = button_backgnd[15] = ipe_pixel[1]; // White
  
  for (j = 1; j < 15; j++) {
    button_backgnd[j] = ipe_pixel[0];
    button_foregnd[j] = ipe_pixel[1];
    if (!color_choice[j].parse(app.color[j])) {
      cerr << "Error: could not parse color specification"
	   << " for Color #" << j << " : " << app.color[j] << "\n";
    } else {
      button_backgnd[j] = color_choice[j].pixel();
      button_foregnd[j] = (0.30 * color_choice[j].col->red +
			   0.59 * color_choice[j].col->green +
			   0.11 * color_choice[j].col->blue) < 0.5 ?
			   ipe_pixel[1] : ipe_pixel[0];
    }
  }
  
  for (j = 0; j < COL_LAST; j++) {
    if (!sys_color[j].parse(app.sys_color[j])) {
      cerr << "Error: could not parse color specification: "
	   << app.sys_color[j] << "\n";
    }
  }
}

//
// Allocate colors for penguins image
//

advertise void allocate_colors(int num, char *names[], unsigned long pixel[])
{
  Boolean message_was_shown = message_shown;
  int was_next_pixel = next_pixel;
  message_shown = TRUE;
  if (num_colors - num > next_pixel)
    next_pixel = num_colors - num;
  
  IColor color;
  
  for (int i = 0; i < num; i++) {
    // allocate ith color
    color.parse(names[2*i+1]);
    pixel[i] = color.pixel();
  }
  next_pixel = was_next_pixel;
  message_shown = message_was_shown;
}
