/*
 * configuration.C
 * 
 * Ipe's configuration window
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Scale.h>
#include <Xm/RowColumn.h>
#include <Xm/TextF.h>
#include <Xm/Text.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>

static Widget configurationWindow;

static Widget arrowSize, markSize;

static Widget printCommand, previewCommand, latexPreamble, psPreamble;

advertise IpeButton enable_undo_bt, enable_latex_box_bt, enable_fifi_bt;
advertise IpeButton enable_interior_bt, enable_zoom_pan_bt, enable_stroke_bt;
advertise IpeButton enable_3_spline_bt;

//
//----------------------------------------------------------------------
//
// SET and READOUT values
//

advertise void set_ps_preamble(char *s)
{
  XmTextSetString(psPreamble, s);
}

advertise char *get_preamble(int which DefaultsTo(0))
{
  static char buf[MAX_LINE_LENGTH];
  char *p = XmTextGetString((which ? psPreamble : latexPreamble));
  strcpy(buf, p);
  XtFree(p);
  return buf;
}

advertise char *get_command(int which)
{
  static char buf[MAX_LINE_LENGTH];
  char *p = XmTextFieldGetString((which ? printCommand : previewCommand));
  strcpy(buf, p);
  XtFree(p);
  return buf;
}

advertise void show_m_ipe_configuration(void)
// set configuration screen according to mipe/ipe mode
{
  XmTextFieldSetString(previewCommand, app.preview_command[mipe_mode]);
  XmTextSetString(latexPreamble,
		  (new_latexpreamble.size() ?
		   &new_latexpreamble[0] : app.latex_preamble[mipe_mode]));

  XtVaSetValues(markSize, XmNvalue, int(mark_size[mipe_mode] * 10.0), NIL);
  XtVaSetValues(arrowSize, XmNvalue, int(arrow_size[mipe_mode] * 10.0), NIL);
}

//
//----------------------------------------------------------------------
//
// SLIDER callbacks
//

static void line_set_arsize(Iobject *ob)
{
  ((Line *) ob)->setarsize(arrow_size[mipe_mode]);
}

static void spline_set_arsize(Iobject *ob)
{
  ((Spline *) ob)->setarsize(arrow_size[mipe_mode]);
}

static void arc_set_arsize(Iobject *ob)
{
  ((Arc *) ob)->setarsize(arrow_size[mipe_mode]);
}

static void mark_set_size(Iobject *ob)
{
  ((Mark *) ob)->setsize(mark_size[mipe_mode]);
}

static void arrow_size_cb(Widget, XtPointer, XtPointer)
// set arrow size
{
  int val;
  XmScaleGetValue(arrowSize, &val);
  
  arrow_size[mipe_mode] = s_coord(val) / 10.0;
    
  if (drawing_mode == IDLE && selection()) {
    pic_traverse(LINE, line_set_arsize);
    pic_traverse(SPLINE, spline_set_arsize);
    pic_traverse(ARC, arc_set_arsize);
    undo_changed();
    redraw_canvas();
  }
}

static void mark_size_cb(Widget, XtPointer, XtPointer)
// set marker size
{
  int val;
  XmScaleGetValue(markSize, &val);
  
  mark_size[mipe_mode] = s_coord(val) / 10.0;

  if (drawing_mode == IDLE && selection()) {
    pic_traverse(MARK, mark_set_size);
    undo_changed();
    redraw_canvas();
  }
}

static void distance_cb(Widget w, XtPointer client_data, XtPointer)
// set distance
{
  s_coord *distance = (s_coord *) client_data;
  int val;
  
  XmScaleGetValue(w, &val);
  *distance = val;
}

//
//----------------------------------------------------------------------
//
// POPUP and POPDOWN window
//

advertise void show_configuration_window(void)
{
  XtManageChild(configurationWindow);
}

static void hide_cb(Widget, XtPointer, XtPointer)
{
  XtUnmanageChild(configurationWindow);
}


//
//----------------------------------------------------------------------
//
// CREATE window
//

advertise void create_configuration_window(void)
{
  configurationWindow =
    XmCreateFormDialog(topLevel, "configurationWindow", NIL, 0);
  
  Widget buttonFrame =
    XtVaCreateManagedWidget("frame",
			    xmFrameWidgetClass, configurationWindow,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_NONE,
			    XmNrightAttachment, XmATTACH_FORM,
			    NIL);

  Widget distanceFrame = 
    XtVaCreateManagedWidget("frame",
			    xmFrameWidgetClass, configurationWindow,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_NONE,
			    NIL);
  
  Widget sliderFrame = 
    XtVaCreateManagedWidget("frame",
			    xmFrameWidgetClass, configurationWindow,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, distanceFrame,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_NONE,
			    NIL);
  
  Widget inputFrame =
    XtVaCreateManagedWidget("frame",
			    xmFrameWidgetClass, configurationWindow,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_WIDGET,
			    XmNrightWidget, buttonFrame,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, sliderFrame,
			    NIL);

  Widget hideButton =
    XtVaCreateManagedWidget("hideButton",
			    xmPushButtonWidgetClass, configurationWindow,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, buttonFrame,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, inputFrame,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, sliderFrame,
			    NIL);
  XtAddCallback(hideButton, XmNactivateCallback, hide_cb, NIL);

  Widget buttonField =
    XtVaCreateManagedWidget("buttonField",
			    xmRowColumnWidgetClass, buttonFrame,
			    XmNnumColumns, 1,
			    XmNpacking, XmPACK_COLUMN,
			    NIL);

  XtVaCreateManagedWidget("buttonFieldLabel",
			  xmLabelWidgetClass, buttonField,
			  NIL);

  enable_undo_bt.realize(buttonField, "enableButton",
			 "Undo", app.enable_undo);
  enable_latex_box_bt.realize(buttonField, "enableButton",
			      "Latex Box", app.enable_latex_box);
  enable_fifi_bt.realize(buttonField, "enableButton",
			 "Fifi", app.enable_fifi);
  enable_interior_bt.realize(buttonField, "enableButton",
			     "Interior Sel.", app.enable_interior);
  enable_zoom_pan_bt.realize(buttonField, "enableButton",
			     "Zoom & Pan", app.enable_zoom_pan);
  enable_stroke_bt.realize(buttonField, "enableButton",
			   "Stroke", app.enable_stroke);
  enable_3_spline_bt.realize(buttonField, "enableButton",
			     "3 Spline", app.enable_3_spline);

  Widget sliderField =
    XtVaCreateManagedWidget("sliderField",
			    xmFormWidgetClass, sliderFrame,
			    NIL);

  markSize =
    XtVaCreateManagedWidget("markSize",
			    xmScaleWidgetClass, sliderField,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_POSITION,
			    XmNdecimalPoints, 1,
			    XmNvalue, int(10.0 * mark_size[mipe_mode] + 0.5),
			    NIL);

  XtAddCallback(markSize, XmNvalueChangedCallback,
		mark_size_cb, XtPointer(0));

  arrowSize =
    XtVaCreateManagedWidget("arrowSize",
			    xmScaleWidgetClass, sliderField,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_WIDGET,
			    XmNrightWidget, markSize,
			    XmNdecimalPoints, 1,
			    XmNvalue, int(10.0 * arrow_size[mipe_mode] + 0.5),
			    NIL);

  XtAddCallback(arrowSize, XmNvalueChangedCallback,
		arrow_size_cb, XtPointer(0));

  Widget distanceField =
    XtVaCreateManagedWidget("sliderField",
			    xmFormWidgetClass, distanceFrame,
			    NIL);

  Widget snapDistance =
    XtVaCreateManagedWidget("snapDistance",
			    xmScaleWidgetClass, distanceField,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_POSITION,
			    XmNdecimalPoints, 0,
			    XmNvalue, int(snap_distance),
			    NIL);

  XtAddCallback(snapDistance, XmNvalueChangedCallback,
		distance_cb, XtPointer(&snap_distance));

  Widget selectDistance =
    XtVaCreateManagedWidget("selectDistance",
			    xmScaleWidgetClass, distanceField,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_WIDGET,
			    XmNrightWidget, snapDistance,
			    XmNdecimalPoints, 0,
			    XmNvalue, int(select_distance),
			    NIL);

  XtAddCallback(selectDistance, XmNvalueChangedCallback,
		distance_cb, XtPointer(&select_distance));

  Widget inputField =
    XtVaCreateManagedWidget("inputField",
			    xmRowColumnWidgetClass, inputFrame,
			    XmNpacking, XmPACK_TIGHT,
			    NIL);

  XtVaCreateManagedWidget("printCommandLabel",
			  xmLabelWidgetClass, inputField, NIL);
  
  printCommand =
    XtVaCreateManagedWidget("commandField",
			    xmTextFieldWidgetClass, inputField,
			    NIL);
  XmTextFieldSetString(printCommand, app.print_command);

  XtVaCreateManagedWidget("previewCommandLabel",
			  xmLabelWidgetClass, inputField, NIL);
  
  previewCommand =
    XtVaCreateManagedWidget("commandField",
			    xmTextFieldWidgetClass, inputField,
			    NIL);

  XtVaCreateManagedWidget("latexPreambleLabel",
			  xmLabelWidgetClass, inputField, NIL);
  
  latexPreamble =
    XtVaCreateManagedWidget("latexPreamble",
			    xmTextWidgetClass, inputField,
			    NIL);

  XtVaCreateManagedWidget("psPreambleLabel",
			  xmLabelWidgetClass, inputField, NIL);
  
  psPreamble =
    XtVaCreateManagedWidget("psPreamble",
			    xmTextWidgetClass, inputField,
			    NIL);
  XmTextSetString(psPreamble, app.ps_preamble);

}

