/*
 * file.C
 * 
 * postscript file i/o for ipe 
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/errno.h>

extern void save_ps_prologue(ostream& fh);

static ostream  *text_save_fh;
static void text_save_latex(Iobject *ob)
{
  ((Text *)ob)->save_latex(*text_save_fh, save_offset);
  text_objects_saved = TRUE;
}

static void ps_save_preamble(ostream& fh, Boolean which,
			     char *name, Boolean prefix)
// saves a preamble, if it is not empty
{
  char *psp = get_preamble(which);
  for (char *p = psp; *p && (*p == ' ' || *p == '\t' || *p == '\n'); p++)
    ;
  if (*p) {
    for (char *q = p + strlen(p) - 1;
	 *q == ' ' || *q == '\t' || *q == '\n'; q--)
      ;
    q[1] = '\0';
    int nlines = 1;
    for (q = p; *q; q++)
      if (*q == '\n')
	nlines++;
    fh << "% " << name << " " << nlines << "\n";
    if (prefix) fh << "%%";
    for (q = p; *q; q++) {
      fh << *q;
      if (*q == '\n' && prefix) fh << "%%";
    }
    fh << "\n";
  }
}

static void save_ipe_format(ostream& fh, int pagenum, Boolean save_template)
{
  pl_boundingbox bb;

  bb.make_empty();
  if (save_template)
    pages[0]->bbox(bb);
  pages[pagenum]->bbox(bb);
  s_coord width = bb.xmax() - bb.xmin();
  s_coord height = bb.ymax() - bb.ymin();
  save_offset = pl_vec(300, 400) - 0.5 * (pl_vec(bb.xmin(), bb.ymin()) +
					  pl_vec(bb.xmax(), bb.ymax()));

  int bbllx = int(bb.xmin() + save_offset.x() - 1.0);
  int bblly = int(bb.ymin() + save_offset.y() - 1.0);
  int bburx = int(bb.xmax() + save_offset.x() + 1.0);
  int bbury = int(bb.ymax() + save_offset.y() + 1.0);
  // EPS pseudo prologue
  fh << "%!PS-Adobe-2.0 EPSF-1.2\n%%Creator: Ipe " << ipe_version << "\n";
  fh << "%%BoundingBox: "
     << bbllx << " " << bblly << " " << bburx << " " << bbury << "\n";
  fh << "%%EndComments\n";
  // LaTeX prologue
  fh << "{\\catcode37=9\\def\\IPEdummy{({{)}} pop\n"
     << "%%}\\makeatletter\\let\\@notdefinable\\relax\n"
     << "%%\\def\\IPEc#1[#2]#3{\\newcommand{#1}[#2]{#3}\\ignorespaces}\\@ifundefined\n"
     << "%%{selectfont}{\\let\\selectfont\\relax\\def\\fontsize#1#2{}}{}\\makeatother\n"
     << "%%\\IPEc\\IPEput[4]{\\put(0,0){\\special{psfile=\\IPEfile}}}\n"
     << "%%\\IPEc\\IPEmp[2]{\\minipage[t]{#1bp}#2\\special{color pop}\\endminipage}\n"
     << "%%\\IPEc\\IPEtext[1]{\\makebox(0,0)[lb]{#1\\special{color pop}}}\n"
     << "%%\\IPEc\\IPEfs[1]{\\IPEcolfs{0 0 0}{#1}}\n"
     << "%%\\IPEc\\IPEcolfs[2]{\\dimen0=#2pt\\fontsize{#2}{1.2\\dimen0}\\selectfont\n"
     << "%%\\special{color push rgb #1}}\n"
     << "%%\\IPEc\\IPEsize[2]{\\unitlength1bp\\ignorespaces}\n";

  // now we get to the actual figure
  fh << "%%\\IPEsize{" << width << "}{" << height << "}\n";
  if (!bb.is_empty()) {
    fh << "%%\\begin{picture}(" << width << "," << height << ")("
       << (bb.xmin() + save_offset.x()) << ","
       << (bb.ymin() + save_offset.y()) << ")\n";
    fh << "%%\\IPEput{"
       << bbllx << "}{" << bblly << "}{" << bburx << "}{" << bbury << "}\n";

    text_save_fh = &fh;
    if (save_template) {
      pages[0]->traverse(TEXT, text_save_latex);
    }
    pages[pagenum]->traverse(TEXT, text_save_latex);
    fh << "%%\\end{picture}\\endinput}\n";

    // save Postscript prologue
    save_ps_prologue(fh);

    fh << "\nIpeDict begin "
       << save_offset.x() << " " << save_offset.y() << " translate\n\n";
    ps_save_preamble(fh, FALSE, "Preamble", TRUE);
    ps_save_preamble(fh, TRUE,  "PSpreamble", FALSE);
    if (save_template)
      pages[0]->save_postscript(fh);
    pages[pagenum]->save_postscript(fh);
    fh << "end %% of Ipe figure\n";
  }
}

static void save_mipe_format(ostream& fh)
{
  fh << "%\\MIPE\n%%Creator: Ipe " << ipe_version << "\n";
  ps_save_preamble(fh, FALSE, "Preamble", TRUE);
  ps_save_preamble(fh, TRUE,  "PSpreamble", FALSE);
  fh << "% Pages " << pages.size() - 1 << "\n";

  // the first object saved is the template

  for (int i = 0; i < pages.size(); i++) {
    pages[i]->save_properties(fh);
  }
  fh << "\n% End\n";
}

static char force_filename[MAX_FILENAME_LENGTH];
static IpeCallback do_after_saving = NIL;

static void overwrite_cb(void)
{
  ps_save_picture(force_filename, 0, TRUE, do_after_saving);
}

static void save_it_cb(char *fname)
{
  ps_save_picture(fname, 0, FALSE, do_after_saving);
}

advertise Boolean ps_save_picture(IpeCallback callback)
{
  return ps_save_picture(filename, 0, TRUE, callback);
}

advertise Boolean ps_save_picture(char *fn DefaultsTo(filename),
				  int pagenum DefaultsTo(0),
				  Boolean force DefaultsTo(TRUE),
				  IpeCallback callback DefaultsTo(NIL))
//     
// save a page or full current picture in a postscript file
// if pagenum == 0: save all pages in ipe/mipe format, depending on mipe_mode
// if pagenum > 0: save only that page including template in ipe format
// if fn is NIL or "", ask for filename
// reconfirm saving to file different from filename, if it exists
// return TRUE if succeeded
//
{
  do_after_saving = callback;
  struct stat statbuf;
  if (!fn || !fn[0]) {
    show_file_selector("File to save picture in", save_it_cb);
    return FALSE;
  }
  if (!force && !stat(fn, &statbuf)) {
    show_choice("File already exists!",
		"Overwrite",  overwrite_cb,
		"Cancel",     NIL,
		NIL, NIL);
    strcpy(force_filename, fn);
    return FALSE;
  }

  if (!force || !strcmp(fn, filename))
    put_msg("writing file ... ");
  
  // make a backup of old file, if necessary
  if (app.make_backups && !lstat(fn, &statbuf)) {
    char buf[512];
    strcpy(buf, fn);
    strcat(buf, "~");
    rename(fn, buf);
  }

  strcpy(newfilename, fn);

  ipestream fh;
  if (!fh.open(fn, "w")) {
    show_message("Cannot open file for writing", fn);
    return FALSE;
  }
  text_objects_saved = FALSE;

  if (!pagenum && mipe_mode)
    save_mipe_format(fh);
  else if (mipe_mode)
    save_ipe_format(fh, pagenum, TRUE);
  else
    save_ipe_format(fh, 1, FALSE);
  
  fh.close();
  if (!fh.good()) {
    show_message("ERROR during writing", "");
    return FALSE;
  }
  if (do_after_saving) {
    (*do_after_saving)();
  }
  return TRUE;
}

// ==================================================================

static char linebuf[MAX_LINE_LENGTH];

advertise char *ps_read_next(istream& fh)
{
  // read words until we find a sole "%"
  do {
    fh >> linebuf;
  } while (strcmp(linebuf, "%") != 0 && fh.good());
  if (!fh.good())
    // read error (could be EOF, then it\'s a format error)
    return NIL;
  // so next word is the keyword
  fh >> linebuf;
//  cerr << "Keyword " << linebuf << "\n";
  return linebuf;
}

advertise Boolean ps_read_entry(istream& fh, int num, char **fields)
{
  int i, len;
  s_coord mx, my;
  char c;
  

  // mark all keywords as undefined
  for (i = 0; i < num ; i++)
    ps_in_defd[i] = FALSE;
  ps_in_color[0][0] = ps_in_color[1][0] = -1.0;
  ps_in_string.newsize(0);
  
  while (TRUE) {
    if (!ps_read_next(fh))
      // read error
      return FALSE;
    // linebuf contains keyword
    if (!strcmp("End", linebuf)) {
      ps_in_string.append('\0');
      for (i = 0; i < num; i++)
	if (fields[i][0] == '!' && !ps_in_defd[i])
	  // undefined field
	  return FALSE;
      return TRUE;
    }
    if (!strcmp("sk", linebuf) || !strcmp("fi", linebuf) ||
	!strcmp("skc", linebuf) || !strcmp("fic", linebuf)) {
      // keyword for colors
      i = (strncmp(linebuf, "fi", 2) == 0);
      if (strlen(linebuf) == 3) {
	// rgb mode
	fh >> ps_in_color[i][0] >> ps_in_color[i][1] >> ps_in_color[i][2];
      } else {
	fh >> linebuf;
	if (!strcmp(linebuf, "x"))
	  ps_in_color[i][0] = -1.0;
	else
	  ps_in_color[i][0] = atof(linebuf);
	ps_in_color[i][1] = ps_in_color[i][2] = ps_in_color[i][0];
      }
    } else {
      // now it must be one of the user\'s keywords
      for (i = 0; i < num && strcmp(fields[i]+2, linebuf) != 0; i++)
	;
      if (i == num) {
	// no? then it\'s an illegal keyword! complain!!
	return FALSE;
      }
      // we have found the field
      ps_in_defd[i] = TRUE;
      switch (fields[i][1]) {
      case '0':
	// we already know everything
	break;
      case '1':
	// read one parameter
	fh >> ps_in_value[i];
	break;
      case '2':
	// read two parameters
	fh >> ps_in_value[i++];
	fh >> ps_in_value[i];
	break;
      case '3':
	// read three parameters
	fh >> ps_in_value[i++];
	fh >> ps_in_value[i++];
	fh >> ps_in_value[i];
	break;
      case '*':
	// read string
	fh.getline(linebuf, MAX_LINE_LENGTH);
	if (ps_in_string.size() > 0)
	  ps_in_string.append('\n');
	len = strlen(linebuf);
	for (i = 1; i < len; i++) 
	  ps_in_string.append(linebuf[i]);
	break;
      case '#':
	// we have to read a list of vertices
	fh >> i;
	ps_in_array.newsize(i);
	for (i = 0; i < ps_in_array.size(); i++) {
	  do {
	    fh.get(c);
	  } while (c != '\n' && fh.good());
	  if (!fh.good())
	    return FALSE;
	  fh >> mx >> my;
	  ps_in_array[i] = pl_vec(mx, my);
	}
	break;
      case '@':
	// we have to read a list of vertices with link information
	fh >> i;
	ps_in_array.newsize(i);
	ps_in_string.newsize(i);
	for (i = 0; i < ps_in_array.size(); i++) {
	  do {
	    fh.get(c);
	  } while (c != '\n' && fh.good());
	  if (!fh.good())
	    return FALSE;
	  fh >> mx >> my;
	  ps_in_array[i] = pl_vec(mx, my);
	  do {
	    fh.get(c);
	  } while (c == ' ' && fh.good());
	  if (!fh.good())
	    return FALSE;
	  ps_in_string[i] = c;
	}
	break;
      case '$':
	// read a bitmap
	ps_read_bitmap(fh);
	break;
      case '[':
	// read transform
	{
	  fh >> ps_in_transform.A[0] >> ps_in_transform.A[1]
	     >> ps_in_transform.A[2] >> ps_in_transform.A[3];
	  ps_in_transform.A[4] = ps_in_transform.A[5] = 0.0;
	}
	break;
      }
    }
  }
}

static void ps_read_preamble(istream& fh, char_rr& to)
// read preamble into to;
{
  int nlines;
  fh >> nlines;
  char ch;
  do {
    fh.get(ch);
  } while (fh.good () && ch != '\n');
  nlines++;
  while (fh.good() && nlines) {
    if (ch == '\n') {
      nlines--;
      if (nlines) {
	do
	  fh.get(ch);
	while (fh.good() && (ch == '%'));
      }
    } else {
      to.append(ch);
      fh.get(ch);
    }
  }
  to.append('\0');
}

advertise int ps_read_file(char *fn, Group* &gr)
// returns -1 if unsuccessful, 0 for IPE picture, number of pages for MIPE
{
  if (!fn || !fn[0]) {
    cerr << "Error: No filename in ps_read_file\n";
    return -1;
  }
  strcpy(newfilename, fn);
  new_pspreamble.newsize(0);
  new_latexpreamble.newsize(0);
  
  struct stat statbuf;
  if (lstat(fn, &statbuf) && errno == ENOENT) {
    // file does not exist --- create new figure
    put_msg("new file");
    int i = strlen(newfilename);
    if (i > 5 && !strcmp(newfilename + i - 5, ".mipe")) {
      // return an empty mipe file
      gr = new Group;
      // empty template
      gr->add_object(new Group);
      // empty first page
      gr->add_object(new Group);
      // return one page
      return 1;
    } else {
      // return empty figure
      gr = new Group;
      return 0;
    }
  }    

  ipestream fh;
  
  if (!fh.open(newfilename, "r")) {
    show_message("cannot open file for reading", newfilename);
    return -1;
  }

  fh >> linebuf;
  Boolean is_mipe = !strncmp(linebuf, "%\\MIPE", 6);

  int npages = 0;
  char *kw;
  Boolean in_preamble = TRUE;

  while (in_preamble && (kw = ps_read_next(fh)) != NIL) {
    if (!is_mipe && !strcmp(kw, "Group")) {
      // read an IPE file
      in_preamble = FALSE;
    } else if (is_mipe && !strcmp(kw, "Pages")) {
      // read a MIPE file
      fh >> npages;
      in_preamble = FALSE;
    } else if (!strcmp(kw, "PSpreamble")) {
      // read PS preamble
      ps_read_preamble(fh, new_pspreamble);
    } else if (!strcmp(kw, "Preamble")) {
      // read Latex preamble
      ps_read_preamble(fh, new_latexpreamble);
    } else {
      kw = NIL;
      in_preamble = FALSE;
    }
  }

  if (!kw) {
    show_message("this is not an IPE file", newfilename);
    fh.close();
    return -1;
  }

  // FORMS
  if (!new_latexpreamble.size()) {
    new_latexpreamble.newsize(strlen(app.latex_preamble[npages ? 1 : 0]) + 1);
    strcpy(&new_latexpreamble[0], app.latex_preamble[npages ? 1 : 0]);
  }

  ps_okay = TRUE;
  use_new_preamble = TRUE;
  gr = new Group(fh, is_mipe);
  use_new_preamble = FALSE;
  fh.close();
  if (!ps_okay) {
    show_message("could not read IPE file", newfilename);
    delete gr;
    return -1;
  }
  return npages;
}
