/*
 * fonts.c
 * 
 * managing the fonts used on the canvas (everything else is standard X)
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"
#ifndef MIXED
#include <X11/Xatom.h>
#endif

#ifdef MIXED

//////////////////////////// GL USE FONT //////////////////////////////

static Font_rr text_font[8];

static fmfonthandle use_font(short f, s_coord size,
			     s_coord& height, s_coord &depth)
// selects font in the required size, returns handle
{
  int i;
  fmfontinfo info;

  if (f < 0 || 7 < f)
    f = 0;
  size *= ((f >= EDIT_FONT_OFFS) ? app.edit_font_scale :
	   app.font_scale);
  if (!text_font[f].size()) {
    text_font[f].newsize(1);
    text_font[f][0].fh = fmfindfont(app.font_name_gl[f]);
    // Check whether existing
    if (!text_font[f][0].fh) {
      show_message("OOPS: couldn't find font", app.font_name_gl[f]);
      return text_font[0][1].fh;
    }
  }
  // Search for the required size.
  for (i=1; i< text_font[f].size() && size != text_font[f][i].size; i++)
    ;
  // create font, if not found
  if (i == text_font[f].size()) {
    text_font[f].resize(text_font[f].size() + 1);
    text_font[f][i].size = size;
    text_font[f][i].fh = fmscalefont(text_font[f][0].fh, size);
  }
  fmfonthandle fh = text_font[f][i].fh;
  fmgetfontinfo(fh, &info);
  height = info.ysize;
  depth = info.yorig;
  return fh;
}

advertise void draw_string(const pl_vec& p, // drawing position
			   short font,	    // Font style
			   s_coord size,	    // Font size
			   int curspos,	    // cursor position (-1 = none)
			   char_rr str,		
			   Boolean parbox)
{
  fmfonthandle fnt;
  s_coord height, depth;

  // Check whether anything has to be done
  if (curspos != 0 && (!str.size() || str[0] == '\0')) return;

  // Set the font to be used and get some info
  fnt = use_font(font, size * zoom_factor, height, depth);
  fmsetfont(fnt);
  
  pl_vec cp(p);
  if (parbox)
    cp += pl_vec(0, -height/(zoom_factor*screen_factor.y()));

  char c;
  char *sol = &str[0];
  char *cur = &str[curspos];
  char *eol;
  
  do {
    // find end of line
    for (eol = sol; *eol && *eol != '\n'; eol++)
      ;
    // Draw string
    c = *eol; *eol = '\0';
    if (*sol) {
      cmov2(cp.x(),cp.y());
      fmprstr(sol);
    }
    *eol = c;
    // determine cursxy position
    if (sol <= cur && cur <= eol) {
      // cursor is on this line
      c = *cur;
      *cur = '\0';
      s_coord tt = fmgetstrwidth(fnt, sol);
      *cur = c;
      s_coord wd = height/(2.0 * zoom_factor * screen_factor.y());
      char buf[2];
      buf[0] = c;
      buf[1] = '\0';
      if (c && c != '\n')
	wd = fmgetstrwidth(fnt, buf) / (zoom_factor*screen_factor.x());
      pl_vec cursp = cp + pl_vec(tt/(zoom_factor*screen_factor.x()), 0);
      pl_vec cursxy = cursp +
	pl_vec(0, -depth/(zoom_factor*screen_factor.y()));
      rectf(cursxy.x(), cursxy.y(), cursxy.x() + wd,
	    cursxy.y() + height/(zoom_factor*screen_factor.y()));
      cmov2(cursp.x(), cursp.y());
      long col = getcolor();
      color(0);
      fmprstr(buf);
      color(short(col));
    }
    sol = eol + 1;
    cp += pl_vec(0, -height/(zoom_factor*screen_factor.y()));
  } while (*eol != '\0');
}

advertise void get_string_size(char_rr &str,
			       int startpos, int endpos,
			       short font, s_coord size,
			       s_coord &width, s_coord &height,
			       s_coord &depth)
// returns measurements of text in bp (independent of zoom_factor!)
{
  fmfonthandle fnt = use_font(font, size, height, depth);
  height /= screen_factor.y();
  depth /= screen_factor.y();
  char tmpch = str[endpos];
  str[endpos] = '\0';
  width = fmgetstrwidth(fnt, &str[startpos]) / screen_factor.x();
  str[endpos] = tmpch;
}

#else
////////////////////////////// X VERSION //////////////////////////////

static Font_rr text_font[8];

inline unsigned long square(unsigned long a) {return a*a;}

static char **font_names[8];
static int   font_count[8];
static XFontStruct *font_info[8];

static XFontStruct *use_font(short f, s_coord size,
			     s_coord& height, s_coord &depth)
// selects font in the required size, returns handle
{
  int i;

  if (f < 0 || 7 < f)
    f = 0;
  size *= ((f >= EDIT_FONT_OFFS) ? app.edit_font_scale :
	   app.font_scale);
  // we are looking for a font with this height in pixels:
  int fsize = int(size * screen_factor.y() + 0.5);
  if (!text_font[f].size()) {
    // we haven't used this font type before:
    //  first get font information
    font_names[f] =
      XListFontsWithInfo(display, app.font_name_x[f], MAX_X_FONTNAMES,
			 &font_count[f], &font_info[f]);
    DEBUG(DBG_GFX, "Looking up font ", app.font_name_x[f]);
    DEBUG(DBG_GFX, "   # of matches found is ", font_count[f]);
    
    // Check whether there is any font
    if (!font_count[f]) {
      show_message("OOPS: couldn't find any font with pattern",
		   app.font_name_x[f]);
      return XQueryFont(display,
			XGContextFromGC(DefaultGC(display, screen_num)));
    }
  }
  // Search for the desired size
  for (i = 0; i< text_font[f].size() && fsize != text_font[f][i].fsize; i++)
    ;
  // not yet allocated, search through possibilities
  if (i == text_font[f].size()) {
    XFontStruct *xfs = font_info[f];
    unsigned long xfsize = xfs->max_bounds.descent + xfs->max_bounds.ascent;
    unsigned long fsqdist = square(xfsize - fsize);
    int k = 1;  		// best match so far
    // now scan through fonts to find the right one
    for (int j = 1; j < font_count[f]; j++) {
      xfs++;
      xfsize = xfs->max_bounds.descent + xfs->max_bounds.ascent;
      if (square(xfsize - fsize) < fsqdist) {
	fsqdist = square(xfsize - fsize);
	k = j;
      }
    }
    // k is the index of the font we want, font_names[f][k] is its name
    DEBUG(DBG_GFX, "The best fitting font is ", font_names[f][k]);
    XFontStruct *yfs = XLoadQueryFont(display, font_names[f][k]);

    // put it in text_font structure
    text_font[f].resize(text_font[f].size() + 1);
    text_font[f][i].fsize = fsize;
    text_font[f][i].xfs   = yfs;
  }
  XFontStruct *xfs = text_font[f][i].xfs;
  // retrieve font height and depth
  depth = xfs->descent;
  height = depth + xfs->ascent;
  return xfs;
}

advertise void draw_string(const pl_vec& p, // drawing position
			   short font,	    // Font style
			   s_coord size,	    // Font size
			   int curspos,	    // cursor position (-1 = none)
			   char_rr str,		
			   Boolean parbox)
{
  XFontStruct *xfs;
  s_coord height, depth;

  // Check whether anything has to be done
  if (curspos != 0 && (!str.size() || str[0] == '\0')) return;

  // Set the font to be used and get some info
  xfs = use_font(font, size * zoom_factor, height, depth);
  XSetFont(display, current_gc, xfs->fid);
  
  pl_vec cp(p);
  tfstack[tfsp].apply(cp);
  int cpx = int(cp.x());
  int cpy = int(cp.y());

  if (parbox)
    cpy += int(height-depth);
  else {
    // Now this is tricky. In Latex, the cpy position is the lower
    // side of the bounding box...
    int font_dir, font_ascent, font_descent;
    XCharStruct xcs;
    XTextExtents(xfs, &str[0], str.size() - 1,
		 &font_dir, &font_ascent, &font_descent, &xcs);
    cpy -= xcs.descent;
  }

  char *sol = &str[0];
  char *cur = &str[curspos];
  char *eol;
  
  do {
    // find end of line
    for (eol = sol; *eol && *eol != '\n'; eol++)
      ;
    // Draw string
    if (*sol) {
      XDrawString(display, drawable, current_gc, cpx, cpy, sol, eol-sol);
    }
    // determine cursxy position
    if (sol <= cur && cur <= eol) {
      // cursor is on this line
      int tt = XTextWidth(xfs, sol, cur - sol);
      int wd = int(height/2.0);
      if (*cur && *cur != '\n')
	wd = XTextWidth(xfs, cur, 1);
      XSetFunction(display, current_gc, GXxor);
      XFillRectangle(display, drawable, current_gc,
		     (cpx + tt), (cpy - int(height - depth)), wd, int(height));
      XSetFunction(display, current_gc, GXcopy);
    }
    sol = eol + 1;
    cpy += int(height);
  } while (*eol != '\0');
}

advertise void get_string_size(char_rr &str,
			       int startpos, int endpos,
			       short font, s_coord size,
			       s_coord &width, s_coord &height,
			       s_coord &depth)
// returns measurements of text in bp (independent of zoom_factor!)
{
  XFontStruct *xfs = use_font(font, size, height, depth);
  height /= screen_factor.y();
  depth /= screen_factor.y();
  width = XTextWidth(xfs, &str[startpos], endpos - startpos)
    / screen_factor.x();
}

//////////////////////////// END VERSIONS /////////////////////////////
#endif

//
// get width of a substring on screen in bp
//

advertise s_coord get_substring_width(char_rr &str,
				    int startpos, int endpos,
				    short font, s_coord size)
{
  if (endpos <= startpos)
    return 0.0;
  s_coord width, height, depth;
  get_string_size(str, startpos, endpos, font, size * zoom_factor,
		  width, height, depth);
  return width / zoom_factor;
}

//
// return the position of the mouse in a string
//

advertise int mouse_position(char_rr &str, const pl_vec& offs,
			     short font, s_coord size)
// return position of mouse mp in string curstr drawn at pos
{
  s_coord height, depth;

  (void) use_font(font, size * zoom_factor, height, depth);
  height /= (screen_factor.y() * zoom_factor);
  depth /= (screen_factor.y() * zoom_factor);
  
  int line = int((height - depth - offs.y()) / height);
  for (int sol = 0; line > 0; line--) {
    while (str[sol] && str[sol] != '\n')
      sol++;
    if (str[sol])
      sol++;
  }
  int i = sol;
  while (str[i] && str[i] != '\n'
	 && get_substring_width(str, sol, i, font, size) < offs.x())
    i++;
  if (i > sol &&
      (get_substring_width(str, sol, i, font, size) +
       get_substring_width(str, sol, i-1, font, size)) / 2.0 > offs.x())
    i--;
  return i;
}

